/*  Screem:  screem-helper.c
 *
 *  Copyright (C) 2002 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>


#include <unistd.h>
#include <fcntl.h>

#include <ctype.h>

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <errno.h>

#include <glade/glade.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "screem-window.h"
#include "screem-helper.h"
#include "screem-page.h"

#include "fileops.h"
#include "support.h"

#include "eggdesktopentries.h"

#include "screemmarshal.h"

static void screem_helper_class_init( ScreemHelperClass *klass );
static void screem_helper_init( ScreemHelper *helper );
static void screem_helper_finalize( GObject *object );

static void screem_helper_set_property(GObject *object, guint prop_id,
				       const GValue *value, GParamSpec *pspec);
static void screem_helper_get_property( GObject *object, guint prop_id,
					GValue *value, GParamSpec *pspec );
static void screem_helper_verb_cb( GtkAction *action, gpointer user_data );

static void child_setup( gpointer data );
static void child_watch( GPid pid, gint status, gpointer data );
static gboolean child_stdout( GIOChannel *source,
		GIOCondition condition, gpointer data );
static gboolean child_stderr( GIOChannel *source,
		GIOCondition condition, gpointer data );

enum {
	PROP_BOGUS,
	PROP_PATHNAME,
	PROP_ICONPATH,
	PROP_HELPER_NAME,
	PROP_INPUT_MODE,
	PROP_OUTPUT_MODE,
	PROP_EXEC_ON_SAVE,
	PROP_EXEC_ON_SAVE_TYPE
};

enum {
	HELPER_FAILED,
	HELPER_ERROR,
	LAST_SIGNAL
};
static guint screem_helper_signals[LAST_SIGNAL] = { 0 };

struct ScreemHelperPrivate {
	/* a list of ScreemWindows */
	GSList *list;

	gchar *name;
	gchar *pathname;
	gchar *iconpath;

	HelperInputMode imode;
	HelperOutputMode omode;

	gboolean exec;
	gchar *exec_type;
	
	/* the following two variables are set in screem_helper_execute(), 
	   only valid in that context + the source cb */
	ScreemPage *page;

	gint std_out;
	gint std_err;
	gint watch;
	GPid pid;

	GIOChannel *out;
	GIOChannel *err;
	gint owatch;
	gint ewatch;

	GString *input;
	ScreemWindow *window;
	GtkWidget *dialog;
};


ScreemHelper* screem_helper_new( const gchar *name, 
				 const gchar *pathname,
				 const gchar *iconpath,
				 HelperInputMode imode,
				 HelperOutputMode omode )
{
	ScreemHelper *helper;
	GType type;
	
	type = screem_helper_get_type();

	helper = SCREEM_HELPER( g_object_new( type, 
					      "name", name,
					      "pathname", pathname,
					      "iconpath", iconpath,
					      "input", imode,
					      "output", omode,
					      NULL ) );

	return helper;
}

ScreemHelper *screem_helper_new_from_file( const gchar *file )
{
	ScreemHelper *helper;

	EggDesktopEntries *entries;
	GError *error;

	gchar *name;
	gchar *pathname;
	gchar *icon;
	HelperInputMode imode;
	HelperOutputMode omode;
	gboolean exec;
	gchar *type;

	helper = NULL;
	error = NULL;
	entries = eggy_desktop_entries_new_from_file( file,
			NULL,
			EGG_DESKTOP_ENTRIES_DISCARD_COMMENTS |
			EGG_DESKTOP_ENTRIES_DISCARD_TRANSLATIONS,
			&error );

	if( entries ) {
		name = eggy_desktop_entries_get_string( entries,
				"Desktop Entry",
				"Name",
				&error );
		error = NULL;
		pathname = eggy_desktop_entries_get_string( entries,
				"Desktop Entry",
				"Exec",
				&error );
		error = NULL;
		icon = eggy_desktop_entries_get_string( entries,
				"Desktop Entry",
				"Icon",
				&error );
		error = NULL;
		imode = eggy_desktop_entries_get_integer( entries,
				"Desktop Entry",
				"X-SCREEM-imode",
				&error );
		error = NULL;
		omode = eggy_desktop_entries_get_integer( entries,
				"Desktop Entry",
				"X-SCREEM-omode",
				&error );
		error = NULL;
		exec = eggy_desktop_entries_get_boolean( entries,
				"Desktop Entry",
				"X-SCREEM-exec",
				&error );
		error = NULL;
		type = eggy_desktop_entries_get_string( entries,
				"Desktop Entry",
				"X-SCREEM-exectype",
				&error );
		
		eggy_desktop_entries_free( entries );

		helper = screem_helper_new( name, pathname,
				icon, imode, omode );

		g_object_set( G_OBJECT( helper ),
				"exec_on_save", exec,
				"exec_type", type,
				NULL );
		g_free( name );
		g_free( pathname );
		g_free( icon );
		g_free( type );
	}

	return helper;
}

gboolean screem_helper_save( ScreemHelper *helper )
{
	gchar *pathname;
	gchar *tmp;

	gchar *name;
	gchar *path;
	gchar *icon;
	HelperInputMode imode;
	HelperOutputMode omode;

	gboolean exec;
	gchar *type;
	
	GString *output;
	gboolean ret;
	
	tmp = screem_get_dot_dir();
	pathname = g_build_path( G_DIR_SEPARATOR_S, tmp,
			"helpers", NULL );
	g_free( tmp );

	ret = FALSE;
	
	if( mkdir_recursive( pathname, 
			GNOME_VFS_PERM_USER_ALL, NULL, NULL ) ) { 
		g_object_get( G_OBJECT( helper ), 
				"name", &name, 
				"pathname", &path,
				"iconpath", &icon, 
				"input", &imode, 
				"output", &omode,
				"exec_on_save", &exec,
				"exec_type", &type,
				NULL );

		tmp = g_strconcat( pathname, G_DIR_SEPARATOR_S,
				name, ".desktop", NULL );
	
		output = g_string_new( "[Desktop Entry]\n" );
		g_string_append_printf( output, "Name=%s\n", name );
		g_string_append_printf( output, "Exec=%s\n", path );
		g_string_append( output, "Comment=Screem Helper\n" );
		g_string_append_printf( output, "Icon=%s\n", icon );
		g_string_append( output, "Type=Application\n" );
		g_string_append( output, "Terminal=0\n" );
		g_string_append_printf( output,
				"X-SCREEM-imode=%i\n", imode );
		g_string_append_printf( output,
				"X-SCREEM-omode=%i\n", omode );
		g_string_append_printf( output,
				"X-SCREEM-exec=%i\n", exec );
		g_string_append_printf( output,
				"X-SCREEM-exectype=%s\n", type );

		ret = save_file( tmp, output->str,
				GNOME_VFS_PERM_USER_ALL, 
				FALSE, NULL );
		g_string_free( output, TRUE );
		g_free( name );
		g_free( path );
		g_free( icon );
		g_free( type );
		
		g_free( tmp );
	} 	
	g_free( pathname );

	return ret;
}

void screem_helper_execute( ScreemHelper *helper, 
		ScreemPage *page, ScreemWindow *window )
{
	ScreemHelperPrivate *priv;
	const gchar *pagepath;
	gchar *dirname;

	gint argc;
	gchar **argv;
	
	GPid pid;
	gint std_in;
	FILE *out;
	
	GError *error;
	const gchar *charset;
	
	GladeXML *xml;
	gchar *gladepath;
	GtkWidget *widget;
	
	gchar *label;
	gchar *tmp;

	gint button;

	gchar *data;
	gint flags;

	
	const gchar *exec;
	GString *path;
	gunichar c;

	ScreemPage *wpage;
	
	priv = helper->private;

	screem_window_clear_messages( window );
	screem_window_clear_errors( window );

	wpage = NULL;
	if( window ) {
		wpage = screem_window_get_document( window );
	}
	
	if( wpage && ! page ) {
		 page = wpage;
	}
	
	if( ! page ) {
		return;
	}

	priv->page = page;
	
	/* get pathname, and strip file: from local methods */
	pagepath = screem_page_get_pathname( page );
	dirname = NULL;
	if( pagepath ) {
		/* need local path for working directory, for
		 * remote files we stay in the current dir */
		if( ! strncmp( "file://", pagepath,
					strlen( "file://" ) ) ) {
			pagepath += strlen( "file://" );
		
			dirname = g_path_get_dirname( pagepath );
		}
	}
	
	/* replace pathnames in the pattern to exec */
	path = g_string_new( "" );
	exec = priv->pathname;
	do {
		tmp = strchr( exec, '%' );
		if( tmp ) {
			g_string_append_len( path, exec,
					tmp - exec );

			tmp = g_utf8_next_char( tmp );
			c = g_utf8_get_char( tmp );
			switch( c ) {
				case 'f':
					if( pagepath ) {
						g_string_append( path,
								pagepath );
					}
					break;
				case '%':
					g_string_append_unichar( path,
							c );
					break;
				default:
					/* unknown escape, leave it in */
					g_string_append_unichar( path,
							c );
					break;
				
			}
			tmp = g_utf8_next_char( tmp );
		} else {
			g_string_append( path, exec );
		}
		exec = tmp;
	} while( exec );

	error = NULL;
	if( ! g_shell_parse_argv( path->str, &argc, &argv, &error ) ) {
		g_string_free( path, TRUE );
		if( error ) {
			g_error_free( error );
		}
		return;
	}
	g_string_free( path, TRUE );
	
	priv->window = window;

	if( ! g_spawn_async_with_pipes( dirname,
			argv, NULL,
			G_SPAWN_DO_NOT_REAP_CHILD,
			child_setup,
			helper,
			&pid,
			&std_in,
			&priv->std_out,
			&priv->std_err,
			&error ) ) {
		/* error starting helper */
		g_signal_emit( G_OBJECT( helper ),
			       screem_helper_signals[ HELPER_FAILED ],
			       0, error->message );
		g_strfreev( argv );
		if( error ) {
			g_error_free( error );
		}
		return;
	}
	g_strfreev( argv );
	priv->pid = pid;
	
	flags = fcntl( priv->std_out, F_GETFL, 0 );
	if( flags != -1 ) {
		fcntl( priv->std_out, F_SETFL, flags | O_NONBLOCK );
	} else {
		return;
	}
	flags = fcntl( priv->std_err, F_GETFL, 0 );
	if( flags != -1 ) {
		fcntl( priv->std_err, F_SETFL, flags | O_NONBLOCK );
	} else {
		return;
	}
	
	gladepath = screem_get_glade_path();
	
	switch( priv->imode ) {
		case SCREEM_HELPER_STDIN_SELECTION:
			out = fdopen( std_in, "w" );
			if( page == wpage ) {
				data = screem_window_get_selection( window );
			} else {
				data = screem_page_get_data( page );
			}
			/* FIXME: convert to system locale */

			fprintf( out, "%s", data );
			g_free( data );
			fclose( out );
			break;
		case SCREEM_HELPER_STDIN_USER:
			out = fdopen( std_in, "w" );
			xml = glade_xml_new( gladepath, 
					"helper_user_input", NULL );
			widget = glade_xml_get_widget( xml, 
					"helper_name" );
			gtk_label_set_text( GTK_LABEL( widget ), 
					priv->name );
			widget = glade_xml_get_widget( xml, 
							"helper_user_input" );
			if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
				GtkWidget *widg;

				widg = glade_xml_get_widget( xml, 
							     "use_selection" );
				if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widg ) ) ) {
					if( wpage == page ) {
						data = screem_window_get_selection( window );
					} else {
						data = screem_page_get_data( page );
					}
				} else {
					GtkTextBuffer *buffer;
					GtkTextIter it;
					GtkTextIter eit;
					
					widg = glade_xml_get_widget( xml, "user_data" );
					buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( widg ) );
					gtk_text_buffer_get_bounds( buffer, &it, &eit );
					data = gtk_text_buffer_get_text( buffer, &it, &eit, TRUE );
				}
				/* FIXME: convert to locale */

				fprintf( out, "%s", data );
				g_free( data );
			} else {
				/* canceled, stop the helper */
				kill( pid, SIGKILL );
			}
			fclose( out );
			gtk_widget_destroy( widget );
		
			g_object_unref( xml );

			break;
		case SCREEM_HELPER_STDIN_NONE:
			break;
	}
	close( std_in );

	priv->input = g_string_new( "" );
	
	/* we want to listen for child exits */
	priv->watch = g_child_watch_add( pid, child_watch, helper );

	/* listen to priv->std_out and priv->std_err */
	priv->out = g_io_channel_unix_new( priv->std_out );
	priv->err = g_io_channel_unix_new( priv->std_err );

	if( ! g_get_charset( &charset ) ) {
		g_io_channel_set_encoding( priv->out, charset, &error );
		if( error ) {
			g_error_free( error );
			error = NULL;
		}
		g_io_channel_set_encoding( priv->err, charset, &error );
		if( error ) {
			g_error_free( error );
			error = NULL;
		}
	}

	priv->owatch = g_io_add_watch( priv->out, 
			G_IO_IN | G_IO_ERR | G_IO_HUP, 
			child_stdout, helper );
	priv->ewatch = g_io_add_watch( priv->err, 
			G_IO_IN | G_IO_ERR | G_IO_HUP, 
			child_stderr, helper );

	/* display helper running dialog */
	xml = glade_xml_new( gladepath, "helper_run", NULL );

	widget = glade_xml_get_widget( xml, "helperlabel" );
	tmp = g_strdup_printf( "Running %s", priv->name );
	label = g_strconcat( "<span weight=\"bold\" size=\"large\">",
			tmp, "</span>", NULL );
	g_free( tmp );
	gtk_label_set_markup( GTK_LABEL( widget ), label );
	g_free( label );
	
	widget = glade_xml_get_widget( xml, "helper_run" );

	priv->dialog = widget;

	do {
		button = gtk_dialog_run( GTK_DIALOG( widget ) );
		switch( button ) {
			case GTK_RESPONSE_CANCEL:
				kill( pid, SIGKILL );
				break;
			default:
				break;
		}
	} while( button != GTK_RESPONSE_CLOSE );

	gtk_widget_destroy( widget );
	priv->dialog = NULL;
	
	g_free( gladepath );
	g_free( dirname );
}

void screem_helper_add_to_window( ScreemHelper *helper, ScreemWindow *window )
{
	gchar *name;
	gchar *pathname;
	gchar *iconpath;

	gchar *uiname;
	gchar *ui;
	GtkAction *action;
	guint merge_id;
	gchar *tmp;

	helper->private->list = 
		g_slist_append( helper->private->list, window );
	
	g_object_get( G_OBJECT( helper ),
		      "name", &name, "pathname", &pathname, 
		      "iconpath", &iconpath, NULL );

	tmp = screem_escape_char( name, ' ' );
	uiname = g_strconcat( "Exec_Helper_", tmp, NULL );
	g_free( tmp );

	tmp = screem_escape_underlines( name );
	action = gtk_action_new( uiname, tmp, pathname, 
			GTK_STOCK_EXECUTE );
	g_free( tmp );
	g_signal_connect( G_OBJECT( action ), "activate",
			G_CALLBACK( screem_helper_verb_cb ), helper );
	gtk_action_group_add_action( GTK_ACTION_GROUP( window->action_group ),
			action );
	ui = g_strconcat( "<ui><menubar><menu action=\"Tools\"><menu action=\"Helpers\">",
			"<menuitem action=\"", uiname, "\" />",
			"</menu></menu></menubar></ui>",
			NULL );
	merge_id = gtk_ui_manager_add_ui_from_string( GTK_UI_MANAGER( window->merge ),
							ui, strlen( ui ), NULL );
	g_object_set_data( G_OBJECT( action ), "merge_id",
				GUINT_TO_POINTER( merge_id ) );

	g_object_set_data( G_OBJECT( action ), "window", window );

	g_free( uiname );
	g_free( ui );

	g_free( pathname );
	g_free( name );
	g_free( iconpath );
	
}

void screem_helper_remove_from_window( ScreemHelper *helper,
				       ScreemWindow *window )
{
	gchar *name;
	gchar *uiname;
	GtkAction *action;
	guint merge_id;
	gchar *tmp;
	
	g_return_if_fail( SCREEM_IS_HELPER( helper ) );
	
	helper->private->list = g_slist_remove( helper->private->list, 
						window );

	g_object_get( G_OBJECT( helper ), "name", &name, NULL );

	tmp = screem_escape_char( name, ' ' );
	uiname = g_strconcat( "Exec_Helper_", tmp, NULL );
	g_free( tmp );

	g_free( name );
	
	action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
						uiname );
	/* action may have already been removed when closing the
	 * window */
	if( action ) {
		merge_id = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( action ),
						"merge_id" ) );
		gtk_ui_manager_remove_ui( GTK_UI_MANAGER( window->merge ), merge_id );
		gtk_action_group_remove_action( GTK_ACTION_GROUP( window->action_group ),
						action );
	}
	g_free( uiname );

}

/* static stuff */
static void screem_helper_verb_cb( GtkAction *action, gpointer user_data )
{
	ScreemHelper *helper;
	ScreemWindow *window;
	
	helper = SCREEM_HELPER( user_data );
	window = SCREEM_WINDOW( g_object_get_data( G_OBJECT( action ), "window" ) );
	screem_helper_execute( helper, NULL, window );
}

static void child_setup( gpointer data )
{


}
static void child_watch( GPid pid, gint status, gpointer data )
{
	ScreemHelper *helper;
	ScreemHelperPrivate *priv;
	ScreemPage *wpage;
	gchar *tmp;

	helper = SCREEM_HELPER( data );
	priv = helper->private;

	if( pid != priv->pid ) {
		return;
	}
		
	wpage = screem_window_get_document( priv->window );
	
	g_spawn_close_pid( pid );

	g_source_remove( priv->watch );
	
	g_source_remove( priv->owatch );
	g_source_remove( priv->ewatch );

	/* read any remaining data */
	child_stdout( priv->out, G_IO_IN, data );
	child_stderr( priv->err, G_IO_IN, data );
		
	g_io_channel_shutdown( priv->out, FALSE, NULL );
	g_io_channel_shutdown( priv->err, FALSE, NULL );
	g_io_channel_unref( priv->out );
	g_io_channel_unref( priv->err );
	
	priv->out = priv->err = NULL;
	
	gdk_threads_enter();
	
	if( WIFEXITED( status ) != 0 ) {
		if( status == 0 ) {
			/* success */
			switch( priv->omode ) {
				case SCREEM_HELPER_STDOUT_SELECTION:
					if( wpage == priv->page ) {
						screem_window_replace_selection( priv->window,
							priv->input->str );
					} else {
						screem_page_set_data( priv->page, priv->input->str );
					}
					
					break;
				case SCREEM_HELPER_STDOUT_INSERT:
					break;
				case SCREEM_HELPER_STDOUT_NONE:
					/* will have been displayed in
					   the windows message display
					   by the callback */
					break;
			}
		} else {
			/* helper returned an error */
			g_signal_emit( G_OBJECT( helper ),
				       screem_helper_signals[ HELPER_ERROR ], 0, status );
		}
	} else {
		/* helper didn't exit normally */
		tmp = g_strdup_printf( _( "%s finished due to an error" ),
				priv->name );
		g_signal_emit( G_OBJECT( helper ),
			       screem_helper_signals[ HELPER_FAILED ],
			       0, tmp );
		g_free( tmp );
	}
	
	g_string_free( priv->input, TRUE );
	if( priv->dialog ) {
		gtk_dialog_response( GTK_DIALOG( priv->dialog ), GTK_RESPONSE_CLOSE );
	}

	gdk_threads_leave();
}

static gboolean child_stdout( GIOChannel *source,
		GIOCondition condition, gpointer data )
{
	ScreemHelper *helper;
	ScreemHelperPrivate *priv;
	GIOStatus status;
	gchar buf[ BUFSIZ + 1 ];
	GError *error;
	gint read;

	helper = SCREEM_HELPER( data );
	priv = helper->private;

	error = NULL;
	status = G_IO_STATUS_AGAIN;

	gdk_threads_enter();
	
	if( condition == G_IO_IN ) {
		status = g_io_channel_read_chars( source,
				buf, BUFSIZ, &read, &error );
		if( status == G_IO_STATUS_NORMAL ||
			status == G_IO_STATUS_EOF ) {
	
			buf[ read ] = '\0';
			if( priv->omode == SCREEM_HELPER_STDOUT_NONE ) {
				screem_window_show_message( priv->window,
						buf, TRUE );
			} else {	
				g_string_append_len( priv->input,
						buf, read );
			}
		}
		if( error  ) {
			g_error_free( error );
		}
	}
	if( condition == G_IO_ERR ) {
		if( priv->dialog ) {
			gtk_dialog_response( GTK_DIALOG( priv->dialog ),
					GTK_RESPONSE_CANCEL );
		}
				
	} else if( condition == G_IO_HUP || status == G_IO_STATUS_EOF ) {
		if( priv->dialog ) {
			gtk_dialog_response( GTK_DIALOG( priv->dialog ),
					GTK_RESPONSE_CLOSE );
		}
	}
	
	gdk_threads_leave();
	
	return TRUE;
}

static gboolean child_stderr( GIOChannel *source,
		GIOCondition condition, gpointer data )
{
	ScreemHelper *helper;
	ScreemHelperPrivate *priv;
	GIOStatus status;
	gchar buf[ BUFSIZ + 1 ];
	GError *error;
	gint read;

	helper = SCREEM_HELPER( data );
	priv = helper->private;

	error = NULL;
	status = G_IO_STATUS_AGAIN;
	
	gdk_threads_enter();
	
	if( condition == G_IO_IN ) {
		status = g_io_channel_read_chars( source,
				buf, BUFSIZ, &read, &error );
		if( status == G_IO_STATUS_NORMAL ||
			status == G_IO_STATUS_EOF ) {
			
			buf[ read ] = '\0';
			screem_window_show_error( priv->window, buf,
					TRUE );
		}
		if( error  ) {
			g_error_free( error );
		}
	}
	if( condition == G_IO_ERR ) {
		/* we don't mind if this occurs on stderr */
		
	} else if( condition == G_IO_HUP || status == G_IO_STATUS_EOF ) {
		/* we don't mind if this occurs on stderr */
	}
	
	gdk_threads_leave();

	return TRUE;
}



/* G Object stuff */

#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void screem_helper_class_init( ScreemHelperClass *klass )
{
	GObjectClass *object_class;
	GParamSpec *spec;
		
	object_class = G_OBJECT_CLASS( klass );
	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_helper_finalize;

	object_class->set_property = screem_helper_set_property;
	object_class->get_property = screem_helper_get_property;

	spec = g_param_spec_string( "pathname", "Helper Path",
			"The Pathname to the Helper", "",
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_PATHNAME, spec );

	spec = g_param_spec_string( "iconpath", "Icon Path",
			"The Pathname to the image file for the Helper icon",
			"", G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_ICONPATH, spec );


	spec = g_param_spec_string( "name", "Helper name",
			"The name of the Helper", "",
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_HELPER_NAME, spec );

	spec = g_param_spec_int( "input", "Input Mode",
			"The input mode of the Helper",
			SCREEM_HELPER_STDIN_NONE,
			SCREEM_HELPER_STDIN_USER,
			SCREEM_HELPER_STDIN_SELECTION,
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_INPUT_MODE, spec );
							
	spec = g_param_spec_int( "output", "Output Mode", 
			"The output mode of the Helper",
			SCREEM_HELPER_STDOUT_NONE,
			SCREEM_HELPER_STDOUT_INSERT,
			SCREEM_HELPER_STDOUT_SELECTION,
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_OUTPUT_MODE, spec );


	spec = g_param_spec_boolean( "exec_on_save", "Execute On Save",
			"Should the helper run automatically when documents are saved",
			FALSE,
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_EXEC_ON_SAVE, spec );

	spec = g_param_spec_string( "exec_type", "Execute Type",
			"The type of document to execute on when it is saved", "",
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class,
					 PROP_EXEC_ON_SAVE_TYPE, spec );
	
	screem_helper_signals[ HELPER_FAILED ] = 
		g_signal_new( "helper_failed",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemHelperClass, 
					       helper_failed ),
			      NULL, NULL,
			      screem_marshal_VOID__STRING,
			      G_TYPE_NONE, 1,
			      G_TYPE_STRING );

	screem_helper_signals[ HELPER_ERROR ] = 
		g_signal_new( "helper_error",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemHelperClass, 
					       helper_error ),
			      NULL, NULL,
			      screem_marshal_VOID__INT,
			      G_TYPE_NONE, 1,
			      G_TYPE_INT );
}

static void screem_helper_init( ScreemHelper *helper )
{
	helper->private = g_new0( ScreemHelperPrivate, 1 );

}

static void screem_helper_finalize( GObject *object )
{
	ScreemHelper *helper;

	helper = SCREEM_HELPER( object );

	/* we need to remove outselves from all windows */
	while( helper->private->list ) {
		ScreemWindow *window = 
			SCREEM_WINDOW( helper->private->list->data );

		screem_helper_remove_from_window( helper, window );
	}

	if( helper->private->name )
		g_free( helper->private->name );
	if( helper->private->pathname )
		g_free( helper->private->pathname );
	if( helper->private->iconpath )
		g_free( helper->private->iconpath );

	g_free( helper->private );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}


static void screem_helper_set_property( GObject *object, guint prop_id,
					const GValue *value, GParamSpec *pspec)
{
	ScreemHelper *helper;
	ScreemHelperPrivate *private;
	const gchar *val;
	gint mode;
	
	helper = SCREEM_HELPER( object );
	private = helper->private;

	switch( prop_id ) {
	case PROP_PATHNAME:
		val = g_value_get_string( value );
		if( private->pathname )
			g_free( private->pathname );
		if( val ) {
			private->pathname = g_strdup( val );
		} else {
			private->pathname = NULL;
		}
		break;
	case PROP_ICONPATH:
		val = g_value_get_string( value );
		if( private->iconpath )
			g_free( private->iconpath );
		if( val ) {
			private->iconpath = g_strdup( val );
		} else {
			private->iconpath = NULL;
		}
		break;
	case PROP_HELPER_NAME:
		val = g_value_get_string( value );
		if( private->name ) {
			g_free( private->name );
		}
		if( val ) {
			private->name = g_strdup( val );
		} else {
			private->name = NULL;
		}
		break;
	case PROP_INPUT_MODE:
		mode = g_value_get_int( value );
		if( mode < 0 || mode > SCREEM_HELPER_STDIN_USER ) {
			mode = SCREEM_HELPER_STDIN_NONE;
		}
		private->imode = mode;
		break;
	case PROP_OUTPUT_MODE:
		mode = g_value_get_int( value );
		if( mode < 0 || mode > SCREEM_HELPER_STDOUT_INSERT ) {
			mode = SCREEM_HELPER_STDOUT_NONE;
		}
		private->omode = mode;
		break;
	case PROP_EXEC_ON_SAVE:
		private->exec = g_value_get_boolean( value );
		break;
	case PROP_EXEC_ON_SAVE_TYPE:
		val = g_value_get_string( value );
		g_free( private->exec_type );
		private->exec_type = g_strdup( val );
		break;
	default:
		break;
	}
}

static void screem_helper_get_property( GObject *object, guint prop_id,
					GValue *value, GParamSpec *pspec )
{
	ScreemHelper *helper;
	ScreemHelperPrivate *private;

	helper = SCREEM_HELPER( object );
	private = helper->private;

	switch( prop_id ) {
	case PROP_PATHNAME:
		if( private->pathname ) {
			g_value_set_string( value, private->pathname );
		} else {
			g_value_set_string( value, "" );
		}
		break;
	case PROP_ICONPATH:
		if( private->iconpath ) {
			g_value_set_string( value, private->iconpath );
		} else {
			g_value_set_string( value, "" );
		}
		break;
	case PROP_HELPER_NAME:
		if( private->name ) {
			g_value_set_string( value, private->name );
		} else {
			g_value_set_string( value, "" );
		}
		break;
	case PROP_INPUT_MODE:
		g_value_set_int( value, private->imode );
		break;
	case PROP_OUTPUT_MODE:
		g_value_set_int( value, private->omode );
		break;
	case PROP_EXEC_ON_SAVE:
		g_value_set_boolean( value, private->exec );
		break;
	case PROP_EXEC_ON_SAVE_TYPE:
		if( private->exec_type ) {
			g_value_set_string( value, private->exec_type );
		} else {
			g_value_set_string( value, "" );
		}
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID( object, prop_id, pspec );
		break;
	}
}



GType screem_helper_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemHelperClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_helper_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemHelper ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_helper_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemHelper",
					       &info, 0 );
	}

	return type;
}
