/*  Screem:  screem-editor.c
 *
 *  The editor widget
 *
 *  Copyright (C) 1999-2003  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <ctype.h>

#include <gmodule.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include <gtksourceview/gtksourceview.h>
#include <gtksourceview/gtksourcebuffer.h>
#include <gtksourceview/gtksourceprintjob.h>
#include <gtk/gtk.h>

#include <gdk/gdkkeysyms.h>

#include <gconf/gconf-client.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>

#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock-icons.h>
#include <libgnome/gnome-url.h>

#include <glib/gi18n.h>

#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-job.h>
#include <libgnomeprintui/gnome-print-dialog.h>
#include <libgnomeprintui/gnome-print-job-preview.h>

#include "screem-application.h"
#include "screem-window.h"
#include "screem-editor.h"
#include "screem-site.h"
#include "screem-markup.h"
#include "screem-page.h"
#include "screem-plugin.h"
#include "screem-search.h"

#include "screem-tagtree.h"

#include "fileops.h"
#include "support.h"

#include "pageUI.h"

#include "screemmarshal.h"

typedef enum _Drop_actions {
	DEFAULT_ACTION = 0,
        RELATIVE_PATH,
        FULL_PATH,
        TAG,
        ATTRIBUTE,
        INLINE     /* inserts the file contents (text files only) */
} DropActions;

typedef struct {
	ScreemEditor *editor;
	guint pos;
	gboolean adj;
	gdouble vadj;
} ScreemEditorDelaySetPos;

typedef void (*AutocompleteAfterFunc)( ScreemEditor *editor );

static void screem_editor_init( ScreemEditor *editor );
static void screem_editor_class_init( ScreemEditorClass *editor );
static void screem_editor_finalize( GObject *object );
static void screem_editor_window_set( ScreemView *view );
static void screem_editor_size_request( GtkWidget *widget, 
					GtkRequisition *req );
static void screem_editor_realize( GtkWidget *widget, 
		ScreemEditor *editor );
static void screem_editor_show( GtkWidget *widget );
static void screem_editor_hide( GtkWidget *widget );


static void screem_editor_display( ScreemView *view );
static void screem_editor_display_page( ScreemEditor *editor, 
					ScreemPage *page );

static gboolean screem_editor_motion( GtkWidget *widget, 
				      GdkDragContext *context, 
				      gint x, gint y, guint time, 
				      ScreemEditor *editor );
static void screem_editor_drop( GtkWidget *widget, GdkDragContext *context,
				gint x, gint y, 
				GtkSelectionData *selectionData,
				guint info, guint time, ScreemEditor *editor );
static void screem_editor_selection_received( ScreemEditor *editor,
		GtkSelectionData *sel, guint time,
		gpointer data );

static gboolean screem_editor_keypress( GtkWidget *widget, GdkEventKey *event,
					ScreemEditor *editor );
gboolean html_key_press( ScreemEditor *editor, GdkEventKey *event, 
			 ScreemEditorMacro *macro );

static gboolean screem_editor_tip( ScreemEditor *editor );
static gboolean screem_editor_tooltip( ScreemEditor *editor );

static void screem_editor_colour_notify( GConfClient *client,
					 guint cnxn_id,
					 GConfEntry *entry,
					 gpointer data );
static void screem_editor_wrap_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data );
static void screem_editor_tabwidth_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data );
static void screem_editor_autocomplete_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data );
static void screem_editor_spaces_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data );
static void screem_editor_margin_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data );
static void screem_editor_margin_col_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data );

static void screem_editor_clip_received( GtkClipboard *clipboard,
		GdkAtom *targets, gint count, gpointer data );
static void screem_editor_text_received( GtkClipboard *clipboard,
					 const gchar *text,
					 gpointer data );
static void screem_editor_complex_received( GtkClipboard *clipboard,
					 GtkSelectionData *sel,
					 gpointer data );
static void screem_editor_do_cut_paste( ScreemEditor *editor,
		gboolean cut );

static void screem_editor_toggle_overwrite( GtkTextView *view,
					    ScreemEditor *editor );

static gboolean html_editor_tip( ScreemEditor *editor, 
				 gint pos, gchar *text );

static gboolean screem_editor_drop_uris( ScreemEditor *editor,
					 DropActions action,
					 const gchar *selectionData,
					 GdkAtom target );

static gboolean set_pos( ScreemEditorDelaySetPos *delay );
static void screem_editor_set_pos_delayed( ScreemEditor *editor, 
					guint pos, 
					gboolean adj, gdouble val );

static void screem_editor_buffer_change( ScreemPage *page, GParamSpec *spec,
					 ScreemEditor *editor );

static gboolean screem_editor_autocomplete_popup_event( GtkWidget *widget, 
							GdkEventKey *event,
							gpointer data );
static void screem_editor_autocomplete_popup( ScreemEditor *editor, 
					      const GSList *list, 
					      guint32 term,
					      const gchar *str,
					      AutocompleteAfterFunc after );

static void screem_editor_attr_after( ScreemEditor *editor );

static GtkWidget *screem_editor_attribute_menu( ScreemEditor *editor,
					ScreemDTD *dtd, 
					const gchar *element );
static GtkWidget *screem_editor_allowed_menu( ScreemEditor *editor,
					ScreemDTD *dtd, 
					const gchar *element );
static void screem_editor_insert_attr_cb( GtkWidget *item, const gchar *attribute );

static void screem_editor_learn_more( GtkWidget *widget,
				ScreemEditor *editor );

static void screem_editor_get_popup_pos( ScreemEditor *editor, gint pos,
		gint *x, gint *y );
static void screem_editor_tip_popup( ScreemEditor *editor, gint pos,
		const gchar *message, gboolean hide );
static gint strcmp_len( gconstpointer a, gconstpointer b );

static gboolean screem_editor_check_func( ScreemEditor *editor );
static void screem_editor_mark_set( GtkTextBuffer *buffer,
		const GtkTextIter *location, GtkTextMark *mark,
		ScreemEditor *editor );

struct ScreemEditorPrivate {
	GConfClient *client;
	
	/* previous cursor position mark name, based
	 * off the name of the window we are in */
	gchar *markname;
	
	/* the text widget */
	GtkWidget *view;
	GtkWidget *sw;

	/* current page */
	ScreemPage *page;

	gboolean popup_open;
	gboolean inserted;

	/* gconf notify handles */
	GSList *notifies;
	
	gint tooltip_timer;
	
	gboolean overwrite;

	gboolean autocomplete;

	gboolean spaces;

	DropActions action;

	gchar *clipboard;
	gboolean set_clip;

	GtkTooltips *functip;
	gint funcpos;
	guint tipidle;
	guint tiphide;
};

#define HTML_ATOM_NAME "text/html"
#define UTF8_ATOM_NAME "UTF8_STRING"
#define CTEXT_ATOM_NAME "COMPOUND_TEXT"
#define STRING_ATOM_NAME "STRING"
#define TARGETS_ATOM_NAME "TARGETS"

#define URI_LIST_ATOM_NAME "text/uri-list"
#define MOZ_URL_ATOM_NAME "text/x-moz-url"
#define NETSCAPE_ATOM_NAME "_NETSCAPE_URL"
#define HTTP_ATOM_NAME "x-url/http"
#define FTP_ATOM_NAME "x-url/ftp"
#define COLOR_ATOM_NAME "application/x-color"
#define PLAIN_ATOM_NAME "text/plain"
#define UNICODE_ATOM_NAME "text/unicode"

static const GtkTargetEntry drop_types[] = {
	{ HTML_ATOM_NAME, 0, 0 },
        { URI_LIST_ATOM_NAME, 0, 0 },
	{ MOZ_URL_ATOM_NAME, 0, 0 },
        { NETSCAPE_ATOM_NAME, 0, 0 },
        { HTTP_ATOM_NAME, 0, 0 },
        { FTP_ATOM_NAME, 0, 0 },
        { COLOR_ATOM_NAME, 0, 0 },
	{ UTF8_ATOM_NAME, 0, 0 },
	{ UNICODE_ATOM_NAME, 0, 0 },
	{ PLAIN_ATOM_NAME, 0, 0 }
};
static const gint num_drop_types = G_N_ELEMENTS( drop_types );

/* DnD menu */
static void editor_dnd_cb( GtkAction *action, gpointer data );

static const gchar *editor_dnd_menu = "\
<ui>\
<popup action=\"editor_dnd_menu\">\
<menuitem action=\"insert relative filename\"/>\
<menuitem action=\"insert complete filename\"/>\
<menuitem action=\"insert tag\"/>\
<menuitem action=\"insert tag attribute\"/>\
<menuitem action=\"insert inline\"/>\
<separator name=\"dndsep1\"/>\
<menuitem action=\"cancel drag\"/>\
</popup>\
</ui>";
static GtkActionEntry editor_dnd_actions[] = {
	{ "editor_dnd_menu", "", NULL, NULL, NULL, NULL },

	{ "insert relative filename", NULL,
	  N_( "Insert relative filename" ), NULL,
	  N_( "Insert filename relative to the current document" ),
	  G_CALLBACK( editor_dnd_cb ) },
	  
	{ "insert complete filename", NULL,
	  N_( "Insert complete filename" ), NULL,
	  N_( "Insert full filename" ),
	  G_CALLBACK( editor_dnd_cb ) },

	{ "insert tag", NULL,
	  N_( "Insert tag" ), NULL,
	  N_( "Insert an HTML tag, based on what type the drop is" ),
	  G_CALLBACK( editor_dnd_cb ) },

	{ "insert tag attribute", NULL,
	  N_( "Insert tag attribute" ), NULL,
	  N_( "Insert an HTML attribute, based on what type the drop is" ),
	  G_CALLBACK( editor_dnd_cb ) },

	{ "insert inline", NULL,
	  N_( "Insert inline" ), NULL,
	  N_( "Insert the file into the current document" ),
	  G_CALLBACK( editor_dnd_cb ) },

	  { "cancel drag", NULL,
	  N_( "Cancel drag" ), "",
	  N_( "Cancel the drag and drop operation" ),
	  G_CALLBACK( editor_dnd_cb ) },
};
static guint editor_dnd_actions_n = G_N_ELEMENTS( editor_dnd_actions );

typedef gboolean (*Tipcb)( ScreemEditor *editor, gint pos, gchar *text );

static gboolean screem_editor_popup_key( GtkWidget *widget, 
						ScreemEditor *editor );
static gboolean screem_editor_popup( GtkWidget *widget, GdkEventButton *event,
					ScreemEditor *editor);

enum {
	ON = 0,
	OFF,
	BY_SET
};


enum {
	OVERWRITE,
	LAST_SIGNAL
};
static guint screem_editor_signals[ LAST_SIGNAL ] = { 0 };

ScreemEditor *screem_editor_new( ScreemWindow *window )
{
	ScreemEditor *editor;
	GType type;

	type = screem_editor_get_type();

	editor = SCREEM_EDITOR( g_object_new( type, "window", window,
					      NULL ) );

	return editor;
}

static void editor_dnd_cb( GtkAction *action, gpointer data )
{
	ScreemEditor *editor;
	gchar *name;
	gint i;
	gint item;

	editor = SCREEM_EDITOR( data );
	
	g_object_get( G_OBJECT( action ), "name", &name, NULL );
	item = -1;
	for( i = 1; i < editor_dnd_actions_n; ++ i ) {
		if( ! strcmp( name, editor_dnd_actions[ i ].name ) ) {
			item = i - 1;
			break;
		}
	}
	g_object_set_data( G_OBJECT( editor ), "dnd_popup_item",
			GINT_TO_POINTER( item ) );
}

static gboolean screem_editor_popup_key( GtkWidget *widget, ScreemEditor *editor )
{
	screem_editor_popup( widget, NULL, editor );

	return TRUE;
}

gboolean screem_editor_button_release( GtkWidget *widget, 
				       GdkEventButton *event,
				       ScreemEditor *editor )
{
	ScreemEditorPrivate *private;

	private = editor->private;

	if( event && event->button == 1 &&
	    event->window ==
	    gtk_text_view_get_window( GTK_TEXT_VIEW( private->view ),
				      GTK_TEXT_WINDOW_LEFT ) ) {
		/* left click on the line numbers,
		   we want to be able to drag and select
		   entire lines, so lets cheat on pass the
		   event onto the main window, this callback is needed
		   so the drag is released properly */
		GdkEventButton child_event;

		child_event = *event;
		child_event.window =
			gtk_text_view_get_window( GTK_TEXT_VIEW(private->view),
						  GTK_TEXT_WINDOW_TEXT );
		gtk_widget_event( private->view, (GdkEvent*)&child_event );
	}

	return FALSE;
}

static gboolean screem_editor_popup( GtkWidget *widget, GdkEventButton *event,
					ScreemEditor *editor )
{
	gint cpos;
	gint pos2;
    	gint pos;
	gint tag_pos;
	gint tag_len;
	gchar *text;
	gchar *tag;
	gchar *tag2;
	GtkWidget *menu;
	GtkWidget *submenu;
	GtkWidget *menu_item;
	GtkTextBuffer *buffer;

	gboolean tag_known;

	ScreemPage *page;
	ScreemDTD *dtd;

	const gchar *pathname;
	const gchar *mime_type;

	gint start;
	gint len;
	gboolean sel;

	ScreemEditorPrivate *private;
	
	gboolean feature_markup;

	GSList *addeditems = NULL;
	GSList *tmp;

	GtkTextTagTable *table;
	GtkTextTag *itag;
	GtkTextIter it;
	
	GtkUIManager *merge;
	
	private = editor->private;
	page = private->page;

	if( ! page ) {
		return FALSE;
	}

	table = gtk_text_buffer_get_tag_table( GTK_TEXT_BUFFER( page ) );
	itag = gtk_text_tag_table_lookup( table, 
			SCREEM_INVALID_MARKUP_TAG );
	
	pathname = screem_page_get_pathname( page );

	if( event && event->button == 1 &&
	    event->window ==
	    gtk_text_view_get_window( GTK_TEXT_VIEW( private->view ),
				      GTK_TEXT_WINDOW_LEFT ) ) {
		/* left click on the line numbers,
		   we want to be able to drag and select
		   entire lines, so lets cheat on pass the
		   event onto the main window */
		GdkEventButton child_event;

		child_event = *event;
		child_event.window =
			gtk_text_view_get_window( GTK_TEXT_VIEW(private->view),
						  GTK_TEXT_WINDOW_TEXT );
		/* we do this so we select from the start of the line */
		child_event.x = 0;
		gtk_widget_event( private->view, (GdkEvent*)&child_event );
		return TRUE;
	}

	g_object_get( G_OBJECT( editor ), 
			"uimanager", &merge,
			NULL );

	if( event && event->button != 3 ) {
		g_object_unref( merge );
		return FALSE;
	}

	private->popup_open = TRUE;

	if( event ) {
		g_signal_stop_emission_by_name( G_OBJECT( widget ),
						"button_press_event" );
	} else {
		g_signal_stop_emission_by_name( G_OBJECT( widget ),
						"popup_menu" );
	}
	
	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( private->view ) );
	mime_type = screem_page_get_mime_type( page );
	cpos = screem_editor_get_pos( editor );

	/* get position of click */
	if( event ) {
		gint x;
		gint y;
		gint bx;
		gint by;

		x = (gint)event->x + 0.5;
		y = (gint)event->y + 0.5;
		gtk_text_view_window_to_buffer_coords(GTK_TEXT_VIEW(private->view ),
						      GTK_TEXT_WINDOW_TEXT,
						      x, y, &bx, &by );
		gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(private->view),
						   &it, bx, by);
		pos2 = pos = gtk_text_iter_get_offset( &it );
	} else {
		pos2 = pos = cpos;
	}

	sel = screem_editor_has_selection( editor, &start, &len );
	screem_editor_set_pos( editor, pos );

	if( sel ) {
		screem_editor_select_region( editor, start, len - start );
	}

	/* create the menu */
       	menu = gtk_ui_manager_get_widget( GTK_UI_MANAGER( merge ),
					"/editormenu" );
	g_object_unref( merge );
	
	/* id the tag we are inside so we can offer attribute editing */
	
	text = screem_page_get_data( page );

	feature_markup = screem_page_is_markup( page );

	if( feature_markup &&
	    screem_markup_is_tag( text, pos, &pos, NULL ) ) {
		tag_pos = pos;
		tag = screem_markup_next_tag( text, pos, NULL, &pos, &tag2 );
		tag_len = pos - tag_pos + 1;

		dtd = screem_page_get_dtd( page );
		menu_item = screem_editor_attribute_menu( editor, dtd, tag2 );
		if( menu_item ) {
			gtk_menu_shell_prepend( GTK_MENU_SHELL( menu ), 
						menu_item );
			addeditems = g_slist_prepend( addeditems, menu_item );
		}
		tag_known = (gboolean)menu_item;

		/* add the wizard option */
		if( (menu_item = screem_plugin_tag_wizard( SCREEM_VIEW( editor ), 
							   tag2, tag,
							   tag_pos, 
							   tag_len ) ) ) {
			gtk_menu_shell_append( GTK_MENU_SHELL( menu ),
					       menu_item );
			addeditems = g_slist_prepend( addeditems, menu_item );
		}

		g_free( tag );

		/* add the learn option */
		if( itag && gtk_text_iter_has_tag( &it, itag ) ) {
			menu_item = gtk_menu_item_new_with_label( _( "Learn More..." ) );
			gtk_widget_show( menu_item );
			gtk_menu_shell_append( GTK_MENU_SHELL( menu ), menu_item );
			addeditems = g_slist_prepend( addeditems, menu_item );
			g_signal_connect( G_OBJECT( menu_item ), 
					"activate", 
					G_CALLBACK( screem_editor_learn_more ), 
					editor );
		}
		g_free( tag2 );
	} else if( feature_markup ) {
		/* add the insert option */
		tag2 = screem_page_query_context( page, pos, FALSE,
			TRUE, NULL, NULL, NULL, NULL );

		dtd = screem_page_get_dtd( page );

		if( tag2 && 
		    ( menu_item = screem_editor_allowed_menu( editor, dtd, tag2 ) ) ) {
			gtk_menu_shell_append( GTK_MENU_SHELL( menu ), 
					       menu_item );
			addeditems = g_slist_prepend( addeditems, menu_item );
		}
	}
	g_free( text );

	menu_item = gtk_menu_item_new_with_mnemonic( _( "Input _Methods" ) );
	addeditems = g_slist_prepend( addeditems, menu_item );
	gtk_widget_show( menu_item );
	submenu = gtk_menu_new ();
	gtk_menu_item_set_submenu( GTK_MENU_ITEM( menu_item ), submenu );
	gtk_menu_shell_append( GTK_MENU_SHELL( menu ), menu_item );
	gtk_im_multicontext_append_menuitems( GTK_IM_MULTICONTEXT( GTK_TEXT_VIEW(private->view)->im_context ),
						GTK_MENU_SHELL( submenu ) );

	/* display menu */
	private->inserted = FALSE;
	screem_popup_menu_do_popup_modal( menu, 0, 0, event, 0, private->view );

	for( tmp = addeditems; tmp; tmp = tmp->next ) {
		gtk_container_remove( GTK_CONTAINER( menu ), 
					GTK_WIDGET( tmp->data ) );
	}
	g_slist_free( addeditems );
	
	if( ! private->inserted )
		screem_editor_set_pos( editor, cpos );
	
	if( sel ) {
		screem_editor_select_region( editor, start, len - start );
	}

	private->popup_open = FALSE;

	return TRUE;
}

void screem_editor_insert( ScreemEditor *editor, gint pos, const gchar *text )
{
	gint len;
	gint start;
	const gchar *pathname;
	const gchar *mime_type;

	GtkTextBuffer *buffer;
	GtkTextIter it;

	buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	if( ! text || ! editor->private->page )
		return;

	pathname = screem_page_get_pathname( editor->private->page );
	mime_type = screem_page_get_mime_type( editor->private->page );

	len = gtk_text_buffer_get_char_count( buffer );
	if( pos == -1 ) {
		pos = screem_editor_get_pos( editor );
	}

	if( pos > len ) {
		pos = len;
	}

	start = pos;

	len = strlen( text );

	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );
	gtk_text_buffer_insert( buffer, &it, text, len );

	gtk_widget_grab_focus( GTK_WIDGET( editor ) );
}

static void focus( GtkWidget *widget )
{
	ScreemEditor *editor;

	editor = SCREEM_EDITOR( widget );
	gtk_widget_grab_focus( editor->private->view );
}

static gboolean screem_editor_focus( GtkWidget *widget )
{
	ScreemEditor *editor;

	editor = SCREEM_EDITOR( widget );

gdk_threads_enter();
	
	gtk_widget_grab_focus( editor->private->view );

	gdk_threads_leave();

	return FALSE;
}

gchar *screem_editor_get_text( ScreemEditor *editor, guint start, guint len )
{
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter eit;

	guint end = len;

	buffer =
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	if( end == 0 ) {
		end = gtk_text_buffer_get_char_count( buffer );
	} else {
		end += start;
	}

	gtk_text_buffer_get_iter_at_offset( buffer, &it, start );
	gtk_text_buffer_get_iter_at_offset( buffer, &eit, end );

	return gtk_text_buffer_get_text( buffer, &it, &eit, TRUE );
}

gboolean screem_editor_has_selection( ScreemEditor *editor, guint *start, 
				      guint *end )
{
	guint temp;
	gboolean ret;
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter eit;

	buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	if( ! start )
		start = &temp;
	if( ! end )
		end = &temp;

	ret = gtk_text_buffer_get_selection_bounds( buffer, &it, &eit );

	*start = gtk_text_iter_get_offset( &it );
	*end = gtk_text_iter_get_offset( &eit );

	return ret;
}

void screem_editor_select_region( ScreemEditor *editor, guint start, guint len )
{
        GtkTextBuffer* buffer = NULL;
        GtkTextIter it;
	GtkTextIter eit;

        buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	if( len == 0 ) {
		len = gtk_text_buffer_get_char_count( buffer );
	}

	gtk_text_buffer_get_iter_at_offset( buffer, &it, start );
	gtk_text_buffer_get_iter_at_offset( buffer, &eit, start + len );

        gtk_text_buffer_place_cursor( buffer, &eit );

        gtk_text_buffer_move_mark( buffer,
				   gtk_text_buffer_get_mark(buffer, 
							    "selection_bound"),
				   &it );
	gtk_text_view_scroll_mark_onscreen(GTK_TEXT_VIEW(editor->private->view),
					gtk_text_buffer_get_insert( buffer ) );
}

void screem_editor_cut( ScreemEditor *editor )
{
	screem_editor_do_cut_paste( editor, TRUE );
}

void screem_editor_copy( ScreemEditor *editor )
{
	screem_editor_do_cut_paste( editor, FALSE );
}

void screem_editor_paste( ScreemEditor *editor )
{
	GdkDisplay *display;
	GtkClipboard *clip;
	GdkAtom atom;
	
	display = gtk_widget_get_display( GTK_WIDGET( editor ) );
	clip = gtk_clipboard_get_for_display( display,
			GDK_SELECTION_CLIPBOARD );

	g_object_set_data( G_OBJECT( editor ), "encode", 
			GINT_TO_POINTER( FALSE ) );
	atom = gdk_atom_intern( TARGETS_ATOM_NAME, FALSE );
	gtk_clipboard_request_targets( clip,
			screem_editor_clip_received, editor );
}

void screem_editor_paste_unformatted( ScreemEditor *editor )
{
	GdkDisplay *display;
	GtkClipboard *clip;
	
	display = gtk_widget_get_display( GTK_WIDGET( editor ) );
	clip = gtk_clipboard_get_for_display( display,
			GDK_SELECTION_CLIPBOARD );

	g_object_set_data( G_OBJECT( editor ), "encode", 
			GINT_TO_POINTER( FALSE ) );
	gtk_clipboard_request_text( clip, 
			screem_editor_text_received,
			editor );
}

void screem_editor_paste_encoded( ScreemEditor *editor )
{
	GdkDisplay *display;
	GtkClipboard *clip;
	GdkAtom atom;
	
	display = gtk_widget_get_display( GTK_WIDGET( editor ) );
	clip = gtk_clipboard_get_for_display( display,
			GDK_SELECTION_CLIPBOARD );

	g_object_set_data( G_OBJECT( editor ), "encode", 
			GINT_TO_POINTER( TRUE ) );
	atom = gdk_atom_intern( TARGETS_ATOM_NAME, FALSE );
	gtk_clipboard_request_targets( clip,
			screem_editor_clip_received, editor );
}

void screem_editor_clear_selection( ScreemEditor *editor )
{
	GtkTextBuffer *buffer;

	buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	gtk_text_buffer_delete_selection( buffer, FALSE, TRUE );

	gtk_text_view_scroll_mark_onscreen(GTK_TEXT_VIEW(editor->private->view),
					gtk_text_buffer_get_insert( buffer ) );

}

void screem_editor_set_pos( ScreemEditor *editor, guint pos )
{
	GtkTextView *view;
	GtkTextBuffer *buffer;
	GtkTextIter it;

	view = GTK_TEXT_VIEW( editor->private->view );

	buffer = gtk_text_view_get_buffer( view );

	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );

	gtk_text_buffer_place_cursor( buffer, &it );

	gtk_text_view_scroll_to_iter( view, &it, 0.0, FALSE, 0.0, 0.0 );
}

guint screem_editor_get_pos( ScreemEditor *editor )
{
	GtkTextBuffer *buffer;
	GtkTextMark *mark;
	GtkTextIter it;

	buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));
	mark = gtk_text_buffer_get_insert( buffer );

	gtk_text_buffer_get_iter_at_mark( buffer, &it, mark );

	return gtk_text_iter_get_offset( &it );
}

void screem_editor_delete_forward( ScreemEditor *editor, guint pos, guint len )
{
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter eit;

	buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));
	if( len == 0 ) {
		len = gtk_text_buffer_get_char_count( buffer );
	}
	
	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );
	gtk_text_buffer_get_iter_at_offset( buffer, &eit, pos + len );

	gtk_text_buffer_delete( buffer, &it, &eit );
}

void screem_editor_insert_markup( ScreemEditor *editor, 
				  const gchar *open_element, 
				  const gchar *close_element )
{
	gboolean has_selection;
	guint start;
	guint end;
	gint pos;
	gint ipos;
	ScreemPage *page;

	page = editor->private->page;
	
	if( ! page )
		return;

	pos = screem_editor_get_pos( editor );
	ipos = pos;

	has_selection = screem_editor_has_selection( editor, &start, &end );

	if( has_selection ) {
		if( open_element ) {
			screem_editor_insert( editor, start, open_element );
                        end += strlen( open_element );
		}
		pos = end;
		if( close_element ) {
                        screem_editor_insert( editor, end, close_element );
                        pos += strlen( close_element );
                }
	} else {
		if( open_element ) {
			screem_editor_insert( editor, pos, open_element );
			pos += strlen( open_element );
		}
		if( close_element ) {
			screem_editor_insert( editor, pos, close_element );
			pos += strlen( close_element );
		}

		if( open_element )
			pos = ipos + strlen( open_element );
		else if( close_element )
			pos = ipos + strlen( close_element );
	}
	screem_editor_set_pos( editor, pos );

	if( editor->private->popup_open )
		editor->private->inserted = TRUE;
}

void screem_editor_insert_file( ScreemEditor *editor, const gchar *filename )
{
	GtkWidget *window;
	GString *data;
	gint pos;
	gchar *tmp;
	gchar *base;
	gchar *primary;
	GError *error;

	window = NULL;
	g_object_get( G_OBJECT( editor ), "window", &window, NULL );

	data = load_file( filename, NULL, NULL, &error ); 
	if( data ) {
		pos = screem_editor_get_pos( editor );
		screem_editor_insert( editor, pos, data->str );
		g_string_free( data, TRUE );
	} else {
		tmp = gnome_vfs_unescape_string_for_display( filename );
		base = g_path_get_basename( tmp );
		primary = g_strdup_printf( _( "Could not open recent page: %s" ), base );
		screem_hig_alert( GTK_STOCK_DIALOG_ERROR,
				primary, error->message,
				GTK_WIDGET( window ) );
		g_free( base );
		g_free( primary );
		g_free( tmp );
		g_error_free( error );
	}
}

void screem_editor_undo( ScreemView *view )
{
	ScreemEditor *editor;
	ScreemPage *page;
	guint pos;

	editor = SCREEM_EDITOR( view );
	page = editor->private->page;

	if( page && gtk_source_buffer_can_undo( GTK_SOURCE_BUFFER( page ) ) ) {
		gtk_source_buffer_undo( GTK_SOURCE_BUFFER( page ) );
	}
	pos = screem_editor_get_pos( editor );
	screem_editor_set_pos( editor, pos );
}

void screem_editor_redo( ScreemView *view )
{
	ScreemEditor *editor;
	ScreemPage *page;
	guint pos;

	editor = SCREEM_EDITOR( view );
	page = editor->private->page;

	if( page && gtk_source_buffer_can_redo( GTK_SOURCE_BUFFER( page ) ) ) {
		gtk_source_buffer_redo( GTK_SOURCE_BUFFER( page ) );
	}
	pos = screem_editor_get_pos( editor );
	screem_editor_set_pos( editor, pos );
}

gint screem_editor_auto_indent( ScreemEditor *editor, gint pos )
{
	guint depth;
	guint start;
	guint end;
	GString *indent;
	GtkTextIter it;
	GtkTextIter *copy;
	GtkTextBuffer *buffer;
	gunichar c;
	guint offset;
	gboolean corrected;

	guint sel_start;
	guint sel_end;
	gint sel_offset;
	gboolean built;

	gchar *temp;
	
	gboolean spaces;
	guint tab_width;
	
	spaces = editor->private->spaces;
	tab_width = gtk_source_view_get_tabs_width( GTK_SOURCE_VIEW( editor->private->view ) );
	
	if( ! screem_editor_has_selection( editor, &sel_start, &sel_end ) ) {
		sel_start = pos;
		sel_end = sel_start;
	}

	buffer =gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	/* convert to line numbers */
	gtk_text_buffer_get_iter_at_offset( buffer, &it, sel_start );
	sel_start = gtk_text_iter_get_line( &it );
	gtk_text_buffer_get_iter_at_offset( buffer, &it, sel_end );
	sel_end = gtk_text_iter_get_line( &it );

	for( sel_offset = 0, built = FALSE; sel_start <= sel_end;
		++ sel_start ) {
		pos = sel_start;
		gtk_text_buffer_get_iter_at_line( buffer, &it, pos );
		pos = gtk_text_iter_get_offset( &it );

		temp = screem_page_query_context( SCREEM_PAGE( buffer ),
				pos - sel_offset, FALSE, ! built, 
				&depth, &start, &end, NULL );
		g_free( temp );
		built = TRUE;
	
		/* if the first char on a line is < then we will have
		   depth 1 > than we should */
		corrected = FALSE;
		if( depth != 0 && gtk_text_iter_get_char( &it ) == '<' ) {
			depth --;
			corrected = TRUE;
		}
		
		copy = gtk_text_iter_copy( &it );
		offset = 0;
		
		do {
			offset ++;
			c = gtk_text_iter_get_char( copy );
			if( ! g_unichar_isspace( gtk_text_iter_get_char( copy ) ) ||
			    c == '\n' ) {
				break;
			}
			gtk_text_iter_forward_char( copy );
		} while( ! gtk_text_iter_is_end( copy ) );
		
		gtk_text_buffer_delete( buffer, &it, copy );
		gtk_text_iter_free( copy );
		gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );
		
		/* handle close of context correctly */
		copy = gtk_text_iter_copy( &it );
		c = gtk_text_iter_get_char( copy );
		if( c == '<' ) {
			gtk_text_iter_forward_char( copy );
			if( ! gtk_text_iter_is_end( copy ) ) {
				c = gtk_text_iter_get_char( copy );
			}
			if( c == '/' ) {
				/* closing tag, does it close the context? */
				do {
					gtk_text_iter_forward_char( copy );
					if( gtk_text_iter_is_end( copy ) ) {
						break;
					}
					c = gtk_text_iter_get_char( copy );
				} while( c != '>' );
			}
			pos = gtk_text_iter_get_offset( copy ) + 1;
			pos += offset;
			/* is pos outside the current context? */
			if( pos >= end && depth != 0 && ! corrected ) {
				depth --;
			}
		}
		gtk_text_iter_free( copy );
	
		if( ! spaces ) {	
			indent = g_string_new_len( "", depth );
			memset( indent->str, '\t', depth );
		} else {
			indent = g_string_new_len( "", depth * tab_width );
			memset( indent->str, ' ', depth * tab_width );
		}

		gtk_text_buffer_insert( buffer, &it, indent->str, indent->len );
		
		sel_offset -= offset;
		sel_offset += 1 + indent->len;
		
		g_string_free( indent, TRUE );
	}
	
	pos = gtk_text_iter_get_offset( &it );
	pos += sel_offset;

	return pos;
}

void screem_editor_indent( ScreemEditor *editor, guint pos )
{
	guint sel_start;
	guint sel_end;
	GtkTextBuffer *buffer;
	GtkTextIter it;
	gboolean spaces;
	guint tab_width;
	GString *space_str;
	
	spaces = editor->private->spaces;
	tab_width = gtk_source_view_get_tabs_width( GTK_SOURCE_VIEW( editor->private->view ) );
	space_str = NULL;
	if( spaces ) {
		space_str = g_string_new_len( "", tab_width );
		memset( space_str->str, ' ', tab_width );
	}
	
	if( ! screem_editor_has_selection( editor, &sel_start, 
					   &sel_end ) ) {
		sel_start = pos;
		sel_end = sel_start;
	}

	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( editor->private->view ) );

	/* convert to line numbers */
	gtk_text_buffer_get_iter_at_offset( buffer, &it, sel_start );
	sel_start = gtk_text_iter_get_line( &it );
	gtk_text_buffer_get_iter_at_offset( buffer, &it, sel_end );
	sel_end = gtk_text_iter_get_line( &it );

	for( ; sel_start <= sel_end; ++ sel_start ) {
		pos = sel_start;
		gtk_text_buffer_get_iter_at_line( buffer, &it, pos );
		pos = gtk_text_iter_get_offset( &it );
		
		if( ! spaces ) {
			gtk_text_buffer_insert( buffer, &it, "\t", 1 );
		} else {
			gtk_text_buffer_insert( buffer, &it, space_str->str, space_str->len );
		}
	}
	if( spaces ) {
		g_string_free( space_str, TRUE );
	}
}

void screem_editor_unindent( ScreemEditor *editor, guint pos )
{
	guint sel_start;
	guint sel_end;
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter eit;
	gboolean spaces;
	guint tab_width;
	GString *space_str;
	gint i;
	gchar *tmp;
	gboolean erase;
	
	spaces = editor->private->spaces;
	tab_width = gtk_source_view_get_tabs_width( GTK_SOURCE_VIEW( editor->private->view ) );
	space_str = NULL;
	if( spaces ) {
		space_str = g_string_new_len( "", tab_width );
		memset( space_str->str, ' ', tab_width );
	}
	
	if( ! screem_editor_has_selection( editor, &sel_start, 
					   &sel_end ) ) {
		sel_start = pos;
		sel_end = sel_start;
	}

	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( editor->private->view ) );

	/* convert to line numbers */
	gtk_text_buffer_get_iter_at_offset( buffer, &it, sel_start );
	sel_start = gtk_text_iter_get_line( &it );
	gtk_text_buffer_get_iter_at_offset( buffer, &it, sel_end );
	sel_end = gtk_text_iter_get_line( &it );

	for( ; sel_start <= sel_end; ++ sel_start ) {
		gtk_text_buffer_get_iter_at_line( buffer, &it, sel_start );
		eit = it;
		gtk_text_iter_forward_char( &eit );

		erase = FALSE;
		if( ! spaces ) {
			erase = ( gtk_text_iter_get_char( &it ) == '\t' );
		} else {
			for( i = 0; i < tab_width; ++ i ) {
				gtk_text_iter_forward_char( &eit );
			}
			tmp = gtk_text_buffer_get_text( buffer, &it, &eit, TRUE );
			erase = ( ! strcmp( space_str->str, tmp ) );
			g_free( tmp );			
		}
		if( erase ) {
			gtk_text_buffer_delete( buffer, &it, &eit );
		}
	}
	if( spaces ) {
		g_string_free( space_str, TRUE );
	}
}


void screem_editor_goto_line( ScreemEditor *editor, gint line )
{
	ScreemPage *page;
	GtkTextIter it;
	GtkTextView *view;
	gint pos;

	page = editor->private->page;

	if( ! page ) {
		return;
	}

	view = GTK_TEXT_VIEW(editor->private->view);
	gtk_text_buffer_get_iter_at_line( GTK_TEXT_BUFFER( page ), &it, line - 1 );
	gtk_text_view_scroll_to_iter( view, &it, 0.0, FALSE, 0.0, 0.0 );
	pos = gtk_text_iter_get_offset( &it );
	
	screem_editor_set_pos( editor, pos );
}

void screem_editor_encode_text( ScreemEditor *editor, gboolean urienc )
{
	gchar *text;
	gchar *ret;
	guint start = 0;
	guint len = 0;
	ScreemPage *page;

	page = editor->private->page;

	if( ! page )
		return;

	if( screem_editor_has_selection( editor, &start, &len ) )
		len = len - start;

	text = screem_editor_get_text( editor, start, len );

	if( ! urienc ) {
		ret = screem_markup_encode_text( text );
	} else {
		ret = gnome_vfs_escape_string( text );		
	}

	screem_editor_delete_forward( editor, start, len );
	screem_editor_insert( editor, start, ret );

	g_free( ret );
	g_free( text );
}

gchar* screem_editor_get_word( ScreemEditor *editor, gint pos )
{
	ScreemEditorPrivate *priv;
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter eit;
	gunichar c;
	gchar *word;

	priv = editor->private;
	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( priv->view ) );

	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );
	c = gtk_text_iter_get_char( &it );
	if( c == 0 || c == '\n' || c == '\r' ) {
		pos --;
	}
	
	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );

	c = gtk_text_iter_get_char( &it );
	if( g_unichar_isspace( c ) ) {
		return NULL;
	}
	
	for( ; ! gtk_text_iter_is_start( &it ); gtk_text_iter_backward_char( &it ) ) {
		c = gtk_text_iter_get_char( &it );
		if( c != '_' && ! g_unichar_isalnum( c ) ) {
			gtk_text_iter_forward_char( &it );
			break;
		}
	}
	gtk_text_buffer_get_iter_at_offset( buffer, &eit, pos );
	for( ; ! gtk_text_iter_is_end( &eit ); gtk_text_iter_forward_char( &eit ) ) {
		c = gtk_text_iter_get_char( &eit );
		if( c != '_' && ! g_unichar_isalnum( c ) ) {
			break;
		}
	}

	word = NULL;
	if( gtk_text_iter_compare( &it, &eit ) ) {
		word = gtk_text_iter_get_visible_text( &it, &eit );
	}

	return word;
}

static void screem_editor_set_colours( ScreemEditor *editor )
{
	GdkColor back;
	GdkColor text;
	GConfClient *client;
	ScreemEditorPrivate *private;
 	PangoFontDescription *font = NULL;
	gchar *value;

	private = editor->private;
	client = private->client;

	if( ! gconf_client_get_bool( client, "/apps/screem/editor/themecolours", NULL ) ) {
		value = gconf_client_get_string( client, "/apps/screem/editor/back",
						 NULL );
		if( ! value ) {
			value = g_strdup( "#ffffffffffff" );
		}
		gdk_color_parse( value, &back );
		g_free( value );
	
		gtk_widget_modify_base( GTK_WIDGET( private->view ),
					GTK_STATE_NORMAL, &back );

		value = gconf_client_get_string( client, "/apps/screem/editor/text",
						 NULL );
		if( ! value ) {
			value = g_strdup( "#000000000000" );
		}
		gdk_color_parse( value, &text );

		g_free( value );

		gtk_widget_modify_text( GTK_WIDGET( private->view ),
					GTK_STATE_NORMAL, &text );
	} else {
		GtkRcStyle *rc_style;

		rc_style = gtk_widget_get_modifier_style( GTK_WIDGET( private->view ) );
		rc_style->color_flags [GTK_STATE_NORMAL] = 0;
		rc_style->color_flags [GTK_STATE_SELECTED] = 0;
   		rc_style->color_flags [GTK_STATE_ACTIVE] = 0;
		gtk_widget_modify_style( GTK_WIDGET( private->view), rc_style );
	}

	if( ! gconf_client_get_bool( client, "/apps/screem/editor/themefont", NULL ) ) {
		value = gconf_client_get_string( client, "/apps/screem/editor/font",
						 NULL );
		if( value ) {
			font = pango_font_description_from_string( value );
			gtk_widget_modify_font( GTK_WIDGET( private->view ), font );
			if( private->functip->tip_label ) {
				gtk_widget_modify_font( private->functip->tip_label, font  );
			}
			pango_font_description_free( font );
			g_free( value );
		}
	} else {
		GtkRcStyle *rc_style;
	
		rc_style = gtk_widget_get_modifier_style( GTK_WIDGET( private->view ) );
		if( rc_style->font_desc ) {
			pango_font_description_free( rc_style->font_desc );
		}
		rc_style->font_desc = NULL;
		gtk_widget_modify_style( GTK_WIDGET( private->view ), rc_style );
	}
}

static void screem_editor_print_page_cb( GtkSourcePrintJob *job, 
					 ScreemEditor *editor )
{
	gchar *msg;
	
	msg = g_strdup_printf( "Printing %.2f%%",
				100.0 * gtk_source_print_job_get_page( job ) /
				gtk_source_print_job_get_page_count( job ) );
	gdk_threads_enter();
	screem_view_show_message( SCREEM_VIEW( editor ), msg );
	gdk_threads_leave();

	g_free( msg );
}
static void screem_editor_print_finish_cb( GtkSourcePrintJob *job,
					ScreemEditor *editor )
{
	GnomePrintJob *gjob;
	GtkWidget *preview;

	gjob = gtk_source_print_job_get_print_job( job );

	if( g_object_get_data( G_OBJECT( job ), "preview" ) ) {
		preview = gnome_print_job_preview_new( gjob, _( "Print Preview - Screem" ) );
		gtk_widget_show( preview );
	} else {
                gnome_print_job_print( gjob );
	}
	g_object_unref( gjob );
	g_object_unref( job );

}

static GtkSourcePrintJob* screem_editor_print_setup_job( ScreemEditor *editor ) 
{
	GConfClient *client;
	GtkSourcePrintJob *job;
	GtkSourceView *view;
	ScreemPage *page;
	const gchar *pathname;

	client = gconf_client_get_default();
	
	view = GTK_SOURCE_VIEW( editor->private->view );
	page = editor->private->page;

	job = gtk_source_print_job_new( NULL );
	gtk_source_print_job_setup_from_view( job, view );

	if( gconf_client_get_bool( client, 
				"/apps/screem/editor/print/wrap_lines",
				NULL ) ) {
		gtk_source_print_job_set_wrap_mode( job, GTK_WRAP_CHAR );
		if( gconf_client_get_bool( client,
					"/apps/screem/editor/print/split",
					NULL ) ) {
			gtk_source_print_job_set_wrap_mode( job, GTK_WRAP_WORD );
		}
	}

	gtk_source_print_job_set_highlight( job, 
			gconf_client_get_bool( client,
				"/apps/screem/editor/print/highlight",
				NULL ) );
	
	if( gconf_client_get_bool( client, "/apps/screem/editor/print/lines",
				   NULL ) ) {
		gfloat lines;

		lines = gconf_client_get_float( client,
					"/apps/screem/editor/print/nlines",
					NULL );
		gtk_source_print_job_set_print_numbers( job, (gint)lines );
	}
	
	gtk_source_print_job_set_header_format( job,
						"Printed on %A",
						NULL,
						"%F",
						TRUE );
	
	pathname = screem_page_get_pathname( page );
	if( ! pathname ) {
		pathname = _( "Untitled" );
	}
	
	gtk_source_print_job_set_footer_format( job,
						"%T",
						pathname,
						"Page %N/%Q",
						TRUE );
	if( gconf_client_get_bool( client, 
				"/apps/screem/editor/print/headers",
				NULL ) ) {
		gtk_source_print_job_set_print_header( job, TRUE );
		gtk_source_print_job_set_print_footer( job, TRUE );
	}

	
	return job;
}

static void screem_editor_print( ScreemView *view, gboolean preview )
{
	ScreemEditor *editor;
	ScreemPage *page;
	GtkTextIter it;
	GtkTextIter eit;
	GtkSourcePrintJob *job;
	
	editor = SCREEM_EDITOR( view );
	
	job = screem_editor_print_setup_job( editor );
	page = editor->private->page;
		
	if( preview ) {
		gtk_text_buffer_get_bounds( GTK_TEXT_BUFFER( page ), &it, &eit );
	} else {
		GtkWidget *dialog;
		GnomePrintConfig *config;
		gint lines;
		gint res;
	
		gint first;
		gint last;
		GnomePrintRangeType type;
		
		config = gtk_source_print_job_get_config( job );
		
		dialog = g_object_new( GNOME_TYPE_PRINT_DIALOG, 
					"print_config", config, NULL );
	
		gnome_print_dialog_construct( GNOME_PRINT_DIALOG( dialog ),
					      _( "Screem - Print Document " ),
					      GNOME_PRINT_DIALOG_RANGE |
					      GNOME_PRINT_DIALOG_COPIES );

		lines = gtk_text_buffer_get_line_count( GTK_TEXT_BUFFER( page ) );
		gnome_print_dialog_construct_range_page( GNOME_PRINT_DIALOG( dialog ),
							 GNOME_PRINT_RANGE_ALL,/* ||
							 GNOME_PRINT_RANGE_RANGE, */
						 	1, lines, "A", "Lines" );

		res = gtk_dialog_run( GTK_DIALOG( dialog ) );
		switch( res ) {
			case GNOME_PRINT_DIALOG_RESPONSE_PRINT:
				preview = FALSE;
				break;
			case GNOME_PRINT_DIALOG_RESPONSE_PREVIEW:
				preview = TRUE;
				break;
			default:
				gtk_widget_destroy( dialog );
				g_object_unref( job );
				return ;
				break;
		}

		type = gnome_print_dialog_get_range( GNOME_PRINT_DIALOG( dialog ) );
		gnome_print_dialog_get_range_page( GNOME_PRINT_DIALOG( dialog ),
						   &first, &last );

		gtk_text_buffer_get_bounds( GTK_TEXT_BUFFER( page ), &it, &eit );

		gtk_widget_destroy( dialog );
	}
	g_object_set_data( G_OBJECT( job ), "preview", 
			   GINT_TO_POINTER( preview ) );

	if( gtk_source_print_job_print_range_async( job, &it, &eit ) ) {
		g_signal_connect( job, "begin_page", 
				  G_CALLBACK( screem_editor_print_page_cb ),
				  editor );
		g_signal_connect( job, "finished",
				  G_CALLBACK( screem_editor_print_finish_cb ),
				  editor );
	}
}

/* static stuff */
static void screem_editor_display( ScreemView *view )
{
	ScreemPage *page;

	g_object_get( G_OBJECT( view ), "page", &page, NULL );

	screem_editor_display_page( SCREEM_EDITOR( view ), page );

	if( page ) {
		g_object_unref( page );
	}
}

static void screem_editor_display_page( ScreemEditor *editor, 
					ScreemPage *page )
{
	ScreemEditorPrivate *priv;
	gint ppos;
	GtkTextMark *mark;
	GtkTextIter it;
	GtkTextBuffer *buffer;
	
	gboolean active;
	GdkWindow *window;
	GdkCursor *cursor;
	GdkDisplay *display;

	priv = editor->private;
	g_object_set_data( G_OBJECT( editor ), "ppos",
			GINT_TO_POINTER( -1 ) );

	window = gtk_text_view_get_window( GTK_TEXT_VIEW( priv->view ), 
			GTK_TEXT_WINDOW_TEXT );
	if( page ) {
		active = TRUE;
		if( window ) {
			display = gtk_widget_get_display( priv->view );
			cursor = gdk_cursor_new_for_display( display,
					GDK_XTERM );
			gdk_window_set_cursor( window, cursor );
			gdk_cursor_unref( cursor );
		}
		screem_page_set_lang( page );
	} else {
		active = FALSE;
		if( window ) {
			gdk_window_set_cursor( window, NULL );
		}
	}
	
	ppos = screem_editor_get_pos( editor );
	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( priv->view ) );
	gtk_text_view_set_buffer( GTK_TEXT_VIEW( priv->view ),
				  NULL );
	gtk_text_view_set_editable( GTK_TEXT_VIEW( priv->view ),
				    FALSE );
	
	
	if( priv->page ) {
		/* set mark for cursor position */
		gtk_text_buffer_get_iter_at_offset( GTK_TEXT_BUFFER( priv->page ), &it, ppos );
		gtk_text_buffer_create_mark( GTK_TEXT_BUFFER( priv->page ), 
				priv->markname, &it, TRUE );
		
		/* disconnect from page signals */
		g_signal_handlers_disconnect_matched( G_OBJECT( priv->page ),
						      G_SIGNAL_MATCH_DATA,
						      0, 0, NULL, NULL, editor );
	}
	priv->page = NULL;

	if( page ) {
		/* we need to know if the page changes buffer, so
		 * the editor knows it needs to get the new one, this
		 * happens on pathname or mime type change */
		g_signal_connect( G_OBJECT( page ), "notify::pathname",
				  G_CALLBACK( screem_editor_buffer_change ),
				  editor );
		g_signal_connect( G_OBJECT( page ), "notify::mime-type",
				  G_CALLBACK( screem_editor_buffer_change ),
				  editor );

		g_signal_connect_data( G_OBJECT( page ), "mark_set",
			G_CALLBACK( screem_editor_mark_set ),
			editor, NULL, 
			G_CONNECT_AFTER );
		
		gtk_text_view_set_buffer( GTK_TEXT_VIEW( priv->view ),
					  GTK_TEXT_BUFFER( page ) );
		
		gtk_text_view_set_editable( GTK_TEXT_VIEW( priv->view ),
					    TRUE );
		buffer = GTK_TEXT_BUFFER( page );
		priv->page = page;
	} else {
		/* yes we did set this to NULL, but that means we
		   get a newed one back so we can actually clear the display
		   it will be unrefed when we set it to a proper one later */
		buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( priv->view ) );

		gtk_text_view_set_buffer( GTK_TEXT_VIEW( priv->view ),
				buffer );
	}


	mark = gtk_text_buffer_get_mark( buffer, priv->markname );
	ppos = 0;
	if( mark ) {
		gtk_text_buffer_get_iter_at_mark( buffer, &it, mark );
		ppos = gtk_text_iter_get_offset( &it );
	}
	screem_editor_set_pos_delayed( editor, ppos, FALSE, 0.0 );
}

static gboolean screem_editor_motion( GtkWidget *widget, 
				      GdkDragContext *context, 
				      gint x, gint y, guint time, 
				      ScreemEditor *editor )
{
	GdkDragAction action;
       
	GdkModifierType modifiers;
        gdk_window_get_pointer (NULL, NULL, NULL, &modifiers);

	
        if( context->suggested_action != GDK_ACTION_ASK )
                action = GDK_ACTION_COPY;
        else
                action = GDK_ACTION_ASK;

	if ((modifiers & GDK_MOD1_MASK) != 0)
                action = GDK_ACTION_ASK;
        
        gdk_drag_status( context, action, time );

        return TRUE;
}

static void screem_editor_drop( GtkWidget *widget, 
		GdkDragContext *context,
		gint x, gint y, 
		GtkSelectionData *selectionData,
		guint info, guint time, ScreemEditor *editor )
{
	ScreemEditorPrivate *priv;
	ScreemPage *page;
	GtkWidget *popup;
	gint item;
	gint pos;
	DropActions action;

	const gchar *pathname;

	gint bx;
	gint by;
	GtkTextIter it;

	gboolean handled;
	GdkAtom *targets;
	GdkAtom target;
	guint n_targets;
	gint i;

	GtkUIManager *merge;
	
	handled = FALSE;

	priv = editor->private;
	page = priv->page;

	/* stop gtksourceview acting upon the drop */
	g_signal_stop_emission_by_name( G_OBJECT( priv->view ),
			"drag_data_received" );

	if( ! page ) {
		gtk_drag_finish( context, FALSE, FALSE, time );
		return;
	}
			
	pathname = screem_page_get_pathname( page );
	g_object_get( G_OBJECT( editor ), 
			"uimanager", &merge,
			NULL );

	/* if we are doing a right drag then we need to ask the user
	   what to do */
	if( context->action == GDK_ACTION_ASK ) {
	       	popup = gtk_ui_manager_get_widget( GTK_UI_MANAGER( merge ),
					"/editor_dnd_menu" );
	
		g_object_set_data( G_OBJECT( editor ), "dnd_popup_item",
			GINT_TO_POINTER( -1 ) );
                screem_popup_menu_do_popup_modal( popup, 0, 0, 0, 0, 
				priv->view );
		item = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( editor ), "dnd_popup_item" ) );
                switch( item ) {
                case 0: /* insert relative filename */
                        action = RELATIVE_PATH;
                        break;
                case 1: /* insert complete filename */
                        action = FULL_PATH;
                        break;
                case 2: /* insert tag */
                        action = TAG;
                        break;
                case 3: /* insert tag attribute */
                        action = ATTRIBUTE;
                        break;
                case 4: /* insert text inline */
                        action = INLINE;
                        break;
                default:
			gtk_drag_finish( context, FALSE, FALSE, time ); 
			g_object_unref( merge );
			return;
                        break;
                }
        } else {
                action = RELATIVE_PATH;
	}
	g_object_unref( merge );

	/* position the cursor at the drop location */
	gtk_text_view_window_to_buffer_coords(GTK_TEXT_VIEW(priv->view ),
					      GTK_TEXT_WINDOW_WIDGET,
					      x, y, &bx, &by );
	gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(priv->view),
					   &it, bx, by);
	pos = gtk_text_iter_get_offset( &it );

	screem_editor_set_pos( editor, pos );
	priv->action = action;

	/* we prefer text/html if it is available */
	n_targets = 0;
	if( ! gtk_selection_data_get_targets( selectionData, &targets,
			&n_targets ) ) {
		n_targets = 0;
	}
	
	for( i = 0; i < n_targets; ++ i ) {
		target = targets[ i ];
		if( target == gdk_atom_intern( HTML_ATOM_NAME, FALSE ) ) {
			handled = gtk_selection_convert( widget,
					selectionData->selection,
					target, time );
			break;
		}
	}
	if( ! handled ) {
		screem_editor_selection_received( editor,
				selectionData, time, NULL );
	}
	priv->action = DEFAULT_ACTION;
	
	gtk_drag_finish( context, TRUE, 
			( context->action == GDK_ACTION_MOVE ), time );
}

static void screem_editor_selection_received( ScreemEditor *editor,
		GtkSelectionData *sel, guint time,
		gpointer data )
{
	ScreemEditorPrivate *priv;
	ScreemPage *page;
	DropActions action;

	GdkAtom target;
	GtkTextBuffer *buffer;
	GtkTextMark *mark;
	gboolean encode;

	guint16 *colours;
	guchar *text = NULL;
	gchar *temp;
	gchar *tmp;
	guint pos;

	guint len;
	
	guint start;
	guint end;
	
	priv = editor->private;
	page = priv->page;
	action = priv->action;
	
	buffer = GTK_TEXT_BUFFER( page );
	encode = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( editor ),
				"encode" ) );
	target = sel->target;

	if( sel->length < 0 ) {
		g_warning( "ARRRRg\n" );
		return;
	}

	pos = screem_editor_get_pos( editor );

	/* clipboard from mozilla prepends a zero width
	 * non breaking space, so we can check for that,
	 * drag and drop doesn't do this, so try and validate
	 * as utf8 and if that fails assume utf16,
	 *
	 * also check if strlen( data ) + 1 < length, and treat that
	 * as utf16 as well, otherwise we end up with drops of
	 * "<" for instance when dropping from mozilla.
	 * We do +1 as abiword doesn't give the correct length */
	len = sel->length;
	if( ((const gunichar2*)sel->data)[ 0 ] == 0xFEFF ||
	     ! g_utf8_validate( sel->data, -1, NULL ) ||
	     ( strlen( sel->data ) + 1 < len ) ) {
		text = sel->data;
		if( ((const gunichar2*)text)[ 0 ] == 0xFEFF ) {
			text += 2;
			len -= 2;
		}
		/* len / 2 as utf16 uses 2 bytes per char */
		text = g_utf16_to_utf8( (const gunichar2*)text, 
					len / 2,
					NULL, NULL, NULL );
		len = strlen( text );
	} else {
		text = g_strndup( sel->data, len );
	}

	if( encode ) {
		tmp = text;
		text = screem_markup_encode_text( tmp );
		g_free( tmp );
	}
	
	/* handle the different drop types */
	if( target == gdk_atom_intern( URI_LIST_ATOM_NAME, FALSE ) ||
	    target == gdk_atom_intern( MOZ_URL_ATOM_NAME, FALSE ) ||
	    target == gdk_atom_intern( NETSCAPE_ATOM_NAME, FALSE ) ||
	    target == gdk_atom_intern( HTTP_ATOM_NAME, FALSE ) ||
	    target == gdk_atom_intern( FTP_ATOM_NAME, FALSE ) ) {
		screem_editor_drop_uris( editor, action, text, target );
		g_free( text );
		text = NULL;
	} else if( target == gdk_atom_intern( HTML_ATOM_NAME, FALSE ) ||
		   target == gdk_atom_intern( UTF8_ATOM_NAME, FALSE ) ||
		   target == gdk_atom_intern( PLAIN_ATOM_NAME, FALSE ) ||
		   target == gdk_atom_intern( UNICODE_ATOM_NAME, FALSE ) ) {
		/* */
	} else if( target == gdk_atom_intern( STRING_ATOM_NAME, FALSE ) ) {
		text = screem_support_charset_convert( sel->data,
				sel->length, NULL );
		if( encode ) {
			tmp = screem_markup_encode_text( text );
			g_free( text );
			text = tmp;
		}	
	} else if( target == gdk_atom_intern( CTEXT_ATOM_NAME, FALSE ) ) {
		text = gtk_selection_data_get_text( sel );
		if( encode ) {
			tmp = screem_markup_encode_text( text );
			g_free( text );
			text = tmp;
		}
	} else if( target == gdk_atom_intern( COLOR_ATOM_NAME, FALSE ) ) {
		colours = (guint16*)sel->data;
                temp = g_strdup_printf( "#%.2x%.2x%.2x",colours[ 0 ] >> 8, 
                                        colours[ 1 ] >> 8, colours[ 2 ] >> 8 );
		switch( action ) {
		case TAG:
                        text = g_strdup_printf( "<font color=\"%s\"> </font>",
                                                temp );
                        g_free( temp );
                        break;
		case ATTRIBUTE:
			text = screem_page_get_data( page );
                        if( screem_markup_is_tag( text, pos, NULL, NULL ) ) {
                                /* we are inside a tag */
                                g_free( text );
                                text = g_strdup_printf( " color=\"%s\"", temp );
                                screem_editor_insert_attribute( editor, text );
                                g_free( text );
                                return;
                        } else {
                                /* not in a tag, but we will insert it anyway*/
                                g_free( text );
                                text = g_strdup_printf( " color=\"%s\"", temp );
                                g_free( temp );
                        }
                        break;
		default:
			text = temp;
			break;
		}
	} else {
		/* we don't know about this type, and as such
		   this case should never execute */
		g_free( text );
		text = NULL;
	}
	if( text ) {
		if(screem_editor_has_selection( editor, &start, 
					&end ) ) {
			screem_editor_delete_forward( editor, 
					start, end - start );
			pos = screem_editor_get_pos( editor );
		}
		screem_editor_insert( editor, pos, text );
		g_free( text );
		mark = gtk_text_buffer_get_insert( buffer );
		gtk_text_view_scroll_mark_onscreen( GTK_TEXT_VIEW( editor->private->view ), mark );
	}
}

static gboolean screem_editor_keypress( GtkWidget *widget, GdkEventKey *event, 
					ScreemEditor *editor )
{
	ScreemPage *page;

	const gchar *pathname;

	GList *list;
	GModule *self;

	gchar *string;
	ScreemEditorMacro *macro;
	gint pos;

	gboolean press_handled;

	ScreemEditorPrivate *private;
	ScreemApplication *application;
	GConfClient *client;

	GtkTextBuffer *buffer;
	gboolean feature_markup;

	const gchar *mime_type;
	
	private = editor->private;
	client = private->client;

	page = private->page;

	if( ! page )
		return FALSE;

	press_handled = FALSE;

	g_object_get( G_OBJECT( editor ), 
			"app", &application,
			NULL );

	pathname = screem_page_get_pathname( page );

	self = g_module_open( NULL, G_MODULE_BIND_LAZY );

	/* does string match any of the editor keys set by the user? */
	string = NULL;
	if( application->macros ) {
		string = convert_keysym_state_to_string( event->keyval, 
							 event->state );
	}
	macro = NULL;

	if( string ) {
		for( list = application->macros; list; 
				list = list->next ) {
			macro = (ScreemEditorMacro*)list->data;
			if( ! strcmp( string, macro->key_name ) ) {
				if( macro->action == TEXT ) {
					pos = screem_editor_get_pos( editor );
					screem_editor_insert( editor, pos, 
							      macro->text );
					pos += strlen( macro->text );
					screem_editor_set_pos( editor, pos );
					g_free( string );
					g_object_unref( application );
					return TRUE;
				}
				break;
			} else {
				macro = NULL;
			}
		}
		g_free( string );
	}

	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( private->view ) );
	feature_markup = screem_page_is_markup( page );

	if( feature_markup ) {
		press_handled = html_key_press( editor, event, NULL );
	}
	/* handle hiding tooltip */
	if( ! press_handled && 
		event->keyval == GDK_Escape &&
		GTK_WIDGET_VISIBLE( private->functip->tip_window ) ) {
		press_handled = TRUE;
		gtk_widget_hide( private->functip->tip_window );
		if( private->tiphide ) {
			g_source_remove( private->tiphide );
			private->tiphide = 0;
		}
	}
	
	/* offer tag tree auto completion */
	mime_type = screem_page_get_mime_type( page );
	if( private->autocomplete &&
		event->string && *event->string && ! press_handled ) {
		GSList *comp;
		gchar *tmp;
		gchar *word;
		gint pos;
		guint len;
		
		pos = screem_editor_get_pos( editor );
		word = screem_editor_get_word( editor, pos );

		comp = NULL;
		tmp = NULL;
		len = 0;
		if( word ) {
			len = strlen( word );
		}
		if( len > 2 ) {
			/* get completions */
			tmp = g_strdup_printf( "%s%s", 
					word, event->string );
			comp = screem_tag_tree_autocomplete( mime_type, tmp, FALSE );
		}
		g_free( word );
		
		screem_editor_autocomplete_popup( editor, comp, 0, tmp, NULL );
		g_free( tmp );
		if( comp ) {
			g_slist_free( comp );
		}

	}
	
	g_object_unref( application );
	
	return press_handled;
}

gboolean html_key_press( ScreemEditor *editor, GdkEventKey *event, 
			 ScreemEditorMacro *macro )
{
	gchar *text;
	gchar *tag = NULL;
	gchar *tag2;
	const gchar *tmp;
	const gchar *tmp2;
	gchar *format;
	gint pos;
	gint start;
	gint len;
	gint len2;
	gboolean empty = FALSE;
	ScreemPage *page;
	gchar *charset;
	gboolean isent;
	
	ScreemEditorPrivate *private;
	GConfClient *client;
	ScreemDTD *dtd;

	gint entinsert;
	const ScreemDTDElement *elem;
	const GSList *list;
	GSList *tlist;
	gboolean sel;
	gint end;

	guint tab_width;

	/* we don't want to do any checking if they only key being
	pressed is a modifier, or cursor keys.
	this should speed things up a little, as we avoid checking
	the shift on its own when a < is being typed etc. */
	if( ( event->keyval >= GDK_Shift_L &&
		event->keyval <= GDK_Hyper_R ) || 
		( event->keyval >= GDK_Home &&
		  event->keyval <= GDK_End ) )	{
			return FALSE;
	}

	private = editor->private;
	client = private->client;

	tab_width = gtk_source_view_get_tabs_width( GTK_SOURCE_VIEW( private->view ) );

	pos = screem_editor_get_pos( editor );
	page = private->page;

	text = screem_page_get_data( page );

	/* Are we in an inline for scripting languages, 
	   ie  <? ?>  <% %> <?php ?> <!-- --> if so we don't want
	   any features, apart from possibly auto indent,
	   
	   This doesn't work very well, for instance if the php
	   contains < and > then screem_markup_is_tag() will think
	   it has a different tag
	   */
	if( ( screem_markup_is_tag( text, pos, &start, NULL ) ) &&
	    event->keyval != GDK_Tab ) {
		tmp = g_utf8_offset_to_pointer( text, start );
		tmp2 = g_utf8_next_char( tmp );
	    	if( strncmp( "<?xml ", tmp, strlen( "<?xml " ) ) &&
		    ( *tmp2 == '?' || *tmp2 == '%' || *tmp2 == '!' ) ) {
			g_free( text );
			return FALSE;
		}
	}

	switch( event->keyval ) {
	case GDK_Return:
		g_free( text );
		if( gconf_client_get_bool( client, "/apps/screem/editor/auto_indent", NULL ) ) {
			screem_editor_insert( editor, pos, "\n" );
			pos ++;
			screem_editor_auto_indent( editor, pos );
			g_signal_stop_emission_by_name(G_OBJECT(private->view),
						       "key_press_event" );
			return TRUE;
		}
		break;
	case GDK_space:
		if( ( ! gconf_client_get_bool( client, "/apps/screem/editor/inline_tagging", NULL ) ) ||
		    ( ! screem_markup_is_tag( text, pos, &start, NULL ) ) ||
		      ( screem_markup_is_attribute( text, pos, NULL, NULL ) ) ) {
			g_free( text );
			return FALSE;
		}

		if( start > 0 ) {
			start --;
		}
		tag = screem_markup_next_tag(text, start, &start, NULL, &tag2);
		if( ! tag ) {
			return FALSE;
		}
		len = strlen( tag );

		len2 = strlen( tag2 );
		g_free( text );

		if( pos <= start + len2 ) {
			/* not in the tag, or space pressed
			   in the tag name */
			g_free( tag2 );
			g_free( tag );
			return FALSE;
		}
		/* if intelliclose is enabled, and the tag can't be
		   closed, and we are an XML document we should put
		   in the closing / part of the tag if it isn't already
		   present */
		dtd = screem_page_get_dtd( page );
		if( gconf_client_get_bool( client,
					   "/apps/screem/editor/intelliclose",
					   NULL  ) ) {
			ScreemDTDTagExistance close;

			close = SCREEM_DTD_MUST;
			if( dtd ) {
				close = screem_dtd_element_close_state( dtd,
									tag2 );
			}
			if( close == SCREEM_DTD_MUST_NOT && 
				screem_page_is_xml( page ) &&
			    tag[ len - 2 ] != '/' ) {
				screem_editor_insert( editor, start + len - 1,
						      "/" );
				screem_editor_set_pos( editor, pos );
			}
		}
		elem = screem_dtd_valid_element( dtd, tag2 );
		list = NULL;
		tlist = NULL;
		if( elem ) {
			list = screem_dtd_element_get_attrs( elem );
			tlist = NULL;
			if( list ) {
				tlist = g_slist_copy( (GSList*)list );
				list = tlist;
			}
			for( ; tlist; tlist = tlist->next ) {
				tlist->data = (gpointer)screem_dtd_attribute_get_name( tlist->data );
			}
			tlist = (GSList*)list;
		}
		screem_editor_autocomplete_popup( editor, list, GDK_equal, NULL,
						  screem_editor_attr_after );
		if( tlist ) {
			g_slist_free( tlist );
		}

		g_free( tag2 );
		g_free( tag );
		break;
	case GDK_slash:
		if( ! gconf_client_get_bool(client,
					    "/apps/screem/editor/intelliclose",
					    NULL ) ||  
		    ( ! screem_markup_is_tag( text, pos, &start, NULL ) )  ) {
			g_free( text );
			return FALSE;
		}
		tmp = g_utf8_offset_to_pointer( text, pos );
		tmp2 = g_utf8_prev_char( tmp );
		if( ( ( start < ( pos - 1 ) ) && *tmp2 != '<' )
		    || *tmp != '>' ) {
			g_free( text );
			return FALSE;
		}
	
		screem_editor_insert( editor, pos, "/" );
		tag2 = screem_page_query_context( page, pos, FALSE,
			TRUE, NULL, NULL, NULL, NULL );

		if( tag2 && 
		    ! screem_markup_next_tag_close( text, tag2, start + 1 ) ) {
			/* we found a tag that needed closing */
			screem_editor_insert( editor, pos + 1, tag2 );
			g_signal_stop_emission_by_name(G_OBJECT(private->view),
						       "key_press_event");
			screem_editor_set_pos( editor, pos + strlen( tag2 ) + 2 );
			g_free( tag2 );
		} else {
			if( tag2 ) {
				tag =g_strdup_printf(_("%s is already closed"),
						     tag2 );
			} else {
				tag=g_strdup_printf(_("no tag needs closing"));
			}
			screem_view_show_message( SCREEM_VIEW( editor ),
						    tag );
			g_free( tag );
			g_free( tag2 );
		}

		g_free( text );
	   		
		return TRUE;
		break;
	case GDK_greater:
	case GDK_less:
	case GDK_ampersand:
	case GDK_quotedbl:
		tag = NULL;
		isent = screem_markup_is_entity( text, pos, NULL, NULL );
		empty = FALSE;
		tmp = g_utf8_offset_to_pointer( text, pos );
		tmp2 = '\0';
		if( pos > 0 ) {
			tmp2 = g_utf8_prev_char( tmp );
		}
		if( isent && pos > 0 ) {
			empty = ( *tmp2 == '&' && *tmp == ';' );
		}
		if( isent ) {
			if( empty ) {
				tag = (gchar*)screem_markup_char_to_ent( event->keyval );
				tag = g_strdup( tag );
			}
		} else if( event->keyval == GDK_ampersand ) {
			tag = g_strdup( "&;" );
		} else if( event->keyval == GDK_quotedbl ) {
			if( *tmp2 == '=' &&
			    ( *tmp == ' ' || *tmp == '>' ||
				*tmp == '/' ) )  {
				tag = g_strdup( "\"\"" );
			}
		} else if( screem_markup_is_attribute( text, pos - 1, NULL, NULL ) ) {
			/* in a tag, we need an entity for <>" */
			tag = (gchar*)screem_markup_char_to_ent( event->keyval );
			tag = g_strdup_printf( "&%s;", tag );
		} else if( event->keyval == GDK_less && *tmp != '>' ) {
			tag = g_strdup( "<>" );
		}
		if( tag ) {
			/* if we have the selection, erase it */
			sel = screem_editor_has_selection( editor, 
					&start, &end );
			if( sel ) {
				screem_editor_delete_forward( editor, start, end - start );
				pos = screem_editor_get_pos( editor );
			}

			screem_editor_insert( editor, pos, tag );
			if( isent && empty ) {
				pos += strlen( tag );
			}
			pos ++;
			screem_editor_set_pos( editor, pos );
			g_free( tag );
		}
		g_free( text );

		if( tag && event->keyval == GDK_less ) {
			/* get list of valid elements,
			 * FIXME: should only offer elements which
			 * are valid in the current context */
			dtd = screem_page_get_dtd( page );
			list = screem_dtd_get_elements( dtd );
			tlist = g_slist_copy( (GSList*)list );
			for( list = tlist; tlist; tlist = tlist->next ) {
				tlist->data = (gpointer)screem_dtd_element_get_name( tlist->data );
			}

			screem_editor_autocomplete_popup( editor, list, GDK_slash, NULL, NULL );

			g_slist_free( tlist );
		}

		
		return ( tag != NULL );
		break;
	case GDK_Tab:
		tag = NULL;
		isent = screem_markup_is_entity( text, pos, NULL, NULL );
		empty = FALSE;
		tmp = g_utf8_offset_to_pointer( text, pos );
		tmp2 = g_utf8_prev_char( tmp );
		if( isent && pos > 0 ) {
			empty = ( *tmp2 == '&' && *tmp == ';' );
		}
		if( isent ) {
			if( empty ) {
				tag = (gchar*)screem_markup_char_to_ent( event->keyval );
				tag = g_strdup( tag );
				screem_editor_insert( editor, pos, tag );
				if( isent && empty ) {
					pos += strlen( tag );
				}
				pos ++;
				screem_editor_set_pos( editor, pos );
				g_free( tag );
			}
		} else if( gconf_client_get_bool( client, "/apps/screem/editor/auto_indent", NULL ) ) {
			screem_editor_auto_indent( editor, pos );
			tag = (gchar*)0x1;
		}
		g_free( text );
		return ( tag != NULL );
		break;
	default:
		/* get entity name for the key press */
		tag = (gchar*)screem_markup_char_to_ent( event->keyval );
		
		if( tag ) {
			tag = g_strdup( tag );
		} else {
			guint32 uni;

			uni = gdk_keyval_to_unicode( event->keyval );
			
			/* we should try an insert the numerical entity
			   if needed here */
			if( uni > 0x7F && g_unichar_isprint( uni ) ) {
				tag = g_strdup_printf( "#%i", uni );
			}
		}
		
		/* are we performing entity insertion? */
                entinsert = gconf_client_get_int( client,
                                                  "/apps/screem/editor/entityinsertion", NULL );
		charset = screem_page_get_charset( page );
		if( entinsert == BY_SET && 
		    ( ! charset || ! g_strcasecmp( "ISO-8859-1", charset ) ) ) {
			entinsert = OFF;
		}
		g_free( charset );
		/* are we in an entity? */
		isent = screem_markup_is_entity( text, pos, NULL, NULL );
		empty = FALSE;
		tmp = g_utf8_offset_to_pointer( text, pos );
		tmp2 = g_utf8_prev_char( tmp );
		if( isent && pos > 0 ) {
			empty = ( *tmp2 == '&' && *tmp == ';' );
		}
	
		/* if we are in an entity:
		   if it is empty enable entity insertion,
		   if not disable entity insertion,
		   also set up the correct format string for insertion */
		if( isent ) {
			if( empty ) {
				entinsert = ON;
			} else {
				entinsert = OFF;
			}
			format = "%s";
		} else if( *tmp == ';' ) {
			format = "&%s";
		} else if( pos > 0 && *tmp2 == '&' ) {
			format = "%s;";
		} else {
			format = "&%s;";
		}
		
		if( entinsert == ON && tag ) {
			gchar *temp;
			
			temp = tag;
			tag = g_strdup_printf( format, tag );
			g_free( temp );
			
			screem_editor_insert( editor, pos, tag );
			pos += strlen( tag );
			if( ! empty ) {
				pos ++;
			}
			screem_editor_set_pos( editor, pos );
		}
		if( tag ) {
			g_free( tag );
		}
		g_free( text );
		return ( entinsert == ON && ( tag != NULL ) );
		break;
	}
	return FALSE;
}

static gboolean screem_editor_tip( ScreemEditor *editor )
{
	ScreemEditorPrivate *private;

	private = editor->private;

	if( private->tooltip_timer ) {
		g_source_remove( private->tooltip_timer );
	}
	private->tooltip_timer = g_timeout_add( 2000,
			(GSourceFunc)screem_editor_tooltip, editor );

	return FALSE;
}

static gboolean screem_editor_tooltip( ScreemEditor *editor )
{
	ScreemEditorPrivate *priv;
	gint pos;
	gchar *text;
	ScreemPage *page;
	const gchar *mime_type;
	GModule *self;
	gboolean handled;

	GdkWindow *win;
	gint x;
	gint y;
	GdkModifierType mask;
	gint bx;
	gint by;
	GtkTextIter it;

	GtkTextBuffer *buffer;
	gboolean feature_markup;

	priv = editor->private;
	page = priv->page;
	
	priv->tooltip_timer = 0;

	if( ! page ) {
		return FALSE;
	}

	gdk_threads_enter();
	
	mime_type = screem_page_get_mime_type( page );

	self = g_module_open( NULL, G_MODULE_BIND_LAZY );

	win = gtk_text_view_get_window( GTK_TEXT_VIEW( priv->view ),
				        GTK_TEXT_WINDOW_TEXT);
	gdk_window_get_pointer( win, &x, &y, &mask );

	gtk_text_view_window_to_buffer_coords(GTK_TEXT_VIEW(priv->view),
					      GTK_TEXT_WINDOW_TEXT,
					      x, y, &bx, &by );
	gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(priv->view),
					   &it, bx, by);
	pos = gtk_text_iter_get_offset( &it );

	text = screem_page_get_data( page );

	handled = FALSE;

	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( priv->view ) );
	feature_markup = screem_page_is_markup( page );

	if( feature_markup ) {
		handled = html_editor_tip( editor, pos, text );
	}

	/* offer tag tree based tips */
	if( ! handled ) {
		GSList *comp;
		GSList *tmp;
		gchar *word;
		gchar *tword;

		/* FIXME: this is a problem, css for example will be
		 * word:, functions are word( etc, checking for
		 * each possible suffix isn't very nice */
		word = screem_editor_get_word( editor, pos );

		comp = NULL;
		if( word ) {
			/* get completions */
			comp = screem_tag_tree_autocomplete( mime_type, word, TRUE );
			if( comp ) {
				tword = g_strconcat( word, "(", NULL );
				tmp = g_slist_find_custom( comp,
						tword,
						(GCompareFunc)
						strcmp_len );
				if( ! tmp ) {
					tmp = g_slist_find_custom( comp,
						word,
						(GCompareFunc)
						strcmp );

				}
				g_free( tword );
				g_free( word );
				if( tmp ) {
					screem_editor_tip_popup( editor,
							pos, tmp->data,
							TRUE );
				}
			}
		}
		
		if( comp ) {
			g_slist_free( comp );
		}
	}
	
	g_free( text );
	
	gdk_threads_leave();
	
	return FALSE;
}

static gboolean html_editor_tip( ScreemEditor *editor,
				 gint pos, gchar *text )
{
	ScreemPage *page;
	ScreemDTD *dtd;
	gchar *tag;
	gchar *name;
	const ScreemDTDElement *element;
	gchar *message;

	page = editor->private->page;
	
	if( screem_markup_is_tag( text, pos, &pos, NULL ) ) {
		tag = screem_markup_next_tag( text, pos, NULL, &pos, &name );
		/* so we can tooltip PHP/ASP etc functions */
		if( tag[ 1 ] == '?' || tag[ 1 ] == '!' ) {
			g_free( name );
			return FALSE;
		}
		/* we have the tag, look it up in the DTD */
		dtd = screem_page_get_dtd( page );
		element = screem_dtd_valid_element( dtd, name );
		if( ! element && *name != '/' ) {
			message = g_strdup_printf( _("%s is not a valid element of this documents DTD" ), name );
			screem_view_show_message( SCREEM_VIEW( editor ),
					message );

			screem_editor_tip_popup( editor, pos, message,
					TRUE );
			g_free( message );
		} else if( *name != '/' ) {
			/* its valid, but is it allowed here? */
			message = g_strdup( "" );

			g_free( message );
		}

		g_free( name );
		g_free( tag );
		return TRUE;
	}

	return FALSE;
}

static void screem_editor_colour_notify( GConfClient *client,
					 guint cnxn_id,
					 GConfEntry *entry,
					 gpointer data )
{
	screem_editor_set_colours( SCREEM_EDITOR( data ) );
}

static void screem_editor_wrap_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data )
{
	ScreemEditor *editor;
	GtkWrapMode wmode;
	
	editor = SCREEM_EDITOR( data );

	wmode = GTK_WRAP_NONE;
	if( entry->value && entry->value->type == GCONF_VALUE_BOOL &&
		gconf_value_get_bool( entry->value ) ) {

		wmode = GTK_WRAP_WORD;
	}
	gtk_text_view_set_wrap_mode( GTK_TEXT_VIEW( editor->private->view ), 
				     wmode);
}

static void screem_editor_tabwidth_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data )
{
	ScreemEditor *editor;
	gfloat width;	
	editor = SCREEM_EDITOR( data );

	if( entry->value && entry->value->type == GCONF_VALUE_FLOAT ) {

		width = gconf_value_get_float( entry->value );
		
		gtk_source_view_set_tabs_width( GTK_SOURCE_VIEW( editor->private->view ), (gint)width );

	}
}

static void screem_editor_autocomplete_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data )
{
	ScreemEditor *editor;
	
	editor = SCREEM_EDITOR( data );

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		editor->private->autocomplete = gconf_value_get_bool( entry->value );
	}
}

static void screem_editor_spaces_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data )
{
	ScreemEditor *editor;
	
	editor = SCREEM_EDITOR( data );

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		editor->private->spaces = gconf_value_get_bool( entry->value );
		gtk_source_view_set_insert_spaces_instead_of_tabs( GTK_SOURCE_VIEW( editor->private->view ), editor->private->spaces );
	}
}
static void screem_editor_margin_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data )
{
	ScreemEditor *editor;
	
	editor = SCREEM_EDITOR( data );

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		gtk_source_view_set_show_margin( GTK_SOURCE_VIEW( editor->private->view ),
										gconf_value_get_bool( entry->value ) );
	}
}

static void screem_editor_margin_col_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data )
{
	ScreemEditor *editor;
	gfloat col;	
	editor = SCREEM_EDITOR( data );

	if( entry->value && entry->value->type == GCONF_VALUE_FLOAT ) {

		col = gconf_value_get_float( entry->value );
		
		gtk_source_view_set_margin( GTK_SOURCE_VIEW( editor->private->view ), 
			(gint)col );
	}
}

static void screem_editor_clip_received( GtkClipboard *clipboard,
		GdkAtom *targets, gint count, gpointer data )
{
	ScreemEditor *editor;
	
	GdkAtom preferred;
	gint i;
	gint j;
	
	static gchar const *fmts[] = {
		HTML_ATOM_NAME,
		NULL
	};
	static gchar const *str_fmts[] = {
		UTF8_ATOM_NAME,
		CTEXT_ATOM_NAME,
		STRING_ATOM_NAME,
		NULL
	};
	
	editor = SCREEM_EDITOR( data );

	preferred = GDK_NONE;
	for( i = 0; fmts[ i ] && preferred == GDK_NONE; ++ i ) {
		GdkAtom atom;

		atom = gdk_atom_intern( fmts[ i ], FALSE );
		for( j = 0; j < count && preferred == GDK_NONE; ++ j ) {
			if( targets[ j ] == atom ) {
				preferred = atom;
			}
		}
	}
	for( i = 0; str_fmts[ i ] && preferred == GDK_NONE; ++ i ) {
		GdkAtom atom;

		atom = gdk_atom_intern( str_fmts[ i ], FALSE );
		for( j = 0; j < count && preferred == GDK_NONE; ++ j ) {
			if( targets[ j ] == atom ) {
				preferred = atom;
			}
		}
	}
	
	if( preferred != GDK_NONE ) {
		gtk_clipboard_request_contents( clipboard,
				preferred, 
				screem_editor_complex_received,
				data );
	}
}

static void screem_editor_text_received( GtkClipboard *clipboard,
					 const gchar *text,
					 gpointer data )
{
	ScreemEditor *editor;
	GtkSelectionData *sel;

	if( ! text ) {
		return;
	}
	
	editor = SCREEM_EDITOR( data );

	sel = g_new0( GtkSelectionData, 1 );
	sel->data = (gchar*)text;
	sel->target = gdk_atom_intern( UTF8_ATOM_NAME, FALSE );
	sel->length = strlen( text );
	sel->format = 8;
	sel->display = gtk_widget_get_display( GTK_WIDGET( data ) );

	screem_editor_selection_received( editor, sel, 
			GDK_CURRENT_TIME, NULL );

	g_free( sel );
}

static void screem_editor_complex_received( GtkClipboard *clipboard,
					 GtkSelectionData *sel,
					 gpointer data )
{
	ScreemEditor *editor;

	editor = SCREEM_EDITOR( data );
	screem_editor_selection_received( editor, sel, 
			GDK_CURRENT_TIME, NULL );
}

static void screem_editor_clip_get_cb( GtkClipboard *clip,
		GtkSelectionData *sel, guint info, 
		ScreemEditor *editor )
{
	GdkAtom html;
	gchar *text;

	html = gdk_atom_intern( HTML_ATOM_NAME, FALSE );

	/* as html is just text we do the same no matter
	 * what the application requested */
	text = editor->private->clipboard;
	gtk_selection_data_set( sel, sel->target, 8, text,
			strlen( text ) );
}

static gint screem_editor_clip_clear_cb( GtkClipboard *clip,
		ScreemEditor *editor )
{
	if( editor->private->set_clip ) {
		editor->private->set_clip = FALSE;
	} else {
		g_free( editor->private->clipboard );
		editor->private->clipboard = NULL;
	}

	return TRUE;
}

static void screem_editor_do_cut_paste( ScreemEditor *editor,
		gboolean cut )
{
	GtkTextBuffer *buffer;
	GtkTextMark *mark;
	GdkDisplay *display;
	GtkClipboard *clip;
	guint start;
	guint end;
	GtkTextIter it;
	GtkTextIter eit;

	gchar *text;
	static GtkTargetEntry const targets[] = {
		{ HTML_ATOM_NAME, 0, 0 },
		{ UTF8_ATOM_NAME, 0, 0 },
		{ CTEXT_ATOM_NAME, 0, 0 },
		{ STRING_ATOM_NAME, 0, 0 }
	};
	guint n_targets = G_N_ELEMENTS( targets );

	
	buffer = 
		gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor->private->view));

	display = gtk_widget_get_display( GTK_WIDGET( editor ) );

	if( screem_editor_has_selection( editor, &start, &end ) ) {
		gtk_text_buffer_get_selection_bounds( buffer,
				&it, &eit );
		text = gtk_text_buffer_get_text( buffer, &it, &eit,
				TRUE );

		if( cut ) {
			gtk_text_buffer_delete_selection( buffer,
					FALSE, TRUE );
		}
		
		mark = gtk_text_buffer_get_insert( buffer );
		gtk_text_view_scroll_mark_onscreen( GTK_TEXT_VIEW( editor->private->view ), mark );
	
		clip = gtk_clipboard_get_for_display( display,
				GDK_SELECTION_CLIPBOARD );
		gtk_clipboard_set_with_owner( clip, targets, n_targets,
				(GtkClipboardGetFunc)screem_editor_clip_get_cb,
				(GtkClipboardClearFunc)screem_editor_clip_clear_cb,
				G_OBJECT( editor ) );
		clip = gtk_clipboard_get_for_display( display,
				GDK_SELECTION_PRIMARY );
		gtk_clipboard_set_with_owner( clip, targets, n_targets,
				(GtkClipboardGetFunc)screem_editor_clip_get_cb,
				(GtkClipboardClearFunc)screem_editor_clip_clear_cb,
				G_OBJECT( editor ) );
		g_free( editor->private->clipboard );
		editor->private->set_clip = TRUE;
		editor->private->clipboard = text;
	}
}

static void screem_editor_toggle_overwrite( GtkTextView *view,
					    ScreemEditor *editor )
{
	editor->private->overwrite = ! editor->private->overwrite;

	g_signal_emit( G_OBJECT( editor ),
			screem_editor_signals[ OVERWRITE ],
			0, editor->private->overwrite );
}

static gboolean screem_editor_drop_uris( ScreemEditor *editor,
					 DropActions action,
					 const gchar *selectionData,
					 GdkAtom target )
{
	gboolean online;
	GList *uris;
	GList *tmp;
	GnomeVFSURI *uri;
	
	ScreemPage *page;
	const gchar *pathname;
	gchar *mime_type;
	gboolean ret;

	guint pos;
	guint i;

	gchar **astext;
	
	g_object_get( G_OBJECT( editor ), "online", &online, NULL );
	page = editor->private->page;

	ret = FALSE;

	pathname = screem_page_get_pathname( page );

	uris = gnome_vfs_uri_list_parse( selectionData );
	astext = g_strsplit( selectionData, "\r\n", -1 );
	
	for( i = 0, tmp = uris; tmp; tmp = tmp->next, ++ i ) {
		gchar *url;
		uri = (GnomeVFSURI*)tmp->data;
		
		url = gnome_vfs_uri_to_string( uri, 0 );
		if( online || gnome_vfs_uri_is_local( uri ) ) {
			mime_type = screem_get_mime_type( url, FALSE );
		} else {
			mime_type = NULL;
		}
		/* INLINE action selected, can only do this,
		   if we have a mime type that is for a text file */
		if( action == INLINE && 
		    screem_page_is_mime_type_page( mime_type ) ) {
			screem_editor_insert_file( editor,
						   url );
			g_free( url );
			url = NULL;
			ret = TRUE;
		} else if( action != FULL_PATH && pathname ) {
			gchar *dir;
			gchar *temp;
			
			dir = g_path_get_dirname( pathname );
			temp = url;
			url = relative_path( url, dir );
			g_free( dir );
				g_free( temp );
		}
		if( action == TAG ) {
			gchar *temp;

			temp = url;
			if( mime_type && ! strncmp(mime_type, "image/", 
						   strlen("image/") ) ) {
				url = g_strdup_printf( "<img src=\"%s\" alt=\"\">", temp );
			} else if( target == gdk_atom_intern( MOZ_URL_ATOM_NAME, FALSE ) && tmp->next ) {
				const gchar *title;

				/* astext should be 1/2 the size of
				   the url list */
				title = strchr( astext[ ( i / 2 ) ],
						'\n' );
				if( title ) {
					title ++;
					if( ! strcmp( title, temp ) ) {
						title = "";
					}
				} else {
					title = "";
				}
				url = g_strdup_printf( "<a href=\"%s\">%s</a>", temp, title );
			} else {
				url = g_strdup_printf( "<a href=\"%s\"> </a>",
						       temp );
			}
			g_free( temp );
		}
		if( url ) {
			pos = screem_editor_get_pos( editor );
			screem_editor_insert( editor, pos, url );
			g_free( url );
			ret = TRUE;
		}
		g_free( mime_type );	
		gnome_vfs_uri_unref( uri );

		/* skip title for text/x-moz-url */
		if( target == gdk_atom_intern( MOZ_URL_ATOM_NAME, 
					FALSE ) && ! ( i % 2 ) ) {
			if( tmp->next ) {
				tmp = tmp->next;
				gnome_vfs_uri_unref( tmp->data );
			}
		}
	}
	g_list_free( uris );
	g_strfreev( astext );

	return ret;
}

static gboolean set_pos( ScreemEditorDelaySetPos *delay )
{
	ScreemEditor *editor;
	GtkTextView *view;

	g_return_val_if_fail( delay != NULL, TRUE );
	g_return_val_if_fail( SCREEM_IS_EDITOR( delay->editor ), TRUE );
	
	editor = delay->editor;
	view = GTK_TEXT_VIEW( editor->private->view );

gdk_threads_enter();
	
	screem_editor_set_pos( delay->editor, delay->pos );
	if( delay->adj ) {
		gtk_adjustment_set_value( view->vadjustment, delay->vadj );
	}
	g_free( delay );

	gdk_threads_leave();
	
	return FALSE;
}

static void screem_editor_set_pos_delayed( ScreemEditor *editor, 
					guint pos, 
					gboolean adj, gdouble val )
{
	ScreemEditorDelaySetPos *delay;

	g_return_if_fail( SCREEM_IS_EDITOR( editor ) );
	
	delay = g_new0(ScreemEditorDelaySetPos, 1 );
	delay->editor = editor;
	delay->pos = pos;
	delay->adj = adj;
	if( adj ) {
		delay->vadj = val;
	}
	g_idle_add_full( G_PRIORITY_DEFAULT_IDLE, 
			 (GSourceFunc)set_pos, delay, NULL );
	g_idle_add_full( G_PRIORITY_DEFAULT_IDLE,
			 (GSourceFunc)screem_editor_focus, 
			 editor, NULL );
}

static void screem_editor_buffer_change( ScreemPage *page, 
		GParamSpec *spec, ScreemEditor *editor )
{
	screem_editor_display_page( editor, page );
}

static void screem_editor_autocomplete_popup_delete( GtkWidget *widget,
						     gpointer data )
{
	gchar *str;
	ScreemEditor *editor;
	AutocompleteAfterFunc after;
	guint pos;

	str = g_object_get_data( G_OBJECT( widget ), "string" );
	after = g_object_get_data( G_OBJECT( widget ), "after" );

	editor = SCREEM_EDITOR( data );
	screem_editor_insert( editor, -1, str );

	if( g_str_has_suffix( str, ")" ))
	{
		pos = screem_editor_get_pos( editor );
		screem_editor_set_pos ( editor, pos - 1 );
		screem_editor_check_func( editor );
	}

	if( after && *str != '\0' ) {
		after( editor );
	}
	
	g_free( str );
}

static gboolean screem_editor_autocomplete_activate( GtkWidget *widget,
						     GtkTreePath *path,
						     GtkTreeViewColumn *coli,
						     gpointer data )
{
	return screem_editor_autocomplete_popup_event( widget, NULL, data );
}

static gboolean screem_editor_autocomplete_popup_event( GtkWidget *widget, 
							GdkEventKey *event,
							gpointer data )
{
	ScreemEditor *editor;
	GdkEventKey child;
	GtkWidget *popup;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreePath *path;
	GtkTreeViewColumn *column;
	gchar *str;
	gchar *temp;

	guint32 uni;
	
	gboolean dosel;
	gboolean ret;
	
	guint32 term;
	gsize len;
	guint pos;

	GtkAccelGroup *accel;
	GtkWindow *window;
	
	editor = SCREEM_EDITOR( data );
	g_object_get( G_OBJECT( editor ), "window", &window, NULL );

	pos = screem_editor_get_pos( editor );
	
	popup = gtk_widget_get_toplevel( widget );
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	str = g_object_get_data( G_OBJECT( widget ), "string" );
	accel = g_object_get_data( G_OBJECT( popup ), "accel" );
	ret = dosel = FALSE;

	term = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( widget ), 
				"term" ) );
	if( ! event ) {
		event = &child;
		event->keyval = GDK_Return;
	} else if( event->keyval == term && *str == '\0' ) {
		gtk_widget_destroy( popup );
		ret = TRUE;
		/* pass term onto the editor */
		child = *event;
		child.window =	gtk_text_view_get_window( GTK_TEXT_VIEW(editor->private->view),
						  	  GTK_TEXT_WINDOW_TEXT );
		gtk_widget_event( editor->private->view, (GdkEvent*)&child );
		return ret;	
	}

	switch( event->keyval ) {
		case GDK_space:
			child = *event;
			child.window =	gtk_text_view_get_window( GTK_TEXT_VIEW(editor->private->view),
							  	  GTK_TEXT_WINDOW_TEXT );
			gtk_widget_event( editor->private->view, (GdkEvent*)&child );
			/* deliberate fall through */
		case GDK_Escape:
			g_free( str );
			g_object_set_data( G_OBJECT( widget ),
					   "string", g_strdup( "" ) );
			gtk_widget_destroy( popup );
			ret = TRUE;
			break;
		case GDK_Up:
			if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
				GtkTreePath *path;

				path = gtk_tree_model_get_path( model, &it );

				if( ! gtk_tree_path_prev( path ) ) {
					gtk_tree_model_get_iter_first( model, &it );
				} else {
					gtk_tree_model_get_iter( model, &it, path );
				}

				gtk_tree_path_free( path );
			} else {
				gtk_tree_model_get_iter_first( model, &it );
			}
			gtk_tree_selection_select_iter( sel, &it );
			path = gtk_tree_model_get_path( model, &it );
			column = gtk_tree_view_get_column( GTK_TREE_VIEW(widget),
							   0 );
			gtk_tree_view_scroll_to_cell( GTK_TREE_VIEW(  widget ),
						      path, column, FALSE, 
						      0.0, 0.0 );
			gtk_tree_path_free( path );
			ret = TRUE;
			break;
		case GDK_Down:
			if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
				gtk_tree_model_iter_next( model, &it );

			} else {
				gtk_tree_model_get_iter_first( model, &it );
			}
			gtk_tree_selection_select_iter( sel, &it );
			path = gtk_tree_model_get_path( model, &it );
			if( path ) {
				column = gtk_tree_view_get_column( GTK_TREE_VIEW(widget),
								0 );
				gtk_tree_view_scroll_to_cell( GTK_TREE_VIEW( widget ),
							      path, column, FALSE, 
							      0.0, 0.0 );
				gtk_tree_path_free( path );
			}
			ret = TRUE;
			break;
		case GDK_Return:
			if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
				GValue value = { 0 };
				temp = str;
				
				/* delete previously inserted text */
				len = g_utf8_strlen( str, -1 );
				if( len > 0 ) {
					screem_editor_delete_forward( editor, pos - len, len );
				}
				gtk_tree_model_get_value( model, &it, 0, &value );
				str = g_strdup( g_value_get_string( &value ) );
				g_value_unset( &value );
				g_object_set_data( G_OBJECT( widget ), "string", str );
				g_free( temp );
			}
			gtk_widget_destroy( popup );
			ret = TRUE;
			break;
		case GDK_BackSpace:
			len = g_utf8_strlen( str, -1 );
			if( len > 0 ) {
				temp = g_malloc0( strlen( str ) );
				g_utf8_strncpy( temp, str, len - 1 );
				g_free( str );
				str = temp;
				g_object_set_data( G_OBJECT( widget ), 
						   "string", str );
				ret = TRUE;
				dosel = TRUE;
				screem_editor_delete_forward( editor, pos - 1, 1 );
			}
			break;
		default:
			uni = gdk_keyval_to_unicode( event->keyval );
			if( screem_find_accel( accel, event ) ) {
				ret = FALSE;
				dosel = FALSE;
				g_free( str );
				g_object_set_data( G_OBJECT( widget ),
						   "string", 
						   g_strdup( "" ) );
				gtk_widget_destroy( popup );
				gtk_accel_groups_activate( G_OBJECT( window ),
						event->keyval,
						event->state );
			} else if( g_unichar_isprint( uni ) ) {
				gchar out[ 6 ];

				len = g_utf8_strlen( str, -1 );
				memset( out, '\0', 6 );
				g_unichar_to_utf8( uni, out );
				temp = str;
				str = g_strconcat( str, out, NULL );
				g_free( temp );
				g_object_set_data( G_OBJECT( widget ), 
						   "string", str );
				ret = TRUE;
				dosel = TRUE;
				screem_editor_insert( editor, pos, out );
			}
			break;
	}
	
	/* possibly need to select a new row in the list view */
	if( dosel ) {
		model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
		if( gtk_tree_model_get_iter_first( model, &it ) ) {
			GValue value = { 0 };
			gboolean selected;
			selected = FALSE;
			do {
				gtk_tree_model_get_value( model, &it, 0, &value );
				if( ! strncmp( str, g_value_get_string( &value ), 
						strlen( str ) ) ) {
					gtk_tree_selection_select_iter( sel, &it );
					path = gtk_tree_model_get_path( model, &it );
					column = gtk_tree_view_get_column( GTK_TREE_VIEW(widget),
									0 );
					gtk_tree_view_scroll_to_cell( GTK_TREE_VIEW( widget ),
								      path, column, FALSE, 
								      0.0, 0.0 );
					gtk_tree_path_free( path );
					g_value_unset( &value );
					selected = TRUE;
					break;
				}
				g_value_unset( &value );
			} while( gtk_tree_model_iter_next( model, &it ) );
			if( ! selected ) {
				g_free( str );
				g_object_set_data( G_OBJECT( widget ),
						   "string", g_strdup( "" ) );
				gtk_widget_destroy( popup );
				ret = TRUE;
			}
		}
	}
	
	return ret;
}

static void screem_editor_autocomplete_popup( ScreemEditor *editor, 
		const GSList *list, guint32 term, const gchar *str,
		AutocompleteAfterFunc after )
{
	GtkWindow *window;
	GtkWidget *popup;
	GtkWidget *sw;
	GtkWidget *widget;
	GtkListStore *store;
	GtkTreeIter it;
	gint winx;
	gint winy;
	gint rheight;
	gint rwidth;
	gint pheight;
	gint ypad;
	guint inum;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
     
	GtkAccelGroup *accel;
	GtkUIManager *merge;
	
	GtkStyle *style;
	
	g_return_if_fail( SCREEM_IS_EDITOR( editor ) );
	
	if( ! list ) {
		return;
	}
	
	inum = g_slist_length( (GSList*)list );

	g_object_get( G_OBJECT( editor ), 
			"window", &window, 
			"uimanager", &merge,
			NULL );

	/* get rect for cursor position */
	winx = winy = 0;
	screem_editor_get_popup_pos( editor, -1, &winx, &winy );
	widget = editor->private->view;
	
	style = gtk_rc_get_style( widget );

	popup = gtk_window_new( GTK_WINDOW_POPUP );
	accel = gtk_ui_manager_get_accel_group( GTK_UI_MANAGER( merge ) );
	g_object_unref( merge );

	g_object_set_data( G_OBJECT( popup ), "accel", accel );
	
	store = gtk_list_store_new( 1, G_TYPE_STRING );

	while( list ) {
		gtk_list_store_append( store, &it );
		gtk_list_store_set( GTK_LIST_STORE( store ), &it, 0, list->data, -1 );
		list = list->next;
	}

	sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
					GTK_POLICY_NEVER,
					GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
						GTK_SHADOW_IN );
	gtk_container_add( GTK_CONTAINER( popup ), sw );
	
	widget = gtk_tree_view_new_with_model( GTK_TREE_MODEL( store ) );

	if( GTK_IS_STYLE( style ) ) {
		gtk_widget_modify_font( widget, style->font_desc );
	}
	
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "completion" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_columns_autosize( GTK_TREE_VIEW( widget ) );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text", 0, NULL );
	gtk_tree_view_set_headers_visible( GTK_TREE_VIEW( widget ), FALSE );
	gtk_tree_view_set_rules_hint( GTK_TREE_VIEW( widget ), TRUE );
	g_object_unref( store );

	gtk_container_add( GTK_CONTAINER( sw ), widget );

	gtk_window_set_transient_for( GTK_WINDOW( popup ), GTK_WINDOW( window ) );
	gtk_window_set_skip_taskbar_hint( GTK_WINDOW( popup ), TRUE );
	gtk_window_set_skip_pager_hint( GTK_WINDOW( popup ), TRUE );
	gtk_window_set_decorated( GTK_WINDOW( popup ), FALSE );
	gtk_window_set_modal( GTK_WINDOW( popup ), TRUE );
	
	gtk_window_move( GTK_WINDOW( popup ), winx, winy ); 

	gtk_window_set_focus( GTK_WINDOW( popup ), widget );

	g_signal_connect( G_OBJECT( widget ), "key_press_event",
			  G_CALLBACK( screem_editor_autocomplete_popup_event ),
			  editor );
	g_signal_connect( G_OBJECT( widget ), "destroy",
			  G_CALLBACK( screem_editor_autocomplete_popup_delete ),
			  editor );
	g_signal_connect_after( G_OBJECT( widget ), "row_activated",
			  G_CALLBACK( screem_editor_autocomplete_activate ),
			  editor );

	if( ! str ) {
		str = "";
	}
	g_object_set_data( G_OBJECT( widget ), "string", g_strdup( str ) );
	g_object_set_data( G_OBJECT( widget ), "term", GINT_TO_POINTER( term ) );
	g_object_set_data( G_OBJECT( widget ), "after", after );
	gtk_widget_show_all( popup );

	gtk_cell_renderer_get_size( renderer, widget, NULL, NULL, NULL, &rwidth, &rheight );

	g_object_get( G_OBJECT( renderer ), "ypad", &ypad, NULL);
	rheight += ypad;

	if( inum < 5 ) {
		pheight = inum * rheight;
	} else {
		pheight = 5 * rheight;
	}

	/* FIXME: scrolledwindow doesn't want to shrink ... */
	gtk_widget_set_size_request( widget, rwidth, pheight);
	gtk_widget_show( widget );
}

static void screem_editor_attr_after( ScreemEditor *editor )
{
	ScreemPage *page;
	GtkTextBuffer *buffer;
	gchar *attr;
	gchar *elem;
	GtkTextIter it;
	GtkTextIter *eit;
	guint pos;

	page = editor->private->page;
	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( editor->private->view ) );
	pos = screem_editor_get_pos( editor );
	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );
	eit = gtk_text_iter_copy( &it );
	if( gtk_text_iter_backward_word_start( &it ) ) {
		attr = gtk_text_buffer_get_text( buffer, &it, eit, TRUE );
		while( gtk_text_iter_backward_char( &it ) &&
			gtk_text_iter_get_char( &it ) != '<' ) {}
	
		elem = NULL;
		if( gtk_text_iter_forward_char( &it ) ) {
			gunichar c;
			gtk_text_iter_free( eit );
			eit = gtk_text_iter_copy( &it );
			do {
				c = gtk_text_iter_get_char( &it );
			} while( c != ' ' && c != '>' && 
				gtk_text_iter_forward_char( &it ) );
			elem = gtk_text_buffer_get_text( buffer, eit, &it, TRUE );
		}	
	} else {
		attr = elem = NULL;
	}
	gtk_text_iter_free( eit );
	
	screem_editor_insert( editor, -1, "=\"\"" );

	pos = screem_editor_get_pos( editor );
	pos --;
	screem_editor_set_pos( editor, pos );

	if( attr ) {
		if( elem ) {
			ScreemDTD *dtd;
			const ScreemDTDAttribute *attribute;
			const GSList *list;
			
			dtd = screem_page_get_dtd( page );

			attribute = screem_dtd_valid_attr( dtd, elem, attr );
			list = NULL;
			if( attribute ) {
				list = screem_dtd_attribute_get_values( attribute );
			}
			screem_editor_autocomplete_popup( editor, list, 0, NULL, NULL );

			g_free( elem );
		}
		
		g_free( attr );
	}

}

static GtkWidget *screem_editor_attribute_menu( ScreemEditor *editor,
					ScreemDTD *dtd, 
					const gchar *element )
{
	GtkWidget *menu;
	GtkWidget *item;
	GtkWidget *subitem;
	GtkWidget *submenu;

	const ScreemDTDElement *el;
	const GSList *list;
	const GSList *values;

	if( ! dtd ) {
		return NULL;
	}

	el = screem_dtd_valid_element( dtd, element );

	if( ! el ) {
		return NULL;
	}

	item = gtk_menu_item_new_with_label( element );
	g_object_set_data( G_OBJECT( item ), "editor", editor );
	gtk_widget_show( item );

	menu = gtk_menu_new();
	gtk_menu_item_set_submenu( GTK_MENU_ITEM( item ), menu );

	list = screem_dtd_element_get_attrs( el );
	for( ; list; list = list->next ) {
		const ScreemDTDAttribute *attr;
		const gchar *name;
		const gchar *default_value;
		
		attr = (const ScreemDTDAttribute*)list->data;
		name = screem_dtd_attribute_get_name( attr );
		default_value = screem_dtd_attribute_get_default( attr );

		subitem = gtk_menu_item_new_with_label( name );
		g_object_set_data( G_OBJECT( subitem ), "editor", editor );
		gtk_widget_show( subitem );
		gtk_menu_shell_append( GTK_MENU_SHELL( menu ), subitem );

		submenu = NULL;
		values = screem_dtd_attribute_get_values( attr );
		if( values || default_value ) {
			submenu = gtk_menu_new();
			gtk_menu_item_set_submenu( GTK_MENU_ITEM( subitem ),
						   submenu );
		}
		for( ; values || default_value; values = values->next ) {
			gchar *val;
			gchar *data;

			if( values ) {
				data = g_strdup( values->data );
			} else {
				data = g_strdup( default_value );
				default_value = NULL;
			}
			val = NULL;
			if( *data == '%' ) {
				val = screem_dtd_lookup_entity( dtd,
								data, 
								TRUE );
			}
			if( ! val ) {
				val = g_strdup( data );
			}
			g_free( data );
			
			if( ! values ) {
				data = g_strdup_printf( "%s (%s)", val,
							_( "Default" ) );
				subitem = gtk_separator_menu_item_new();
				gtk_widget_show( subitem );
				gtk_menu_shell_prepend( GTK_MENU_SHELL( submenu ), subitem );
				subitem = gtk_menu_item_new_with_label( data );
				g_free( data );
			} else {
				subitem = gtk_menu_item_new_with_label( val );
			}
			g_object_set_data( G_OBJECT( subitem ), 
					   "editor", editor );
			gtk_widget_show( subitem );
			
			data = g_strconcat( name, "=\"", val, "\"",
					    NULL );
			g_free( val );
			g_signal_connect_data( G_OBJECT( subitem ),
					       "activate",
					       G_CALLBACK(screem_editor_insert_attr_cb),
					       data,
					       (GClosureNotify)g_free,
					       0 );
			if( values ) {
				gtk_menu_shell_append( GTK_MENU_SHELL(submenu),
						       subitem );
			} else {
				gtk_menu_shell_prepend( GTK_MENU_SHELL( submenu ), subitem );
				break;
			}
		}
	}

	return item;
}

static void screem_editor_insert_allowed( GtkWidget *widget, gpointer data )
{
	ScreemEditor *editor;
	ScreemPage *page;
	const gchar *tag;
	ScreemDTD *dtd;
	ScreemDTDTagExistance state;
	gchar *open;
	gchar *close;
	
	editor = SCREEM_EDITOR( data );
	page = SCREEM_PAGE( gtk_text_view_get_buffer( GTK_TEXT_VIEW( editor->private->view ) ) );
	
	if( page ) {
		dtd = screem_page_get_dtd( page );
		
		tag = (const gchar*)g_object_get_data( G_OBJECT( widget ), 
				"tag" );
		state = SCREEM_DTD_MUST;
		if( dtd ) {
			state = screem_dtd_element_close_state( dtd, tag );
		}

		close = NULL;
		if( state != SCREEM_DTD_MUST_NOT ) {
			close = g_strconcat( "</", tag, ">", NULL );
		} 
		
		if( state == SCREEM_DTD_MUST_NOT &&
				screem_page_is_xml( page ) ) {
			open = g_strconcat( "<", tag, " />", NULL );
		} else {
			open = g_strconcat( "<", tag, ">", NULL );
		}
	
		screem_editor_insert_markup( editor, open, close );

		g_free( open );
		g_free( close );
	}
}

static GtkWidget *screem_editor_allowed_menu( ScreemEditor *editor,
					ScreemDTD *dtd, 
					const gchar *element )
{
	GtkWidget *menu;
	GtkWidget *item;
	GtkWidget *subitem;
	const ScreemDTDElement *el;
	
	item = NULL;

	if( dtd ) {
		el = screem_dtd_valid_element( dtd, element );
		
		if( el ) {
			const GSList *list;
			
			item = gtk_menu_item_new_with_label( _( "Valid Elements") );
			gtk_widget_show( item );

			menu = NULL;

			list = screem_dtd_element_get_allowed( el );

			if( list ) {
				menu = gtk_menu_new();
				gtk_menu_item_set_submenu( GTK_MENU_ITEM(item),
							   menu );
			}
			for( ; list; list = list->next ) {
				subitem = gtk_menu_item_new_with_label( (const gchar*)list->data );
				g_object_set_data( G_OBJECT( subitem ),
						"tag", list->data );
				gtk_widget_show( subitem );
				gtk_menu_shell_append( GTK_MENU_SHELL( menu ), subitem );
				g_signal_connect( G_OBJECT( subitem ),
						"activate",
						G_CALLBACK( screem_editor_insert_allowed ),
						editor );
			}
		}
	}

	return item;
}

static void screem_editor_insert_attr_cb( GtkWidget *item, const gchar *attribute )
{
	ScreemEditor *editor;

	editor = SCREEM_EDITOR( g_object_get_data( G_OBJECT( item ),
						   "editor" ) );
	
	screem_editor_insert_attribute( editor, attribute );
}

void screem_editor_insert_attribute( ScreemEditor *editor, 
		const gchar *attribute )
{
	gint pos;
	gint cpos;
	gint start;
	gchar *insert;
	gchar *text;
	gchar *tag;
	gchar *attr;
	ScreemPage *page;

	gchar *temp;

	page = editor->private->page;
	
	if( ! page || ! attribute )
		return;

        cpos = pos = screem_editor_get_pos( editor );
        text = screem_page_get_data( page );

        if( ! screem_markup_is_tag( text, pos, &pos, NULL ) ) {
		g_free( text );
		return;
        }
        start = pos;
        /* pos == the start of the tag */
        tag = screem_markup_next_tag( text, pos, NULL, &pos, NULL );

  	/* does the tag already have this attribute?
           = match:  [A-Za-z]*=\"[^"]*\"
           other: [A-Za-z]*
        */
      	temp = g_strconcat( " ", attribute, NULL );
        if( ( attr = find_text( temp, "=", NULL, NULL ) ) ) {
                attr = g_strndup( temp, attr - temp );
                insert = g_strdup_printf( "%s=\\\"[^\\\"]*\\\"", attr );
                g_free( attr );
        } else {
                insert = g_strdup( temp );
        }

        /* now we try and replace it */
        if( ! ( attr = find_text( tag, insert, temp, NULL ) ) ) {
                /* we didn't find it so we just insert it */
                g_free( insert );
		/* pos is at the end of the tag, however, we may
		   be dealing with xml, hence we need to insert
		   the attribute before a possible closing / */
		if( text[ pos - 2 ] == '/' ) {
			pos --;
		}
		pos --;
		insert = g_strdup_printf( " %s", attribute );
                screem_editor_insert( editor, pos, insert );
		pos += strlen( attribute ) + 1;
	} else {
                /* erase all the tag's attributes */
		g_free( tag );
                tag = attr;
		screem_editor_delete_forward( editor, start, pos - start  );
                insert = g_strdup( tag );
                screem_editor_insert( editor, start, insert );
		pos = start + strlen( insert ) - 1;
        }

        g_free( text );
	g_free( tag );
	g_free( insert );
	g_free( temp );

	editor->private->inserted = TRUE;

	screem_editor_set_pos( editor, pos );
}

static gboolean predicate( gunichar ch, gpointer data )
{
	return ( ch == (gunichar)data );
}

static void screem_editor_learn_more( GtkWidget *widget,
				ScreemEditor *editor )
{
	ScreemEditorPrivate *priv;
	guint pos;
	GtkTextIter it;
	GtkTextIter eit;
	GtkTextBuffer *buffer;
	GtkTextTagTable *table;
	GtkTextTag *tag;
	gchar *txt;
	gchar *tmp;
	gchar *tmp2;
	gchar *attr;
	ScreemDTD *dtd;
	const gchar *publicid;
	gchar *url;
	
	priv = editor->private;
	pos = screem_editor_get_pos( editor );
	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( priv->view ) );
	table = gtk_text_buffer_get_tag_table( buffer );
	tag = gtk_text_tag_table_lookup( table, 
			SCREEM_INVALID_MARKUP_TAG );
	
	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );
	eit = it;

	gtk_text_iter_backward_to_tag_toggle( &it, tag );
	gtk_text_iter_forward_to_tag_toggle( &eit, tag );

	/* it and eit are now the invalid part, this may be
	 * a full tag, or just an attribute in a tag */
	tmp = gtk_text_buffer_get_text( buffer, &it, &eit, TRUE );

	if( *tmp != '<' ) {
		/* txt is an attribute, we want the tag name as well */
		gtk_text_iter_backward_find_char( &it, predicate, 
						GUINT_TO_POINTER( '<' ),
						NULL );
		gtk_text_iter_forward_char( &it );
		eit = it;
		gtk_text_iter_forward_word_end( &eit );
		attr = tmp;
		txt = gtk_text_buffer_get_text( buffer, &it, &eit, TRUE );
	} else {
		txt = g_utf8_strchr( tmp, -1, ' ' );
		tmp2 = g_utf8_strchr( tmp, -1, '>' );
		if( ( ! txt ) || ( tmp2 < txt ) ) {
			txt = tmp2;
			if( ! txt ) {
				txt = tmp + strlen( tmp ) - 1;
			}
		}
		txt = g_strndup( tmp + 1, txt - tmp - 1 );
		g_free( tmp );
		attr = NULL;
	}

	dtd = screem_page_get_dtd( SCREEM_PAGE( buffer ) );
	publicid = screem_dtd_get_public_id( dtd );

	/* txt == tag name, attr = attribute name or NULL,
	 * publicid is the DOCTYPE public identifier in use */
	url = g_strconcat( "http://www.screem.org/invalid.php?",
			"doctype=", publicid,
			"&tag=", txt,
			"&attr=", attr,
			NULL );
	
	gnome_url_show( url, NULL );
	
	g_free( url );
	g_free( txt );
	g_free( attr );
}

static void screem_editor_get_popup_pos( ScreemEditor *editor, gint pos,
		gint *x, gint *y )
{
	GtkWidget *widget;
	GtkTextBuffer *buffer;
	gchar *word;
	GtkTextIter tit;
	GdkRectangle rect;
	GdkWindow *gwindow;
	gint winx;
	gint winy;
	gint orgx;
	gint orgy;
	
	g_assert( x && y );
		
	/* get rect for cursor position */
	widget = editor->private->view;
	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( widget ) );
	if( pos == -1 ) {
		pos = screem_editor_get_pos( editor );
	}
	word = screem_editor_get_word( editor, pos );
	if( word ) {
		if( strcmp( word, ">" ) == 0 ) {
			pos++;
		} else {
			pos -= strlen( word );
		}
	}
	gtk_text_buffer_get_iter_at_offset( buffer, &tit, pos );
	gtk_text_view_get_iter_location( GTK_TEXT_VIEW( widget ), 
			&tit, &rect );
	
	gtk_text_view_buffer_to_window_coords( GTK_TEXT_VIEW( widget ),
			GTK_TEXT_WINDOW_WIDGET, 
			rect.x, rect.y + rect.height, &winx, &winy );
	
	gwindow = gtk_text_view_get_window( GTK_TEXT_VIEW( widget ),
			GTK_TEXT_WINDOW_WIDGET );
	gdk_window_get_origin( gwindow, &orgx, &orgy );

	*x = orgx + winx;
	*y = orgy + winy;
	
	g_free( word );
}

static gboolean timed_hide( gpointer data )
{
	ScreemEditor *editor;
	ScreemEditorPrivate *priv;

	editor = SCREEM_EDITOR( data );
	priv = editor->private;
	
	gtk_widget_hide( priv->functip->tip_window );
	priv->funcpos = 0;
	
	return FALSE;
}

static void screem_editor_tip_popup( ScreemEditor *editor, gint pos,
		const gchar *message, gboolean hide )
{
	GConfClient *client;
	ScreemEditorPrivate *priv;
	GtkTooltips *tip;
	GtkWindow *window;
	gint x;
	gint y;
	
	client = gconf_client_get_default();
	
	if( message && gconf_client_get_bool( client,
				"/apps/screem/editor/tooltips",
				NULL ) ) {
		priv = editor->private;
		tip = priv->functip;
		
		g_object_get( G_OBJECT( editor ), 
			"window", &window, NULL );

		x = y = 0;
		screem_editor_get_popup_pos( editor, pos, &x, &y );
	
		gtk_label_set_text( GTK_LABEL( tip->tip_label ),
				message );

		gtk_window_move( GTK_WINDOW( tip->tip_window ), x, y );
		gtk_widget_show_all( tip->tip_window );

		if( priv->tiphide ) {
			g_source_remove( priv->tiphide );
			priv->tiphide = 0;
		}
		if( hide ) {
			priv->tiphide = g_timeout_add( 2000, 
					timed_hide, editor ); 
		}
	}

	g_object_unref( client );
}

static gint strcmp_len( gconstpointer a, gconstpointer b )
{
	return strncmp( a, b, strlen( b ) );
}


static gint screem_editor_get_func( ScreemEditor *editor, gchar **func )
{
	ScreemEditorPrivate *priv;
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter sit;
	gunichar c;
	gunichar sc;
	gint pos;
	gint funccount;
	gint spos;
	gboolean singleline;
	gboolean lbr;
	
	gint ret;
	
	funccount = 0;
	lbr = FALSE;

	priv = editor->private;
	if( ! priv->page ) {
		return 0;
	}
	
	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( priv->view ) );
	pos = screem_editor_get_pos( editor );
	
	gtk_text_buffer_get_iter_at_offset( buffer, &it, pos );
	c = gtk_text_iter_get_char( &it );
	
	if( c == 0 || c == '\n' || c == '\r') {
		gtk_text_iter_backward_char( &it );
		c = gtk_text_iter_get_char( &it );
		if( c == ')' ) {
			return 0;
		}
		if( c != '(' ) {
			gtk_text_iter_backward_char( &it );
			c = gtk_text_iter_get_char( &it );
		}
	} else {
		gtk_text_iter_backward_char( &it );
		c = gtk_text_iter_get_char( &it );
	}
	
	singleline = screem_page_is_feature( priv->page,
	                                     "single-line-functions" );
	ret = 0;
	spos = 0; /* shut gcc up, can't be used initialised as
		     it is only used when lbr is TRUE, and
		     gets initialised when we set lbr to TRUE */
	while( ! gtk_text_iter_is_start( &it ) &&
	       ( ( singleline && c != '\n' && c != '\r' ) ||
	       ( !singleline && c != ';' ) ) && c != 0 ) {

		if( !lbr && ( c == '\n' || c == '\r' ) ) {
			lbr = TRUE;
			sit = it;
			
			gtk_text_iter_forward_char( &sit );
			sc = gtk_text_iter_get_char( &sit );

			while( g_unichar_isspace( sc ) && ! gtk_text_iter_is_end( &it ) ) {
				gtk_text_iter_forward_char( &sit );
				sc = gtk_text_iter_get_char( &sit );
			}
			
			spos = gtk_text_iter_get_offset( &sit );
		}

		if( c == '(' && funccount == 0 ) {
			gint funcpos;

			funcpos = gtk_text_iter_get_offset( &it );
			funcpos++;

			gtk_text_iter_backward_char( &it );
			c = gtk_text_iter_get_char( &it );
			
			while( g_unichar_isspace( c ) ) {
				gtk_text_iter_backward_char( &it );
				c = gtk_text_iter_get_char( &it );
			}
		
			if( c != 0 && c != ')' ) {
				pos = gtk_text_iter_get_offset( &it );
				*func = screem_editor_get_word( editor,
						pos );
				if( lbr ) {
					ret = spos;
				} else {
					ret = funcpos;
				}
			} 
			break;
		} else if( c == '(' ) {
			funccount --;
		} else if( c == ')' ) {
			funccount ++;
		}
		
		gtk_text_iter_backward_char( &it );
		c = gtk_text_iter_get_char( &it );
	}

	return ret;
}

static gboolean screem_editor_check_func( ScreemEditor *editor )
{
	gchar *func;
	gint funcpos;
	ScreemEditorPrivate *priv;
	GtkTooltips *functip;

	priv = editor->private;
	functip = priv->functip;

	func = NULL;
	funcpos = screem_editor_get_func( editor, &func );
	
	if( funcpos > 0 && priv->funcpos != funcpos) {
		gchar *tmp;
		GSList *comp;
		GSList *tcomp;
		const gchar *mime_type;
		const gchar *tip;

		if( priv->funcpos != 0 ) {
			gtk_widget_hide( functip->tip_window );
			priv->funcpos = 0;
		}
		
		mime_type = screem_page_get_mime_type( priv->page );
		comp = screem_tag_tree_autocomplete( mime_type, 
				func, TRUE );

		tip = NULL;
		if( comp ) {
			tmp = g_strconcat( func, "(", NULL);
			/* use this, we have multiple matches
			 * based on prefix, so if the first one
			 * in the list is fgetss for example we
			 * would never get a tip for fgets as
			 * it wouldn't match comp->data */
			tcomp = g_slist_find_custom( comp, tmp,
						(GCompareFunc)
						strcmp_len );
			g_free( tmp );
			if( tcomp ) {
				tip = tcomp->data;
			}
			g_slist_free( comp );
		}
		if( tip ) {
			screem_editor_tip_popup( editor, funcpos,
					tip, FALSE );
			priv->funcpos = funcpos;
		}
	} else if( priv->funcpos && ! funcpos ) {
		gtk_widget_hide( functip->tip_window );
		priv->funcpos = 0;
	}
	g_free( func );

	priv->tipidle = 0;
	
	return FALSE;
}

static void screem_editor_mark_set( GtkTextBuffer *buffer,
		const GtkTextIter *location, GtkTextMark *mark,
		ScreemEditor *editor )
{
	ScreemEditorPrivate *priv;
	
	if( buffer ) {
		priv = editor->private;
	
		if( priv->tipidle ) {
			g_source_remove( priv->tipidle );
			priv->tipidle = 0;
		}
		priv->tipidle = g_timeout_add( 250,
				(GSourceFunc)screem_editor_check_func,
				editor );
	}
}

/* G Object stuff */

#define PARENT_TYPE SCREEM_TYPE_VIEW

static gpointer parent_class;

static void screem_editor_class_init( ScreemEditorClass *klass )
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = G_OBJECT_CLASS( klass );
	widget_class = (GtkWidgetClass *)klass;
	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_editor_finalize;

	widget_class->show = screem_editor_show;
	widget_class->hide = screem_editor_hide;
	widget_class->size_request = screem_editor_size_request;

	widget_class->grab_focus = focus;

	screem_editor_signals[ OVERWRITE ] = 
		g_signal_new( "overwrite",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemEditorClass, 
					       overwrite),
			      NULL, NULL,
			      screem_marshal_VOID__BOOLEAN,
			      G_TYPE_NONE, 1,
			      G_TYPE_BOOLEAN );
}

static void screem_editor_init( ScreemEditor *editor )
{
	GtkWidget *sw;
	GtkWidget *text;
	GdkPixbuf *pixbuf;
	GtkWrapMode wmode;
	gfloat tabwidth;
	ScreemEditorPrivate *private;
	gint handle;

	SCREEM_VIEW( editor )->window_set = screem_editor_window_set;
	SCREEM_VIEW( editor )->display = screem_editor_display;
	SCREEM_VIEW( editor )->print = screem_editor_print;
	SCREEM_VIEW( editor )->undo = screem_editor_undo;
	SCREEM_VIEW( editor )->redo = screem_editor_redo;

	private = editor->private = g_new0( ScreemEditorPrivate, 1 );

	private->client = gconf_client_get_default();
	gconf_client_add_dir( private->client, "/apps/screem/editor",
			      GCONF_CLIENT_PRELOAD_ONELEVEL, NULL );

	private->popup_open = FALSE;
	private->inserted = FALSE;
	private->tooltip_timer = 0;
	
	private->view = text = gtk_source_view_new();
	g_signal_connect( G_OBJECT( text ), "realize",
			G_CALLBACK( screem_editor_realize ),
			editor );
	gtk_source_view_set_show_line_numbers( GTK_SOURCE_VIEW(private->view),
						TRUE );
	gtk_source_view_set_show_line_markers( GTK_SOURCE_VIEW(private->view),
						TRUE );
	tabwidth = gconf_client_get_float( private->client,
			"/apps/screem/editor/tabwidth",
			NULL );
	if( tabwidth == 0.0 ) {
		tabwidth = 8.0;
	}
	gtk_source_view_set_tabs_width( GTK_SOURCE_VIEW( private->view ), (gint)tabwidth );
	gtk_source_view_set_auto_indent( GTK_SOURCE_VIEW( private->view ),
					TRUE );
	pixbuf = gtk_widget_render_icon( GTK_WIDGET( private->view ),
					 "Screem_Bookmark",
					 GTK_ICON_SIZE_MENU, "" );
	gtk_source_view_set_marker_pixbuf( GTK_SOURCE_VIEW( private->view ),
					   BOOKMARK_MARKER,
					   pixbuf );

	wmode = GTK_WRAP_NONE;
	if( gconf_client_get_bool( private->client,
				"/apps/screem/editor/wordwrap",
				NULL ) ) {
		wmode = GTK_WRAP_WORD;
	}
	gtk_text_view_set_wrap_mode( GTK_TEXT_VIEW( text ), 
				     wmode );
	gtk_text_view_set_editable( GTK_TEXT_VIEW( text ),
				    FALSE );
		
	gtk_widget_show( text );

	sw = private->sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_widget_show( sw );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ), 
                                        GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
						GTK_SHADOW_IN );
	gtk_container_add( GTK_CONTAINER( sw ), text );

	g_signal_connect( G_OBJECT( text ), "button_press_event",
			  G_CALLBACK( screem_editor_popup ), editor );
	
	g_signal_connect( G_OBJECT( text ), "button_release_event",
			  G_CALLBACK( screem_editor_button_release ),
			  editor );

	g_signal_connect( G_OBJECT( text ), "popup_menu",
			  G_CALLBACK( screem_editor_popup_key ), editor );
	
	g_signal_connect( G_OBJECT( text ), "key_press_event",
			  G_CALLBACK( screem_editor_keypress ), editor );
	
	gtk_drag_dest_set( text,
			   GTK_DEST_DEFAULT_MOTION | 
			   GTK_DEST_DEFAULT_HIGHLIGHT,
			   drop_types, num_drop_types,
			   GDK_ACTION_COPY | GDK_ACTION_ASK );
	
	g_signal_connect( G_OBJECT( text ), "drag_data_received",
			  G_CALLBACK( screem_editor_drop ), editor );
	g_signal_connect( G_OBJECT( text ), "drag_motion",
			  G_CALLBACK( screem_editor_motion ), editor );

	g_signal_connect( G_OBJECT( editor->private->view ),
			  "toggle_overwrite",
			  G_CALLBACK( screem_editor_toggle_overwrite ),
			  editor );

	private->autocomplete = gconf_client_get_bool( private->client,
			"/apps/screem/editor/autocomplete",
			NULL );
	
	private->spaces = gconf_client_get_bool( private->client,
			"/apps/screem/editor/spaces_instead_of_tabs",
			NULL );
	gtk_source_view_set_insert_spaces_instead_of_tabs( GTK_SOURCE_VIEW( private->view ), private->spaces );
	
	gtk_source_view_set_show_margin( GTK_SOURCE_VIEW( editor->private->view ),
					gconf_client_get_bool( private->client,
						"/apps/screem/editor/right_margin",	NULL ) );
	gtk_source_view_set_margin( GTK_SOURCE_VIEW( editor->private->view ),
					(gint)gconf_client_get_float( private->client,
						"/apps/screem/editor/margin_column", NULL ) );
	
	handle = gconf_client_notify_add( private->client,
					 "/apps/screem/editor/back",
					 screem_editor_colour_notify,
					 editor, NULL, NULL );
	private->notifies = g_slist_prepend( private->notifies,
							GINT_TO_POINTER( handle ) );
	
	handle = gconf_client_notify_add( private->client,
					 "/apps/screem/editor/text",
					 screem_editor_colour_notify,
					 editor, NULL, NULL );
	private->notifies = g_slist_prepend( private->notifies,
							GINT_TO_POINTER( handle ) );
	
	handle = gconf_client_notify_add( private->client,
					 "/apps/screem/editor/font",
					 screem_editor_colour_notify,
					 editor, NULL, NULL );
	private->notifies = g_slist_prepend( private->notifies,
							GINT_TO_POINTER( handle ) );
	
	handle = gconf_client_notify_add( private->client,
					 "/apps/screem/editor/themecolours",
					 screem_editor_colour_notify,
					 editor, NULL, NULL );
	private->notifies = g_slist_prepend( private->notifies,
							GINT_TO_POINTER( handle ) );
					 
	handle = gconf_client_notify_add( private->client,
					 "/apps/screem/editor/themefont",
					 screem_editor_colour_notify,
					 editor, NULL, NULL );
	private->notifies = g_slist_prepend( private->notifies,
							GINT_TO_POINTER( handle ) );
					 
	handle = gconf_client_notify_add( private->client,
				"/apps/screem/editor/wordwrap",
				screem_editor_wrap_notify,
				editor, NULL, NULL );
	private->notifies = g_slist_prepend( private->notifies,
							GINT_TO_POINTER( handle ) );
				
	handle = gconf_client_notify_add( private->client,
				"/apps/screem/editor/tabwidth",
				screem_editor_tabwidth_notify,
				editor, NULL, NULL );
	private->notifies = g_slist_prepend( private->notifies,
							GINT_TO_POINTER( handle ) );
				
	handle = gconf_client_notify_add( private->client,
				"/apps/screem/editor/autocomplete",
				screem_editor_autocomplete_notify,
				editor, NULL, NULL );
	private->notifies = g_slist_prepend( private->notifies,
							GINT_TO_POINTER( handle ) );
				
	handle = gconf_client_notify_add( private->client,
				"/apps/screem/editor/spaces_instead_of_tabs",
				screem_editor_spaces_notify,
				editor, NULL, NULL );
	private->notifies = g_slist_prepend( private->notifies,
							GINT_TO_POINTER( handle ) );

	handle = gconf_client_notify_add( private->client,
				"/apps/screem/editor/right_margin",
				screem_editor_margin_notify,
				editor, NULL, NULL );
	private->notifies = g_slist_prepend( private->notifies,
							GINT_TO_POINTER( handle ) );

	handle = gconf_client_notify_add( private->client,
				"/apps/screem/editor/margin_column",
				screem_editor_margin_col_notify,
				editor, NULL, NULL );
	private->notifies = g_slist_prepend( private->notifies,
							GINT_TO_POINTER( handle ) );

	g_signal_connect_data( G_OBJECT( text ),
			       "motion_notify_event",
			       G_CALLBACK( screem_editor_tip ), 
			       editor,
			       NULL, G_CONNECT_SWAPPED );

	gtk_container_add( GTK_CONTAINER( editor ), sw );


	/* clipboard setup */
	gtk_selection_add_target( GTK_WIDGET( editor ),
			GDK_SELECTION_PRIMARY,
			gdk_atom_intern( HTML_ATOM_NAME, FALSE ),
			0 );
	gtk_selection_add_target( GTK_WIDGET( editor ),
			GDK_SELECTION_PRIMARY,
			gdk_atom_intern( UTF8_ATOM_NAME, FALSE ),
			0 );
	gtk_selection_add_target( GTK_WIDGET( editor ),
			GDK_SELECTION_CLIPBOARD,
			gdk_atom_intern( HTML_ATOM_NAME, FALSE ),
			0 );
	gtk_selection_add_target( GTK_WIDGET( editor ),
			GDK_SELECTION_CLIPBOARD,
			gdk_atom_intern( UTF8_ATOM_NAME, FALSE ),
			0 );

	/* tooltip stuff */
	private->functip = gtk_tooltips_new();
	gtk_tooltips_force_window( private->functip );
}

static void screem_editor_finalize( GObject *object )
{
	ScreemEditor *editor;
	ScreemEditorPrivate *private;
	GSList *tmp;
	gint handle;
	
	editor = SCREEM_EDITOR( object );
	private = editor->private;

	g_free( private->markname );
	
	for( tmp = private->notifies; tmp; tmp = tmp->next ) {
		handle = GPOINTER_TO_INT( tmp->data );
		gconf_client_notify_remove( private->client, handle );
	}
	g_slist_free( private->notifies );
	
	g_object_unref( private->client );

	if( private->tooltip_timer ) {
		g_source_remove( private->tooltip_timer );
	}
	if( private->tipidle ) {
		g_source_remove( private->tipidle );
	}
	if( private->tiphide ) {
		g_source_remove( private->tiphide );
	}

	gtk_object_destroy( GTK_OBJECT( private->functip ) );
	
	g_free( private );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static void screem_editor_window_set( ScreemView *view )
{
	ScreemEditor *editor;
	ScreemEditorPrivate *private;
	GError *error;
	GtkUIManager *merge;
	GtkActionGroup *group;

	editor = SCREEM_EDITOR( view );
	private = editor->private;

	g_object_get( G_OBJECT( view ), 
			"uimanager", &merge,
			"actions", &group,
			NULL );
	
	private->markname = g_strdup_printf( "ScreemEditor_%s\n",
					"fixme" );
		
	private->overwrite = TRUE;
	screem_editor_toggle_overwrite( GTK_TEXT_VIEW( private->view ),
					editor );
	
	/* dnd popup */
	gtk_action_group_add_actions( GTK_ACTION_GROUP( group ),
			editor_dnd_actions, editor_dnd_actions_n,
			editor );
	error = NULL;
	if( ! gtk_ui_manager_add_ui_from_string( GTK_UI_MANAGER( merge ),
				editor_dnd_menu, 
				strlen( editor_dnd_menu ),
				&error ) ) {
		g_message( "Editor dnd menu string error = %s", 
				error->message );
		g_error_free( error );
	}

	g_object_unref( merge );
	g_object_unref( group );
}	

static void screem_editor_size_request( GtkWidget *widget, GtkRequisition *req )
{
	GTK_WIDGET_CLASS( parent_class )->size_request( widget, req );
}

static void screem_editor_realize( GtkWidget *widget, 
		ScreemEditor *editor )
{
	GdkWindow *window;
	
	if( ! editor->private->page ) {
		window = gtk_text_view_get_window( GTK_TEXT_VIEW( widget ), 
				GTK_TEXT_WINDOW_TEXT );
		gdk_window_set_cursor( window, NULL );
	}
}

static void screem_editor_show( GtkWidget *widget )
{
	ScreemEditor *editor;
	ScreemEditorPrivate *priv;
	gboolean active;
	GdkWindow *window;
	GdkDisplay *display;
	GdkCursor *cursor;
	
	editor = SCREEM_EDITOR( widget );
	priv = editor->private;
	
	screem_editor_set_colours( editor );

	GTK_WIDGET_CLASS( parent_class )->show( widget );

	gtk_widget_show( priv->view );
	gtk_widget_show( priv->sw );

	window = gtk_text_view_get_window( GTK_TEXT_VIEW( priv->view ), 
			GTK_TEXT_WINDOW_TEXT );
	if( priv->page ) {
		active = TRUE;
		if( window ) {
			display = gtk_widget_get_display( priv->view );
			cursor = gdk_cursor_new_for_display( display,
					GDK_XTERM );
			gdk_window_set_cursor( window, cursor );
			gdk_cursor_unref( cursor );
		}
	} else {
		active = FALSE;
		if( window ) {
			gdk_window_set_cursor( window, NULL );
		}
	}
}

static void screem_editor_hide( GtkWidget *widget )
{
	ScreemEditor *editor;
	ScreemEditorPrivate *private;
	
	editor = SCREEM_EDITOR( widget );
	private = editor->private;
	
	GTK_WIDGET_CLASS( parent_class )->hide( widget );
}

GType screem_editor_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemEditorClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_editor_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemEditor ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_editor_init
		};
		
		type = g_type_register_static( PARENT_TYPE,
					       "ScreemEditor",
					       &info, 0 );
	}

	return type;
}
