//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [mean_kep]=CL_ex_meanLyddane(osc_kep,er,mu, zonals)
// Lyddane orbit propagation analytical model (mean elements)
//
// Calling Sequence
// [mean_kep] = CL_ex_meanLyddane(osc_kep [,er,mu,zonals])
//
// Description
// <itemizedlist><listitem>
// Computes the mean Keplerian elements <emphasis role="bold">mean_kep</emphasis> from the osculating Keplerian elements <emphasis role="bold">osc_kep</emphasis>.
// </listitem>
// <listitem> The orbital elements are the following:  
// <para><inlinemediaobject><imageobject><imagedata fileref="kep_par.gif"/></imageobject></inlinemediaobject></para>
// </listitem>
// <listitem>Warnings :
// <para> - This function does not work for inclinations close to the critical inclinations (63.43494882 deg and 116.5650512 deg)</para>
// <para> - This function does not work for eccentricities higher than 0.9</para></listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// osc_kep: Osculating Keplerian elements [sma;ecc;inc;pom;raan;anm] (6xN)
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Gravitational constant [m^3/s^2] (default value is %CL_mu)
// zonals: (optional) Vector of zonals coefficients J1 to Jn (troncated to J5) to be used (default is %CL_j1jn(1:5)) (1 x N)
// mean_kep: Mean Keplerian elements [sma;ecc;inc;pom;raan;anm] (6xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_ex_lyddane
// CL_ex_lyddaneMan
//
// Bibliography
// 1 CNES - MSLIB FORTRAN 90, Volume E (me_lyddane_mean)
//
// Examples
// osc_kep_t1 = [42166712 ; 7.9e-3 ; CL_deg2rad([97.2 ; 89 ; 125 ; 0])];
// [mean_kep_t1] = CL_ex_meanLyddane(osc_kep_t1);
// t1 = 12584;
// t2 = t1;
// [mean_kep_t2,osc_kep_t2] = CL_ex_lyddane(t1,mean_kep_t1,t2)

//

//************************************************************************
// But:  Calcul des parametres meanens du modele d'extrapolation d'orbite de Lydanne
// ===
//
// Note d'utilisation: les unites de distance sont en metres OBLIGATOIREMENT
// ==================  (pour mu, er, et a)
//
//                     l'eccentricite doit appartenir a [ 0. , 0.9 [
//                     le domaine d'erreur est          < 0. et >= 0.9
//
//                     l'inclinaison  doit appartenir a [ 0. , pi ]
//                     et non proches des inclinaisons critiques definies par
//                     (pm_i_critique_non_retro, pm_i_critique_retro) a pm_eps_i_critique pres.
//************************************************************************

//************************************************************************
// But:  Calcul des parametres meanens du modele d'extrapolation d'orbite de Lydanne
// ===
//
// Note d'utilisation: les unites de distance sont en metres OBLIGATOIREMENT
// ==================  (pour mu, er, et a)
//
//                     l'eccentricite doit appartenir a [ 0. , 0.9 [
//                     le domaine d'erreur est          < 0. et >= 0.9
//
//                     l'inclinaison  doit appartenir a [ 0. , pi ]
//                     et non proches des inclinaisons critiques definies par
//                     (pm_i_critique_non_retro, pm_i_critique_retro) a pm_eps_i_critique pres.
//************************************************************************


// Declarations:
if(~exists('%CL_eqRad')) then global %CL_eqRad; end;
if(~exists('%CL_j1jn')) then global %CL_j1jn; end;
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('zonals','local') then zonals=%CL_j1jn(1:5); end

nmax=51

osc = CL_oe_kep2cirEqua(osc_kep);

n = ones(1,size(osc,2))           // initialisation du numero d'iterations
convergence = zeros(1,size(osc,2))// initialisation de l'indicateur de convergence

eps1000 = 1000 * %eps // epsilon pour test de convergence

//la date est inutile dans l'appel a me_lyddane: on l'initialise a 1 j 0 sec,
dat = 1.0

mean_local=zeros(osc)
ecart = zeros(osc)
ecart_admi=zeros(osc)
mean_sortie=zeros(osc)
oscn=zeros(osc)

//for security...
osc(6,:) = CL_rMod(osc(6,:),2*%pi)

// initialisation des meanens aux osculateurs en entree et passage des angles dans l'intervalle [0,2*pi]
mean_local(1,:) = osc(1,:)
mean_local(2,:) = osc(2,:)
mean_local(3,:) = osc(3,:)
mean_local(4,:) = osc(4,:)
mean_local(5,:) = osc(5,:)
mean_local(6,:) = CL_rMod(osc(6,:),2*%pi)

ecart_admi(1,:) = eps1000*(1 + abs(mean_local(1,:)))
ecart_admi(2,:) = eps1000*(1 + abs(mean_local(2,:)))
ecart_admi(3,:) = eps1000*(1 + abs(mean_local(3,:)))
ecart_admi(4,:) = eps1000*(1 + abs(mean_local(4,:)))
ecart_admi(5,:) = eps1000*(1 + abs(mean_local(5,:)))
ecart_admi(6,:) = eps1000*(1 + abs(mean_local(6,:)))

// debut d'iteration
ii = find((n < nmax) & (~convergence))
while ii~=[]  // iterations tant que le nombre maximum d'iterations n'est pas atteint et convergence non atteinte

  //calcul des osculateurs correspondant aux meanens (a la meme date)
  mean_sortie_prov=[];
  oscn_prov=[];
  [mean_sortie_prov,oscn_prov]=CL_ex_lyddane(dat,CL_oe_cirEqua2kep(mean_local(:,ii)),dat,er,mu,zonals);
  mean_sortie(:,ii) = CL_oe_kep2cirEqua(mean_sortie_prov)
  oscn(:,ii) = CL_oe_kep2cirEqua(oscn_prov)

  //calcul des ecarts entre parametres osculateurs calcules et ceux de depart.
  //Ces ecarts sont ramenes sur [-pi,pi].
  ecart(1,ii) = osc(1,ii) - oscn(1,ii);
  ecart(2,ii) = osc(2,ii) - oscn(2,ii);
  ecart(3,ii) = osc(3,ii) - oscn(3,ii);
  ecart(4,ii) = osc(4,ii) - oscn(4,ii);
  ecart(5,ii) = osc(5,ii) - oscn(5,ii);
  ecart(6,ii) = osc(6,ii) - oscn(6,ii)

  jj1 = find( (ecart(6,:) < (-%pi)) & ((n < nmax) & (~convergence)) );
  jj2 = find( (ecart(6,:) > %pi) & ((n < nmax) & (~convergence)) );
  ecart(6,jj1) = ecart(6,jj1) + 2*%pi;
  ecart(6,jj2) = ecart(6,jj2) - 2*%pi

  //  test de convergence
  nn = find( ...
            ( abs(ecart(1,:)) <= ecart_admi(1,:) ) &...
            ( abs(ecart(2,:)) <= ecart_admi(2,:) ) &...
            ( abs(ecart(3,:)) <= ecart_admi(3,:) ) &...
            ( abs(ecart(4,:)) <= ecart_admi(4,:) ) &...
            ( abs(ecart(5,:)) <= ecart_admi(5,:) ) &...
            ( abs(ecart(6,:)) <= ecart_admi(6,:) ) &...
            ( (n < nmax) & (~convergence) )...
            );
  jj = find( ~(...
                ( abs(ecart(1,:)) <= ecart_admi(1,:) ) &...
                ( abs(ecart(2,:)) <= ecart_admi(2,:) ) &...
                ( abs(ecart(3,:)) <= ecart_admi(3,:) ) &...
                ( abs(ecart(4,:)) <= ecart_admi(4,:) ) &...
                ( abs(ecart(5,:)) <= ecart_admi(5,:) ) &...
                ( abs(ecart(6,:)) <= ecart_admi(6,:) ) &...
                ( (n < nmax) & (~convergence) ) ...
              ) ...
           );

  convergence(nn) = 1

   //  reinitialisation des elements means
  mean_local(1,jj) = mean_local(1,jj)+ecart(1,jj);
  mean_local(2,jj) = mean_local(2,jj)+ecart(2,jj);
  mean_local(3,jj) = mean_local(3,jj)+ecart(3,jj);
  mean_local(4,jj) = mean_local(4,jj)+ecart(4,jj);
  mean_local(5,jj) = mean_local(5,jj)+ecart(5,jj);
  mean_local(6,jj) = CL_rMod(mean_local(6,jj)+ecart(6,jj),2*%pi);

  n(ii) = n(ii) + 1
  ii = find((n < nmax) & (~convergence))

end

if or(~convergence) then
  CL__error("No convergence mean Lyddane in "+string(ii))
end // pas de convergence

// affectation des parametres de sortie
mean_kep = CL_oe_cirEqua2kep(mean_local);

endfunction
