//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [output] = CL_op_ssoJ2(type_output,par1,par2,er,mu,j2,rotr_pla_sun)
// Semi major axis, eccentricity or inclination for a sun synchronous orbit (J2)
//
// Calling Sequence
// [output] = CL_op_ssoJ2(type_output,par1,par2[,er[,mu[,j2[,rotr_pla_sun]]]])
// [sma] = CL_op_ssoJ2('a',ecc,inc[,er[,mu[,j2[,rotr_pla_sun]]]])
// [ecc] = CL_op_ssoJ2('e',sma,inc[,er[,mu[,j2[,rotr_pla_sun]]]])
// [inc] = CL_op_ssoJ2('i',sma,ecc[,er[,mu[,j2[,rotr_pla_sun]]]])
//
// Description
// <itemizedlist><listitem>
// This function computes orbital parameters (semi major axis, eccentricity or inclination) ensuring sun-synchronicity so that 
// the orbit's ascending node precession rate is equal to the Sun apparent rotation rate <emphasis role="bold">rotr_pla_sun</emphasis>.(See equation (1))
// <para> - If <emphasis role="bold">type_output</emphasis>='a', <emphasis role="bold">par1</emphasis> is eccentricity, <emphasis role="bold">par2</emphasis> is inclination and semi major axis is computed (<emphasis role="bold">output</emphasis>).</para>
// <para> - If <emphasis role="bold">type_output</emphasis>='e', <emphasis role="bold">par1</emphasis> is semi major axis, <emphasis role="bold">par2</emphasis> is inclination and eccentricity is computed (<emphasis role="bold">output</emphasis>).</para>
// <para> - If <emphasis role="bold">type_output</emphasis>='i', <emphasis role="bold">par1</emphasis> is semi major axis, <emphasis role="bold">par2</emphasis> is eccentricity and inclination is computed (<emphasis role="bold">output</emphasis>)</para>
// <para>Note: an error is returned is input parameters are out of range. </para>
// <para/><inlinemediaobject><imageobject><imagedata fileref="helio.gif"/></imageobject></inlinemediaobject></listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// type_output: String defining the parameter to be computed (output). It can be 'a' for semi major axis, 'e' for eccentricity or 'i' for inclination
// par1: Semi major axis if type_output is 'e' or 'i' ; eccentricity if type_output is 'a' [m] (1xN or 1x1)
// par2: Inclination if type_output is 'a' or 'e' ; eccentricity if type_output is 'i' [rad] (1xN or 1x1)
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Gravitational constant [m^3/s^2] (default value is %CL_mu)
// j2: (optional) Second zonal harmonic term (default is %CL_j2)
// rotr_pla_sun : (optional) Mean apparent rotation rate of the Sun around the planet (default is %CL_rotrEarthSun) (1 x 1)
// output: Semi major axis [m] if type_output is 'a' ; eccentricity if type_output is 'e' ; inclination [rad] if type_output is 'i' (1xN). Output value is %nan if no valid result exists. 
//
// Authors
// CNES - DCT/SB
//
// Examples
// // Compute inclination for sun-synchronicity
// sma = 7078.e3;
// ecc = 0.01;
// inc = CL_op_ssoJ2 ('i', sma, ecc)
//
// // Compute eccentricity for sun-synchronicity
// sma = 7078.e3;
// inc = CL_deg2rad(98.15);
// ecc = CL_op_ssoJ2 ('e', sma, inc)
//
// // Compute semi major axis for sun-synchronicity
// ecc = 0.01;
// inc = CL_deg2rad(97);
// sma = CL_op_ssoJ2 ('a', ecc, inc)

// Declarations:
if(~exists('%CL_eqRad')) then global %CL_eqRad; end;
if(~exists('%CL_j2')) then global %CL_j2; end;
if(~exists('%CL_mu')) then global %CL_mu; end;
if(~exists('%CL_rotrEarthSun')) then global %CL_rotrEarthSun; end;

// Code:

Npar1 = size(par1,2)
Npar2 = size(par2,2)
N = max(Npar1,Npar2)
if ~( (Npar1==1|Npar1==N) & (Npar2==1|Npar2==N) ) then CL__error('bad size of input arguments'); end

if ~exists('rotr_pla_sun','local') then rotr_pla_sun=%CL_rotrEarthSun; end
if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('j2','local') then j2=%CL_j2; end


K = -3.0 * j2 * (er.^2) * sqrt(mu) / (2.0 * rotr_pla_sun) // auxiliary (scalar) variable

if type_output == 'a'
  // calculation of semi major axis

  ecc = par1 .* ones(1:N);
  inc = par2 .* ones(1:N);

  I = find (ecc < 0 | ecc >= 1); 
  if ~isempty(I); CL__error('eccentricity out of range'); end
  I = find (inc < 0 | inc > %pi+%eps); 
  if ~isempty(I); CL__error('inclination out of range'); end

  // output = semi major axis
  tmp = K * cos(inc) ./ ((1-ecc.^2).^2);
  I = find (tmp > 0); 
  output = %nan * ones(1:N);  
  output(I) = tmp(I) .^ (1/3.5); 

elseif type_output == 'e'
  // calculation of eccentricity

  sma = par1 .* ones(1:N);
  inc = par2 .* ones(1:N);

  I = find (sma <= 0); 
  if ~isempty(I); CL__error('semi major axis out of range'); end
  I = find (inc < 0 | inc > %pi+%eps); 
  if ~isempty(I); CL__error('inclination out of range'); end

  // output = excentricity
  tmp = K * cos(inc) ./ (sma .^ (3.5));
  I = find (tmp >= 0 & tmp < 1); 
  output = %nan * ones(1:N);  
  output(I) = sqrt(1-sqrt(tmp(I))); 

elseif type_output == 'i'
  // calculation of inclination

  sma = par1 .* ones(1:N);
  ecc = par2 .* ones(1:N);

  I = find (sma <= 0); 
  if ~isempty(I); CL__error('semi major axis out of range'); end
  I = find (ecc < 0 | ecc >= 1); 
  if ~isempty(I); CL__error('eccentricity out of range'); end

  // output = inclination
  tmp = (sma.^ (3.5) .* (1-ecc.^2).^2) / K;
  I = find (tmp >= -1 & tmp <= 1); 
  output = %nan * ones(1:N);
  output(I) = acos(tmp(I)); 

else

  CL__error('unknown value for type_output');

end

endfunction
