//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function kep = CL_kp_characteristics(sma,ecc,v,er,mu)
// Various Keplerian characteristics
//
// Calling Sequence
// kep = CL_kp_characteristics(sma,ecc,v[,er[,mu]])
//
// Description
// <itemizedlist><listitem>
// This function computes many keplerian parameters for elliptic or hyperbolic orbits. Parabolic orbits
// are not handled. 
// <para>The computed parameters are : </para>
// <para>- standard quantities: </para>
// <para> semi major axis, eccentricity, apoapsis radius, periapsis radius,
// apoapsis altitude, periapsis altitude, velocity at apoapsis, velocity at periapsis, energy,
// law of areas' constant, period, semi minor axis, conic's p parameter, conic's c parameter (angular
// momentum), mean motion, velocity at infinity on an hyperbola, distance from asymptote to
// focus for an hyperbola, escape velocity at apoapsis, escape velocity at periapsis,
// escape delta-V at apoapsis, escape delta-V at periapsis.</para>
// <para>- position-related quantities : </para>
// <para> true anomaly, mean anomaly, eccentric anomaly, time
// from periapsis, radius, x cartesian coordinate, y cartesian coordinate, slope of velocity vector, velocity.</para>
// <para>The output argument <emphasis role="bold">kep</emphasis> is a structure that contains everything. </para>
// <para>Parameters that are irrelevant are given the value %nan. Example: kep.period is %nan for hyperbolic orbits. </para>
// <para> <emphasis role="bold">kep</emphasis> is defined as follows:</para>
// <para>- kep.orb_type: type of orbit (1->elliptical, 2->hyperbolic)</para>
// <para>- kep.sma: semi major axis [m]</para>
// <para>- kep.ecc: eccentricity</para>
// <para>- kep.ra: apoapsis radius [m]</para>
// <para>- kep.rp: periapsis radius [m]</para>
// <para>- kep.ha: apoapsis altitude (= kep.ra minus planet (equatorial) radius) [m]</para>
// <para>- kep.hp: periapsis altitude (= kep.rp minus planet (equatorial) radius) [m]</para>
// <para>- kep.va: velocity at apoapsis [m/s]</para>
// <para>- kep.vp: velocity at periapsis [m/s]</para>
// <para>- kep.vinf: velocity at infinity for an hyperbolic orbit (%nan otherwise) [m/s]</para>
// <para>- kep.period: orbit's period [s]</para>
// <para>- kep.mm: orbit's mean motion [rad/s]</para>
// <para>- kep.energy: orbit's energy [J]</para>
// <para>- kep.c_area: C constant for the law of areas [m^2/s]</para>
// <para>- kep.l_con: conic's l parameter (semi latus rectum)</para>
// <para>- kep.c_con: conic's c parameter (linear eccentricity) [m]</para>
// <para>- kep.smb: semi minor axis [m]</para>
// <para>- kep.d_asymt: distance asymptote-focus for an hyperbolic orbit (%nan otherwise) [m]</para>
// <para>- kep.vlp: escape velocity at periapsis [m/s]</para>
// <para>- kep.vla: escape velocity at apoapsis [m/s]</para>
// <para>- kep.dvp: escape delta-V at periapsis [m/s]</para>
// <para>- kep.dva: escape delta-V at apoapsis [m/s]</para>
// <para>- kep.v: true anomaly [rad]</para>
// <para>- kep.M: mean anomaly [rad]</para>
// <para>- kep.E: eccentric anomaly [rad]</para>
// <para>- kep.taninf: true anomaly at infinity (only for hyperbolic orbits, %nan otherwise) [rad]</para>
// <para>- kep.tperi: time from periapsis [s]</para>
// <para>- kep.r: norm of radius vector [m]</para>
// <para>- kep.x: x cartesian coordinate [m]</para>
// <para>- kep.y: y cartesian coordinate [m]</para>
// <para>- kep.slope: slope of velocity vector [rad]</para>
// <para>- kep.vel: velocity [m/s]</para>
// <para>NB: the linear eccentricity (kep.c_con) is the distance between the center and the focus
// (or one of the two foci). The latus rectum is the chord parallel to the directrix and passing
// through the focus (or one of the two foci). The semi latus rectum (kep.l_con) is half
// of the latus rectum.</para>
// </listitem>
// <listitem>
// Note: Deprecated names. 
// <para> A few names have changed but are still recognized. They should be avoided 
// as could disappear in the future. These names are: </para>
// <para> - k : same as energy </para>
// <para> - C_area : same as c_area </para>
// <para> - d_foy : same as d_asymt</para>
// <para> - anvinf : same as taninf</para>
// <para> - date: same as tperi</para>
// <para> - R: same as r </para>
// <para> - v_slope: same as slope </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// sma: semi major axis [m] (1xN or 1x1)
// ecc: eccentricity  (1xN or 1x1)
// v: true anomaly [rad] (1xN or 1x1)
// mu: (optional) gravitational constant [m^3/s^2] (default value is %CL_mu)
// er: (optional) planet equatorial radius [m] (default is %CL_eqRad)
// kep: structure containing all the data (1xN)
//
// See also
// CL_kp_params
// CL_kp_E2v
// CL_kp_v2E
// CL_kp_E2M
// CL_kp_M2E
// CL_op_rarp2ae
// CL_op_rava2ae
// CL_op_rpvinf2ae
// CL_op_rpvp2ae
//
// Authors
// CNES - DCT/SB
//
// Examples
// sma = 7078.e3;
// ecc = 0.0001;
// v = %pi/4;
// kep = CL_kp_characteristics(sma,ecc,v)

// Declarations:
if(~exists('%CL_epsilon')) then global %CL_epsilon; end;
if(~exists('%CL_eqRad')) then global %CL_eqRad; end;
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

Nsma = size(sma,2);
Necc = size(ecc,2);
Nv = size(v,2);
N = max(Nsma,Necc,Nv);

if ~( (Nsma==1|Nsma==N) & (Necc==1|Necc==N) & (Nv==1|Nv==N) ) 
   CL__error('Wrong sizes for input arguments'); 
end

if (Nsma==1); sma = sma*ones(1,N); end
if (Necc==1); ecc = ecc*ones(1,N); end
if (Nv==1); v = v*ones(1,N); end


if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('er','local') then er=%CL_eqRad; end

// keplerian parameters
  orb_type = zeros(ecc);
  ih = find(ecc>1);
  ie = find(ecc<1);
  orb_type(ih) = 2; //hyperbolic orbit
  orb_type(ie) = 1; //elliptic orbit

  if find(ecc >= 1-%CL_epsilon.parab & ecc <= 1+%CL_epsilon.parab) ~= [] 
     CL__error('Parabola case not handled');
  end

  // calculation of main and secondary kepler parameters
  [ra,rp,ha,hp,va,vp,nmoy,period,vinf,anvinf,dinf,vlibp,vliba,l_con,c_con,k,caire,smb,..
  anm,anex,temps,r,x,y,pen,vit] = CL__kp_keplerParams(sma,ecc,v,orb_type,er,mu)

  // calculation of velocity increments (apoapsis and periapsis) only for elliptic orbits
  delta_va = %nan * ones(va)
  delta_vp = %nan * ones(vp)
  delta_va(ie) = vliba(ie)-va(ie)
  delta_vp(ie) = vlibp(ie)-vp(ie)

  kep = struct( ..
    'orb_type', orb_type, ..
    'sma', sma, ..
    'ecc', ecc, ..
    'ra', ra, ..
    'rp', rp, ..
    'ha', ha, ..
    'hp', hp, ..
    'va', va, ..
    'vp', vp, ..
    'vinf', vinf, ..
    'period', period, ..
    'mm', nmoy, ..
    'k', k, ..  // DEPRECATED
    'C_area', caire, .. // DEPRECATED
    'l_con', l_con, ..
    'c_con', c_con, ..
    'smb', smb, ..
    'd_foy', dinf, .. // DEPRECATED
    'vlp', vlibp, ..
    'vla', vliba, ..
    'dvp', delta_vp, ..
    'dva', delta_va, ..
    'v', v, ..
    'M', anm, ..
    'E', anex, ..
    'anvinf', anvinf, .. // DEPRECATED
    'date', temps, .. // DEPRECATED
    'R', r, .. // DEPRECATED
    'x', x, ..
    'y', y, ..
    'v_slope', pen, .. // DEPRECATED
    'vel', vit, ..
    'energy', k, ..   // NEW NAME 
    'c_area', caire, ..  // NEW NAME 
    'd_asymt', dinf, .. // NEW NAME 
    'taninf', anvinf, .. // NEW NAME 
    'tperi', temps, .. // NEW NAME 
    'r', r, .. // NEW NAME 
    'slope', pen .. // NEW NAME 
  );


endfunction
