//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [kep,jacob]=CL_oe_car2kep(pos_car,vel_car,mu)
// Cartesian to keplerian orbital elements
//
// Calling Sequence
// [kep[,jacob]] = CL_oe_car2kep(pos_car,vel_car[,mu])
//
// Description
// <itemizedlist><listitem>
// Converts position & and velocity to classical keplerian orbital elements for 
// elliptic (non circular and non equatorial), hyperbolic or parabolic orbit.
// <para>The transformation jacobian is optionally computed.</para>
// <para>See <link linkend="OrbitalElements">Orbital elements</link> for more details</para>
// </listitem>
// <listitem>Notes on the computed orbital elements : 
// <para> - for eccentricites close to 0, &#969; and M are not defined independently. Only &#969; + M is.</para>
// <para> - for inclinations close to 0, &#969; and &#937; are not defined independently. Only &#969; + &#937; is.</para>
// <para> - for inclinations and eccentricities close to 0, &#969;, &#937; and M are not defined independently. Only &#969; + &#937; + M is.</para>
// </listitem>
// <listitem>Jacobian transformation is not exact for any of the above cases 
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// pos_car: position [X;Y;Z] [m] (3xN)
// vel_car: velocity [Vx;Vy;Vz] [m/s] (3xN)
// mu : (optional) gravitational constant. [m^3/s^2] (default value is %CL_mu)
// kep: classical keplerian orbital elements [sma;ecc;inc;pom;raan;anm] [m,rad] (6xN)
// jacob: (optional) transformation jacobian d(sma,ecc,inc,pom,raan,anm)/d(X,Y,Z,Vx,Vy,Vz) (6x6xN)
//
// Bibliography
// 1 CNES - MSLIB FORTRAN 90, Volume V (mv_car_kep)
//
// See also
// CL_oe_kep2car
// CL_oe_car2cir
// CL_oe_car2cirEqua
// CL_co_car2sph
// CL_co_car2ell
//
// Authors
// CNES - DCT/SB
//
// Examples
// // Example 1 : elliptical orbit
// pos = [-26655.e3 ; 29881.e3 ; 0];
// vel = [-1125 ; -1122 ; 195];
// [kep,jacob] = CL_oe_car2kep(pos,vel)
// [pos2,vel2,jacob2] = CL_oe_kep2car(kep);
// jacob*jacob2 // = identity
// [pos2-pos ; vel2-vel]
//
// // Example 2 : hyperbolic orbit
// pos = [0; 6933.e3; 0];
// vel = [0; 0; 12739.];
// [kep,jacob] = CL_oe_car2kep(pos,vel)
// [pos2,vel2,jacob2] = CL_oe_kep2car(kep);
// jacob*jacob2 // = identity
// [pos2-pos ; vel2-vel]

// Declarations:
if(~exists('%CL_epsilon')) then global %CL_epsilon; end;
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

jacob = [];

Np = size(pos_car,2);
Nv = size(vel_car,2);
if(Np ~= Nv) then CL__error('Position and velocity vectors must have the same size'); end

[lhs,rhs] = argn();
if ~exists('mu','local') then mu=%CL_mu; end
if lhs==2
  compute_jacob=%t
  jacob = zeros(6,6,Np)
else
  compute_jacob=%f
end

kep = zeros(6,Np)

norm_pos = CL_colNorm(pos_car)
norm_vel = CL_colNorm(vel_car)
kinetic_moment = CL_cross(pos_car,vel_car)
norm_kinetic = CL_colNorm(kinetic_moment)

i_pos = find(norm_pos <= 0)
i_vel = find(norm_vel <= 0)
i_colinear = find(norm_kinetic <= 0)

if i_pos~=[] then CL__error('Norm of position vector <= 0'); end
if i_vel~=[] then CL__error('Norm of velocity vector <= 0'); end
if i_colinear~=[] then CL__error('Norm of angular momemtum <= 0: position and velocity colinear'); end

// Ancien Code :
//pos_vel = CL_dot(pos_car,vel_car) //scalar product position*velocity
//parameter = (norm_kinetic.^2)./mu //parameter p = (rxv)^2/mu = c^2/mu; p>0 because mu>0
//norm_vel_sq = norm_vel.^2
//one_over_a = 2.0./norm_pos - norm_vel_sq./mu  //1/a
//coefk = 0.5.*norm_vel_sq - mu./norm_pos //eccentricity
//e2 = 1 + 2.*parameter.*coefk./mu        //valid for any orbit type: k = v2/2-|mu|/r
//eccentricity = sqrt(e2) //e^2 = 1 + 2.c^2.k/mu^2

// Variables intermédiaires
pos_vel = CL_dot(pos_car,vel_car) ;                // scalar product position*velocity
one_over_a = 2.0 ./ norm_pos - norm_vel.^2 / mu ;  // 1/a
energy = norm_vel.^2 / 2 - mu ./ norm_pos ;        // E = v^2/2 - mu/r

eccentricity = zeros(1,Np);
parameter = zeros(1,Np);

// Hyperbolic and parabolic :
i_hp = find(energy >= 0);                                     // indices hyperboliques ou paraboliques 
  if( i_hp ~= [])
    parameter(i_hp) = (norm_kinetic(i_hp).^2) / mu ;            // parameter p = (rxv)^2/mu = c^2/mu; p>0 because mu>0
    e2 = 1 + 2 * parameter(i_hp) .* (energy(i_hp) / mu );       // e^2 = 1 + 2.c^2.E/mu^2
    eccentricity(i_hp) = real(sqrt(e2));                        // real pour gestion des erreurs numériques
  end
   
// Elliptic (Formule plus précise pour les excentricités faibles):
i_e = find(energy < 0);                                      // indices elliptiques 
  if( i_e ~= [])
    esinE = pos_vel(i_e) ./ sqrt(mu./one_over_a(i_e));
    ecosE = 1 - norm_pos(i_e) .* one_over_a(i_e);
    E = atan(esinE,ecosE);
    cosE = cos(E);
    sinE = sin(E);
  
    ii = find( abs(cosE) > abs(sinE) );
    eccentricity(i_e(ii)) = ecosE(ii) ./ cosE(ii);
    ii = find( abs(cosE) <= abs(sinE) );
    eccentricity(i_e(ii)) = esinE(ii) ./ sinE(ii);
  end
 
//***********************************************
//calcul transformation cartesian --> keplerian
//following orbit type
//***********************************************
//elliptic orbit
ie = find(eccentricity <= 1-%CL_epsilon.parab)
if ie~=[]

  ie_a = find( (eccentricity <= 1-%CL_epsilon.parab) & (one_over_a <= 0) )
  if ie_a~=[] then CL__error('Negative or infinite semi-major axis'); end

  if compute_jacob
    [kepi,jac] = CL__oe_car2kepEllip(pos_car(:,ie),vel_car(:,ie),eccentricity(:,ie),norm_pos(:,ie),...
                                   norm_vel(:,ie),one_over_a(:,ie),kinetic_moment(:,ie),pos_vel(:,ie),mu)
    jacob(:,:,ie) = jac
  else
    [kepi] = CL__oe_car2kepEllip(pos_car(:,ie),vel_car(:,ie),eccentricity(:,ie),norm_pos(:,ie),...
                                   norm_vel(:,ie),one_over_a(:,ie),kinetic_moment(:,ie),pos_vel(:,ie),mu)
  end
  kep(:,ie) = kepi

end

//hyperbolic orbit
ih = find(eccentricity>=1+%CL_epsilon.parab)
if ih~=[]

  one_over_a = abs(one_over_a)  //we use absolute value of 1/a in the hyperbolic case

  ih_1a = find( (eccentricity>=1+%CL_epsilon.parab)&(one_over_a <= 0) )
  if ih_1a~=[] then CL__error('Negative or infinite semi-major axis'); end

  if compute_jacob
    [kepi,jac] = CL__oe_car2kepHyperb(pos_car(:,ih),vel_car(:,ih),parameter(:,ih),eccentricity(:,ih),...
                                   norm_pos(:,ih),norm_vel(:,ih),kinetic_moment(:,ih),...
                                   norm_kinetic(:,ih),one_over_a(:,ih),pos_vel(:,ih),mu)
    jacob(:,:,ih) = jac
  else
    [kepi] = CL__oe_car2kepHyperb(pos_car(:,ih),vel_car(:,ih),parameter(:,ih),eccentricity(:,ih),...
                                   norm_pos(:,ih),norm_vel(:,ih),kinetic_moment(:,ih),...
                                   norm_kinetic(:,ih),one_over_a(:,ih),pos_vel(:,ih),mu)
  end
  kep(:,ih) = kepi

end

//parabolic orbit
ip = find( (eccentricity>1-%CL_epsilon.parab)&(eccentricity<1+%CL_epsilon.parab) )
if ip~=[]

  if compute_jacob
    [kepi,jac] = CL__oe_car2kepParab(pos_car(:,ip),vel_car(:,ip),parameter(:,ip),eccentricity(:,ip),...
                                   norm_pos(:,ip),kinetic_moment(:,ip),norm_kinetic(:,ip),pos_vel(:,ip),mu)
    jacob(:,:,ip) = jac
  else
    [kepi] = CL__oe_car2kepParab(pos_car(:,ip),vel_car(:,ip),parameter(:,ip),eccentricity(:,ip),...
                                   norm_pos(:,ip),kinetic_moment(:,ip),norm_kinetic(:,ip),pos_vel(:,ip),mu)
  end
  kep(:,ip) = kepi

end

//if (compute_jacob & size(jacob,3)==1) then jacob=jacob(:,:,1); end

endfunction
