/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <gdk/gdkkeysyms.h>
#include "sg_plot_tools.h"
#include "sg.h"
#include "sg_dialogs.h"
#include "sg_arrange_dialog.h"
#include "sg_layer.h"
#include "sg_menu.h"
#include "../pixmaps/pixmaps.h"
#include "../pixmaps/fit_page.xpm"
#include "math.h"

#define ZOOM 2.0000000000000
#define NTOP 2

static void     toolbox_autoscale                       (GtkButton *button,
                                                         gpointer data);
static void     toolbox_fitpage                         (GtkButton *button,
                                                         gpointer data);
static void     toolbox_fitpage_h                         (GtkButton *button,
                                                         gpointer data);
static void     toolbox_fitpage_v                         (GtkButton *button,
                                                         gpointer data);
static void     toolbox_up                              (GtkButton *button,
                                                         gpointer data);
static void     toolbox_down                            (GtkButton *button,
                                                         gpointer data);
static void     toolbox_left                            (GtkButton *button,
                                                         gpointer data);
static void     toolbox_right                           (GtkButton *button,
                                                         gpointer data);
static void     toolbox_zoomin                          (GtkButton *button,
                                                         gpointer data);
static void     toolbox_zoomout                         (GtkButton *button,
                                                         gpointer data);
static void     toolbox_select                          (GtkButton *button,
                                                         gpointer data);
static gint     tool_select_region                      (GtkPlotCanvas *canvas,
                                                         gdouble xmin,
                                                         gdouble xmax,
                                                         gdouble ymin,
                                                         gdouble ymax,
                                                         gpointer data);
static void     toolbox_reset                           (GtkButton *button,
                                                         gpointer data);
static void     toolbox_rotate_x                        (GtkButton *button,
							 GdkEventButton *event,
                                                         gpointer data);
static void     toolbox_rotate_y                        (GtkButton *button,
							 GdkEventButton *event,
                                                         gpointer data);
static void     toolbox_rotate_z                        (GtkButton *button,
							 GdkEventButton *event,
                                                         gpointer data);
static void     button_release                        	(GtkButton *button,
							 GdkEventButton *event,
                                                         gpointer data);
static void     init_tool                               (SGplot *plot);
static void 	plot_print 				(GtkWidget *widget, 
							 gpointer data);
static void 	plot_new2d 				(GtkWidget *widget, 
							 gpointer data);
static void 	plot_new3d 				(GtkWidget *widget, 
							 gpointer data);
static void 	plot_new_polar				(GtkWidget *widget, 
							 gpointer data);
static void 	plot_arrange				(GtkWidget *widget, 
							 gpointer data);
static void 	plot_layer_control			(GtkWidget *widget, 
							 gpointer data);
static void 	plot_fit				(GtkWidget *widget, 
							 gpointer data);
static void 	plot_zoom_out				(GtkWidget *widget, 
							 gpointer data);
static void 	plot_zoom_in				(GtkWidget *widget, 
							 gpointer data);
static void 	plot_refresh				(GtkWidget *widget, 
							 gpointer data);




static void
toolbox_left(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  gdouble step;

  plot = (SGplot *)data;
  real_plot = GTK_PLOT(plot->active_layer->real_plot);

  step = real_plot->bottom->ticks.step;

  if(GTK_IS_PLOT3D(real_plot)) 
    gtk_plot3d_set_xrange(GTK_PLOT3D(real_plot), 
                          real_plot->xmin - step, 
                          real_plot->xmax - step);
  else if(GTK_IS_PLOT_POLAR(real_plot))
    gtk_plot_polar_rotate(GTK_PLOT_POLAR(real_plot),
                          GTK_PLOT_POLAR(real_plot)->rotation + step);
  else
    gtk_plot_set_xrange(real_plot, 
                        real_plot->xmin - step, 
                        real_plot->xmax - step);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));

}

static void
toolbox_right(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  gdouble step;

  plot = (SGplot *)data;
  real_plot = GTK_PLOT(plot->active_layer->real_plot);

  step = real_plot->bottom->ticks.step;

  if(GTK_IS_PLOT3D(real_plot)) 
    gtk_plot3d_set_xrange(GTK_PLOT3D(real_plot), 
                          real_plot->xmin + step, 
                          real_plot->xmax + step);
  else if(GTK_IS_PLOT_POLAR(real_plot))
    gtk_plot_polar_rotate(GTK_PLOT_POLAR(real_plot), 
                          GTK_PLOT_POLAR(real_plot)->rotation - step);
  else
    gtk_plot_set_xrange(real_plot, 
                        real_plot->xmin + step, 
                        real_plot->xmax + step);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
}

static void
toolbox_up(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  gdouble step;

  plot = (SGplot *)data;
  real_plot = GTK_PLOT(plot->active_layer->real_plot);

  step = real_plot->left->ticks.step;

  if(GTK_IS_PLOT3D(real_plot)) 
    gtk_plot3d_set_yrange(GTK_PLOT3D(real_plot), 
                          real_plot->ymin + step, 
                          real_plot->ymax + step);
  else if(GTK_IS_PLOT_POLAR(real_plot))
    gtk_plot_set_yrange(real_plot, 
                        real_plot->ymin, 
                        real_plot->ymax + step);
  else
    gtk_plot_set_yrange(real_plot, 
                        real_plot->ymin + step, 
                        real_plot->ymax + step);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
}

static void
toolbox_down(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  gdouble step;

  plot = (SGplot *)data;
  real_plot = GTK_PLOT(plot->active_layer->real_plot);

  step = real_plot->left->ticks.step;

  if(GTK_IS_PLOT3D(real_plot)) 
    gtk_plot3d_set_yrange(GTK_PLOT3D(real_plot), 
                          real_plot->ymin - step, 
                          real_plot->ymax - step);
  else if(GTK_IS_PLOT_POLAR(real_plot))
    gtk_plot_set_yrange(real_plot, 
                        real_plot->ymin, 
                        real_plot->ymax - step);
  else
    gtk_plot_set_yrange(real_plot, 
                        real_plot->ymin - step, 
                        real_plot->ymax - step);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
}

static void
toolbox_autoscale(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;

  plot = (SGplot *)data;
  real_plot = GTK_PLOT(plot->active_layer->real_plot);

  if(GTK_IS_PLOT3D(real_plot))
    gtk_plot3d_autoscale(GTK_PLOT3D(real_plot));
  else
    sg_layer_autoscale(plot->active_layer,sg_autoscale_left,sg_autoscale_right,sg_autoscale_top,
                      sg_autoscale_bottom);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
}

static void
toolbox_fitpage(GtkButton *button, gpointer data)
{
  sg_plot_fit_page((SGplot *)data);
}

static void
toolbox_fitpage_h(GtkButton *button, gpointer data)
{
  sg_plot_fit_page_h((SGplot *)data);
}

static void
toolbox_fitpage_v(GtkButton *button, gpointer data)
{
  sg_plot_fit_page_v((SGplot *)data);
}


static void
toolbox_zoomin(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;

  plot = (SGplot *)data;
  real_plot = GTK_PLOT(plot->active_layer->real_plot);

  if(GTK_IS_PLOT3D(real_plot)) return;

  if(GTK_IS_PLOT_POLAR(real_plot)){
    gtk_plot_set_yrange(real_plot, 
                        real_plot->ymin, 
                        real_plot->ymax * ZOOM);
  } else {
    gtk_plot_set_yrange(real_plot, 
                        real_plot->ymin * ZOOM, 
                        real_plot->ymax * ZOOM);

    gtk_plot_set_xrange(real_plot, 
                        real_plot->xmin * ZOOM, 
                        real_plot->xmax * ZOOM);
  }

  gtk_plot_axis_set_ticks(real_plot, GTK_PLOT_AXIS_X,
                     real_plot->bottom->ticks.step * ZOOM,
                     real_plot->bottom->ticks.nminor);

  gtk_plot_axis_set_ticks(real_plot, GTK_PLOT_AXIS_Y,
                     real_plot->left->ticks.step * ZOOM,
                     real_plot->left->ticks.nminor);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
}

static void
toolbox_zoomout(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;

  plot = (SGplot *)data;
  real_plot = GTK_PLOT(plot->active_layer->real_plot);

  if(GTK_IS_PLOT3D(real_plot)) return;

  if(GTK_IS_PLOT_POLAR(real_plot)){
    gtk_plot_set_yrange(real_plot, 
                        real_plot->ymin, 
                        real_plot->ymax / ZOOM);
  } else {
    gtk_plot_set_yrange(real_plot, 
                        real_plot->ymin / ZOOM, 
                        real_plot->ymax / ZOOM);

    gtk_plot_set_xrange(real_plot, 
                        real_plot->xmin / ZOOM, 
                        real_plot->xmax / ZOOM);
  }

  gtk_plot_axis_set_ticks(real_plot, GTK_PLOT_AXIS_X,
                     real_plot->bottom->ticks.step / ZOOM,
                     real_plot->bottom->ticks.nminor);

  gtk_plot_axis_set_ticks(real_plot, GTK_PLOT_AXIS_Y,
                     real_plot->left->ticks.step / ZOOM,
                     real_plot->left->ticks.nminor);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
}

static void
toolbox_reset(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;

  plot = (SGplot *)data;
  real_plot = GTK_PLOT(plot->active_layer->real_plot);

  if(!GTK_IS_PLOT3D(real_plot)) return;

  gtk_plot3d_reset_angles(GTK_PLOT3D(real_plot));

  gtk_plot3d_rotate_y(GTK_PLOT3D(real_plot), 30.0);
  gtk_plot3d_rotate_z(GTK_PLOT3D(real_plot), 60.0);
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
}

static void
toolbox_rotate_x(GtkButton *button, GdkEventButton *event, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  gdouble angle = 10.;

  if(event->type != GDK_BUTTON_PRESS) return;

  if(event->button != 1){
    button->button_down = FALSE;
    event->button = 1;
    angle = -10.;
  }

  plot = (SGplot *)data;
  real_plot = GTK_PLOT(plot->active_layer->real_plot);

  if(!GTK_IS_PLOT3D(real_plot)) return;

  gtk_plot3d_rotate_vector(GTK_PLOT3D(real_plot), 
			   &GTK_PLOT3D(real_plot)->e1, angle, 0., 0.);
  gtk_plot3d_rotate_vector(GTK_PLOT3D(real_plot), 
			   &GTK_PLOT3D(real_plot)->e2, angle, 0., 0.);
  gtk_plot3d_rotate_vector(GTK_PLOT3D(real_plot), 
			   &GTK_PLOT3D(real_plot)->e3, angle, 0., 0.);
  gtk_signal_emit_by_name(GTK_OBJECT(real_plot), "update", FALSE);
  gtk_signal_emit_by_name(GTK_OBJECT(real_plot), "changed");
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
}

static void
toolbox_rotate_z(GtkButton *button, GdkEventButton *event, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  gdouble angle = 10.;

  if(event->type != GDK_BUTTON_PRESS) return;

  if(event->button != 1){
    button->button_down = FALSE;
    event->button = 1;
    angle = -10.;
  }

  plot = (SGplot *)data;
  real_plot = GTK_PLOT(plot->active_layer->real_plot);

  if(!GTK_IS_PLOT3D(real_plot)) return;

  gtk_plot3d_rotate_vector(GTK_PLOT3D(real_plot), 
			   &GTK_PLOT3D(real_plot)->e1, 0., angle, 0.);
  gtk_plot3d_rotate_vector(GTK_PLOT3D(real_plot), 
			   &GTK_PLOT3D(real_plot)->e2, 0., angle, 0.);
  gtk_plot3d_rotate_vector(GTK_PLOT3D(real_plot), 
			   &GTK_PLOT3D(real_plot)->e3, 0., angle, 0.);
  gtk_signal_emit_by_name(GTK_OBJECT(real_plot), "update", FALSE);
  gtk_signal_emit_by_name(GTK_OBJECT(real_plot), "changed");
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
}

static void
toolbox_rotate_y(GtkButton *button, GdkEventButton *event, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  gdouble angle = 10.;

  if(event->type != GDK_BUTTON_PRESS) return;

  if(event->button != 1){
    button->button_down = FALSE;
    event->button = 1;
    angle = -10.;
  }

  plot = (SGplot *)data;
  real_plot = GTK_PLOT(plot->active_layer->real_plot);

  if(!GTK_IS_PLOT3D(real_plot)) return;

  gtk_plot3d_rotate_vector(GTK_PLOT3D(real_plot), 
			   &GTK_PLOT3D(real_plot)->e1, 0., 0., angle);
  gtk_plot3d_rotate_vector(GTK_PLOT3D(real_plot), 
			   &GTK_PLOT3D(real_plot)->e2, 0., 0., angle);
  gtk_plot3d_rotate_vector(GTK_PLOT3D(real_plot), 
			   &GTK_PLOT3D(real_plot)->e3, 0., 0., angle);
  gtk_signal_emit_by_name(GTK_OBJECT(real_plot), "update", FALSE);
  gtk_signal_emit_by_name(GTK_OBJECT(real_plot), "changed");
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
}

static gint
tool_select_region(GtkPlotCanvas *canvas, 
                   gdouble x1, gdouble y1,
                   gdouble x2, gdouble y2,
                   gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  GtkPlotCanvasChild *child = NULL;
  gdouble xmin, ymin, xmax, ymax;
  gint px1, px2, py1, py2;

  xmin = MIN(x1, x2);
  ymin = MIN(y1, y2);
  xmax = MAX(x1, x2);
  ymax = MAX(y1, y2);

  plot = (SGplot *)data;

  real_plot = GTK_PLOT(plot->active_layer->real_plot);

  switch(plot->tool){
    case SG_TOOL_ZOOM:
      if(x1 == x2 || y1 == y2){
         gtk_plot_canvas_refresh(canvas);
         return TRUE;
      }
      gtk_plot_canvas_get_pixel(canvas, xmin, ymin, &px1, &py1);
      gtk_plot_canvas_get_pixel(canvas, xmax, ymax, &px2, &py2);
      gtk_plot_get_point(real_plot, px1, py1, &xmin, &ymax);
      gtk_plot_get_point(real_plot, px2, py2, &xmax, &ymin);
      while(real_plot->left->ticks.step > ymax - ymin){
          gtk_plot_axis_set_major_ticks(real_plot, GTK_PLOT_AXIS_Y, 
					real_plot->left->ticks.step / 2.); 
      }
      while(real_plot->bottom->ticks.step > xmax - xmin){
          gtk_plot_axis_set_major_ticks(real_plot, GTK_PLOT_AXIS_X, 
					real_plot->bottom->ticks.step / 2.); 
      }
      gtk_plot_set_range(real_plot, xmin, xmax, ymin, ymax); 
      gtk_plot_canvas_paint(canvas);
      gtk_plot_canvas_refresh(canvas);
      return TRUE;
    case SG_TOOL_RECTANGLE:
      if(x1 == x2 || y1 == y2){
         gtk_plot_canvas_refresh(canvas);
         return TRUE;
      }
      child = gtk_plot_canvas_put_rectangle(canvas, x1, y1, x2, y2,
                                            GTK_PLOT_LINE_SOLID, 0, 
                                            NULL, NULL, (GtkPlotBorderStyle)1, FALSE);
      gtk_plot_canvas_paint(canvas);
      gtk_plot_canvas_refresh(canvas);
      return TRUE;
    case SG_TOOL_ELLIPSE:
      if(x1 == x2 || y1 == y2){
         gtk_plot_canvas_refresh(canvas);
         return TRUE;
      }
      child = gtk_plot_canvas_put_ellipse(canvas, x1, y1, x2, y2,
                                          GTK_PLOT_LINE_SOLID, 0, 
                                          NULL, NULL, FALSE);
      gtk_plot_canvas_paint(canvas);
      gtk_plot_canvas_refresh(canvas);
      return TRUE;
    case SG_TOOL_LINE_ARROW:
      child = gtk_plot_canvas_put_line(canvas, x1, y1, x2, y2, 
                                       GTK_PLOT_LINE_SOLID, 0, NULL,
                                       GTK_PLOT_ARROW_END);
      gtk_plot_canvas_paint(canvas);
      gtk_plot_canvas_refresh(canvas);
      return TRUE;
    case SG_TOOL_LINE:
      child = gtk_plot_canvas_put_line(canvas, x1, y1, x2, y2, 
                                       GTK_PLOT_LINE_SOLID, 0, NULL,
                                       GTK_PLOT_ARROW_NONE);
      gtk_plot_canvas_paint(canvas);
      gtk_plot_canvas_refresh(canvas);
      return TRUE;
  }

  return FALSE;
}


static void
toolbox_select(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  SGtoolbox *toolbox;

  plot = (SGplot *)data;

  if(!plot->active_layer) return;

  real_plot = GTK_PLOT(plot->active_layer->real_plot);

  toolbox = SG_TOOLBOX(plot->toolbox);

  GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                              GTK_PLOT_CANVAS_DND_FLAGS);

  if(button == GTK_BUTTON(toolbox->button[0][0])){
        plot->tool = SG_TOOL_ARROW;
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                  GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                    GTK_PLOT_CANVAS_CAN_DND_POINT |
                                    GTK_PLOT_CANVAS_CAN_SELECT_POINT |
                                    GTK_PLOT_CANVAS_CAN_SELECT); 
  }

  if(button == GTK_BUTTON(toolbox->button[0][1])){
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                    GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                  GTK_PLOT_CANVAS_CAN_SELECT);
        plot->tool = SG_TOOL_ZOOM;
  }

  if(button == GTK_BUTTON(toolbox->button[1][0])){
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                  GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                  GTK_PLOT_CANVAS_CAN_SELECT_POINT);
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                    GTK_PLOT_CANVAS_CAN_SELECT |
                                    GTK_PLOT_CANVAS_CAN_SELECT_ITEM); 
        plot->tool = SG_TOOL_POINTER;
  }

  if(button == GTK_BUTTON(toolbox->button[1][1])){
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                  GTK_PLOT_CANVAS_CAN_SELECT_POINT);
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                    GTK_PLOT_CANVAS_CAN_SELECT |
                                    GTK_PLOT_CANVAS_CAN_SELECT_ITEM |
                                    GTK_PLOT_CANVAS_CAN_DND_POINT |
                                    GTK_PLOT_CANVAS_CAN_DND);
        plot->tool = SG_TOOL_MARKERS;
  }

  if(button == GTK_BUTTON(toolbox->button[2][0])){
        plot->tool = SG_TOOL_TEXT;
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
				    GTK_PLOT_CANVAS_CAN_SELECT |
                                    GTK_PLOT_CANVAS_DND_FLAGS);
  }

/*
  if(button == GTK_BUTTON(toolbox->button[2][0])){
        plot->tool = SG_TOOL_LINE;
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                    GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                  GTK_PLOT_CANVAS_CAN_SELECT);
  }
*/
  if(button == GTK_BUTTON(toolbox->button[2][1])){
        plot->tool = SG_TOOL_LINE_ARROW;
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                    GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                  GTK_PLOT_CANVAS_CAN_SELECT);
  }
  if(button == GTK_BUTTON(toolbox->button[3][0])){
        plot->tool = SG_TOOL_RECTANGLE;
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                    GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                  GTK_PLOT_CANVAS_CAN_SELECT);
  }
  if(button == GTK_BUTTON(toolbox->button[3][1])){
        plot->tool = SG_TOOL_ELLIPSE;
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                    GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot->real_canvas),
                                  GTK_PLOT_CANVAS_CAN_SELECT);
  }

  if(plot->tool != SG_TOOL_POINTER && plot->tool != SG_TOOL_MARKERS)
    gtk_label_set_text(GTK_LABEL(plot->label), "");
}


void
sg_plot_toolbox_init(SGplot *plot)
{
 gint i, j;

 for(i = 0; i < 2; i++){
   for(j = 0; j < 4; j++){
      gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->button[j][i]),
                         "toggled",
                         GTK_SIGNAL_FUNC(toolbox_select), plot);
   }
 }

 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->button[3+NTOP][0]),
                    "clicked",
                    GTK_SIGNAL_FUNC(toolbox_zoomout), plot);

 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->button[3+NTOP][1]),
                    "clicked",
                    GTK_SIGNAL_FUNC(toolbox_zoomin), plot);

 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->button[2+NTOP][1]),
                    "clicked",
                    GTK_SIGNAL_FUNC(toolbox_fitpage), plot);

 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->button[2+NTOP][0]),
                    "clicked",
                    GTK_SIGNAL_FUNC(toolbox_autoscale), plot);

 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->button[4+NTOP][0]),
                    "clicked",
                    GTK_SIGNAL_FUNC(toolbox_up), plot);

 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->button[4+NTOP][1]),
                    "clicked",
                    GTK_SIGNAL_FUNC(toolbox_down), plot);

 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->button[5+NTOP][0]),
                    "clicked",
                    GTK_SIGNAL_FUNC(toolbox_left), plot);

 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->button[5+NTOP][1]),
                    "clicked",
                    GTK_SIGNAL_FUNC(toolbox_right), plot);

 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->reset),
                    "clicked",
                    GTK_SIGNAL_FUNC(toolbox_reset), plot);

 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->rotate_x),
                    "button_press_event",
                    GTK_SIGNAL_FUNC(toolbox_rotate_x), plot);
 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->rotate_x),
                    "button_release_event",
                    GTK_SIGNAL_FUNC(button_release), NULL);

 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->rotate_y),
                    "button_press_event",
                    GTK_SIGNAL_FUNC(toolbox_rotate_y), plot);
 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->rotate_y),
                    "button_release_event",
                    GTK_SIGNAL_FUNC(button_release), NULL);

 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->rotate_z),
                    "button_press_event",
                    GTK_SIGNAL_FUNC(toolbox_rotate_z), plot);
 gtk_signal_connect(GTK_OBJECT(SG_TOOLBOX(plot->toolbox)->rotate_z),
                    "button_release_event",
                    GTK_SIGNAL_FUNC(button_release), NULL);

 gtk_signal_connect(GTK_OBJECT(plot->real_canvas), "select_region",
                    GTK_SIGNAL_FUNC(tool_select_region), plot);


 init_tool(plot);
}

static void
init_tool(SGplot *plot)
{
  gint row = 0, col = 0;

  switch(plot->tool){
   case SG_TOOL_ARROW:
      row = 0;
      col = 0;
      break;
   case SG_TOOL_ZOOM:
      row = 0;
      col = 1;
   case SG_TOOL_POINTER:
      row = 1;
      col = 0;
      break;
   case SG_TOOL_MARKERS:
      row = 1;
      col = 1;
      break;
   case SG_TOOL_TEXT:
      row = 2;
      col = 0;
      break;
   case SG_TOOL_LINE:
/*
      row = 2;
      col = 0;
*/
      break;
   case SG_TOOL_LINE_ARROW:
      row = 2;
      col = 1;
      break;
   case SG_TOOL_RECTANGLE:
      row = 3;
      col = 0;
      break;
   case SG_TOOL_ELLIPSE:
      row = 3;
      col = 1;
      break;
  }
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SG_TOOLBOX(plot->toolbox)->button[row][col]), TRUE);
                           
}

GtkWidget *
sg_plot_build_toolbar(SGplot *plot)
{
  GnomeUIInfo plot_toolbar[] =
  {
      GNOMEUIINFO_ITEM_DATA("2D", "New 2D layer", plot_new2d,
                             plot,
                             new_2d_xpm),
      GNOMEUIINFO_ITEM_DATA("3D", "New 3D layer", plot_new3d,
                             plot,
                             new_3d_xpm),
      GNOMEUIINFO_ITEM_DATA("Polar", "New polar layer", plot_new_polar,
                             plot,
                             new_polar_xpm),
      GNOMEUIINFO_SEPARATOR,
      GNOMEUIINFO_ITEM_DATA("Print", "Print plot", plot_print,
                             plot,
                             print_xpm),
      GNOMEUIINFO_SEPARATOR,
      GNOMEUIINFO_ITEM_DATA("Refresh", "Refresh datasets in plot", plot_refresh,
                             plot,
                             reload_xpm),
      GNOMEUIINFO_ITEM_DATA("Arrange", "Arrange layers", plot_arrange,
                             plot,
                             arrange_xpm),
      GNOMEUIINFO_ITEM_DATA("Control", "Lauch layer control dialog", 
			     plot_layer_control,
                             plot,
                             layer_control_xpm),
      GNOMEUIINFO_SEPARATOR,
      GNOMEUIINFO_ITEM_DATA("Fit Page", "Fit to window size", plot_fit,
                             plot,
                             fit5_xpm),
      GNOMEUIINFO_ITEM_DATA("Zoom in", "Larger page view", plot_zoom_in,
                             plot,
                             zoomin_xpm),
      GNOMEUIINFO_ITEM_DATA("Zoom out", "Smaller page view", plot_zoom_out,
                             plot,
                             zoomout_xpm),
      GNOMEUIINFO_END
  };

#ifdef WITH_GNOME
  gnome_app_create_toolbar(GNOME_APP(plot) , plot_toolbar);
#else
  sg_ui_create_toolbar(GTK_BIN(plot)->child, plot_toolbar);
#endif /* WITH_GNOME */

  return NULL;
}

static void
plot_print (GtkWidget *widget, gpointer data)
{
  active_plot = (SGplot *) data;
  sg_plot_export();
}

static void
plot_new2d (GtkWidget *widget, gpointer data)
{
  SGlayer *layer;

  active_plot = (SGplot *) data;

  layer = sg_layer_new(SG_LAYER_2D, .65, .45);
  sg_plot_add_layer(active_plot, layer, .175, .15);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(active_plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(active_plot->real_canvas));
}

static void
plot_new3d (GtkWidget *widget, gpointer data)
{
  SGlayer *layer;

  active_plot = (SGplot *) data;

  layer = sg_layer_new(SG_LAYER_3D, .65, .45);
  sg_plot_add_layer(active_plot, layer, .175, .15);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(active_plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(active_plot->real_canvas));
}

static void
plot_new_polar (GtkWidget *widget, gpointer data)
{
  SGlayer *layer;

  active_plot = (SGplot *) data;

  layer = sg_layer_new(SG_LAYER_POLAR, .75, .45);
  sg_plot_add_layer(active_plot, layer, .175, .15);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(active_plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(active_plot->real_canvas));
}

static void
plot_refresh (GtkWidget *widget, gpointer data)
{
  active_plot = (SGplot *) data;
  sg_plot_refresh_datasets(active_plot);
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(active_plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(active_plot->real_canvas));
}

static void
plot_arrange (GtkWidget *widget, gpointer data)
{
  active_plot = (SGplot *) data;
  sg_arrange_dialog(GTK_PLOT_CANVAS(active_plot->real_canvas));
}

static void
plot_fit (GtkWidget *widget, gpointer data)
{
  sg_plot_fit_page((SGplot *)data);
}

static void
plot_layer_control (GtkWidget *widget, gpointer data)
{
  gchar path[2000];

  active_plot = (SGplot *) data;
  g_snprintf(path, 2000, "%s", active_plot->name);

  sg_layer_control(path);
}

static void
plot_zoom_in (GtkWidget *widget, gpointer data)
{
  active_plot = (SGplot *) data;
  sg_plot_rescale(active_plot, active_plot->scale + .1);
}

static void
plot_zoom_out (GtkWidget *widget, gpointer data)
{
  active_plot = (SGplot *) data;
  sg_plot_rescale(active_plot, active_plot->scale - .1);
}

static void     
button_release                     	(GtkButton *button,
					 GdkEventButton *event,
                                         gpointer data)
{
  event->button = 1;
}
