#  Copyright (C) 1999-2012
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc OpenFits {{layer {}}} {
    global current

    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    # just in case (could be invoked via a menu keyshortcut)
    if {$current(frame) == {}} {
	CreateFrame
    }

    if {$layer == {mask}} {
	if {![MaskLoad]} {
	    return
	}
    }

    StartLoad
    LoadFits $fn $layer
    FinishLoad
}

proc OpenMECubeFits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    StartLoad
    LoadMECubeFits $fn
    FinishLoad
}

proc OpenSliceFits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    StartLoad
    LoadSliceFits $fn
    FinishLoad
}

proc OpenMultiFrameFits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    StartLoad
    LoadMultiFrameFits $fn
    FinishLoad
}

proc OpenMosaicImageIRAFFits {{layer {}}} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    if {$layer == {mask}} {
	if {![MaskLoad]} {
	    return
	}
    }

    StartLoad
    LoadMosaicImageIRAFFits $fn $layer
    FinishLoad
}

proc OpenMosaicIRAFFits {{layer {}}} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    if {$layer == {mask}} {
	if {![MaskLoad]} {
	    return
	}
    }

    StartLoad
    LoadMosaicIRAFFits $fn $layer
    FinishLoad
}

proc OpenMosaicImageWCSFits {{layer {}}} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    set sys {}
    if [MosaicWCSDialog sys] {
	if {$layer == {mask}} {
	    if {![MaskLoad]} {
		return
	    }
	}

	StartLoad
	LoadMosaicImageWCSFits $sys $fn $layer
	FinishLoad
    }
}

proc OpenMosaicWCSFits {{layer {}}} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    set sys {}
    if [MosaicWCSDialog sys] {
	if {$layer == {mask}} {
	    if {![MaskLoad]} {
		return
	    }
	}

	StartLoad
	LoadMosaicWCSFits $sys $fn $layer
	FinishLoad
    }
}

proc OpenMosaicImageWFPC2Fits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    StartLoad
    LoadMosaicImageWFPC2Fits $fn
    FinishLoad
}

proc OpenRGBImageFits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    StartLoad
    LoadRGBImageFits $fn
    FinishLoad
}

proc OpenRGBCubeFits {} {
    set fn [OpenFileDialog fitsfbox]
    if {[string length $fn] == 0} {
	return
    }

    StartLoad
    LoadRGBCubeFits $fn
    FinishLoad
}

proc OpenVar {} {
    set fn [OpenFileDialog fitsvarfbox]
    if {[string length $fn] == 0} {
	return
    }

    if {![catch {set ch [open "$fn"]}]} {
	fconfigure $ch -translation binary -encoding binary
	global vardata
	set vardata [read -nonewline $ch]
	close $ch

	StartLoad
	LoadVar vardata $fn {}
	FinishLoad
    }
}

# Load

proc LoadFits {fn {layer {}}} {
    global current
    global pds9
    global loadParam
    global marker

    set loadParam(load,type) mmapincr
    set loadParam(load,layer) $layer
    set loadParam(file,type) fits
    set loadParam(file,mode) {}
    set loadParam(file,name) $fn
    ConvertFile
    # save load type, since ProcessLoad will clear loadParam
    if {$loadParam(load,type) == "mmapincr"} {
	set mmap 1
    } else {
	set mmap 0
    }
    ProcessLoad

    # now autoload markers
    if {$pds9(automarker) && $mmap} {
	# now, load fits[REGION] if present
	set id [string first "\[" $fn]
	if {$id > 0} {
	    set base [string range $fn 0 [expr $id-1]]
	} else {
	    set base $fn
	}

	set reg "${base}\[REGION\]"
	if {[$current(frame) fitsy has ext "\"$reg\""]} {
	    RealizeDS9
	    catch {
		$current(frame) marker load fits "\"$reg\"" $marker(color) $marker(dashlist) $marker(width) "\{$marker(font) $marker(font,size) $marker(font,weight) $marker(font,slant)\}"
	    }
	}
    }
}

proc LoadSFits {hdr fn {layer {}}} {
    global current
    global ds9
    global loadParam
    global marker

    set loadParam(load,type) smmap
    set loadParam(load,layer) $layer
    set loadParam(file,type) fits
    set loadParam(file,mode) {}
    set loadParam(file,name) $fn
    set loadParam(file,header) $hdr
    ProcessLoad
}

proc LoadMECubeFits {fn} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(load,layer) {}
    set loadParam(file,type) fits
    set loadParam(file,mode) {ext cube}
    set loadParam(file,name) $fn
    ConvertFile
    ProcessLoad
}

proc LoadSliceFits {fn} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(load,layer) {}
    set loadParam(file,type) fits
    set loadParam(file,mode) {slice}
    set loadParam(file,name) $fn
    ConvertFile
    ProcessLoad
}

proc LoadMultiFrameFits {fn} {
    global loadParam
    global current
    global ds9

    set ext 0
    set cnt 0
    set did 0
    set need 0

    # start with new frame?
    switch -- [$current(frame) get type] {
	base {
	    if [$current(frame) has fits] {
		CreateFrame
		set did 1
	    }
	}
	rgb -
	3d {
	    CreateFrame
	    set did 1
	}
    }

    while {1} {

	# create a new frame
	if {$need} {
	    CreateFrame
	    set did 1
	}

	set loadParam(load,type) mmapincr
	set loadParam(load,layer) {}
	set loadParam(file,type) fits
	set loadParam(file,mode) {}
	set loadParam(file,name) "$fn\[$ext\]"
	ConvertFile 

	set rr [ProcessLoad 0]
	if  {$rr != {}} {
	    if {$ext} {
		# reset xpa error
		global errorInfo
		set errorInfo {}

		if {$did} {
		    DeleteCurrentFrame
		    incr ds9(next,num) -1
		}
		if {!$cnt} {
		    Error "$rr"
		}
		break;
	    }
	} else {
	    # ignore any bin tables
	    if {![$current(frame) has fits bin]}  {
		incr cnt
		set need 1
	    } else {
		set need 0
	    }
	}

	incr ext
    }

    # go into tile mode if more than one
    if {$cnt && $current(display) != "tile"} {
	set current(display) tile
	DisplayMode
    }
}

proc LoadMosaicImageIRAFFits {fn {layer {}}} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(load,layer) $layer
    set loadParam(file,type) fits
    set loadParam(file,mode) {mosaic image iraf}
    set loadParam(file,name) $fn
    ConvertFile
    ProcessLoad
}

proc LoadMosaicIRAFFits {fn {layer {}}} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(load,layer) $layer
    set loadParam(file,type) fits
    set loadParam(file,mode) {mosaic iraf}
    set loadParam(file,name) $fn
    ConvertFile
    ProcessLoad
}

proc LoadMosaicIRAFSFits {hdr fn {layer {}}} {
    global loadParam

    set loadParam(load,type) smmap
    set loadParam(load,layer) $layer
    set loadParam(file,type) fits
    set loadParam(file,mode) {mosaic iraf}
    set loadParam(file,name) $fn
    set loadParam(file,header) $hdr
    ProcessLoad
}

proc LoadMosaicImageWCSFits {sys fn {layer {}}} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(load,layer) $layer
    set loadParam(file,type) fits
    set loadParam(file,mode) [list mosaic image $sys]
    set loadParam(file,name) $fn
    ConvertFile
    ProcessLoad
}

proc LoadMosaicWCSFits {sys fn {layer {}}} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(load,layer) $layer
    set loadParam(file,type) fits
    set loadParam(file,mode) [list mosaic $sys]
    set loadParam(file,name) $fn
    ConvertFile
    ProcessLoad
}

proc LoadMosaicWCSSFits {sys hdr fn {layer {}}} {
    global loadParam

    set loadParam(load,type) smmap
    set loadParam(load,layer) $layer
    set loadParam(file,type) fits
    set loadParam(file,mode) [list mosaic $sys]
    set loadParam(file,name) $fn
    set loadParam(file,header) $hdr
    ProcessLoad
}

proc LoadMosaicImageWFPC2Fits {fn} {
    global loadParam

    set loadParam(load,type) mmapincr
    set loadParam(load,layer) {}
    set loadParam(file,type) fits
    set loadParam(file,mode) {mosaic image wfpc2}
    set loadParam(file,name) $fn
    ConvertFile
    ProcessLoad
}

proc LoadRGBImageFits {fn} {
    global loadParam
    global current

    switch -- [$current(frame) get type] {
	base -
	3d {Error [msgcat::mc {Unable to load RGB image into a non-rgb frame}]}
	rgb {
	    set loadParam(load,type) mmapincr
	    set loadParam(load,layer) {}
	    set loadParam(file,type) fits
	    set loadParam(file,mode) {rgb image}
	    set loadParam(file,name) $fn
	    ConvertFile
	    ProcessLoad
	}
    }
}

proc LoadRGBCubeFits {fn} {
    global loadParam
    global current

    switch -- [$current(frame) get type] {
	base -
	3d {Error [msgcat::mc {Unable to load RGB image into a non-rgb frame}]}
	rgb {
	    set loadParam(load,type) mmapincr
	    set loadParam(load,layer) {}
	    set loadParam(file,type) fits
	    set loadParam(file,mode) {rgb cube}
	    set loadParam(file,name) $fn
	    ConvertFile
	    ProcessLoad
	}
    }
}

proc LoadRGBCubeSFits {hdr fn} {
    global loadParam
    global current

    switch -- [$current(frame) get type] {
	base -
	3d {Error [msgcat::mc {Unable to load RGB image into a non-rgb frame}]}
	rgb {
	    set loadParam(load,type) smmap
	    set loadParam(load,layer) {}
	    set loadParam(file,type) fits
	    set loadParam(file,mode) {rgb cube}
	    set loadParam(file,name) $fn
	    set loadParam(file,header) $hdr
	    ProcessLoad
	}
    }
}

proc LoadVar {varname fn mode {layer {}}} {
    global loadParam

    set loadParam(load,type) var
    set loadParam(load,layer) $layer
    set loadParam(var,name) $varname
    set loadParam(file,type) fits
    set loadParam(file,name) "$fn"
    set loadParam(file,mode) $mode
    ProcessLoad
}

# Support

proc MosaicWCSDialog {varname} {
    upvar $varname var
    global ed

    set w {.wcs}

    set ed(ok) 0
    set ed(sys) wcs
    set ed(label) WCS

    DialogCreate $w [msgcat::mc {Load Mosaic}] ed(ok)

    # Param
    set f [ttk::frame $w.param]

    ttk::label $f.title -text [msgcat::mc {Select Coordinate System }]
    ttk::menubutton $f.sys -textvariable ed(label) \
	-menu $f.sys.m -width 10

    menu $f.sys.m
    $f.sys.m add radiobutton -label [msgcat::mc {WCS}] \
	-variable ed(sys) -value "wcs" -command [list set ed(label) WCS]
    $f.sys.m add separator
    foreach l {a b c d e f g h i j k l m n o p q r s t u v w x y z} {
	$f.sys.m add radiobutton -variable ed(sys) \
	    -label "[msgcat::mc {WCS}] $l" \
	    -value "wcs$l" \
	    -command [list set ed(label) "[msgcat::mc {WCS}] $l"]
    }

    grid $f.title $f.sys

    # Buttons
    set f [ttk::frame $w.buttons]
    ttk::button $f.ok -text [msgcat::mc {OK}] -command {set ed(ok) 1} \
	-default active 
    ttk::button $f.cancel -text [msgcat::mc {Cancel}] -command {set ed(ok) 0}
    pack $f.ok $f.cancel -side left -expand true -padx 2 -pady 4

    bind $w <Return> {set ed(ok) 1}

    # Fini
    ttk::separator $w.sep -orient horizontal
    pack $w.buttons $w.sep -side bottom -fill x
    pack $w.param -side top -fill both -expand true

    DialogCenter $w 
    DialogWait $w ed(ok)
    DialogDismiss $w

    if {$ed(ok)} {
	set var $ed(sys)
    }

    set rr $ed(ok)
    unset ed
    return $rr
}

# File

proc ProcessFileCmd {varname iname sock fn} {
    upvar $varname var
    upvar $iname i

    global current

    set layer {}

    switch -- [string tolower [lindex $var $i]] {
	new {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		rgbimage -
		rgbcube -
		rgbarray {
		    incr i
		    CreateRGBFrame
		}
		default {
		    CreateFrame
		}
	    }
	}
	mask {
	    incr i
	    set layer mask
	}
    }

    switch -- [string tolower [lindex $var $i]] {
	save {
	    # backward compatibility, use SAVE and SAVEIMAGE
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		gz {
		    incr i
		    $current(frame) save fits image file [lindex $var $i] gz
		}
		resample {
		    incr i
		    switch -- [string tolower [lindex $var $i]] {
			gz {
			    incr i
			    $current(frame) save fits resample file [lindex $var $i] gz
			}
			default {
			    $current(frame) save fits resample file [lindex $var $i]
			}
		    }
		}
		default {
		    $current(frame) save fits image file [lindex $var $i]
		}
	    }
	}

	default {
	    StartLoad
	    switch -- [string tolower [lindex $var $i]] {
		array {
		    incr i
		    LoadArray [lindex $var $i] $layer
		}

		fits {
		    incr i
		    switch -- [lindex $var $i] {
			slice {
			    incr i
			    LoadSliceFits [lindex $var $i]
			}
			default {
			    LoadFits [lindex $var $i] $layer
			}
		    }
		}
		sfits {
		    incr i
		    LoadSFits [lindex $var $i] [lindex $var [expr $i+1]] $layer
		    incr i
		}

		datacube -
		mecube -
		medatacube {
		    incr i
		    LoadMECubeFits [lindex $var $i]
		}

		multiframe {
		    incr i
		    LoadMultiFrameFits [lindex $var $i]
		}

		mosaicimage {
		    incr i
		    switch -- [lindex $var $i] {
			iraf {
			    incr i
			    LoadMosaicImageIRAFFits [lindex $var $i] $layer
			}
			wfpc2 {
			    incr i
			    LoadMosaicImageWFPC2Fits [lindex $var $i]
			}
			default {
			    if {[string range [lindex $var $i] 0 2] == "wcs"} {
				LoadMosaicImageWCSFits [lindex $var $i] [lindex $var [expr $i+1]] $layer
			    incr i
			    } else {
				LoadMosaicImageWCSFits wcs [lindex $var $i] $layer
			    }
			}
		    }
		}
		mosaic {
		    incr i
		    switch -- [lindex $var $i] {
			iraf {
			    incr i
			    LoadMosaicIRAFFits [lindex $var $i] $layer
			}
			default {
			    if {[string range [lindex $var $i] 0 2] == "wcs"} {
				LoadMosaicWCSFits [lindex $var $i] [lindex $var [expr $i+1]] $layer
				incr i
			    } else {
				LoadMosaicWCSFits wcs [lindex $var $i] $layer
			    }
			}
		    }
		}
		smosaic {
		    incr i
		    switch -- [lindex $var $i] {
			iraf {
			    incr i
			    LoadMosaicIRAFSFits [lindex $var $i] [lindex $var [expr $i+1]] $layer
			    incr i
			}
			default {
			    if {[string range [lindex $var $i] 0 2] == "wcs"} {
				LoadMosaicWCSSFits [lindex $var $i] [lindex $var [expr $i+1]] [lindex $var [expr $i+2]] $layer
			    incr i 2
			    } else {
				LoadMosaicWCSSFits wcs [lindex $var $i] [lindex $var [expr $i+1]] $layer
			    }
			    incr i
			}
		    }
		}

		mosaicimageiraf {
		    # backward compatibility
		    incr i
		    LoadMosaicImageIRAFFits [lindex $var $i] $layer
		}
		mosaiciraf {
		    # backward compatibility
		    incr i
		    LoadMosaicIRAFFits [lindex $var $i] $layer
		}
		mosaicimagewcs {
		    # backward compatibility
		    incr i
		    LoadMosaicImageWCSFits wcs [lindex $var $i] $layer
		}
		mosaicwcs {
		    # backward compatibility
		    incr i
		    LoadMosaicWCSFits wcs [lindex $var $i] $layer
		}
		mosaicwfpc2 -
		mosaicimagewfpc2 {
		    # backward compatibility
		    incr i
		    LoadMosaicImageWFPC2Fits [lindex $var $i]
		}

		photo {
		    incr i
		    switch -- [lindex $var $i] {
			slice {
			    incr i
			    LoadSlicePhoto [lindex $var $i]
			}
			default {
			    LoadPhoto [lindex $var $i]
			}
		    }
		}
		rgbcube {
		    incr i
		    LoadRGBCubeFits [lindex $var $i]
		}
		srgbcube {
		    incr i
		    LoadRGBCubeSFits [lindex $var $i] [lindex $var [expr $i+1]]
		    incr i
		}
		rgbimage {
		    incr i
		    LoadRGBImageFits [lindex $var $i]
		}
		rgbarray {
		    incr i
		    LoadRGBArray [lindex $var $i]
		}

		url {
		    incr i
		    LoadURL [lindex $var $i]
		}

		slice {
		    incr i
		    LoadSliceFits [lindex $var $i]
		}
		default {
		    LoadFits [lindex $var $i] $layer
		}
	    }
	    FinishLoad
	}
    }
}

proc ProcessSendFileCmd {proc id param} {
    global current

    if {$current(frame) != {}} {
	$proc $id "[$current(frame) get fits file name full]\n"
    }
}

proc ProcessFitsCmd {varname iname sock fn} {
    upvar $varname var
    upvar $iname i

    global loadParam
    global current

    set loadParam(file,type) fits
    set loadParam(load,layer) {}

    switch -- [string tolower [lindex $var $i]] {
	new {
	    incr i
	    CreateFrame
	}
	mask {
	    incr i
	    set loadParam(load,layer) mask
	}
    }

    # mode/name
    set loadParam(file,mode) {}
    set loadParam(file,name) {}

    switch -- [string tolower [lindex $var $i]] {
	datacube -
	mecube -
	medatacube {
	    incr i
	    set loadParam(file,mode) {ext cube}
	    set loadParam(file,name) [lindex $var $i]
	}

	mosaicimage {
	    incr i
	    switch -- [lindex $var $i] {
		iraf -
		wfpc2 {
		    set loadParam(file,mode) [list mosaic image [lindex $var $i]]
		    incr i
		    set loadParam(file,name) [lindex $var $i]
		}
		default {
		    if {[string range [lindex $var $i] 0 2] == "wcs"} {
			set loadParam(file,mode) [list mosaic image [lindex $var $i]]
			incr i
			set loadParam(file,name) [lindex $var $i]
		    } else {
			set loadParam(file,mode) [list mosaic image wcs]
			set loadParam(file,name) [lindex $var $i]
		    }
		}
	    }
	}
	mosaic {
	    incr i
	    switch -- [lindex $var $i] {
		iraf {
		    set loadParam(file,mode) [list mosaic [lindex $var $i]]
		    incr i
		    set loadParam(file,name) [lindex $var $i]
		}
		default {
		    if {[string range [lindex $var $i] 0 2] == "wcs"} {
			set loadParam(file,mode) [list mosaic [lindex $var $i]]
			incr i
			set loadParam(file,name) [lindex $var $i]
		    } else {
			set loadParam(file,mode) [list mosaic wcs]
			set loadParam(file,name) [lindex $var $i]
		    }
		}
	    }
	}

	mosaicimageiraf {
	    # backward compatibility
	    incr i
	    set loadParam(file,mode) {mosaic image iraf}
	    set loadParam(file,name) [lindex $var $i]
	}
	mosaiciraf {
	    # backward compatibility
	    incr i
	    set loadParam(file,mode) {mosaic iraf}
	    set loadParam(file,name) [lindex $var $i]
	}
	mosaicimagewcs {
	    # backward compatibility
	    incr i
	    set loadParam(file,mode) {mosaic image wcs}
	    set loadParam(file,name) [lindex $var $i]
	}
	mosaicwcs {
	    # backward compatibility
	    incr i
	    set loadParam(file,mode) {mosaic wcs}
	    set loadParam(file,name) [lindex $var $i]
	}
	mosaicwfpc2 -
	mosaicimagewfpc2 {
	    # backward compatibility
	    incr i
	    set loadParam(file,mode) {mosaic image wfpc2}
	    set loadParam(file,name) [lindex $var $i]
	}

	rgbimage {
	    incr i
	    switch -- [$current(frame) get type] {
		base -
		3d {
		    Error [msgcat::mc {Unable to load RGB image into a non-rgb frame}]
		    return
		}
		rgb {
		    incr i
		    set loadParam(file,mode) {rgb image}
		    set loadParam(file,name) [lindex $var $i]
		}
	    }
	}
	rgbcube {
	    incr i
	    switch -- [$current(frame) get type] {
		base -
		3d {
		    Error [msgcat::mc {Unable to load RGB image into a non-rgb frame}]
		    return
		}
		rgb {
		    incr i
		    set loadParam(file,mode) {rgb cube}
		    set loadParam(file,name) [lindex $var $i]
		}
	    }
	}
	slice {
	    incr i
	    set loadParam(file,mode) {slice}
	    set loadParam(file,name) [lindex $var $i]
	}

	default {
	    set loadParam(file,name) [lindex $var $i]
	}
    }
    if {$loadParam(file,name) == {}} {
	set loadParam(file,name) stdin
    }

    StartLoad
    if {$sock != {}} {
	set loadParam(load,type) socketgz
	set loadParam(socket,id) $sock
    } else {
	set loadParam(load,type) allocgz
	set loadParam(file,fn) $fn
    }
    ProcessLoad
    FinishLoad
}

proc ProcessSendFitsCmd {proc id param sock fn} {
    global current

    if {$current(frame) == {}} {
	return
    }

    switch -- [string tolower [lindex $param 0]] {
	size {
	    set sys [lindex $param 1] 
	    set sky [lindex $param 2] 
	    set format [lindex $param 3]
	    if {$sys == {} && $sky == {} && $format == {}} {
		$proc $id "[$current(frame) get fits size]\n"
	    } else {
		FixSpec sys sky format image fk5 degrees
		$proc $id "[$current(frame) get fits size $sys $sky $format]\n"
	    }
	}
	width {$proc $id "[$current(frame) get fits width]\n"}
	height {$proc $id "[$current(frame) get fits height]\n"}
	bitpix {$proc $id "[$current(frame) get fits bitpix]\n"}
	depth {$proc $id "[$current(frame) get fits depth 2]\n"}
	header {
	    switch [llength $param] {
		1 {ProcessSend $proc $id {} $fn {.txt} \
			"[$current(frame) get fits header 1]\n"}
		2 {ProcessSend $proc $id {} $fn {.txt} \
			"[$current(frame) get fits header [lindex $param 1]]\n"}
		3 {$proc $id "[string trim [$current(frame) get fits header 1 keyword [lindex $param 2]]]\n"}
		4 {$proc $id "[string trim [$current(frame) get fits header [lindex $param 1] keyword [lindex $param 3]]]\n"}
	    }
	}
	type {
	    if {[$current(frame) has fits bin]} {
		$proc $id "table\n"
	    } else {
		$proc $id "image\n"
	    }
	}
	table {
	    if {$sock != {}} {
		$current(frame) save fits table socket $sock [lindex $param 1]
	    } elseif {$fn != {}} {
		if {[lindex $param 1] != {}} {
		    append fn ".fits.[lindex $param 1]"
		} else {
		    append fn {.fits}
		}
		$current(frame) save fits table file "\{$fn\}" [lindex $param 1]
		$proc $id {} $fn
	    }
	}
	image {
	    if {$sock != {}} {
		$current(frame) save fits image socket $sock [lindex $param 1]
	    } elseif {$fn != {}} {
		if {[lindex $param 1] != {}} {
		    append fn ".fits.[lindex $param 1]"
		} else {
		    append fn {.fits}
		}
		$current(frame) save fits image file "\{$fn\}" [lindex $param 1]
		$proc $id {} $fn
	    }
	}
	resample {
	    if {$sock != {}} {
		$current(frame) save fits resample socket $sock [lindex $param 1]
	    } elseif {$fn != {}} {
		if {[lindex $param 1] != {}} {
		    append fn ".fits.[lindex $param 1]"
		} else {
		    append fn {.fits}
		}
		$current(frame) save fits resample file "\{$fn\}" [lindex $param 1]
		$proc $id {} $fn
	    }
	}
	gz {
	    if {$sock != {}} {
		$current(frame) save fits image socket $sock gz
	    } elseif {$fn != {}} {
		append fn {.fits.gz}
		$current(frame) save fits image file "\{$fn\}" gz
		$proc $id {} $fn
	    }
	}
	default {
	    if {$sock != {}} {
		$current(frame) save fits image socket $sock
	    } elseif {$fn != {}} {
		append fn {.fits}
		$current(frame) save fits image file "\{$fn\}"
		$proc $id {} $fn
	    }
	}
    }
}

proc ProcessShmCmd {varname iname ml} {
    upvar $varname var
    upvar $iname i

    global loadParam
    global current
    global ds9

    StartLoad
    set done 0
    while {!$done} {

	# defaults
	set loadParam(load,type) shared
	set loadParam(load,layer) {}
	set loadParam(file,type) fits
	set loadParam(file,mode) {}

	set nn [lindex $var [expr $i+4]]
	if {$nn == {} || [string range $nn 0 0] == "-"} {
	    set def 1
	} else {
	    set def 0
	}

	switch -- [lindex $var $i] {
	    key -
	    shmid {
		if {$ml} {
		    MultiLoad
		}
		set loadParam(shared,idtype) [lindex $var $i]
		set loadParam(shared,id) [lindex $var [expr $i+1]]
		set loadParam(file,name) [lindex $var [expr $i+2]]
		incr i 2
	    }

	    fits {
		if {$ml} {
		    MultiLoad
		}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    sfits {
		if {$ml} {
		    MultiLoad
		}
		set loadParam(load,type) sshared
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,hdr) [lindex $var [expr $i+2]]
		set loadParam(shared,id) [lindex $var [expr $i+3]]
		set loadParam(file,name) [lindex $var [expr $i+4]]
		incr i 4
	    }

	    mosaicimage {
		if {$ml} {
		    MultiLoad
		}
		if {$def} {
		    set loadParam(file,mode) {mosaic image iraf}
		    set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		    set loadParam(shared,id) [lindex $var [expr $i+2]]
		    set loadParam(file,name) [lindex $var [expr $i+3]]
		    incr i 3
		} else {
		    set loadParam(file,mode) \
			[list mosaic image [lindex $var [expr $i+1]]]
		    set loadParam(shared,idtype) [lindex $var [expr $i+2]]
		    set loadParam(shared,id) [lindex $var [expr $i+3]]
		    set loadParam(file,name) [lindex $var [expr $i+4]]
		    incr i 4
		}
	    }
	    mosaic {
		if {$def} {
		    set loadParam(file,mode) {mosaic iraf}
		    set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		    set loadParam(shared,id) [lindex $var [expr $i+2]]
		    set loadParam(file,name) [lindex $var [expr $i+3]]
		    incr i 3
		} else {
		    set loadParam(file,mode) \
			[list mosaic [lindex $var [expr $i+1]]]
		    set loadParam(shared,idtype) [lindex $var [expr $i+2]]
		    set loadParam(shared,id) [lindex $var [expr $i+3]]
		    set loadParam(file,name) [lindex $var [expr $i+4]]
		    incr i 4
		}
	    }
	    smosaic {
		set loadParam(load,type) sshared
		set loadParam(file,mode) \
		    [list mosaic [lindex $var [expr $i+1]]]
		set loadParam(shared,idtype) [lindex $var [expr $i+2]]
		set loadParam(shared,hdr) [lindex $var [expr $i+3]]
		set loadParam(shared,id) [lindex $var [expr $i+4]]
		set loadParam(file,name) [lindex $var [expr $i+5]]
		incr i 5
	    }

	    mosaicimageiraf {
		# backward compatibility
		if {$ml} {
		    MultiLoad
		}
		set loadParam(file,mode) {mosaic image iraf}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    mosaiciraf {
		# backward compatibility
		set loadParam(file,mode) {mosaic iraf}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    mosaicimagewcs {
		# backward compatibility
		if {$ml} {
		    MultiLoad
		}
		set loadParam(file,mode) {mosaic image wcs}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    mosaicwcs {
		# backward compatibility
		set loadParam(file,mode) {mosaic wcs}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    mosaicimagewfpc2 {
		# backward compatibility
		if {$ml} {
		    MultiLoad
		}
		set loadParam(file,mode) {mosaic image wfpc2}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }

	    rgbcube {
		if {$ml} {
		    CreateRGBFrame
		}
		set loadParam(file,mode) {rgb cube}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    srgbcube {
		if {$ml} {
		    CreateRGBFrame
		}
		set loadParam(load,type) sshared
		set loadParam(file,mode) {rgb cube}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,hdr) [lindex $var [expr $i+2]]
		set loadParam(shared,id) [lindex $var [expr $i+3]]
		set loadParam(file,name) [lindex $var [expr $i+4]]
		incr i 4
	    }
	    rgbimage {
		if {$ml} {
		    CreateRGBFrame
		}
		set loadParam(file,mode) {rgb image}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    rgbarray {
		if {$ml} {
		    CreateRGBFrame
		}
		set loadParam(file,type) array
		set loadParam(file,mode) {rgb cube}
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }
	    array {
		if {$ml} {
		    MultiLoad
		}
		set loadParam(file,type) array
		set loadParam(shared,idtype) [lindex $var [expr $i+1]]
		set loadParam(shared,id) [lindex $var [expr $i+2]]
		set loadParam(file,name) [lindex $var [expr $i+3]]
		incr i 3
	    }

	    default {
		if {$ml} {
		    MultiLoad
		}
		set loadParam(shared,idtype) key
		set loadParam(shared,id) [lindex $var $i]
		set loadParam(file,name) [lindex $var [expr $i+1]]
		incr i 1
	    }
	}

	ProcessLoad

	# more to come?
	incr i
	if {([lindex $var $i] == "-shm") || 
	    ([lindex $var $i] == "shm")} {
	    set done 0
	    incr i
	} else {
	    set done 1
	    incr i -1
	}
    }
    FinishLoad
}

proc ProcessSendShmCmd {proc id param} {
    global current

    if {$current(frame) != {}} {
	$proc $id "[$current(frame) get fits file name full]\n"
    }
}

