// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "vect.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Vect::Vect(FrameBase* p, const Vector& ptr1,
	   const Vector& ptr2,
	   int arr,
	   const char* clr, int wth, const char* fnt, const char* txt, 
	   unsigned short prop, const char* cmt, 
	   const List<Tag>& tg, const List<CallBack>& cb) 
  : Line(p, ptr1, ptr2, 0, arr, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  strcpy(type,"vector");
}

Vect::Vect(FrameBase* p, const Vector& pt,
	   double mag, double ang,
	   int arr,
	   const char* clr, int wth, const char* fnt, const char* txt, 
	   unsigned short prop, const char* cmt, 
	   const List<Tag>& tg, const List<CallBack>& cb) 
  : Line(p, pt, pt, 0, arr, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  strcpy(type,"vector");
  p2 = Vector(mag,0) * Rotate(ang) * flip_ * Translate(p1);
  updateBBox();
}

void Vect::setPoints(const Vector& pt, double mag, double ang)
{
  p1 = pt;
  p2 = Vector(mag,0) * Rotate(ang) * flip_ * Translate(p1);
  updateBBox();

  doCallBack(CallBack::EDITCB);
}

// list

void Vect::list(ostream& str, CoordSystem sys, SkyFrame sky, 
		SkyFormat format, int conj, int strip)
{
  if (!strip) {
    FitsImage* ptr = parent->findFits(center);
    listPre(str, sys, sky, ptr, strip, 1);

    switch (sys) {
    case IMAGE:
    case PHYSICAL:
    case DETECTOR:
    case AMPLIFIER:
      {
	Vector v1 = ptr->mapFromRef(p1,sys);
	Vector v2 = ptr->mapFromRef(p2,sys);
	double mag = (p2-p1).length();
	double ang = (p2-p1).angle();
	str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] << ','
	    << ptr->mapLenFromRef(mag,sys) << ',' 
	    << radToDeg(parent->mapAngleFromRef(ang,sys)) << ')';
      }
      break;
    default:
      if (ptr->hasWCS(sys)) {
	if (ptr->hasWCSEqu(sys)) {
	  switch (format) {
	  case DEGREES:
	    {
	      Vector v1 = ptr->mapFromRef(p1,sys,sky);
	      Vector v2 = ptr->mapFromRef(p2,sys,sky);
	      double mag = (p2-p1).length();
	      double ang = (p2-p1).angle();
	      str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
		  << ','  << ptr->mapLenFromRef(mag,sys,ARCSEC) << "\""
		  << ',' << radToDeg(parent->mapAngleFromRef(ang,sys,sky)) 
		  << ')';
	    }
	    break;
	  case SEXAGESIMAL:
	    {
	      char buf[64];
	      char ra1[16], ra2[16];
	      char dec1[16], dec2[16];
	      double mag = (p2-p1).length();
	      double ang = (p2-p1).angle();
	      {
		ptr->mapFromRef(p1,sys,sky,format,buf,64);
		string x(buf);
		istringstream wcs(x);
		wcs >> ra1 >> dec1;
	      }
	      str << type << '(' << ra1 << ',' << dec1 
		  << ',' << ptr->mapLenFromRef(mag,sys,ARCSEC) << "\"" 
		  << ',' << radToDeg(parent->mapAngleFromRef(ang,sys,sky))
		  << ')';
	    }
	    break;
	  }
	}
	else {
	  Vector v1 = ptr->mapFromRef(p1,sys);
	  Vector v2 = ptr->mapFromRef(p2,sys);
	  double mag = (p2-p1).length();
	  double ang = (p2-p1).angle();
	  str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
	      << ','  << ptr->mapLenFromRef(mag,sys) 
	      << ',' << radToDeg(parent->mapAngleFromRef(ang,sys)) << ')';
	}
      }
    }

    listPost(str, conj, strip);
  }
}

void Vect::listPost(ostream& str, int conj, int strip)
{
  if (conj)
    str << " ||";

  FitsImage* ptr = parent->findFits(center);
  str << " vector=" << p2Arrow;
  listProperties(str, 0);
}
