/* Copyright (C) 1999-2005
 * Smithsonian Astrophysical Observatory, Cambridge, MA, USA
 * For conditions of distribution and use, see copyright notice in "copyright"
 */

%option noyywrap
%option caseless
%option never-interactive
%option c++

%{
  #include <stdio.h>
  #include <stdlib.h>
  #include <string.h>

  #include "util.h"
  #include "prosparser.H"

  extern prosFlexLexer* proslexx;
  extern int prossign;
%}

%x DISCARD

D   [0-9]
E   [Ee][+-]?{D}+

/* rules */

%%

<DISCARD>[\n]	{ // special case-- #\n
		  BEGIN INITIAL;
		  yyless(0);             // put back the terminator
		  strcpy(proslval.str,""); // feed a blank string
		  return STRING;
		}

<DISCARD>[^\n]* {  // Discard reset of line
		  BEGIN INITIAL;
		  strncpy(proslval.str,yytext,256);
		  return STRING;
		}

annulus			{return ANNULUS_;}
b1950			{return B1950_;}
box			{return BOX_;}
circle			{return CIRCLE_;}
debug			{return DEBUG_;}
ellipse			{return ELLIPSE_;}
ecliptic		{return ECLIPTIC_;}
equatorial		{return EQUATORIAL_;}
galactic		{return GALACTIC_;}
j2000			{return J2000_;}
logical			{return LOGICAL_;}
n			{return N_;}
off			{return OFF_;}
on			{return ON_;}
physical		{return PHYSICAL_;}
point			{return POINT_;}
polygon			{return POLYGON_;}
rotbox			{return ROTBOX_;}
version			{return VERSION_;}

[+-]?{D}+	{ // Integer
		  proslval.integer = atoi(yytext);
		  prossign = (yytext[0]!='-') ? 1 : -1;
		  return INT;
		}

[+-]?{D}+"."?({E})? |
[+-]?{D}*"."{D}+({E})? { // Real Number
		  proslval.real = atof(yytext);
		  return REAL;
		}

[+-]?{D}+"."?d |
[+-]?{D}*"."{D}+d { // degrees
		  yytext[yyleng-1] = '\0';
		  proslval.real = atof(yytext);
		  return DEGREE;
		}	

[+-]?{D}+"."?r |
[+-]?{D}*"."{D}+r { // radians
		  yytext[yyleng-1] = '\0';
		  proslval.real = atof(yytext);
		  return RADIAN;
		}

{D}+"."?' |
{D}*"."{D}+'	{ // minutes of arc
		  yytext[yyleng-1] = '\0';
		  proslval.real = atof(yytext);
		  return ARCMINUTE;
		}	

{D}+"."?\" |
{D}*"."{D}+\"	{ // seconds of arc
		  yytext[yyleng-1] = '\0';
		  proslval.real = atof(yytext);
		  return ARCSECOND;
		}

[+-]?{D}+h	{ // hours
		  yytext[yyleng-1] = '\0';
		  proslval.integer = atoi(yytext);
		  prossign = (yytext[0]!='-') ? 1 : -1;
		  return HOUR;
		}	

{D}+m		{ // minutes
		  yytext[yyleng-1] = '\0';
		  proslval.integer = atoi(yytext);
		  return MINUTE;
		}	

{D}+"."?s |
{D}*"."{D}+s	{ // seconds
		  yytext[yyleng-1] = '\0';
		  proslval.real = atof(yytext);
		  return SECOND;
		}

[+-]?{D}+:{D}+:{D}+"."? |
[+-]?{D}+:{D}+:{D}*"."{D}+ { // Sexagesimal
		  strncpy(proslval.str,yytext,256);
		  return SEXSTR;
		}	

\"[^\"\n]*\" | 
\'[^\'\n]*\'	{ // Quoted String
		  strncpy(proslval.str,yytext+1,yyleng-2); // skip the " " 
	          proslval.str[yyleng-2] = '\0'; // Remove the '"' 
		  return STRING;
		}

\{[^\}\n]*\} 	{ // Quoted String
		  strncpy(proslval.str,yytext+1,yyleng-2); // skip the '{'
	          proslval.str[yyleng-2] = '\0'; // Remove the '}'
		  return STRING;
		}

[0-9A-Za-z]+	{ // General String
		  strncpy(proslval.str,yytext,256);
		  return STRING;
		}

[ \t]+		{ // White Spaces
		} 

\r\n		{ // windows line feed
		  return '\n';
		}

\\n		{ // fake line feed
		  return '\n';
		}

\n		{ // linefeed
		  return '\n';
		}

<<EOF>>		{ // eof
		  return EOF_;
		}

.		{ // Else, return the char
		  return yytext[0];
		}

%%

void prosDiscard(int doit)
{
  if (proslexx)
    proslexx->begin(DISCARD, doit);
}

void prosFlexLexer::begin(int which, int doit)
{
  BEGIN which;
  if (doit)
    yyless(0);
}
