// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

%{
#define YYPARSE_PARAM fr
#define YYDEBUG 1

#define FRB ((FrameBase*)fr)
#define FR ((Frame*)fr)
#define FRRGB ((FrameRGB*)fr)
#define FITSPTR (FRB->findFits(1))

#include <math.h>
#include <string.h>
#include <iostream.h>

#include "framebase.h"
#include "frame.h"
#include "framergb.h"
#include "fitsimage.h"
#include "util.h"
#include "vector.h"
#include "coord.h"
#include "fvcontour.h"

extern int frlex(void);
extern void frerror(const char*);

int frsign;
int frsign2;

static CoordSystem currentCoord = IMAGE;
static SkyFrame currentSky = FK5;

static unsigned short defaultProps = Marker::SELECT | Marker::HIGHLITE |
		Marker::EDIT | Marker::MOVE | Marker::ROTATE | 
		Marker::DELETE | Marker::INCLUDE | Marker::SOURCE;
static unsigned short currentProps;
static char currentColor[16];
static int currentWidth;
static char currentFont[32];
static char currentText[80];

static List<Tag> taglist;
static List<CallBack> cblist;

static unsigned short propQMask;
static unsigned short propQValue;

static void setProps(unsigned short* props, unsigned short prop, int value);
%}

%union {
  char chr;
  char str[2048];
  void* ptr;
  int integer;
  double real;
  double vector[3];
}

%type <real> numeric
%type <integer> yesno

%type <real> angle
%type <real> optangle
%type <integer> precision
%type <real> sexagesimal
%type <vector> coord
%type <integer> coordSystem
%type <integer> wcsSystem
%type <integer> internalSystem
%type <integer> skyFrame
%type <integer> skyFormat

%type <integer> markerProperty
%type <integer> markerCallBack
%type <integer> markerFormat
%type <integer> markerLayer
%type <integer> pscolorspace
%type <integer> scaleType
%type <integer> shmType
%type <integer> colormapMode
%type <integer> contourmethod
%type <integer> incrLoad
%type <integer> gridType
%type <integer> compressType
%type <integer> fileNameType
%type <integer> smoothFunction

%token <real> REAL
%token <integer> INT
%token <str> STRING
%token <ptr> POINTER

%token <integer> HOUR
%token <integer> MINUTE
%token <real> SECOND
%token <real> DEGREE
%token <real> ARCMINUTE
%token <real> ARCSECOND
%token <real> RADIAN

%token <str> SEXSTR

%token ABOUT_
%token ALLOC_
%token ALLOCGZ_
%token ALIGN_
%token ALL_
%token AMPLIFIER_
%token ANALYSIS_
%token ANGLE_
%token ANNULUS_
%token APPEND_
%token ARCMIN_
%token ARCSEC_
%token AREA_
%token ARRAY_
%token ARROW_
%token AST_
%token AUTO_
%token AVERAGE_
%token B1950_
%token BACK_
%token BASE_
%token BBOX_
%token BEGIN_
%token BG_
%token BIN_
%token BITPIX_
%token BLOCK_
%token BOX_
%token BOXANNULUS_
%token BOXCAR_
%token BOXCIRCLE_
%token BPANDA_
%token BUFFER_
%token BW_
%token CALLBACK_
%token CANVAS_
%token CATALOG_
%token CENTER_
%token CHANNEL_
%token CIRCLE_
%token CIAO_
%token CLEAR_
%token CLICK_
%token CLIP_
%token COLORMAP_
%token COLOR_
%token COLORSCALE_
%token COLORSPACE_
%token COLS_
%token COMMAND_
%token COMPASS_
%token COMPOSITE_
%token CONTOUR_
%token COORDINATES_
%token COPY_
%token COUNT_
%token CPANDA_
%token CREATE_
%token CROSS_
%token CROSSHAIR_
%token CUBE_
%token CURSOR_
%token CUT_
%token CMYK_
%token DATA_
%token DATAMIN_
%token DATASEC_
%token DEBUG_
%token DEGREES_
%token DEFAULT_
%token DELETE_
%token DEPTH_
%token DETECTOR_
%token DIAMOND_
%token DS9_
%token EDIT_
%token ECLIPTIC_
%token ELLIPSE_
%token ELLIPSEANNULUS_
%token END_
%token EPANDA_
%token EQUATORIAL_
%token ERASE_
%token EXPORT_
%token EXT_
%token FACTOR_
%token FALSE_
%token FAST_
%token FILE_
%token FILTER_
%token FIT_
%token FITS_
%token FITSY_
%token FIXED_
%token FK4_
%token FK5_
%token FONT_
%token FRONT_
%token FULL_
%token FUNCTION_
%token GALACTIC_
%token GAUSSIAN_
%token GET_
%token GLOBAL_
%token GRAPHICS_
%token GRAY_
%token GRID_
%token GZ_
%token HANDLE_
%token HAS_
%token HEADER_
%token HEIGHT_
%token HIDE_
%token HIGH_
%token HIGHLITE_
%token HISTEQU_
%token HISTOGRAM_
%token HMS_
%token HORIZONTAL_
%token ICRS_
%token ID_
%token IIS_
%token IMAGE_
%token INCLUDE_
%token INCR_
%token INFO_
%token INTEGER_
%token IRAF_
%token IRAFMIN_
%token J2000_
%token JPEG_
%token KEY_
%token KEYWORD_
%token LABEL_
%token LENGTH_
%token LEVEL_
%token LINE_
%token LINEAR_
%token LIST_
%token LOAD_
%token LOCAL_
%token LOG_
%token LOW_
%token MAGNIFIER_
%token MAP_
%token MARK_
%token MARKER_
%token MASTER_
%token MESSAGE_
%token METHOD_
%token MINMAX_
%token MMAP_
%token MMAPINCR_
%token MOSAIC_
%token MODE_
%token MOTION_
%token MOVE_
%token NAME_
%token NAN_
%token NEW_
%token NEXT_
%token NO_
%token NONE_
%token NOW_
%token NUMBER_
%token OBJECT_
%token OFF_
%token ON_
%token ONLY_
%token OPTION_
%token ORIENT_
%token PAN_
%token PANNER_
%token PARAM_
%token PARSER_
%token PASTE_
%token PERF_
%token PHYSICAL_
%token PIXELS_
%token PNG_
%token POINT_
%token POLYGON_
%token POSTSCRIPT_
%token PPM_
%token PRESERVE_
%token PROJECTION_
%token PROPERTY_
%token PUBLICATION_
%token PROS_
%token RADIUS_
%token REGION_
%token REPLACE_
%token RESAMPLE_
%token RESET_
%token RESOLUTION_
%token RGB_
%token ROOT_
%token ROTATE_
%token RULER_
%token SAMPLE_
%token SAOIMAGE_
%token SAOTNG_
%token SAVE_
%token SCAN_
%token SCIENTIFIC_
%token SCOPE_
%token SEGMENT_
%token SELECT_
%token SET_
%token SEXAGESIMAL_
%token SHARED_
%token SHIFT_
%token SHMID_
%token SHOW_
%token SIZE_
%token SKY_
%token SLAVE_
%token SLICE_
%token SMMAP_
%token SMOOTH_
%token SOCKET_
%token SOCKETGZ_
%token SOURCE_
%token SQRT_
%token SQUARED_
%token SSHARED_
%token STATUS_
%token SUM_
%token SYSTEM_
%token TABLE_
%token TAG_
%token TEMPLATE_
%token TEXT_
%token THICK_
%token TIFF_
%token TO_
%token TOGGLE_
%token TOPHAT_
%token TRUE_
%token TYPE_
%token UNDO_
%token UNHIGHLITE_
%token UNLOAD_
%token UNSELECT_
%token UPDATE_
%token USER_
%token VALUE_
%token VAR_
%token VIEW_
%token VECTOR_
%token VERSION_
%token VERTEX_
%token VERTICAL_
%token WARP_
%token WCS_
%token WCSA_
%token WCSB_
%token WCSC_
%token WCSD_
%token WCSE_
%token WCSF_
%token WCSG_
%token WCSH_
%token WCSI_
%token WCSJ_
%token WCSK_
%token WCSL_
%token WCSM_
%token WCSN_
%token WCSO_
%token WCSP_
%token WCSQ_
%token WCSR_
%token WCSS_
%token WCST_
%token WCSU_
%token WCSV_
%token WCSW_
%token WCSX_
%token WCSY_
%token WCSZ_
%token WCS0_
%token WFPC2_
%token WIDTH_
%token XY_
%token YES_
%token ZMAX_
%token ZSCALE_
%token ZOOM_

%%

command : DEBUG_ debug

	| BIN_ bin
	| BG_ COLOR_ STRING {FRB->bgColorCmd($3);}
	| CENTER_ {FRB->centerCmd();}
	| CLEAR_ {FRB->clearCmd();}
	| CLIP_ clip
	| COLORMAP_ colormap
	| COLORSCALE_ scaleType 
	  {FRB->colorScaleCmd((FrScale::ColorScaleType)$2);}
	| CONTOUR_ contour
	| CROSSHAIR_ crosshair
	| DATASEC_ yesno {FRB->DATASECCmd($2);}
	| EXPORT_ export
	| FITSY_ fitsy
	| GET_ get
	| GRID_ grid
	| HAS_ has
	| HIDE_ {FRB->hideCmd();}
	| HIGHLITE_ yesno {FRB->highliteCmd($2);}
	| IIS_ iis
	| LOAD_ load
	| MAGNIFIER_ magnifier
	| MARKER_ markerLayer marker
	| MOSAIC_ FAST_ yesno {FRB->mosaicFastCmd($3);}
	| NAN_ COLOR_ STRING {FRB->nanColorCmd($3);}
	| ORIENT_ orient
	| PAN_ pan
	| PANNER_ panner
	| POSTSCRIPT_ postscript
	| RESET_ {FRB->resetCmd();}
	| REGION_ region
	| RGB_ rgb
	| ROTATE_ rotate
	| SAVE_ save
	| SHOW_ {FRB->showCmd();}
	| SMOOTH_ smooth
	| UNLOAD_ {FRB->unloadFitsCmd();}
	| UPDATE_ update
	| VERSION_ {FRB->msg("Frame 1.0");}
	| WARP_ warp
	| WCS_ wcs
	| ZOOM_ zoom
	;

numeric	: REAL {$$=$1;}
	| INT {$$=$1;}
	;

debug	: yesno {yydebug=$1;}
	| AST_ yesno {DebugAST=$2;}
	| MOSAIC_ yesno {DebugMosaic=$2;}
	| PARSER_ yesno {yydebug=$2;}
	| PERF_ yesno {DebugPerf=$2;}
	| WCS_ yesno {DebugWCS=$2;}
	| BIN_ yesno {DebugBin=$2;}
	| GZ_ yesno {DebugGZ=$2;}
	| RGB_ yesno {DebugRGB=$2;}
	;

yesno	: INT {$$=($1 ? 1 : 0);}

	| YES_ {$$=1;}
	| 'Y' {$$=1;}
	| ON_ {$$=1;}
	| TRUE_ {$$=1;}

	| NO_ {$$=0;}
	| 'N' {$$=0;}
	| OFF_ {$$=0;}
	| FALSE_ {$$=0;}
	;

precision : /* empty */ {$$ = FrameBase::DEFAULT;}
	| DEFAULT_ {$$ = FrameBase::DEFAULT;}
	| FIXED_ {$$ = FrameBase::FIXED;}
	| SCIENTIFIC_ {$$ = FrameBase::SCIENTIFIC;}
	| INTEGER_ {$$ = FrameBase::INTEGER;}
	;

fileNameType : /* empty */ {$$ = FrameBase::ROOTBASE;}
	| ROOT_ BASE_ {$$ = FrameBase::ROOTBASE;}
	| FULL_ BASE_ {$$ = FrameBase::FULLBASE;}
	| ROOT_ {$$ = FrameBase::ROOT;}
	| FULL_ {$$ = FrameBase::FULL;}
	;

optangle : /* empty */ {$$ = 0;}
	| angle {$$ = $1;}
	;

angle	: numeric {$$ = degToRad($1);} /* assume degree */
	| DEGREE {$$ = degToRad($1);}
	| RADIAN {$$=$1;}
	;

sexagesimal: SEXSTR {$$ = parseDMS($1);}
	;

coord	: sexagesimal sexagesimal
	{
	  Vector r;
	  if (currentSky == GALACTIC || currentSky == ECLIPTIC) 
	    r = Vector($1,$2);
	  else
	    r = Vector($1*360./24.,$2);

	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| HOUR MINUTE SECOND {frsign2 = frsign;} INT ARCMINUTE ARCSECOND
        {
	  Vector r = Vector(hmsToDegree(frsign2,$1,$2,$3), 
	    dmsToDegree(frsign,$5,$6,$7));

	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| INT ARCMINUTE ARCSECOND {frsign2 = frsign;} INT ARCMINUTE ARCSECOND
        {
	  Vector r = Vector(dmsToDegree(frsign2,$1,$2,$3),
	    dmsToDegree(frsign,$5,$6,$7));

	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| numeric numeric 
	{
	  $$[0] = $1;
	  $$[1] = $2;
	  $$[2] = 1;
	}
	;

coordSystem :IMAGE_ {$$ = currentCoord = IMAGE;}
	| PHYSICAL_ {$$ = currentCoord = PHYSICAL;}
	| DETECTOR_ {$$ = currentCoord = DETECTOR;}
	| AMPLIFIER_ {$$ = currentCoord = AMPLIFIER;}
	| wcsSystem {$$ = (CoordSystem)$1;}
	;

wcsSystem : WCS_ {$$ = currentCoord = WCS;}
	| WCSA_ {$$ = currentCoord = WCSA;}
	| WCSB_ {$$ = currentCoord = WCSB;}
	| WCSC_ {$$ = currentCoord = WCSC;}
	| WCSD_ {$$ = currentCoord = WCSD;}
	| WCSE_ {$$ = currentCoord = WCSE;}
	| WCSF_ {$$ = currentCoord = WCSF;}
	| WCSG_ {$$ = currentCoord = WCSG;}
	| WCSH_ {$$ = currentCoord = WCSH;}
	| WCSI_ {$$ = currentCoord = WCSI;}
	| WCSJ_ {$$ = currentCoord = WCSJ;}
	| WCSK_ {$$ = currentCoord = WCSK;}
	| WCSL_ {$$ = currentCoord = WCSL;}
	| WCSM_ {$$ = currentCoord = WCSM;}
	| WCSN_ {$$ = currentCoord = WCSN;}
	| WCSO_ {$$ = currentCoord = WCSO;}
	| WCSP_ {$$ = currentCoord = WCSP;}
	| WCSQ_ {$$ = currentCoord = WCSQ;}
	| WCSR_ {$$ = currentCoord = WCSR;}
	| WCSS_ {$$ = currentCoord = WCSS;}
	| WCST_ {$$ = currentCoord = WCST;}
	| WCSU_ {$$ = currentCoord = WCSU;}
	| WCSV_ {$$ = currentCoord = WCSV;}
	| WCSW_ {$$ = currentCoord = WCSW;}
	| WCSX_ {$$ = currentCoord = WCSX;}
	| WCSY_ {$$ = currentCoord = WCSY;}
	| WCSZ_ {$$ = currentCoord = WCSZ;}
	;

internalSystem : CANVAS_ {$$ = CANVAS;}
	| PANNER_ {$$ = PANNER;}
	;

skyFrame : /* empty */ {$$ = currentSky = FK5;}
	| FK4_ {$$ = currentSky = FK4;}
	| B1950_ {$$ = currentSky = FK4;}
	| FK5_ {$$ = currentSky = FK5;}
	| J2000_ {$$ = currentSky = FK5;}
	| ICRS_ {$$ = currentSky = ICRS;}
	| GALACTIC_ {$$ = currentSky = GALACTIC;}
	| ECLIPTIC_ {$$ = currentSky = ECLIPTIC;}
	;

skyFormat : /* empty */ {$$=DEGREES;}
	| DEGREES_ {$$=DEGREES;}
	| SEXAGESIMAL_ {$$=SEXAGESIMAL;}
	| HMS_ {$$=HMS;}
	| ARCMIN_ {$$=ARCMIN;}
	| ARCSEC_ {$$=ARCSEC;}
	;

scaleType: LINEAR_ {$$ = FrScale::LINEARSCALE;}
	| LOG_ {$$ = FrScale::LOGSCALE;}
	| SQUARED_ {$$ = FrScale::SQUAREDSCALE;}
	| SQRT_ {$$ = FrScale::SQRTSCALE;}
	| HISTEQU_ {$$ = FrScale::HISTEQUSCALE;}
	;	

shmType	: /* empty */ {$$ = FrameBase::SHMID;}
	| SHMID_ {$$ = FrameBase::SHMID;}
	| KEY_ {$$ = FrameBase::KEY;}
	;

incrLoad: /* empty */ {$$ = FrameBase::LOADALL;}
	| ALL_ {$$ = FrameBase::LOADALL;}
	| INCR_ {$$ = FrameBase::INCR;}
	;

compressType: /* empty */ {$$ = FrameBase::NOCOMPRESS;}
	| NONE_ {$$ = FrameBase::NOCOMPRESS;}
	| GZ_ {$$ = FrameBase::GZ;}
	;

bin	: ABOUT_ numeric numeric {FRB->binAboutCmd(Vector($2,$3));}
	| COLS_ STRING STRING STRING {FRB->binColsCmd($2,$3,$4);}
	| DEPTH_ INT {FRB->binDepthCmd($2);}
	| FACTOR_ binFactor
	| FUNCTION_ binFunction
	| BUFFER_ SIZE_ INT {FRB->binBufferSizeCmd($3);}
	| TO_ binTo
	| FILTER_ STRING {FRB->binFilterCmd($2);}
	;

binFactor : numeric numeric {FRB->binFactorCmd(Vector($1,$2));}
	| numeric numeric ABOUT_ numeric numeric
	  {FRB->binFactorAboutCmd(Vector($1,$2), Vector($4,$5));}
	| TO_ numeric numeric {FRB->binFactorToCmd(Vector($2,$3));}
	| TO_ numeric numeric ABOUT_ numeric numeric
	  {FRB->binFactorToAboutCmd(Vector($2,$3), Vector($5,$6));}
	;

binFunction : AVERAGE_ {FRB->binFunctionCmd(FitsHist::AVERAGE);}
	| SUM_ {FRB->binFunctionCmd(FitsHist::SUM);}
	;

binTo	: FIT_ {FRB->binToFitCmd();}
	| numeric numeric ABOUT_ CENTER_ STRING STRING STRING 
	  {FRB->binCmd(Vector($1,$2), $5, $6, $7);}
	| numeric numeric INT numeric numeric ABOUT_ CENTER_ 
	    STRING STRING STRING STRING
	  {FRB->binCmd(Vector($1,$2), $3, Vector($4,$5), $8, $9, $10, $11);}
	| numeric numeric ABOUT_ numeric numeric STRING STRING STRING 
	  {FRB->binCmd(Vector($1,$2), Vector($4,$5), $6, $7, $8);}
	| numeric numeric INT numeric numeric ABOUT_ numeric numeric 
	    STRING STRING STRING STRING
	  {FRB->binCmd(Vector($1,$2), $3, Vector($4,$5), Vector($7,$8), 
	    $9, $10, $11, $12);}
	;

clip	: SCOPE_ clipScope
	| MODE_ clipMode
	| MINMAX_ clipMinMax
	| USER_ clipUser
	| ZSCALE_ PARAM_ numeric INT INT {FRB->clipZScaleParamCmd($3,$4,$5);}
	| PRESERVE_ yesno {FRB->clipPreserveCmd($2);}
	;

clipUser: /* empty */ numeric numeric {FRB->clipUserCmd($1,$2);}
	| LOW_ numeric {FRB->clipUserLowCmd($2);} 
	| HIGH_ numeric {FRB->clipUserHighCmd($2);} 
	;

clipScope: GLOBAL_ {FRB->clipScopeCmd(FrScale::GLOBAL);}
	| LOCAL_ {FRB->clipScopeCmd(FrScale::LOCAL);}
	;

clipMode: numeric {FRB->clipModeCmd($1);}
	| MINMAX_ {FRB->clipModeCmd(FrScale::MINMAX);}
	| ZSCALE_ {FRB->clipModeCmd(FrScale::ZSCALE);}
	| ZMAX_ {FRB->clipModeCmd(FrScale::ZMAX);}
	| USER_ {FRB->clipModeCmd(FrScale::USERCLIP);}
	;

clipMinMax : MODE_ clipMinMaxMode
	| SAMPLE_ INT {FRB->clipMinMaxParamCmd($2);}
	;

clipMinMaxMode : AUTO_ {FRB->clipMinMaxModeCmd(FrScale::AUTOSCAN);}
	| SCAN_ {FRB->clipMinMaxModeCmd(FrScale::SCAN);}
	| SAMPLE_ {FRB->clipMinMaxModeCmd(FrScale::SAMPLE);}
	| DATAMIN_ {FRB->clipMinMaxModeCmd(FrScale::DATAMIN);}
	| IRAFMIN_ {FRB->clipMinMaxModeCmd(FrScale::IRAFMIN);}
	;

colormap : INT numeric numeric INT POINTER POINTER INT 
	  {FR->colormapCmd($1, $2, $3, $4, (unsigned short*)$5, 
	    (unsigned char*)$6, $7);}
	| RGB_ numeric numeric numeric numeric numeric numeric POINTER INT
	  {FRRGB->colormapCmd($2,$3,$4,$5,$6,$7,(unsigned char*)$8,$9);}
	| AREA_ INT colormapMode 
	  {FRB->colormapAreaCmd($2,(FrameBase::ColormapMode)$3);}
	| BEGIN_ colormapBegin
	| MOTION_ INT numeric numeric INT POINTER POINTER INT
	  {FR->colormapMotionCmd($2, $3, $4, $5, (unsigned short*)$6,
	    (unsigned char*)$7, $8);}
	| MOTION_ RGB_ 
	    numeric numeric numeric numeric numeric numeric POINTER INT
	  {FRRGB->colormapMotionCmd($3,$4,$5,$6,$7,$8,(unsigned char*)$9,$10);}
	| END_ INT numeric numeric INT POINTER POINTER INT
	  {FR->colormapEndCmd($2, $3, $4, $5, (unsigned short*)$6,
	    (unsigned char*)$7, $8);}
	| END_ RGB_ numeric numeric numeric numeric numeric numeric POINTER INT
	  {FRRGB->colormapEndCmd($3,$4,$5,$6,$7,$8,(unsigned char*)$9,$10);}
	;

colormapBegin : /* empty */ {FRB->colormapBeginCmd();}
	| numeric numeric {FRB->colormapBeginCmd(Vector($1,$2));}
	;

colormapMode: CENTER_ {$$ = FrameBase::CENTER;}
	| CLICK_ {$$ = FrameBase::CLICK;}
	;

contour	: COLOR_ STRING {FRB->contourSetColorCmd($2);}
	| COPY_ coordSystem skyFrame 
	  {FRB->contourCopyCmd((CoordSystem)$2, (SkyFrame)$3);}
	| CREATE_ contourCreate
	| DELETE_ {FRB->contourDeleteCmd();}
	| DELETE_ ALL_ {FRB->contourDeleteAllCmd();}
	| LOAD_ STRING INT STRING coordSystem skyFrame
	  {FRB->contourLoadCmd($2, $3, $4, (CoordSystem)$5, (SkyFrame)$6);}
	| PASTE_ STRING INT POINTER coordSystem skyFrame
	  {FRB->contourPasteCmd($2, $3, $4, (CoordSystem)$5, (SkyFrame)$6);}
	| SAVE_ STRING coordSystem skyFrame
	  {FRB->contourSaveCmd($2, (CoordSystem)$3, (SkyFrame)$4);}
	| WIDTH_ INT {FRB->contourSetLineWidthCmd($2);}
	;

contourCreate : STRING INT INT numeric contourmethod 
	  {FRB->contourCreateCmd($1,$2,$3,$4,$5);}
	| STRING INT numeric contourmethod STRING
	  {FRB->contourCreateCmd($1,$2,$3,$4,$5);}
	| POLYGON_ markerProperties
	  {FRB->createContourPolygonCmd(currentColor,currentWidth, 
	    currentFont, currentText, currentProps, NULL, taglist,cblist);}
	;

contourmethod : SMOOTH_ {$$ = FVContour::SMOOTH;}
	| BLOCK_ {$$ = FVContour::BLOCK;}
	;

crosshair: BEGIN_ MOTION_ internalSystem numeric numeric
	  {FRB->crosshairBeginCmd(Vector($4,$5), (InternalSystem)$3);}
	| MOTION_ internalSystem numeric numeric
	  {FRB->crosshairMotionCmd(Vector($3,$4), (InternalSystem)$2);}
	| internalSystem numeric numeric
	  {FRB->crosshairCmd(Vector($2,$3), (InternalSystem)$1);}
	| BEGIN_ MOTION_ coordSystem coord
	  {FRB->crosshairBeginCmd(Vector($4), (CoordSystem)$3);}
	| MOTION_ coordSystem coord
	  {FRB->crosshairMotionCmd(Vector($3), (CoordSystem)$2);}
	| coordSystem skyFrame coord
	  {FRB->crosshairCmd(Vector($3), (CoordSystem)$1, (SkyFrame)$2);}
	| yesno {FRB->crosshairCmd($1);}
	| WARP_ numeric numeric {FRB->crosshairWarpCmd(Vector($2,$3));}
	;

export	: FITS_ STRING 
	| JPEG_ STRING numeric {FRB->exportJPEGCmd($2,$3);}
	| TIFF_ STRING
	| PNG_ STRING
	| PPM_ STRING
	;

fitsy	: HAS_ EXT_ STRING {FRB->fitsyHasExtCmd($3);}
	;

get	: BG_ COLOR_ {FRB->getBgColorCmd();}
	| BIN_ getBin
	| BITPIX_ {FRB->getBitpixCmd();}
	| CLIP_ getClip
	| COLORMAP_ {FRB->getColormapCmd();}
	| COLORSCALE_ {FRB->getColorScaleCmd();}
	| CONTOUR_ getContour
	| COORDINATES_ getCoord
	| CROSSHAIR_ getCrosshair
	| CURSOR_ getCursor
	| DATA_ getData
	| DATASEC_ {FRB->getDATASECCmd();}
	| FITS_ getFits
	| GRID_ getGrid
	| HEIGHT_ {FRB->getHeightCmd();}
	| HISTOGRAM_ STRING STRING {FRB->getHistogramCmd($2,$3);}
	| HORIZONTAL_ CUT_ STRING STRING numeric numeric internalSystem
	  {FRB->getHorzCutCmd($3,$4,Vector($5,$6),(InternalSystem)$7);}
	| IIS_ getiis
	| INFO_ getInfo
	| MINMAX_ {FRB->getMinMaxCmd();}
	| MARKER_ markerLayer markerGet
	| NAN_ COLOR_ {FRB->getNANColorCmd();}
	| ORIENT_ {FRB->getOrientCmd();}
	| PAN_ getCursor
	| PIXELS_ TABLE_ internalSystem numeric numeric INT STRING
	  {FRB->getPixelTableCmd(Vector($4,$5), (InternalSystem)$3, $6, $7);}
	| RGB_ getRGB
	| ROTATE_ getRotate
	| SMOOTH_ getSmooth
	| TYPE_ {FRB->getTypeCmd();}
	| VALUE_ internalSystem numeric numeric
	  {FRB->getValueCmd(Vector($3,$4),(InternalSystem)$2);}
	| VERTICAL_ CUT_ STRING STRING numeric numeric internalSystem
	  {FRB->getVertCutCmd($3,$4,Vector($5,$6),(InternalSystem)$7);}
	| WCS_ getWCS
	| WIDTH_ {FRB->getWidthCmd();}
	| ZOOM_ getZoom
	;

getBin	: DEPTH_ {FRB->getBinDepthCmd();}
	| FACTOR_ {FRB->getBinFactorCmd();}
	| FUNCTION_ {FRB->getBinFunctionCmd();}
	| BUFFER_ SIZE_ {FRB->getBinBufferSizeCmd();}
	| CURSOR_ {FRB->getBinCursorCmd();}
	| FILTER_ {FRB->getBinFilterCmd();}
	| COLS_ {FRB->getBinColsCmd();}
	| COLS_ MINMAX_ STRING {FRB->getBinColsMinMaxCmd($3);}
	| LIST_ {FRB->getBinListCmd();}
	;

getClip	: getClipMode
	| SCOPE_ {FRB->getClipScopeCmd();}
	| MODE_ {FRB->getClipModeCmd();}
	| MINMAX_ getClipMinMax
	| USER_ LEVEL_ {FRB->getClipUserCmd();}
	| ZSCALE_ PARAM_ {FRB->getClipZScaleParamCmd();}
	| PRESERVE_ {FRB->getClipPreserveCmd();}
	;

getClipMode: /* empty */ {FRB->getClipCmd();}
	| numeric {FRB->getClipCmd($1);}
	| MINMAX_ {FRB->getClipCmd(FrScale::MINMAX);}
	| ZSCALE_ {FRB->getClipCmd(FrScale::ZSCALE);}
	| ZMAX_ {FRB->getClipCmd(FrScale::ZMAX);}
	| USER_ {FRB->getClipCmd(FrScale::USERCLIP);}
	;

getClipMinMax : MODE_ {FRB->getClipMinMaxModeCmd();}
	| SAMPLE_ {FRB->getClipMinMaxSampleCmd();}
	;

getContour: coordSystem skyFrame 
	  {FRB->getContourCmd((CoordSystem)$1,(SkyFrame)$2);}
	| COLOR_ {FRB->getContourColorNameCmd();}
	| LEVEL_ getContourLevel
	| METHOD_ {FRB->getContourMethodCmd();}
	| WIDTH_ {FRB->getContourLineWidthCmd();}
	;

getContourLevel : /* empty */ {FRB->getContourLevelCmd();}
	| INT numeric numeric scaleType 
	  {FRB->getContourLevelCmd($1,$2,$3,(FrScale::ColorScaleType)$4);}
	;

getCoord : internalSystem numeric numeric coordSystem skyFrame skyFormat
	  {FRB->getCoordCmd(Vector($2,$3), (InternalSystem)$1, (CoordSystem)$4,
	    (SkyFrame)$5, (SkyFormat)$6);}
	;

getCrosshair: internalSystem {FRB->getCrosshairCmd((InternalSystem)$1);}
	| coordSystem skyFrame skyFormat precision
	  {FRB->getCrosshairCmd((CoordSystem)$1, (SkyFrame)$2, (SkyFormat)$3,
	    (FrameBase::Precision)$4)}
	| STATUS_ {FRB->getCrosshairStatusCmd();}
	;

getCursor : internalSystem {FRB->getCursorCmd((InternalSystem)$1);}
	| coordSystem skyFrame skyFormat precision
	  {FRB->getCursorCmd((CoordSystem)$1, (SkyFrame)$2, (SkyFormat)$3,
	    (FrameBase::Precision)$4);}
	| PRESERVE_ {FRB->getPanPreserveCmd();}
	;

getData	: coordSystem skyFrame coord numeric numeric STRING
	  {FRB->getDataValuesCmd(1, Vector($3), 
	    (CoordSystem)$1, (SkyFrame)$2, Vector($4,$5), $6);}
	| INT coordSystem skyFrame coord numeric numeric STRING
	  {FRB->getDataValuesCmd($1, Vector($4), 
	    (CoordSystem)$2, (SkyFrame)$3, Vector($5,$6), $7);}
	| internalSystem numeric numeric INT INT
	  {FRB->getDataValuesCmd(Vector($2,$3),(InternalSystem)$1,
	    Vector($4,$5));}
	;

getInfo	: STRING {FRB->getInfoCmd($1);}
 	| internalSystem numeric numeric skyFrame skyFormat STRING
	  {FRB->getInfoCmd(Vector($2,$3), (InternalSystem)$1, 
	    (SkyFrame)$4, (SkyFormat)$5, $6);}
	;

getiis	: POINTER INT INT INT INT {FR->iisGetCmd((char*)$1,$2,$3,$4,$5);}
	| CURSOR_ {FR->iisGetCursorCmd();}
	| WIDTH_ {FRB->getWidthCmd();}
	| HEIGHT_ {FRB->getHeightCmd();}
	| FILE_ NAME_ getIISFileName
	;

getIISFileName : /* empty */ {FRB->iisGetFileNameCmd();}
	| INT {FRB->iisGetFileNameCmd($1);}
	| numeric numeric {FRB->iisGetFileNameCmd(Vector($1,$2));}
	;

getFits	: EXT_ COUNT_ {FRB->getFitsExtCountCmd();}
	| FILE_ NAME_ getFitsFileName
	| HEADER_ INT {FRB->getFitsHeaderCmd($2);}
	| HEADER_ INT KEYWORD_ STRING {FRB->getFitsHeaderKeywordCmd($2,$4);}
	| OBJECT_ NAME_ getFitsObjectName
	| MASTER_ {FRB->getFitsMasterCmd();}
	| SIZE_ {FRB->getFitsSizeCmd();}
	| SIZE_ coordSystem skyFormat
	    {FRB->getFitsSizeCmd((CoordSystem)$2, (SkyFormat)$3);}
	| DEPTH_ {FRB->getFitsDepthCmd();}
	| SLICE_ {FRB->getFitsSliceCmd();}
	;

getFitsFileName: fileNameType 
	  {FRB->getFitsFileNameCmd((FrameBase::FileNameType)$1);}
	| fileNameType internalSystem numeric numeric
	  {FRB->getFitsFileNameCmd(Vector($3,$4), (InternalSystem)$2,
	    (FrameBase::FileNameType)$1);}
	| fileNameType EXT_ INT 
	  {FRB->getFitsFileNameCmd($3, (FrameBase::FileNameType)$1);}
	;

getFitsObjectName: /* empty */ {FRB->getFitsObjectNameCmd(1);}
	| EXT_ INT {FRB->getFitsObjectNameCmd($2);}
	;

getGrid : /* empty */ {FRB->getGridCmd();}
	| OPTION_ {FRB->getGridOptionCmd();}
	;

getRGB	: CHANNEL_ {FRB->getRGBChannelCmd();}
	| SYSTEM_ {FRB->getRGBSystemCmd();}
	| VIEW_ {FRB->getRGBViewCmd();}
	;

getRotate : precision {FRB->getRotateCmd((FrameBase::Precision)$1);}
	;

getSmooth : FUNCTION_  {FRB->getSmoothFunctionCmd();}
	| RADIUS_ {FRB->getSmoothRadiusCmd();}
	;

getWCS	: ALIGN_ {FRB->getWCSAlignCmd();}
	| ZOOM_  wcsSystem precision 
	  {FRB->getWCSZoomCmd((CoordSystem)$2, (FrameBase::Precision)$3);}
	| NAME_ wcsSystem {FRB->getWCSNameCmd((CoordSystem)$2);}
	;

getZoom	: precision {FRB->getZoomCmd((FrameBase::Precision)$1);}
	;

grid	: CREATE_ coordSystem skyFrame skyFormat gridType STRING
	   {FRB->gridCmd((CoordSystem)$2, (SkyFrame)$3, (SkyFormat)$4,
	     (Grid::GridType)$5, $6);}
	| DELETE_ {FRB->gridDeleteCmd();}
	;

gridType : ANALYSIS_ {$$=Grid::ANALYSIS;}
	| PUBLICATION_ {$$=Grid::PUBLICATION;}
	;

has	: AMPLIFIER_ {FRB->hasAmplifierCmd();}
	| BIN_ {FRB->hasBinCmd();}
	| CONTOUR_ {FRB->hasContourCmd();}
	| DATAMIN_ {FRB->hasDATAMINCmd();}
	| DATASEC_ {FRB->hasDATASECCmd();}
	| DETECTOR_ {FRB->hasDetectorCmd();}
	| FITS_ hasFits
	| GRID_ {FRB->hasGridCmd();}
	| IIS_ {FR->hasIISCmd();}
	| IRAFMIN_ {FRB->hasIRAFMINCmd();}
	| MARKER_ hasMarker
	| PHYSICAL_ {FRB->hasPhysicalCmd();}
	| SMOOTH_ {FRB->hasSmoothCmd();}
	| SYSTEM_ coordSystem {FRB->hasSystemCmd((CoordSystem)$2);}
	| WCS_ hasWCS
	;

hasFits	: /* empty */ {FRB->hasFitsCmd();}
	| CUBE_ {FRB->hasFitsCubeCmd();}
	| EXT_ INT {FRB->hasFitsExtCmd($2);}
	| IMAGE_ {FRB->hasFitsImageCmd();}
	| MOSAIC_ {FRB->hasFitsMosaicCmd();}
	| TABLE_ {FRB->hasFitsTableCmd();}
	;

hasMarker : HIGHLITE_ {FRB->hasMarkerHighlitedCmd();}
	| SELECT_ {FRB->hasMarkerSelectedCmd();}
	| PASTE_ {FRB->hasMarkerPasteCmd();}
	| UNDO_ {FRB->hasMarkerUndoCmd();}
	;

hasWCS	: coordSystem {FRB->hasWCSCmd((CoordSystem)$1);}
	| EQUATORIAL_ coordSystem {FRB->hasWCSEquCmd((CoordSystem)$2);}
	| LINEAR_ coordSystem {FRB->hasWCSLinearCmd((CoordSystem)$2);}
	;

iis	: NEW_ INT INT {FR->iisCmd($2,$3);}
	| ERASE_ {FR->iisEraseCmd();}
	| MESSAGE_ STRING {FR->iisMessageCmd($2);}
	| CURSOR_ iiscursor
	| SET_ FILE_ NAME_ iisSetFileName
	| SET_ POINTER INT INT INT INT
	  {FR->iisSetCmd((const char*)$2,$3,$4,$5,$6);}
	| UPDATE_ {FR->iisUpdateCmd();}
	| WCS_ numeric numeric numeric numeric 
	  numeric numeric numeric numeric INT
	    {FR->iisWCSCmd(Matrix($2,$3,$4,$5,$6,$7),Vector($8,$9),$10);}
	;

iisSetFileName : STRING {FRB->iisSetFileNameCmd($1);}
	| STRING INT {FRB->iisSetFileNameCmd($1,$2);}
	;

iiscursor: INT INT CANVAS_
	  {FR->iisSetCursorCmd(Vector($1,$2),CANVAS);}
	| INT INT coordSystem
	  {FR->iisSetCursorCmd(Vector($1,$2),(CoordSystem)$3);}
	| MODE_ yesno {FR->iisCursorModeCmd($2);}
	;

load	: ARRAY_ loadArr
	| FITS_ loadFits
	| INCR_ loadIncr
	;

loadArr : STRING ALLOC_ {FRB->loadArrAllocCmd($1);}
	| STRING ALLOCGZ_ {FRB->loadArrAllocGZCmd($1);}
	| STRING CHANNEL_ STRING {FRB->loadArrChannelCmd($3, $1);}
	| STRING MMAP_ {FRB->loadArrMMapCmd($1);}
	| STRING MMAPINCR_ {FRB->loadArrMMapIncrCmd($1);}
	| STRING SHARED_ shmType INT
	  {FRB->loadArrShareCmd((FrameBase::ShmType)$3, $4, $1);}
	| STRING SOCKET_ INT {FRB->loadArrSocketCmd($3, $1);}
	| STRING SOCKETGZ_ INT {FRB->loadArrSocketGZCmd($3, $1);}
	| STRING VAR_ STRING {FRB->loadArrVarCmd($3, $1);}
	| RGB_ CUBE_ loadArrRGBCube
	;

loadArrRGBCube : STRING ALLOC_ {FRB->loadArrRGBCubeAllocCmd($1);}
	| STRING ALLOCGZ_ {FRB->loadArrRGBCubeAllocGZCmd($1);}
	| STRING CHANNEL_ STRING {FRB->loadArrRGBCubeChannelCmd($3, $1);}
	| STRING MMAP_ {FRB->loadArrRGBCubeMMapCmd($1);}
	| STRING MMAPINCR_ {FRB->loadArrRGBCubeMMapIncrCmd($1);}
	| STRING SHARED_ shmType INT
	  {FRB->loadArrRGBCubeShareCmd((FrameBase::ShmType)$3, $4, $1);}
	| STRING SOCKET_ INT {FRB->loadArrRGBCubeSocketCmd($3, $1);}
	| STRING SOCKETGZ_ INT {FRB->loadArrRGBCubeSocketGZCmd($3, $1);}
	| STRING VAR_ STRING {FRB->loadArrRGBCubeVarCmd($3, $1);}
	;

loadFits: STRING ALLOC_ {FRB->loadFitsAllocCmd($1);}
	| STRING ALLOCGZ_ {FRB->loadFitsAllocGZCmd($1);}
	| STRING CHANNEL_ STRING {FRB->loadFitsChannelCmd($3, $1);}
	| STRING MMAP_ incrLoad 
	  {FRB->loadFitsMMapCmd($1,(FrameBase::LoadMethod)$3);}
	| STRING STRING SMMAP_ incrLoad
	  {FRB->loadFitsSMMapCmd($1,$2, (FrameBase::LoadMethod)$4);}
	| STRING MMAPINCR_ incrLoad 
	  {FRB->loadFitsMMapIncrCmd($1,(FrameBase::LoadMethod)$3);}
	| STRING SHARED_ shmType INT incrLoad
	  {FRB->loadFitsShareCmd((FrameBase::ShmType)$3, $4, $1, 
	    (FrameBase::LoadMethod)$5);}
	| STRING SSHARED_ shmType INT INT incrLoad
	  {FRB->loadFitsSShareCmd((FrameBase::ShmType)$3, $4, $5, $1, 
	    (FrameBase::LoadMethod)$6);}
	| STRING SOCKET_ INT {FRB->loadFitsSocketCmd($3, $1);}
	| STRING SOCKETGZ_ INT {FRB->loadFitsSocketGZCmd($3, $1);}
	| STRING VAR_ STRING incrLoad 
	  {FRB->loadFitsVarCmd($3, $1, (FrameBase::LoadMethod)$4);}
	| DATA_ CUBE_ loadFitsDataCube
	| MOSAIC_ loadFitsMosaic
	| RGB_ loadFitsRGB
	| SLAVE_ STRING POINTER {FRB->loadFitsSlaveCmd($2,(FitsFile*)$3);}
	;

loadFitsDataCube: STRING ALLOC_ {FRB->loadDataCubeAllocCmd($1);}
	| STRING ALLOCGZ_ {FRB->loadDataCubeAllocGZCmd($1);}
	| STRING CHANNEL_ STRING {FRB->loadDataCubeChannelCmd($3, $1);}
	| STRING MMAP_ incrLoad
	  {FRB->loadDataCubeMMapCmd($1,(FrameBase::LoadMethod)$3);}
	| STRING MMAPINCR_ incrLoad
	  {FRB->loadDataCubeMMapIncrCmd($1,(FrameBase::LoadMethod)$3);}
	| STRING SHARED_ shmType INT incrLoad
	  {FRB->loadDataCubeShareCmd((FrameBase::ShmType)$3, $4, $1, 
	    (FrameBase::LoadMethod)$5);}
	| STRING SOCKET_ INT {FRB->loadDataCubeSocketCmd($3, $1);}
	| STRING SOCKETGZ_ INT {FRB->loadDataCubeSocketGZCmd($3, $1);}
	| STRING VAR_ STRING incrLoad
	  {FRB->loadDataCubeVarCmd($3, $1, (FrameBase::LoadMethod)$4);}
	;

loadFitsMosaic : IMAGE_ IRAF_ loadFitsMosaicImageIRAF
	| IRAF_ loadFitsMosaicIRAF
	| IMAGE_ loadFitsMosaicImageWCS
	| IMAGE_ NEXT_ loadFitsMosaicImageNextWCS
	| loadFitsMosaicWCS
	| IMAGE_ WFPC2_ loadFitsMosaicImageWFPC2
	;

loadFitsMosaicImageIRAF : STRING ALLOC_ {FRB->loadMosaicImageIRAFAllocCmd($1);}
	| STRING ALLOCGZ_ {FRB->loadMosaicImageIRAFAllocGZCmd($1);}
	| STRING CHANNEL_ STRING {FRB->loadMosaicImageIRAFChannelCmd($3,$1);}
	| STRING MMAP_ incrLoad 
	  {FRB->loadMosaicImageIRAFMMapCmd($1,(FrameBase::LoadMethod)$3);}
	| STRING MMAPINCR_ incrLoad 
	  {FRB->loadMosaicImageIRAFMMapIncrCmd($1,(FrameBase::LoadMethod)$3);}
	| STRING SHARED_ shmType INT incrLoad
	  {FRB->loadMosaicImageIRAFShareCmd((FrameBase::ShmType)$3, $4,
	    $1, (FrameBase::LoadMethod)$5);}
	| STRING SOCKET_ INT {FRB->loadMosaicImageIRAFSocketCmd($3, $1);}
	| STRING SOCKETGZ_ INT {FRB->loadMosaicImageIRAFSocketGZCmd($3, $1);}
	| STRING VAR_ STRING incrLoad 
	  {FRB->loadMosaicImageIRAFVarCmd($3, $1, (FrameBase::LoadMethod)$4);}
	;

loadFitsMosaicIRAF : STRING ALLOC_ {FRB->loadMosaicIRAFAllocCmd($1);}
	| STRING ALLOCGZ_ {FRB->loadMosaicIRAFAllocGZCmd($1);}
	| STRING CHANNEL_ STRING {FRB->loadMosaicIRAFChannelCmd($3,$1);}
	| STRING MMAP_ incrLoad 
	  {FRB->loadMosaicIRAFMMapCmd($1,(FrameBase::LoadMethod)$3);}
	| STRING STRING SMMAP_ incrLoad 
	  {FRB->loadMosaicIRAFSMMapCmd($1,$2,(FrameBase::LoadMethod)$4);}
	| STRING MMAPINCR_ incrLoad 
	  {FRB->loadMosaicIRAFMMapIncrCmd($1,(FrameBase::LoadMethod)$3);}
	| STRING SHARED_ shmType INT incrLoad
	  {FRB->loadMosaicIRAFShareCmd((FrameBase::ShmType)$3, $4,
	    $1, (FrameBase::LoadMethod)$5);}
	| STRING SSHARED_ shmType INT INT incrLoad
	  {FRB->loadMosaicIRAFSShareCmd((FrameBase::ShmType)$3, $4, $5,
	    $1, (FrameBase::LoadMethod)$6);}
	| STRING SOCKET_ INT {FRB->loadMosaicIRAFSocketCmd($3, $1);}
	| STRING SOCKETGZ_ INT {FRB->loadMosaicIRAFSocketGZCmd($3, $1);}
	| STRING VAR_ STRING incrLoad 
	  {FRB->loadMosaicIRAFVarCmd($3, $1, (FrameBase::LoadMethod)$4);}
	;

loadFitsMosaicImageWCS : wcsSystem STRING ALLOC_ 
	  {FRB->loadMosaicImageWCSAllocCmd((CoordSystem)$1, $2);}
	| wcsSystem STRING ALLOCGZ_ 
	  {FRB->loadMosaicImageWCSAllocGZCmd((CoordSystem)$1, $2);}
	| wcsSystem STRING CHANNEL_ STRING 
	  {FRB->loadMosaicImageWCSChannelCmd((CoordSystem)$1, $4, $2);}
	| wcsSystem STRING MMAP_ incrLoad 
	  {FRB->loadMosaicImageWCSMMapCmd((CoordSystem)$1, $2,
	    (FrameBase::LoadMethod)$4);}
	| wcsSystem STRING MMAPINCR_ incrLoad 
	  {FRB->loadMosaicImageWCSMMapIncrCmd((CoordSystem)$1, $2,
	    (FrameBase::LoadMethod)$4);}
	| wcsSystem STRING SHARED_ shmType INT incrLoad
	  {FRB->loadMosaicImageWCSShareCmd((CoordSystem)$1, 
	    (FrameBase::ShmType)$4, $5, $2, (FrameBase::LoadMethod)$6);}
	| wcsSystem STRING SOCKET_ INT 
	  {FRB->loadMosaicImageWCSSocketCmd((CoordSystem)$1, $4, $2);}
	| wcsSystem STRING SOCKETGZ_ INT 
	  {FRB->loadMosaicImageWCSSocketGZCmd((CoordSystem)$1, $4, $2);}
	| wcsSystem STRING VAR_ STRING incrLoad 
	  {FRB->loadMosaicImageWCSVarCmd((CoordSystem)$1, $4, $2, 
	    (FrameBase::LoadMethod)$5);}
	;

loadFitsMosaicImageNextWCS : wcsSystem STRING ALLOC_ 
	  {FRB->loadMosaicImageNextWCSAllocCmd((CoordSystem)$1, $2);}
	| wcsSystem STRING ALLOCGZ_ 
	  {FRB->loadMosaicImageNextWCSAllocGZCmd((CoordSystem)$1, $2);}
	| wcsSystem STRING CHANNEL_ STRING 
	  {FRB->loadMosaicImageNextWCSChannelCmd((CoordSystem)$1, $4,$2);}
	| wcsSystem STRING MMAP_ incrLoad 
	  {FRB->loadMosaicImageNextWCSMMapCmd((CoordSystem)$1, $2,
	    (FrameBase::LoadMethod)$4);}
	| wcsSystem STRING MMAPINCR_ incrLoad 
	  {FRB->loadMosaicImageNextWCSMMapIncrCmd((CoordSystem)$1, $2,
	    (FrameBase::LoadMethod)$4);}
	| wcsSystem STRING SHARED_ shmType INT incrLoad
	  {FRB->loadMosaicImageNextWCSShareCmd((CoordSystem)$1, 
	    (FrameBase::ShmType)$4, $5, $2, (FrameBase::LoadMethod)$6);}
	| wcsSystem STRING SOCKET_ INT 
	  {FRB->loadMosaicImageNextWCSSocketCmd((CoordSystem)$1, $4, $2);}
	| wcsSystem STRING SOCKETGZ_ INT 
	  {FRB->loadMosaicImageNextWCSSocketGZCmd((CoordSystem)$1, $4, $2);}
	| wcsSystem STRING VAR_ STRING incrLoad 
	  {FRB->loadMosaicImageNextWCSVarCmd((CoordSystem)$1, $4, $2,
	    (FrameBase::LoadMethod)$5);}
	;

loadFitsMosaicWCS : wcsSystem STRING ALLOC_ 
	  {FRB->loadMosaicWCSAllocCmd((CoordSystem)$1, $2);}
	| wcsSystem STRING ALLOCGZ_ 
	  {FRB->loadMosaicWCSAllocGZCmd((CoordSystem)$1, $2);}
	| wcsSystem STRING CHANNEL_ STRING 
	  {FRB->loadMosaicWCSChannelCmd((CoordSystem)$1, $4,$2);}
	| wcsSystem STRING MMAP_ incrLoad 
	  {FRB->loadMosaicWCSMMapCmd((CoordSystem)$1, $2,
	    (FrameBase::LoadMethod)$4);}
	| wcsSystem STRING STRING SMMAP_ incrLoad 
	  {FRB->loadMosaicWCSSMMapCmd((CoordSystem)$1, $2, $3,
	    (FrameBase::LoadMethod)$5);}
	| wcsSystem STRING MMAPINCR_ incrLoad 
	  {FRB->loadMosaicWCSMMapIncrCmd((CoordSystem)$1, $2,
	    (FrameBase::LoadMethod)$4);}
	| wcsSystem STRING SHARED_ shmType INT incrLoad
	  {FRB->loadMosaicWCSShareCmd((CoordSystem)$1, 
	    (FrameBase::ShmType)$4, $5, $2, (FrameBase::LoadMethod)$6);}
	| wcsSystem STRING SSHARED_ shmType INT INT incrLoad
	  {FRB->loadMosaicWCSSShareCmd((CoordSystem)$1, 
	    (FrameBase::ShmType)$4, $5, $6, $2, (FrameBase::LoadMethod)$7);}
	| wcsSystem STRING SOCKET_ INT 
	  {FRB->loadMosaicWCSSocketCmd((CoordSystem)$1, $4, $2);}
	| wcsSystem STRING SOCKETGZ_ INT 
	  {FRB->loadMosaicWCSSocketGZCmd((CoordSystem)$1, $4, $2);}
	| wcsSystem STRING VAR_ STRING incrLoad 
	  {FRB->loadMosaicWCSVarCmd((CoordSystem)$1, $4, $2, 
	    (FrameBase::LoadMethod)$5);}
	;

loadFitsMosaicImageWFPC2 : STRING ALLOC_ 
	  {FRB->loadMosaicImageWFPC2AllocCmd($1);}
	| STRING ALLOCGZ_ {FRB->loadMosaicImageWFPC2AllocGZCmd($1);}
	| STRING CHANNEL_ STRING {FRB->loadMosaicImageWFPC2ChannelCmd($3,$1);}
	| STRING MMAP_ incrLoad 
	  {FRB->loadMosaicImageWFPC2MMapCmd($1,(FrameBase::LoadMethod)$3);}
	| STRING MMAPINCR_ incrLoad 
	  {FRB->loadMosaicImageWFPC2MMapIncrCmd($1,(FrameBase::LoadMethod)$3);}
	| STRING SHARED_ shmType INT incrLoad
	  {FRB->loadMosaicImageWFPC2ShareCmd((FrameBase::ShmType)$3, $4,
	    $1, (FrameBase::LoadMethod)$5);}
	| STRING SOCKET_ INT {FRB->loadMosaicImageWFPC2SocketCmd($3, $1);}
	| STRING SOCKETGZ_ INT {FRB->loadMosaicImageWFPC2SocketGZCmd($3,$1);}
	| STRING VAR_ STRING incrLoad 
	  {FRB->loadMosaicImageWFPC2VarCmd($3, $1, (FrameBase::LoadMethod)$4);}
	;
loadFitsRGB : CUBE_ loadFitsRGBCube
	| IMAGE_ loadFitsRGBImage
	;

loadFitsRGBCube: STRING ALLOC_ {FRB->loadRGBCubeAllocCmd($1);}
	| STRING ALLOCGZ_ {FRB->loadRGBCubeAllocGZCmd($1);}
	| STRING CHANNEL_ STRING {FRB->loadRGBCubeChannelCmd($3,$1);}
	| STRING MMAP_ incrLoad 
	  {FRB->loadRGBCubeMMapCmd($1, (FrameBase::LoadMethod)$3);}
	| STRING STRING SMMAP_ incrLoad 
	  {FRB->loadRGBCubeSMMapCmd($1, $2, (FrameBase::LoadMethod)$4);}
	| STRING MMAPINCR_ incrLoad 
	  {FRB->loadRGBCubeMMapIncrCmd($1, (FrameBase::LoadMethod)$3);}
	| STRING SHARED_ shmType INT incrLoad
	  {FRB->loadRGBCubeShareCmd((FrameBase::ShmType)$3, $4, $1, 
	    (FrameBase::LoadMethod)$5);}
	| STRING SSHARED_ shmType INT INT incrLoad
	  {FRB->loadRGBCubeSShareCmd((FrameBase::ShmType)$3, $4, $5, $1, 
	    (FrameBase::LoadMethod)$6);}
	| STRING SOCKET_ INT  {FRB->loadRGBCubeSocketCmd($3, $1);}
	| STRING SOCKETGZ_ INT {FRB->loadRGBCubeSocketGZCmd($3, $1);}
	| STRING VAR_ STRING incrLoad 
	  {FRB->loadRGBCubeVarCmd($3, $1, (FrameBase::LoadMethod)$4);}

	| SLAVE_ STRING POINTER {FRB->loadRGBCubeSlaveCmd($2,(FitsFile*)$3);}
	;

loadFitsRGBImage: STRING ALLOC_ {FRB->loadRGBImageAllocCmd($1);}
	| STRING ALLOCGZ_ {FRB->loadRGBImageAllocGZCmd($1);}
	| STRING CHANNEL_ STRING {FRB->loadRGBImageChannelCmd($3,$1);}
	| STRING MMAP_ incrLoad 
	  {FRB->loadRGBImageMMapCmd($1, (FrameBase::LoadMethod)$3);}
	| STRING MMAPINCR_ incrLoad 
	  {FRB->loadRGBImageMMapIncrCmd($1, (FrameBase::LoadMethod)$3);}
	| STRING SHARED_ shmType INT incrLoad
	  {FRB->loadRGBImageShareCmd((FrameBase::ShmType)$3, $4, $1, 
	    (FrameBase::LoadMethod)$5);}
	| STRING SOCKET_ INT  {FRB->loadRGBImageSocketCmd($3, $1);}
	| STRING SOCKETGZ_ INT {FRB->loadRGBImageSocketGZCmd($3, $1);}
	| STRING VAR_ STRING incrLoad 
	  {FRB->loadRGBImageVarCmd($3, $1, (FrameBase::LoadMethod)$4);}
	;

loadIncr: DATA_ INT INT INT INT INT {FRB->loadIncrDataCmd($2,$3,$4,$5,$6);}
	| MINMAX_ INT INT INT INT INT {FRB->loadIncrMinMaxCmd($2,$3,$4,$5,$6);}
	| END_ {FRB->loadIncrEndCmd();}
	;

magnifier: yesno {FRB->magnifierCmd($1);}
	| GRAPHICS_ yesno {FRB->magnifierGraphicsCmd($2);}
	| CURSOR_ yesno {FRB->magnifierCursorCmd($2);}
	| STRING INT INT {FRB->magnifierCmd($1, $2, $3);}
	| UPDATE_ {FRB->updateMagnifierCmd();}
	| UPDATE_ numeric numeric {FRB->updateMagnifierCmd(Vector($2, $3));}
	| ZOOM_ numeric {FRB->magnifierZoomCmd($2);}
	;

marker	: COLOR_ STRING {FRB->markerColorCmd($2);}
	| COPY_ {FRB->markerCopyCmd();}
	| COMMAND_ markerFormat STRING
	  {FRB->markerCommandCmd((FrameBase::MarkerFormat)$2,$3);}
	| COMMAND_ markerFormat VAR_ STRING
	  {FRB->markerCommandVarCmd((FrameBase::MarkerFormat)$2,$4);}
	| COMPOSITE_ DELETE_ {FRB->markerCompositeDeleteCmd();}
	| CREATE_ markerCreate
	| CUT_ {FRB->markerCutCmd();}
	| DELETE_ {FRB->markerDeleteCmd();}
	| DELETE_ ALL_ {FRB->markerDeleteAllCmd();}
	| EDIT_ markerEdit
	| FONT_ STRING {FRB->markerFontCmd($2);}

	| INT ANGLE_ angle {FRB->markerAngleCmd($1,$3);}
	| INT ANGLE_ angle coordSystem skyFrame
	  {FRB->markerAngleCmd($1,$3,(CoordSystem)$4, (SkyFrame)$5);}
	| INT ANNULUS_ RADIUS_ 
	    numeric numeric INT 
	    coordSystem skyFormat
	  {FRB->markerAnnulusRadiusCmd($1, 
	    $4, $5, $6,
	    (CoordSystem)$7, (SkyFormat)$8);}
	| INT ANNULUS_ RADIUS_ 
	    STRING coordSystem skyFormat
	  {FRB->markerAnnulusRadiusCmd($1,
	    $4,(CoordSystem)$5,(SkyFormat)$6);}
	| INT BOXANNULUS_ RADIUS_ 
	    numeric numeric numeric INT 
	    coordSystem skyFormat
	  {FRB->markerBoxAnnulusRadiusCmd($1, 
	    Vector($4, $5), Vector($6, $6*$5/$4), $7,
	    (CoordSystem)$8, (SkyFormat)$9);}
	| INT BOXANNULUS_ RADIUS_ 
	    STRING coordSystem skyFormat
	  {FRB->markerBoxAnnulusRadiusCmd($1,
	    $4,(CoordSystem)$5, (SkyFormat)$6);}
	| INT BOX_ RADIUS_ 
	    numeric numeric 
	    coordSystem skyFormat
	  {FRB->markerBoxRadiusCmd($1,
	    Vector($4,$5), 
	    (CoordSystem)$6, (SkyFormat)$7);}
	| INT BPANDA_ EDIT_
	    angle angle INT 
	    numeric numeric numeric INT 
	    coordSystem skyFrame
	  {FRB->markerBpandaEditCmd($1, 
	    $4, $5, $6,
	    Vector($7,$8), Vector($9,$9*$8/$7), $10,
	    (CoordSystem)$11, (SkyFrame)$12);}
	| INT BPANDA_ EDIT_
	    STRING STRING 
	    coordSystem skyFrame 
	    coordSystem skyFormat
	  {FRB->markerBpandaEditCmd($1, 
	    $4, $5, 
	    (CoordSystem)$6, (SkyFrame)$7,
	    (CoordSystem)$8, (SkyFormat)$9);}
	| INT CALLBACK_ markerCallBack STRING STRING
	  {FRB->markerCallBackCmd($1,(CallBack::Type)$3,$4,$5);}
	| INT CIRCLE_ RADIUS_ numeric coordSystem skyFormat
	  {FRB->markerCircleRadiusCmd($1, $4, (CoordSystem)$5, (SkyFormat)$6);}
	| INT COLOR_ STRING {FRB->markerColorCmd($1,$3);}
	| INT COMPASS_ ARROW_ yesno yesno 
	  {FRB->markerCompassArrowCmd($1,$4,$5);}
	| INT COMPASS_ LABEL_ STRING STRING
	  {FRB->markerCompassLabelCmd($1,$4,$5);}
	| INT COMPASS_ RADIUS_ numeric coordSystem skyFormat
	  {FRB->markerCompassRadiusCmd($1,$4,(CoordSystem)$5,(SkyFormat)$6);}
	| INT COMPASS_ SYSTEM_ coordSystem skyFrame
	  {FRB->markerCompassSystemCmd($1, (CoordSystem)$4, (SkyFrame)$5);}
	| INT CREATE_ ANNULUS_ RADIUS_ numeric numeric
	  {FRB->markerAnnulusCreateRadiusCmd($1,Vector($5,$6));}
	| INT CREATE_ BOXANNULUS_ RADIUS_ numeric numeric
	  {FRB->markerBoxAnnulusCreateRadiusCmd($1,Vector($5,$6));}
	| INT CREATE_ BPANDA_ ANGLE_ numeric numeric
	  {FRB->markerBpandaCreateAnglesCmd($1,Vector($5,$6));}
	| INT CREATE_ BPANDA_ RADIUS_ numeric numeric
	  {FRB->markerBpandaCreateRadiusCmd($1,Vector($5,$6));}
	| INT CREATE_ ELLIPSEANNULUS_ RADIUS_ numeric numeric
	  {FRB->markerEllipseAnnulusCreateRadiusCmd($1,Vector($5,$6));}
	| INT CREATE_ EPANDA_ ANGLE_ numeric numeric
	  {FRB->markerEpandaCreateAnglesCmd($1,Vector($5,$6));}
	| INT CREATE_ EPANDA_ RADIUS_ numeric numeric
	  {FRB->markerEpandaCreateRadiusCmd($1,Vector($5,$6));}
	| INT CREATE_ CPANDA_ ANGLE_ numeric numeric
	  {FRB->markerCpandaCreateAnglesCmd($1,Vector($5,$6));}
	| INT CREATE_ CPANDA_ RADIUS_ numeric numeric
	  {FRB->markerCpandaCreateRadiusCmd($1,Vector($5,$6));}
	| INT CREATE_ POLYGON_ VERTEX_ INT numeric numeric
	  {FRB->markerPolygonCreateVertexCmd($1,$5,Vector($6,$7));}
	| INT DELETE_ {FRB->markerDeleteCmd($1);}
	| INT DELETE_ ANNULUS_ INT 
	  {FRB->markerAnnulusDeleteRadiusCmd($1,$4);}
	| INT DELETE_ BOXANNULUS_ INT 
	  {FRB->markerBoxAnnulusDeleteRadiusCmd($1,$4);}
	| INT DELETE_ BPANDA_ INT {FRB->markerBpandaDeleteCmd($1,$4);}
	| INT DELETE_ ELLIPSEANNULUS_ INT 
	  {FRB->markerEllipseAnnulusDeleteRadiusCmd($1,$4);}
	| INT DELETE_ CALLBACK_ markerCallBack STRING
	  {FRB->markerDeleteCallBackCmd($1,(CallBack::Type)$4,$5);}
	| INT DELETE_ EPANDA_ INT {FRB->markerEpandaDeleteCmd($1,$4);}
	| INT DELETE_ CPANDA_ INT {FRB->markerCpandaDeleteCmd($1,$4);}
	| INT DELETE_ POLYGON_ VERTEX_ INT 
	  {FRB->markerPolygonDeleteVertexCmd($1,$5);}
	| INT DELETE_ TAG_ {FRB->markerDeleteTagCmd($1);}
	| INT DELETE_ TAG_ STRING {FRB->markerDeleteTagCmd($1,$4);}
	| INT DELETE_ TAG_ INT {FRB->markerDeleteTagCmd($1,$4);}
	| INT EDIT_ BEGIN_ INT {FRB->markerEditBeginCmd($1,$4);}
	| INT ELLIPSE_ RADIUS_
	    numeric numeric 
	    coordSystem skyFormat
	  {FRB->markerEllipseRadiusCmd($1, 
	    Vector($4, $5),
	    (CoordSystem)$6, (SkyFormat)$7);}
	| INT ELLIPSEANNULUS_ RADIUS_ 
	    numeric numeric numeric INT
	    coordSystem skyFormat
	  {FRB->markerEllipseAnnulusRadiusCmd($1, 
	    Vector($4,$5), Vector($6,$6*$5/$4), $7,
	    (CoordSystem)$8, (SkyFormat)$9);}
	| INT ELLIPSEANNULUS_ RADIUS_ 
	    STRING coordSystem skyFormat
	  {FRB->markerEllipseAnnulusRadiusCmd($1,
	    $4, (CoordSystem)$5, (SkyFormat)$6);}
	| INT EPANDA_ EDIT_
	    angle angle INT 
	    numeric numeric numeric INT 
	    coordSystem skyFrame
	  {FRB->markerEpandaEditCmd($1, 
	    $4, $5, $6,
	    Vector($7,$8), Vector($9,$9*$8/$7), $10,
	    (CoordSystem)$11, (SkyFrame)$12);}
	| INT EPANDA_ EDIT_
	    STRING STRING 
	    coordSystem skyFrame 
	    coordSystem skyFormat
	  {FRB->markerEpandaEditCmd($1, 
	    $4, $5, 
	    (CoordSystem)$6, (SkyFrame)$7,
	    (CoordSystem)$8, (SkyFormat)$9);}
	| INT FONT_ STRING {FRB->markerFontCmd($1,$3);}
	| INT HIGHLITE_ {FRB->markerHighliteCmd($1);}
	| INT HIGHLITE_ ONLY_{FRB->markerHighliteOnlyCmd($1);}
	| INT LINE_ ARROW_ yesno yesno {FRB->markerLineArrowCmd($1,$4,$5);}
	| INT LINE_ POINT_ coordSystem skyFrame coord coord
	  {FRB->markerLineCmd($1, Vector($6), Vector($7), 
	    (CoordSystem)$4, (SkyFrame)$5);}
	| INT MOVE_ numeric numeric 
	  {FRB->markerMoveCmd($1, Vector($3,$4));}
	| INT MOVE_ FRONT_ {FRB->markerFrontCmd($1);}
	| INT MOVE_ BACK_ {FRB->markerBackCmd($1);}
	| INT MOVE_ TO_ internalSystem numeric numeric
	  {FRB->markerMoveToCmd($1, Vector($5,$6), (InternalSystem)$4);}
	| INT MOVE_ TO_ coordSystem skyFrame coord 
	  {FRB->markerMoveToCmd($1,Vector($6),(CoordSystem)$4,(SkyFrame)$5);}
	| INT CPANDA_ EDIT_
	    angle angle INT 
	    numeric numeric INT 
	    coordSystem skyFrame
	  {FRB->markerCpandaEditCmd($1, 
	    $4, $5, $6, 
	    $7, $8, $9,
	    (CoordSystem)$10, (SkyFrame)$11);}
	| INT CPANDA_ EDIT_ 
	    STRING STRING 
	    coordSystem skyFrame 
	    coordSystem skyFormat
	  {FRB->markerCpandaEditCmd($1, 
	    $4, $5,
	    (CoordSystem)$6, (SkyFrame)$7,
	    (CoordSystem)$8, (SkyFormat)$9);}
	| INT POLYGON_ RESET_ numeric numeric coordSystem skyFormat
	  {FRB->markerPolygonResetCmd($1, Vector($4,$5),
	    (CoordSystem)$6, (SkyFormat)$7);}
	| INT PROJECTION_ coordSystem skyFrame coord coord numeric INT 
	    coordSystem skyFormat
	  {FRB->markerProjectionCmd($1, Vector($5), Vector($6),
	    (CoordSystem)$3, (SkyFrame)$4, $7, $8,
	    (CoordSystem)$9, (SkyFormat)$10);}
	| INT PROJECTION_ SYSTEM_ coordSystem skyFrame
	  {FRB->markerProjectionSystemCmd($1,(CoordSystem)$4,(SkyFrame)$5);}
	| INT PROPERTY_ markerProperty yesno 
	  {FRB->markerPropertyCmd($1,$3,$4);}
	| INT ROTATE_ BEGIN_ {FRB->markerRotateBeginCmd($1);}
	| INT RULER_ POINT_ coordSystem skyFrame coord coord
	  {FRB->markerRulerPointCmd($1, Vector($6), Vector($7),
	    (CoordSystem)$4, (SkyFrame)$5);}
	| INT RULER_ SYSTEM_ coordSystem skyFrame coordSystem skyFormat
	  {FRB->markerRulerSystemCmd($1, (CoordSystem)$4, (SkyFrame)$5,
	    (CoordSystem)$6, (SkyFormat)$7);}
	| INT SELECT_ {FRB->markerSelectCmd($1);}
	| INT SELECT_ ONLY_ {FRB->markerSelectOnlyCmd($1);}
	| INT TAG_ STRING {FRB->markerTagCmd($1,$3);}
	| INT TEXT_ STRING {FRB->markerTextCmd($1,$3);}
	| INT UNHIGHLITE_ {FRB->markerUnhighliteCmd($1);}
	| INT UNSELECT_ {FRB->markerUnselectCmd($1);}
	| INT VECTOR_ ARROW_ yesno {FRB->markerVectorArrowCmd($1,$4);}
	| INT VECTOR_ POINT_ coordSystem skyFrame coord 
	    coordSystem skyFormat numeric angle
	  {FRB->markerVectorCmd($1, Vector($6), (CoordSystem)$4, (SkyFrame)$5,
	    $9, (CoordSystem)$7, (SkyFormat)$8, $10);}
	| INT WIDTH_ INT {FRB->markerLineWidthCmd($1,$3);}

	| HIGHLITE_ ALL_ {FRB->markerHighliteAllCmd();}
	| HIGHLITE_ ONLY_ numeric numeric 
	  {FRB->markerHighliteOnlyCmd(Vector($3,$4));}
	| HIGHLITE_ TOGGLE_ numeric numeric
	  {FRB->markerHighliteToggleCmd(Vector($3,$4));}

	| LIST_ markerList
	| LOAD_ markerLoad
	| MOVE_ markerMoveSelected
	| PRESERVE_ yesno {FRB->markerPreserveCmd($2);}
	| PROPERTY_ markerProperty yesno {FRB->markerPropertyCmd($2,$3);}
	| PROPERTY_ markerProperty yesno numeric numeric
	  {FRB->markerPropertyCmd($2,$3,Vector($4,$5));}
	| ROTATE_ BEGIN_ numeric numeric 
	  {FRB->markerRotateBeginCmd(Vector($3,$4));}
	| ROTATE_ MOTION_ numeric numeric INT
	  {FRB->markerRotateMotionCmd(Vector($3,$4),$5);}
	| ROTATE_ END_ {FRB->markerRotateEndCmd();}
	| SAVE_ STRING markerFormat coordSystem skyFrame skyFormat yesno yesno
	  {FRB->markerSaveCmd($2, (FrameBase::MarkerFormat)$3,
	    (CoordSystem)$4, (SkyFrame)$5, (SkyFormat)$6, $7, $8);}
	| SAVE_ TEMPLATE_ STRING {FRB->markerSaveTemplateCmd($3);}
	| SELECT_ markerSelect

	| SHOW_ INT {FRB->markerShowCmd($2);}

	| STRING COLOR_ STRING {FRB->markerColorCmd($1,$3);}
	| STRING COPY_ {FRB->markerCopyCmd($1);}
	| STRING DELETE_ {FRB->markerDeleteCmd($1);}
	| STRING CUT_ {FRB->markerCutCmd($1);}
	| STRING FONT_ STRING {FRB->markerFontCmd($1,$3);}
	| STRING HIGHLITE_ {FRB->markerHighliteCmd($1);}
	| STRING HIGHLITE_ ONLY_ {FRB->markerHighliteOnlyCmd($1);}
	| STRING MOVE_ numeric numeric {FRB->markerMoveCmd($1,Vector($3,$4));}
	| STRING MOVE_ FRONT_ {FRB->markerFrontCmd($1);}
	| STRING MOVE_ BACK_ {FRB->markerBackCmd($1);}
	| STRING MOVE_ TO_ coordSystem skyFrame coord 
	  {FRB->markerMoveToCmd($1,Vector($6),(CoordSystem)$4,(SkyFrame)$5);}
	| STRING PROPERTY_ markerProperty yesno 
	  {FRB->markerPropertyCmd($1,$3,$4);}
	| STRING SELECT_ {FRB->markerSelectCmd($1);}
	| STRING SELECT_ ONLY_ {FRB->markerSelectOnlyCmd($1);}
	| STRING UNHIGHLITE_ {FRB->markerUnhighliteCmd($1);}
	| STRING UNSELECT_ {FRB->markerUnselectCmd($1);}

	| TAG_ EDIT_ STRING STRING {FRB->markerTagEditCmd($3,$4);}
	| TAG_ DELETE_ STRING {FRB->markerTagDeleteCmd($3);}
	| TAG_ DELETE_ ALL_ {FRB->markerTagDeleteAllCmd();}
	| TAG_ STRING {FRB->markerTagCmd($2);}
	| TAG_ UPDATE_ STRING {FRB->markerTagUpdateCmd($3);}

	| PASTE_ {FRB->markerPasteCmd();}
	| PASTE_ coordSystem skyFrame 
	  {FRB->markerPasteCmd((CoordSystem)$2,(SkyFrame)$3);}
	| UNDO_ {FRB->markerUndoCmd();}
	| UNHIGHLITE_ ALL_ {FRB->markerUnhighliteAllCmd();}
	| UNSELECT_ ALL_ {FRB->markerUnselectAllCmd();}
	| WIDTH_ INT {FRB->markerLineWidthCmd($2);}
	;

markerCallBack : SELECT_ {$$ = CallBack::SELECTCB;}
	| UNSELECT_ {$$ = CallBack::UNSELECTCB;}
	| HIGHLITE_ {$$ = CallBack::HIGHLITECB;}
	| UNHIGHLITE_ {$$ = CallBack::UNHIGHLITECB;}
	| BEGIN_ MOVE_  {$$ = CallBack::MOVEBEGINCB;}
	| MOVE_  {$$ = CallBack::MOVECB;}
	| END_ MOVE_  {$$ = CallBack::MOVEENDCB;}
	| BEGIN_ EDIT_ {$$ = CallBack::EDITBEGINCB;}
	| EDIT_ {$$ = CallBack::EDITCB;}
	| END_ EDIT_ {$$ = CallBack::EDITENDCB;}
	| BEGIN_ ROTATE_ {$$ = CallBack::ROTATEBEGINCB;}
	| ROTATE_ {$$ = CallBack::ROTATECB;}
	| END_ ROTATE_ {$$ = CallBack::ROTATEENDCB;}
	| DELETE_ {$$ = CallBack::DELETECB;}
	| TEXT_ {$$ = CallBack::TEXTCB;}
	| COLOR_ {$$ = CallBack::COLORCB;}
	| WIDTH_ {$$ = CallBack::LINEWIDTHCB;}
	| PROPERTY_ {$$ = CallBack::PROPERTYCB;}
	| FONT_ {$$ = CallBack::FONTCB;}
	| UPDATE_ {$$ = CallBack::UPDATECB;}
	;

markerCreate : 
	  CIRCLE_ numeric numeric 
	    numeric 
	    markerProperties
	  {FRB->createCircleCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    $4,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| ANNULUS_ numeric numeric 
	    numeric numeric INT 
	    markerProperties
	  {FRB->createAnnulusCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    $4,$5,$6,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| CPANDA_ numeric numeric 
	    angle angle INT
	    numeric numeric INT 
	    markerProperties
	  {FRB->createCpandaCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    $4,$5,$6,
	    $7,$8,$9,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}

	| ELLIPSE_ numeric numeric 
	    numeric numeric 
	    optangle 
	    markerProperties
	  {FRB->createEllipseCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    Vector($4,$5),
	    $6,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| ELLIPSEANNULUS_ numeric numeric 
	    numeric numeric numeric INT 
	    optangle 
	    markerProperties
	  {FRB->createEllipseAnnulusCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    Vector($4,$5), 
	    Vector($6,$6*$4/$5),$7,
	    $8,
	    currentColor,currentWidth,currentFont, 
	    currentText,currentProps,NULL,taglist,cblist);}
	| EPANDA_ numeric numeric 
	    angle angle INT
	    numeric numeric numeric INT 
	    optangle 
	    markerProperties
	  {FRB->createEpandaCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    $4,$5,$6,
	    Vector($7,$8),
	    Vector($9,$9*$7/$8),$10,
	    $11,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}

	| BOX_ numeric numeric 
	    numeric numeric 
	    optangle 
	    markerProperties
	  {FRB->createBoxCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    Vector($4,$5),
	    $6,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| BOXANNULUS_ numeric numeric 
	    numeric numeric numeric INT 
	    optangle 
	    markerProperties
	  {FRB->createBoxAnnulusCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    Vector($4,$5),Vector($6,$6*$4/$5),$7,
	    $8,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| BPANDA_ numeric numeric 
	    angle angle INT
	    numeric numeric numeric INT 
	    optangle 
	    markerProperties
	  {FRB->createBpandaCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    $4,$5,$6,
	    Vector($7,$8),
	    Vector($9,$9*$7/$8),$10,
	    $11,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}

	| LINE_ numeric numeric 
	    numeric numeric 
	    markerProperties
	  {FRB->createLineCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    Vector($4,$5)*FRB->getCanvasToRef(),
	    0, 0,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| VECTOR_ numeric numeric 
	    numeric numeric 
	    markerProperties
	  {FRB->createVectCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    Vector($4,$5)*FRB->getCanvasToRef(),
	    1,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| PROJECTION_ numeric numeric 
	    numeric numeric 
	    numeric 
	    markerProperties
	  {FRB->createProjectionCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    Vector($4,$5)*FRB->getCanvasToRef(),
	    $6,
	    NULL,NULL,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| PROJECTION_ numeric numeric 
	    numeric numeric 
	    numeric 
	    STRING STRING 
	    markerProperties
	  {FRB->createProjectionCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    Vector($4,$5)*FRB->getCanvasToRef(),
	    $6,
	    $7,$8,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| RULER_ numeric numeric 
	    numeric numeric 
	    coordSystem skyFrame 
	    markerProperties
	  {FRB->createRulerCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    Vector($4,$5)*FRB->getCanvasToRef(),
	    (CoordSystem)$6, (SkyFrame)$7, PHYSICAL, DEGREES,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}

	| CIRCLE_ POINT_ numeric numeric 
	    markerProperties
	  {FRB->createCirclePointCmd(Vector($3,$4)*FRB->getCanvasToRef(),
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| BOX_ POINT_ numeric numeric 
	    markerProperties
	  {FRB->createBoxPointCmd(Vector($3,$4)*FRB->getCanvasToRef(),
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| DIAMOND_ POINT_ numeric numeric 
	    markerProperties
	  {FRB->createDiamondPointCmd(Vector($3,$4)*FRB->getCanvasToRef(),
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| CROSS_ POINT_ numeric numeric 
	    markerProperties
	  {FRB->createCrossPointCmd(Vector($3,$4)*FRB->getCanvasToRef(),
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| 'X' POINT_ numeric numeric 
	    markerProperties
	  {FRB->createExPointCmd(Vector($3,$4)*FRB->getCanvasToRef(),
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| ARROW_ POINT_ numeric numeric 
	    markerProperties
	  {FRB->createArrowPointCmd(Vector($3,$4)*FRB->getCanvasToRef(),
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}
	| BOXCIRCLE_ POINT_ numeric numeric 
	    markerProperties
	  {FRB->createBoxCirclePointCmd(Vector($3,$4)*FRB->getCanvasToRef(),
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}

	| TEXT_ numeric numeric 
	    optangle 
	    markerProperties
	  {FRB->createTextCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    $4,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}

	| POLYGON_ numeric numeric 
	    numeric numeric 
	    markerProperties
	  {FRB->createPolygonCmd(Vector($2,$3)*FRB->getCanvasToRef(),
	    Vector($4,$5),
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}

	| COMPASS_ numeric numeric 
	    numeric 
	    coordSystem skyFrame
	    markerProperties
	  {FRB->createCompassCmd(Vector($2,$3)*FRB->getCanvasToRef(), 
	    $4,
	    "N", "E", 1, 1,
	    (CoordSystem)$5, (SkyFrame)$6,
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}

	| COMPOSITE_ markerProperties
	  {FRB->createCompositeCmd(
	    currentColor,currentWidth,currentFont,
	    currentText,currentProps,NULL,taglist,cblist);}

	| TEMPLATE_ numeric numeric STRING
	  {FRB->createTemplateCmd(Vector($2,$3)*FRB->getCanvasToRef(), $4);}

	| TEMPLATE_ numeric numeric VAR_ STRING
	  {FRB->createTemplateVarCmd(Vector($2,$3)*FRB->getCanvasToRef(), $5);}
	;

markerEdit : BEGIN_ numeric numeric INT 
	  {FRB->markerEditBeginCmd(Vector($2,$3),$4);}
	| MOTION_ numeric numeric INT 
	  {FRB->markerEditMotionCmd(Vector($2,$3),$4);}
	| END_ {FRB->markerEditEndCmd();}
	;

markerFormat : DS9_ {$$ = FrameBase::DS9;}
	| CIAO_ {$$ = FrameBase::CIAO;}
	| SAOTNG_ {$$ = FrameBase::SAOTNG;}
	| SAOIMAGE_ {$$ = FrameBase::SAOIMAGE;}
	| PROS_ {$$ = FrameBase::PROS;}
	| XY_ {$$ = FrameBase::RAWXY;}
	;

markerGet : COLOR_ {FRB->getMarkerColorCmd();}
	| FONT_ {FRB->getMarkerFontCmd();}
	| HANDLE_ numeric numeric {FRB->getMarkerHandleCmd(Vector($2,$3));}
	| ID_ numeric numeric {FRB->getMarkerIdCmd(Vector($2,$3));}
	| ID_ ALL_ {FRB->getMarkerIdAllCmd();}
	| INT ANGLE_ coordSystem skyFrame
	  {FRB->getMarkerAngleCmd($1,(CoordSystem)$3, (SkyFrame)$4);}
	| INT ANNULUS_ RADIUS_ coordSystem skyFormat
	  {FRB->getMarkerAnnulusRadiusCmd($1, (CoordSystem)$4, (SkyFormat)$5);}
	| INT BOXANNULUS_ RADIUS_ coordSystem skyFormat
	  {FRB->getMarkerBoxAnnulusRadiusCmd($1,(CoordSystem)$4,(SkyFormat)$5);}
	| INT BOX_ RADIUS_ coordSystem skyFormat
	  {FRB->getMarkerBoxRadiusCmd($1,(CoordSystem)$4, (SkyFormat)$5);}
	| INT BPANDA_ ANGLE_ coordSystem skyFrame
	  {FRB->getMarkerBpandaAnglesCmd($1, (CoordSystem)$4, (SkyFrame)$5);}
	| INT BPANDA_ RADIUS_ coordSystem skyFormat
	  {FRB->getMarkerBpandaRadiusCmd($1, (CoordSystem)$4, (SkyFormat)$5);}
	| INT CENTER_ coordSystem skyFrame skyFormat 
	  {FRB->getMarkerCenterCmd($1, 
	    (CoordSystem)$3, (SkyFrame)$4, (SkyFormat)$5);}
	| INT CIRCLE_ RADIUS_ coordSystem skyFormat
	  {FRB->getMarkerCircleRadiusCmd($1, (CoordSystem)$4, (SkyFormat)$5);}
	| INT COLOR_ {FRB->getMarkerColorCmd($1);}
	| INT COMPASS_ ARROW_ {FRB->getMarkerCompassArrowCmd($1);}
	| INT COMPASS_ LABEL_ {FRB->getMarkerCompassLabelCmd($1);}
	| INT COMPASS_ SYSTEM_ {FRB->getMarkerCompassSystemCmd($1);}
	| INT ELLIPSE_ RADIUS_ coordSystem skyFormat
	  {FRB->getMarkerEllipseRadiusCmd($1, (CoordSystem)$4, (SkyFormat)$5);}
	| INT ELLIPSEANNULUS_ RADIUS_ coordSystem skyFormat
	  {FRB->getMarkerEllipseAnnulusRadiusCmd($1, 
	    (CoordSystem)$4, (SkyFormat)$5);}
	| INT EPANDA_ ANGLE_ coordSystem skyFrame
	  {FRB->getMarkerEpandaAnglesCmd($1, (CoordSystem)$4, (SkyFrame)$5);}
	| INT EPANDA_ RADIUS_ coordSystem skyFormat
	  {FRB->getMarkerEpandaRadiusCmd($1, (CoordSystem)$4, (SkyFormat)$5);}
	| INT FONT_ {FRB->getMarkerFontCmd($1);}
	| INT LINE_ ARROW_ {FRB->getMarkerLineArrowCmd($1);}
	| INT LINE_ LENGTH_ coordSystem skyFormat
	  {FRB->getMarkerLineLengthCmd($1, (CoordSystem)$4, (SkyFormat)$5);}
	| INT LINE_ POINT_ coordSystem skyFrame skyFormat
	  {FRB->getMarkerLineCmd($1, (CoordSystem)$4,
	    (SkyFrame)$5, (SkyFormat)$6);}
	| INT MAP_ LENGTH_ numeric coordSystem skyFormat
	  {FRB->getMarkerMapLenFromRefCmd($1, $4,
	    (CoordSystem)$5,(SkyFormat)$6);}
	| INT CPANDA_ ANGLE_ coordSystem skyFrame
	  {FRB->getMarkerCpandaAnglesCmd($1, (CoordSystem)$4, (SkyFrame)$5);}
	| INT CPANDA_ RADIUS_ coordSystem skyFormat
	  {FRB->getMarkerCpandaRadiusCmd($1, (CoordSystem)$4, (SkyFormat)$5);}
	| INT PROJECTION_ STRING STRING STRING STRING
	  {FRB->getMarkerProjectionCmd($1,$3,$4,$5,$6);}
	| INT PROJECTION_ POINT_ coordSystem skyFrame skyFormat
	  {FRB->getMarkerProjectionPointsCmd($1, (CoordSystem)$4,
	    (SkyFrame)$5, (SkyFormat)$6);}
	| INT PROJECTION_ LENGTH_ coordSystem skyFormat
	  {FRB->getMarkerProjectionLengthCmd($1, 
	    (CoordSystem)$4, (SkyFormat)$5);}
	| INT PROJECTION_ METHOD_ {FRB->getMarkerProjectionMethodCmd($1);} 
	| INT PROJECTION_ THICK_ coordSystem skyFormat
	  {FRB->getMarkerProjectionWidthCmd($1,(CoordSystem)$4,(SkyFormat)$5);}
	| INT PROJECTION_ SYSTEM_ {FRB->getMarkerProjectionSystemCmd($1);}
	| INT PROPERTY_ {FRB->getMarkerPropertyCmd($1);}
	| INT PROPERTY_ markerProperty {FRB->getMarkerPropertyCmd($1,$3);}
	| INT RULER_ LENGTH_ coordSystem skyFormat
	  {FRB->getMarkerRulerLengthCmd($1,(CoordSystem)$4,(SkyFormat)$5);}
	| INT RULER_ POINT_ coordSystem skyFrame skyFormat
	  {FRB->getMarkerRulerPointCmd($1, (CoordSystem)$4,
	    (SkyFrame)$5, (SkyFormat)$6);}
	| INT RULER_ SYSTEM_ {FRB->getMarkerRulerSystemCmd($1);}
	| INT TAG_ {FRB->getMarkerTagCmd($1);}
	| INT TAG_ INT {FRB->getMarkerTagCmd($1,$3);}
	| INT TEXT_ {FRB->getMarkerTextCmd($1);}
	| INT TYPE_ {FRB->getMarkerTypeCmd($1);}
	| INT VECTOR_ ARROW_ {FRB->getMarkerVectorArrowCmd($1);}
	| INT VECTOR_ LENGTH_ coordSystem skyFormat
	  {FRB->getMarkerVectorLengthCmd($1, (CoordSystem)$4, (SkyFormat)$5);}
	| INT VECTOR_ POINT_ coordSystem skyFrame skyFormat
	  {FRB->getMarkerVectorCmd($1, (CoordSystem)$4,
	    (SkyFrame)$5, (SkyFormat)$6);}
	| INT WIDTH_ {FRB->getMarkerLineWidthCmd($1);}
	| HIGHLITE_ markerGetHighlite
	| HIGHLITE_ NUMBER_ {FRB->getMarkerHighlitedNumberCmd();}
	| POLYGON_ SEGMENT_ numeric numeric
	  {FRB->getMarkerPolygonSegmentCmd(Vector($3,$4));}
	| PRESERVE_ {FRB->getMarkerPreserveCmd();}
	| PROPERTY_ markerProperty {FRB->getMarkerPropertyCmd($2);}
	| SELECT_ markerGetSelect
	| SELECT_ NUMBER_ {FRB->getMarkerSelectedNumberCmd();}

	| SHOW_ {FRB->getMarkerShowCmd();}

	| STRING COLOR_ {FRB->getMarkerColorCmd($1);}
	| STRING FONT_ {FRB->getMarkerFontCmd($1);}
	| STRING PROPERTY_ markerProperty {FRB->getMarkerPropertyCmd($1,$3);}
	| STRING TAG_ {FRB->getMarkerTagCmd($1);}
	| STRING TAG_ NUMBER_ {FRB->getMarkerTagNumberCmd($1);}

	| TAG_ ALL_ {FRB->getMarkerTagsCmd();}
	| TAG_ DEFAULT_ NAME_ {FRB->getMarkerTagDefaultNameCmd();}

	| WIDTH_ {FRB->getMarkerLineWidthCmd();}
	;

markerGetSelect : /* empty */ {FRB->getMarkerSelectedCmd();}
	| numeric numeric {FRB->getMarkerSelectedCmd(Vector($1,$2));}
	;

markerGetHighlite : /* empty */ {FRB->getMarkerHighlitedCmd();}
	| numeric numeric {FRB->getMarkerHighlitedCmd(Vector($1,$2));}
	;

markerInitProp : {
	  strncpy(currentFont, "helvetica 10 normal", 32);
	  strncpy(currentColor, "green", 16);
	  currentWidth = 1;
	  strncpy(currentText, "", 80);
	  currentProps = defaultProps;
	  taglist.deleteAll();
	  cblist.deleteAll();
	}
	;

markerLayer : /* empty */ {FRB->markerLayerCmd(FrameBase::USER);}
	| USER_ {FRB->markerLayerCmd(FrameBase::USER);}
	| CATALOG_  {FRB->markerLayerCmd(FrameBase::CATALOG);}
	;

markerList : markerFormat coordSystem skyFrame skyFormat 
	    yesno yesno markerQuery markerTags
	  {FRB->markerListCmd((FrameBase::MarkerFormat)$1,
	    (CoordSystem)$2, (SkyFrame)$3, (SkyFormat)$4, $5, $6,
	    0, propQMask, propQValue, taglist);}
	| SELECT_ markerFormat coordSystem skyFrame skyFormat 
	    yesno yesno markerQuery markerTags
	  {FRB->markerListCmd((FrameBase::MarkerFormat)$2, 
	    (CoordSystem)$3, (SkyFrame)$4, (SkyFormat)$5, $6, $7,
	    1, propQMask, propQValue, taglist);}
	;

markerLoad : markerFormat STRING
	  {FRB->markerLoadCmd((FrameBase::MarkerFormat)$1,$2);}
	| markerFormat STRING coordSystem skyFrame
	  {FRB->markerLoadCmd((FrameBase::MarkerFormat)$1,$2,
	    (CoordSystem)$3,(SkyFrame)$4);}
	| markerFormat INT
	  {FRB->markerLoadCmd((FrameBase::MarkerFormat)$1,$2);}
	| markerFormat INT coordSystem skyFrame
	  {FRB->markerLoadCmd((FrameBase::MarkerFormat)$1,$2,
	    (CoordSystem)$3,(SkyFrame)$4);}
	| FITS_ STRING STRING INT STRING 
	  {FRB->markerLoadFitsCmd($2, $3, $4, $5);}
	;

markerMoveSelected : numeric numeric {FRB->markerMoveCmd(Vector($1,$2));}
	| FRONT_ {FRB->markerFrontCmd();}
	| BACK_ {FRB->markerBackCmd();}
	| BEGIN_ numeric numeric {FRB->markerMoveBeginCmd(Vector($2,$3));}
	| MOTION_ numeric numeric {FRB->markerMoveMotionCmd(Vector($2,$3));}
	| END_ {FRB->markerMoveEndCmd();}
	| TO_ coordSystem skyFrame coord 
	  {FRB->markerMoveToCmd(Vector($4), (CoordSystem)$2, (SkyFrame)$3);}
	;

markerProps : markerProps markerProp
	| markerProp
	;

markerProp : markerProperty '=' yesno {setProps(&currentProps,$1,$3);}
	| COLOR_ '=' STRING {strncpy(currentColor,$3,16);}
	| WIDTH_ '=' INT {currentWidth = $3;}
	| FONT_ '=' STRING {strncpy(currentFont,$3,32);}
	| TEXT_ '=' STRING {strncpy(currentText,$3,80);}
	| tag
	| callback
	;

markerProperty : NONE_ {$$ = Marker::NONE;}
	| SELECT_ {$$ = Marker::SELECT;}
	| HIGHLITE_ {$$ = Marker::HIGHLITE;}
	| EDIT_ {$$ = Marker::EDIT;}
	| MOVE_  {$$ = Marker::MOVE;}
	| ROTATE_ {$$ = Marker::ROTATE;}
	| DELETE_ {$$ = Marker::DELETE;}
	| INCLUDE_ {$$ = Marker::INCLUDE;}
	| SOURCE_ {$$ = Marker::SOURCE;}
	| FIXED_ {$$ = Marker::FIXED;}
	;

markerProperties : /* empty */ markerInitProp
	| markerInitProp markerProps
	;

markerQuery: /* empty */ {propQMask=0;propQValue=0;}
	| {propQMask=0;propQValue=0;} queries
	;

markerSelect : ALL_ {FRB->markerSelectAllCmd();}
	| ONLY_ numeric numeric {FRB->markerSelectOnlyCmd(Vector($2,$3));}
	| TOGGLE_ {FRB->markerSelectToggleCmd();}
	| TOGGLE_ numeric numeric {FRB->markerSelectToggleCmd(Vector($2,$3));}
	;

queries : queries query
	| query
	;

query	: markerProperty '=' yesno 
	  {propQMask |= $1; setProps(&propQValue,$1,$3);}
	;

markerTags: /* empty */ {taglist.deleteAll()}
	| {taglist.deleteAll()} tags
	;

tags	: tags tag
	| tag
	;

tag	: TAG_ '=' STRING {taglist.append(new Tag($3));}
	;

callback : CALLBACK_ '=' markerCallBack STRING STRING {cblist.append(
	new CallBack(FRB->getInterp(),(CallBack::Type)$3,$4,$5));}
	;

orient	: 'X' {FRB->orientCmd(XX);}
	| 'Y' {FRB->orientCmd(YY);}
	| XY_ {FRB->orientCmd(XY);}
	| NONE_ {FRB->orientCmd(NORMAL);}
	;

pan	: numeric numeric numeric numeric 
	  {FRB->panCmd(Vector($1,$2),Vector($3,$4));}
	| internalSystem numeric numeric
	    {FRB->panCmd(Vector($2,$3), (InternalSystem)$1);}
	| coordSystem skyFrame coord
	    {FRB->panCmd(Vector($3), (CoordSystem)$1, (SkyFrame)$2);}
	| TO_ panTo
	| BBOX_ numeric numeric {FRB->panBBoxCmd(Vector($2,$3));}
	| MOTION_ panMotion
	| PRESERVE_ yesno {FRB->panPreserveCmd($2);}
	;

panTo	: internalSystem numeric numeric
	    {FRB->panToCmd(Vector($2,$3), (InternalSystem)$1);}
	| coordSystem skyFrame coord
	    {FRB->panToCmd(Vector($3), (CoordSystem)$1, (SkyFrame)$2);}
	;

panMotion : BEGIN_ numeric numeric {FRB->panMotionBeginCmd(Vector($2,$3));}
	| numeric numeric {FRB->panMotionCmd(Vector($1,$2));}
	| END_ numeric numeric {FRB->panMotionEndCmd(Vector($2,$3));}
	;

panner	: yesno {FRB->pannerCmd($1);}
	| WCS_ wcsSystem skyFrame 
	  {FRB->pannerCmd((CoordSystem)$2,(SkyFrame)$3);}
	| STRING INT INT {FRB->pannerCmd($1, $2, $3);}
	| UPDATE_ {FRB->updatePannerCmd();}
	;

postscript : COLORSPACE_ pscolorspace 
	  {FRB->psColorSpaceCmd((Widget::PSColorSpace)$2);}
	| LEVEL_ INT {FRB->psLevelCmd($2);}
	| RESOLUTION_ INT {FRB->psResolutionCmd($2);}
	;

pscolorspace : BW_ {$$ = Widget::BW;}
	| GRAY_ {$$ = Widget::GRAY;}
	| RGB_ {$$ = Widget::RGB;}
	| CMYK_ {$$ = Widget::CMYK;}
	;

region	: HIGHLITE_ regionHighlite
	| SELECT_ regionSelect
	;

regionHighlite : BEGIN_ numeric numeric 
	  {FRB->regionHighliteBeginCmd(Vector($2,$3));}
	| MOTION_ numeric numeric 
	  {FRB->regionHighliteMotionCmd(Vector($2,$3));}
	| END_ {FRB->regionHighliteEndCmd();}
	| SHIFT_ END_ {FRB->regionHighliteShiftEndCmd();}
	;

regionSelect : BEGIN_ numeric numeric 
	  {FRB->regionSelectBeginCmd(Vector($2,$3));}
	| MOTION_ numeric numeric {FRB->regionSelectMotionCmd(Vector($2,$3));}
	| END_ {FRB->regionSelectEndCmd();}
	| SHIFT_ END_ {FRB->regionSelectShiftEndCmd();}
	;

rgb	: CHANNEL_ STRING {FRB->setRGBChannelCmd($2);}
	| SYSTEM_ coordSystem {FRB->setRGBSystemCmd((CoordSystem)$2);}
	| VIEW_ INT INT INT {FRB->setRGBViewCmd($2,$3,$4);}
	;

rotate	: numeric {FRB->rotateCmd(degToRad($1));}
	| numeric DEGREES_ {FRB->rotateCmd(degToRad($1));}
	| MOTION_ rotateMotion
	| TO_ numeric {FRB->rotateToCmd(degToRad($2));}
	| TO_ numeric DEGREES_ {FRB->rotateToCmd(degToRad($2));}
	;

rotateMotion : BEGIN_ {FRB->rotateMotionBeginCmd();}
	| numeric {FRB->rotateMotionCmd(degToRad($1));}
	| END_ {FRB->rotateMotionEndCmd();}
	;

save	: ARRAY_ saveArray
	| FITS_ saveFits
	;

saveArray : FILE_ STRING {FRB->saveArrayFileCmd($2);}
	| CHANNEL_ STRING {FRB->saveArrayChannelCmd($2);}
	| SOCKET_ INT {FRB->saveArraySocketCmd($2);}
	;

saveFits: IMAGE_ saveFitsImage
	| TABLE_ saveFitsTable
	| RESAMPLE_ saveFitsResample
	;

saveFitsImage : FILE_ STRING compressType {FRB->saveFitsImageFileCmd($2, $3);}
	| CHANNEL_ STRING compressType {FRB->saveFitsImageChannelCmd($2, $3);}
	| SOCKET_ INT compressType {FRB->saveFitsImageSocketCmd($2, $3);}
	;

saveFitsTable : FILE_ STRING compressType {FRB->saveFitsTableFileCmd($2, $3);}
	| CHANNEL_ STRING compressType {FRB->saveFitsTableChannelCmd($2, $3);}
	| SOCKET_ INT compressType {FRB->saveFitsTableSocketCmd($2, $3);}
	;

saveFitsResample : FILE_ STRING compressType 
	  {FRB->saveFitsResampleFileCmd($2, $3);}
	| CHANNEL_ STRING compressType 
	  {FRB->saveFitsResampleChannelCmd($2, $3);}
	| SOCKET_ INT compressType 
	  {FRB->saveFitsResampleSocketCmd($2, $3);}
	;

smooth	: smoothFunction INT 
	  {FRB->smoothCmd((FrameBase::SmoothFunction)$1,$2);}
	| DELETE_ {FRB->smoothDeleteCmd();}
	;

smoothFunction : BOXCAR_ {$$ = FrameBase::BOXCAR;}
	| GAUSSIAN_ {$$ = FrameBase::GAUSSIAN;}
	| TOPHAT_ {$$ = FrameBase::TOPHAT;}
	;

update	: /* empty */ {FRB->updateFitsCmd(0);}
	| INT numeric numeric numeric numeric
	  {FRB->updateFitsCmd($1,BBox($2,$3,$4,$5),0);}
	| NOW_ {FRB->updateFitsCmd(1);}
	| NOW_ INT numeric numeric numeric numeric
	  {FRB->updateFitsCmd($2,BBox($3,$4,$5,$6),1);}
	| FITS_ SLICE_ INT {FRB->sliceCmd($3);}
	;

warp	: /* empty */ numeric numeric {FRB->warpCmd(Vector($1,$2));}
	| TO_ numeric numeric {FRB->warpToCmd(Vector($2,$3));}
	;

wcs	: ALIGN_ INT wcsSystem skyFrame 
	  {FRB->wcsAlignCmd($2,(CoordSystem)$3,(SkyFrame)$4);}
	| ZOOM_ wcsSystem numeric numeric 
	  {FRB->wcsZoomCmd((CoordSystem)$2, Vector($3,$4));}
	| RESET_ {FRB->wcsResetCmd();}
	| REPLACE_ INT {FRB->wcsReplaceCmd($2);}
	| REPLACE_ STRING {FRB->wcsReplaceCmd($2);}
	| APPEND_ INT {FRB->wcsAppendCmd($2);}
	| APPEND_ STRING {FRB->wcsAppendCmd($2);}
	;

zoom	: numeric numeric {FRB->zoomCmd(Vector($1,$2));}
	| numeric numeric ABOUT_ internalSystem numeric numeric
	  {FRB->zoomAboutCmd(Vector($1,$2),Vector($5,$6),(InternalSystem)$4);}
	| numeric numeric ABOUT_ coordSystem skyFrame coord
	  {FRB->zoomAboutCmd(Vector($1,$2),Vector($6),
	    (CoordSystem)$4,(SkyFrame)$5);}
	| TO_ zoomTo
	;

zoomTo	: FIT_	{FRB->zoomToFitCmd();}
	| FIT_	numeric {FRB->zoomToFitCmd($2);}
	| numeric numeric {FRB->zoomToCmd(Vector($1,$2));}
	| numeric numeric ABOUT_ internalSystem numeric numeric
	  {FRB->zoomToAboutCmd(Vector($1,$2),Vector($5,$6),
	    (InternalSystem)$4);}
	| numeric numeric ABOUT_ coordSystem skyFrame coord
	  {FRB->zoomToAboutCmd(Vector($1,$2), Vector($6),
	    (CoordSystem)$4, (SkyFrame)$5);}
	;

%%

static void setProps(unsigned short* props, unsigned short prop, int value)
{
  if (value)
    *props |= prop;
  else
    *props &= ~prop;
}
