// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "marker.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

static int markerSeqID = 1;

// Tag

Tag::Tag(const Tag& t)
{
  tag_ = dupstr(t.tag_);
}

Tag::Tag(const char* t)
{
  tag_ = dupstr(t);
}

Tag& Tag::operator=(const Tag& t) 
{
  tag_ = dupstr(t.tag_);
  return *this;
}

Tag::~Tag()
{
  if (tag_)
    delete [] tag_;
}

void Tag::set(const char* t)
{
  if (tag_)
    delete tag_;

  tag_ = dupstr(t);
}

// Marker Members Public

Marker::Marker(const Marker& a)
{
  id = a.id;
  strcpy(type, a.type);
  parent = a.parent;

  center = a.center;
  bbox = a.bbox;
  allBBox = a.allBBox;

  angle = a.angle;
  flip_ = a.flip_;

  numHandle = a.numHandle;
  if (numHandle) {
    handle = new Vector[numHandle];
    for (int i=0; i<numHandle; i++)
      handle[i] = a.handle[i];
  }
  else
    handle = NULL;

  colorName = dupstr(a.colorName);
  color = a.color;
  lineWidth = a.lineWidth;
  properties = a.properties;
  selected = a.selected;
  highlited = a.highlited;

  text = dupstr(a.text);
  font = Tk_GetFont(parent->interp, parent->tkwin, Tk_NameOfFont(a.font));
  comment = dupstr(a.comment);

  display = a.display;
  gc = a.gc;

  // disable Callbacks by default
  doCB = 0;

  tags = a.tags;
  callbacks = a.callbacks;

  previous_ = NULL;
  next_ = NULL;
}

Marker::Marker(FrameBase* p, const Vector& ctr, 
	       double ang,
	       const char* clr, int w,
	       const char* f, const char* t, unsigned short prop, 
	       const char* c, const List<Tag>& tg, const List<CallBack>& cb)
{
  id = markerSeqID++;
  type[0] = '\0';
  parent = p;

  center = ctr;

  angle = ang;
  if (!parent->isIIS())
    flip_ = FlipY();

  handle = NULL;
  numHandle = 0;

  colorName = dupstr(clr);
  color = parent->getColor(colorName);
  lineWidth = w;
  properties = prop;
  selected = 0;
  highlited = 0;

  text = dupstr(t);
  font = Tk_GetFont(parent->interp, parent->tkwin, f);
  comment = dupstr(c);

  display = parent->display;
  gc = parent->markerGC;

  doCB = 1;

  tags = tg;
  callbacks = cb;

  previous_ = NULL;
  next_ = NULL;
}

Marker::~Marker()
{
  if (colorName)
    delete [] colorName;

  if (text)
    delete [] text;

  if (comment)
    delete [] comment;

  if (font)
    Tk_FreeFont(font);

  if (handle)
    delete [] handle;

  doCallBack(CallBack::DELETECB);
}

void Marker::deleteCBs()
{
  callbacks.deleteAll();
}

void Marker::newIdentity()
{
  id = markerSeqID++;

  doCB = 1;
  deleteCBs();
  updateBBox();
}

void Marker::updateCoords(const Matrix& mx)
{
  center*=mx;
  updateBBox();
}

void Marker::draw(Drawable drawable)
{
  renderHandles(drawable);
  renderText(drawable, parent->refToWidget, SRC);
  render(drawable, parent->refToWidget, SRC);
}

void Marker::drawXOR()
{
  renderText(Tk_WindowId(parent->tkwin), parent->refToWindow, XOR);
  render(Tk_WindowId(parent->tkwin), parent->refToWindow, XOR);
}

void Marker::drawMagnifier(const Matrix& mx)
{
  renderText(parent->magnifierPixmap, mx, SRC);
  render(parent->magnifierPixmap, mx, SRC);
}

void Marker::moveTo(const Vector& v)
{
  center=v;
  updateBBox();
  doCallBack(CallBack::MOVECB);
}

void Marker::moveBegin()
{
  doCallBack(CallBack::MOVEBEGINCB);
}

void Marker::move(const Vector& v)
{
  center+=v;
  updateBBox();
  doCallBack(CallBack::MOVECB);
}

void Marker::moveEnd()
{
  doCallBack(CallBack::MOVEENDCB);
}

void Marker::editBegin(int)
{
  doCallBack(CallBack::EDITBEGINCB);
}

void Marker::edit(const Vector& v, int h)
{
  doCallBack(CallBack::EDITCB);
}

void Marker::editEnd()
{
  doCallBack(CallBack::EDITENDCB);
}

void Marker::rotateBegin()
{
  doCallBack(CallBack::ROTATEBEGINCB);
}

void Marker::rotate(const Vector& v, int h)
{
  // v is in ref coords
  // handles are in canvas coords

  double a = (v * Translate(-center) * flip_).angle();
  double b = ((handle[h-1]*parent->canvasToRef * Translate(-center) * flip_)).angle();
  angle -= a-b;

  updateBBox();
  doCallBack(CallBack::ROTATECB);
}

void Marker::rotateEnd()
{
  doCallBack(CallBack::ROTATEENDCB);
}

void Marker::ps(int mode)
{
  if (properties & SOURCE)
    psLineNoDash();
  else
    psLineDash();

  // set color
  psColor(mode, colorName);

  // do text
  if (text && font) {
    ostringstream str;

    Tcl_DString psFont;
    Tcl_DStringInit(&psFont);
    int psSize = Tk_PostscriptFontName(font, &psFont);
    str << '/' << Tcl_DStringValue(&psFont) 
	<< " findfont " << psSize << " scalefont setfont" << endl;
    Tcl_DStringFree(&psFont);

    Vector bbc = bbox.center();
    Vector t =  Vector(bbc[0], bbox.ll[1]);

    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);

    str << t.TkCanvasPs(parent->canvas) << " moveto" << endl
	<< '(' << psQuote(text) << ')' << endl
	<< "dup stringwidth pop 2 div 0 exch sub " 
	<< metrics.descent << " rmoveto show" << endl
	<< ends;

    Tcl_AppendResult(parent->interp, (char*)str.str().c_str(), NULL);
  }
}

void Marker::select() {
  // only call the CB if not already selected
  if (!selected)
    doCallBack(CallBack::SELECTCB);
  selected = 1;
}

void Marker::unselect() {
  // only call the CB if already selected
  if (selected)
    doCallBack(CallBack::UNSELECTCB);
  selected = 0;
}

void Marker::toggleSelect() {
  selected = !selected;
  if (selected)
    doCallBack(CallBack::SELECTCB);
  else
    doCallBack(CallBack::UNSELECTCB);
}

void Marker::highlite() {
  // only call the CB if not already highlited
  if (!highlited)
    doCallBack(CallBack::HIGHLITECB);
  highlited = 1;
}

void Marker::unhighlite() {
  // only call the CB if already highlited
  if (highlited)
    doCallBack(CallBack::UNHIGHLITECB);
  highlited = 0;
}

void Marker::toggleHighlite() {
  highlited = !highlited;
  if (highlited)
    doCallBack(CallBack::HIGHLITECB);
  else
    doCallBack(CallBack::UNHIGHLITECB);
}

Vector Marker::getHandle(int h)
{
  if (h<numHandle)
    return handle[h];
  else
    return Vector();
}

void Marker::setAngle(double a)
{
  angle = a;
  updateBBox();

  doCallBack(CallBack::ROTATECB);
}

void Marker::setColor(const char* clr)
{
  if (colorName)
    delete [] colorName;

  colorName = dupstr(clr);
  color = parent->getColor(colorName);

  doCallBack(CallBack::COLORCB);
}

void Marker::setLineWidth(int w)
{
  lineWidth = w;

  doCallBack(CallBack::LINEWIDTHCB);
}

void Marker::setFont(const char* f)
{
  if (font)
    Tk_FreeFont(font);

  if (f)
    font = Tk_GetFont(parent->interp, parent->tkwin, f);
  else
    font = Tk_GetFont(parent->interp, parent->tkwin, "helvetica");

  updateBBox();
  doCallBack(CallBack::FONTCB);
}

const char* Marker::getFont()
{
  if (font)
    return Tk_NameOfFont(font);
  else
    return NULL;
}

void Marker::addTag(const char* tg)
{
  Tag* t = new Tag(tg);
  tags.append(t);
}

void Marker::editTag(const char* from, const char* to)
{
  // change any tags
  {
    Tag* t = tags.head();
    while (t) {
      if (!strcmp(t->tag(),from)) {
	t->set(to);
      }
      t=t->next();
    }
  }

  // now, remove duplicates
  {
    Tag* t = tags.head();
    while (t) {
      Tag* tt=t->next();
      while (tt) {
	if (!strcmp(t->tag(),tt->tag())) {
	  Tag* ntt = tags.extractNext(tt);
	  delete tt;
	  tt = ntt;
	}
	else
	  tt=tt->next();
      }
      t=t->next();
    }
  }
}

void Marker::deleteTags()
{
  tags.deleteAll();
}

void Marker::deleteTag(const char* tg)
{
  Tag* t = tags.head();
  while (t) {
    if (!strcmp(t->tag(),tg)) {
      tags.extractNext(t);
      delete t;
      return;
    }
    t = t->next();
  }
}

void Marker::deleteTag(int w)
{
  Tag* t = tags.head();
  for (int i=0; i<w; i++)
    if (t)
      t = t->next();
    else
      break;

  if (t) {
    tags.extractNext(t);
    delete t;
  }
}

const char* Marker::getTag()
{
  Tag* t = tags.head();
  if (t)
    return t->tag();
  else
    return NULL;
}

const char* Marker::getNextTag()
{
  Tag* t = tags.next();
  if (t)
    return t->tag();
  else
    return NULL;
}

const char* Marker::getTag(int w)
{
  Tag* t = tags.head();
  for (int i=0; i<w; i++)
    if (t)
      t = t->next();
    else
      break;

  if (t)
    return t->tag();
  else
    return NULL;
}

int Marker::hasTag(const char* tg)
{
  Tag* t = tags.head();
  while (t) {
    if (!strcmp(t->tag(),tg))
      return 1;
    t = t->next();
  }
  return 0;
}

int Marker::onHandle(const Vector& v)
{
  // return handle number
  // work last to first for annuli

  for (int i=numHandle-1; i>=0; i--) {
    BBox bb(handle[i]);
    bb.expand(2);
    if (bb.isIn(v))
      return i+1;
  }
  return 0;
}

int Marker::getProperty(unsigned short which)
{
  return (properties & which) ? 1 : 0;
}

void Marker::setText(const char* str)
{
  if (text)
    delete [] text;
  text = dupstr(str);

  updateBBox();
  doCallBack(CallBack::TEXTCB);
}

void Marker::setProperty(unsigned short prop, int value)
{
  if (value)
    properties |= prop;
  else
    properties &= ~prop;

  if (prop == FIXED) // bbox will change
    updateBBox();

  doCallBack(CallBack::PROPERTYCB);
}

int Marker::addCallBack(CallBack::Type t, const char* proc, const char* arg)
{
  CallBack* cb = new CallBack(parent->interp, t, proc, arg);
  if (cb) {
    callbacks.append(cb);
    return TCL_OK;
  }

  return TCL_ERROR;
}

void Marker::deleteCallBack(CallBack::Type t)
{
  CallBack* cb = callbacks.head();
  while (cb) {
    if (cb->type() == t) {
      CallBack* next = callbacks.extractNext(cb);
      delete cb;
      cb = next;
    }
    else
      cb = cb->next();
  }
}

int Marker::deleteCallBack(CallBack::Type t, const char* proc)
{
  CallBack* cb = callbacks.head();
  while (cb) {
    if (cb->type() == t && (!strcmp(cb->proc(), proc))) {
      callbacks.extractNext(cb);
      delete cb;
      return TCL_OK;
    }
    else
      cb = cb->next();
  }

  return TCL_ERROR;
}

int Marker::isVisible(const BBox& b)
{
  // assume visible, prove otherwise
  // all coords are in canvas coords

  BBox bb(b);
  return 
    !((allBBox.ur[0] < bb.ll[0]) ||
    (allBBox.ll[0] > bb.ur[0]) ||
    (allBBox.ur[1] < bb.ll[1]) ||
    (allBBox.ll[1] > bb.ur[1]));
}

void Marker::doCallBack(CallBack::Type t)
{
  if (!doCB)
    return;

  ostringstream str;

  str << id << ends;

  CallBack* cb=callbacks.head();
  while (cb) {
    if (cb->type() == t)
      if (cb->eval(str.str().c_str()))
	cerr << "Marker Internal Error: Unable to eval CallBack "
	     << cb->proc() << endl << "  :" << parent->interp->result << endl;
    cb=cb->next();
  }
}

// Protected

double Marker::calcAngle()
{
  switch (parent->getOrientation()) {
  case NONE:
  case XY:
    return angle + parent->getRotation();
    break;
  case XX:
  case YY:
    return -angle + parent->getRotation();
    break;
  }
}

double Marker::calcAngle(double a)
{
  switch (parent->getOrientation()) {
  case NONE:
  case XY:
    return a + parent->getRotation();
    break;
  case XX:
  case YY:
    return -a + parent->getRotation();
    break;
  }
}

void Marker::psColor(int mode, const char* cn)
{
  ostringstream str;

  switch ((FrameBase::PSColorSpace)mode) {
  case FrameBase::BW:
  case FrameBase::GRAY:
    psColorGray(cn, str);
    str << " setgray";
    break;
  case FrameBase::RGB:
    psColorRGB(cn, str);
    str << " setrgbcolor";
    break;
  case FrameBase::CMYK:
    psColorCMYK(cn, str);
    str << " setcmykcolor";
    break;
  }
  str << endl << ends;

    Tcl_AppendResult(parent->interp, (char*)str.str().c_str(), NULL);
}

void Marker::psLineNoDash()
{
  ostringstream str;
  str << lineWidth << " setlinewidth" << endl;
  str << "[] 0 setdash" << endl;
  str << ends;
  Tcl_AppendResult(parent->interp, (char*)str.str().c_str(), NULL);
}

void Marker::psLineDash()
{
  ostringstream str;
  str << lineWidth << " setlinewidth" << endl;
  str << "[8 3] 0 setdash" << endl;
  str << ends;
  Tcl_AppendResult(parent->interp, (char*)str.str().c_str(), NULL);
}

void Marker::calcAllBBox()
{
  allBBox = bbox;

  if (text && font) {
    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);
    int width = Tk_TextWidth(font, text, strlen(text));

    Vector bbc = bbox.center();
    Vector ll =  Vector(bbc[0], bbox.ll[1]) * Translate(-width/2., 0);
    Vector ur = ll * Translate(width, -(metrics.linespace));

    allBBox.bound(ll);
    allBBox.bound(ur);
  }
}

void Marker::renderHandles(Drawable drawable)
{
  XSetForeground(display, gc, color);
  setLineNoDash();

  // handles are of length 5

  if (selected && canSelect())
    for (int i=0; i<numHandle; i++) {
      Vector v = (handle[i]*parent->canvasToWidget - Vector(2,2)).round();
      XFillRectangle(display, drawable, gc, (int)v[0], (int)v[1], 5, 5);
    }
}

void Marker::renderText(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  switch (mode) {
  case SRC:
    XSetForeground(display, gc, color);
    XSetClipRectangles(display, gc, 0, 0, parent->rectWidget, 1, Unsorted);
    break;
  case XOR:
    XSetForeground(display, gc, parent->getWhiteColor());
    XSetClipRectangles(display, gc, 0, 0, parent->rectWindow, 1, Unsorted);
    break;
  }
  setLineNoDash();

  if (font)
    XSetFont(display, gc, Tk_FontId(font));

  if (text && font) {
    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);
    int width = Tk_TextWidth(font, text, strlen(text));

    BBox bb = bbox * parent->canvasToRef * mx;
    Vector bbc = bb.center();
    Vector t =  Vector(bbc[0], bb.ll[1]) * 
      Translate(-width/2., -metrics.descent);
    XDrawString(display, drawable, gc, (int)t[0], (int)t[1], 
		text, strlen(text));
  }
}

void Marker::setGC(RenderMode mode)
{
  switch (mode) {
  case SRC:
    XSetForeground(display, gc, color);
    XSetClipRectangles(display, gc, 0, 0, parent->rectWidget, 1, Unsorted);
    if (properties & SOURCE)
      setLineNoDash();
    else
      setLineDash();
    break;
  case XOR:
    XSetForeground(display, gc, parent->getWhiteColor());
    XSetClipRectangles(display, gc, 0, 0, parent->rectWindow, 1, Unsorted);
    setLineDash();
    break;
  }
}

void Marker::setLineDash()
{
  if (highlited && canHighlite())
    XSetLineAttributes(display,gc,lineWidth*2,LineOnOffDash,CapButt,JoinMiter);
  else
    XSetLineAttributes(display,gc,lineWidth,LineOnOffDash,CapButt,JoinMiter);

  char dlist[] = {8,3};
  XSetDashes(display, gc, 0, dlist, 2);
}

void Marker::setLineNoDash()
{
  if (highlited && canHighlite())
    XSetLineAttributes(display, gc, lineWidth*2,LineSolid, CapButt, JoinMiter);
  else
    XSetLineAttributes(display, gc, lineWidth, LineSolid, CapButt, JoinMiter);
}

Vector* Marker::arrow(const Vector& p, const Vector& dd)
{
  Vector d = dd;      // stupid, however, no const warnings

  const int tip = 6;  // length from end of line to tip of arrow
  const int tail = 2; // length from end of line to tails of arrow
  const int wc = 2;   // width of arrow at end of line
  const int wt = 3;   // width of arrow at tails

  Vector* a = new Vector[6];
  a[0] = Vector(0, tip);
  a[1] = Vector(-wc, 0);
  a[2] = Vector(-wt, -tail);
  a[3] = Vector(0, 0);
  a[4] = Vector(wt, -tail);
  a[5] = Vector(wc, 0);

  Matrix m = Translate(0,-tip) * 
    Scale(1.5) * 
    Rotate(-M_PI/2) * 
    Rotate(-d.angle()) * 
    Translate(p);

  for (int i=0; i<6; i++) {
    a[i] = (a[i] * m).round();
  }

  return a;
}

void Marker::renderArrow(Drawable drawable, const Vector& p, const Vector& d)
{
  Vector* a = arrow(p,d);
  XPoint aa[6];
  for (int i=0; i<6; i++) {
    aa[i].x = (short)a[i][0];
    aa[i].y = (short)a[i][1];
  }

  XFILLPOLYGON(display, drawable, gc, aa, 6, Nonconvex, CoordModeOrigin);
  delete [] a;
}

void Marker::psArrow(const Vector& p, const Vector& d)
{
  Vector* a = arrow(p,d);
  for (int i=0; i<6; i++) {
    ostringstream str;
    if (i==0)
      str << "newpath " << endl
	  << a[i].TkCanvasPs(parent->canvas) << " moveto" << endl << ends;
    else
      str << a[i].TkCanvasPs(parent->canvas) << " lineto" << endl << ends;

    Tcl_AppendResult(parent->interp, (char*)str.str().c_str(), NULL);
  }

  ostringstream str;
  str << "closepath fill" << endl << ends;
  Tcl_AppendResult(parent->interp, (char*)str.str().c_str(), NULL);

  delete [] a;
}

double Marker::mapLenFromRef(double d, CoordSystem sys, SkyFormat format)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
      return ptr->mapLenFromRef(d,sys);
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSEqu(sys))
	return ptr->mapLenFromRef(d,sys,format);
      else
	return ptr->mapLenFromRef(d,sys);
    }
  }
  return 0;
}

Matrix Marker::fwdRefMatrix()
{
  return Rotate(angle) * flip_ * Translate(center);
}

Matrix Marker::fwdCanvasMatrix()
{
  return Rotate(angle) * flip_ * Translate(center) * parent->refToCanvas;
}

Matrix Marker::bckRefMatrix()
{
  return parent->canvasToRef * Translate(-center) * flip_ * Rotate(-angle);
}

Matrix Marker::bckCanvasMatrix()
{
  return Translate(-center) * flip_ * Rotate(-angle);
}

// list

void Marker::listPre(ostream& str, CoordSystem sys, SkyFrame sky, 
		     FitsImage* ptr, int strip, int hash)
{
  // no props for semicolons
  if (!strip) {
    switch (sys) {
    case IMAGE:
    case PHYSICAL:
    case DETECTOR:
    case AMPLIFIER:
      if (parent->isMosaic())
	str << "# tile " << parent->findFits(ptr) << endl;

      break;
    default:
      if (!ptr->hasWCSEqu(sys))
	if (parent->isMosaic())
	  str << "# tile " << parent->findFits(ptr) << endl;
    }

    if (hash)
      str << "# ";
  }
	
  if (!(properties&INCLUDE))
    str << '-';
}

void Marker::listPost(ostream& str, int conj, int strip)
{
  // no props for semicolons
  if (!strip) {
    if (conj)
      str << " ||";

    listProperties(str, 1);
  }
  else {
    if (conj)
      str << "||";
    else
      str << ';';
  }
}

void Marker::listProperties(ostream& str, int hash)
{
  if (strcmp("green",colorName) ||
      strcmp("helvetica 10 normal",getFont()) ||
      text[0] ||
      !(properties&SELECT) ||
      !(properties&HIGHLITE) ||
      !(properties&EDIT) ||
      !(properties&MOVE) ||
      !(properties&ROTATE) ||
      !(properties&DELETE) ||
      (properties&FIXED) ||
      !(properties&SOURCE) ||
      (tags.count() > 0) ||
      (comment && *comment)) {

    if (hash)
      str << " #";
    listProps(str);
  }

  str << endl;
}

void Marker::listProps(ostream& str)
{
  if (strcmp("green",colorName))
    str << " color=" << colorName;

  if (lineWidth != 1)
    str << " width=" << lineWidth;

  if (strcmp("helvetica 10 normal",getFont()))
    str << " font=\"" << getFont() << '"';

  if (text && *text) // only list text if there is something to list
    str << " text={" << text << '}';

  if (!(properties&SELECT))
    str << " select=0";

  if (!(properties&HIGHLITE))
    str << " highlite=0";

  if (!(properties&EDIT))
    str << " edit=0";

  if (!(properties&MOVE))
    str << " move=0";

  if (!(properties&ROTATE))
    str << " rotate=0";

  if (!(properties&DELETE))
    str << " delete=0";

  if (properties&FIXED)
    str << " fixed=1";

  if (!(properties&SOURCE))
    str << " background";

  // tags
  Tag* t = tags.head();
  while (t) {
    str << " tag={" << t->tag() << '}';
    t = t->next();
  }

  if (comment && *comment) 
    str << ' ' << comment;
}

void Marker::listCiaoPre(ostream& str)
{
  if (!(properties&INCLUDE))
    str << '-';
}

void Marker::listProsPost(ostream& str, int strip)
{
  str << (strip ? ';' : '\n');
}

void Marker::listCiaoPost(ostream& str, int conj, int strip)
			  
{
  if (conj)
    str << " |";

  str << (strip ? ';' : '\n');
}

void Marker::listSAOtngPre(ostream& str, int strip)
{
  if (!strip && text[0])
    str << '#' << text << endl;

  if (properties&INCLUDE)
    str << '+';
  else
    str << '-';
}

void Marker::listSAOtngPost(ostream& str, int strip)
{
  if (!strip) {
    str << " # ";
    if (comment && *comment)
      str << comment;
    else if (!(properties&SOURCE))
      str << "background";
    else
      str << colorName;
  }

  str << (strip ? ';' : '\n');
}

void Marker::listSAOimagePre(ostream& str)
{
  if (!(properties&INCLUDE))
    str << '-';
}

void Marker::listSAOimagePost(ostream& str, int strip)
{
  str << (strip ? ';' : '\n');
}

void Marker::listXY(ostream& str, CoordSystem sys, SkyFrame sky,
		    SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    str << setprecision(8) << ptr->mapFromRef(center,sys);
    break;
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  str << setprecision(8) << ptr->mapFromRef(center,sys,sky);
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra >> dec;
	    str << ra << ' ' << dec;
	  }
	  break;
	}
      }
      else {
	str << setprecision(8) << ptr->mapFromRef(center,sys);
      }
    }
    break;
  }

  str << (strip ? ';' : '\n');
}

// special composite funtionallity

void Marker::setComposite(const Matrix& mx, double aa)
{
  center *= mx;
  angle += aa;
  updateBBox();
}

void Marker::setComposite(unsigned short p, const char* clr, int w, int h)
{
  properties = p;
  lineWidth = w;
  if (colorName)
    delete [] colorName;

  colorName = dupstr(clr);
  color = parent->getColor(colorName);
  highlited = h;
}

