// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#ifndef __grid_h__
#define __grid_h__

#include <tk.h>
#include "vector.h"
#include "coord.h"

class FrameBase;
class Grid;

struct FitsStore;

class Attribute {
 public:
  enum Style {SOLID,DASH};
  enum Font {
    DEFAULT=0,TIMES,HELVETICA,SYMBOL,COURIER,
    DEFAULTBOLD=10,TIMESBOLD,HELVETICABOLD,SYMBOLBOLD,COURIERBOLD,
    DEFAULTITALIC=20,TIMESITALIC,HELVETICAITALIC,SYMBOLITALIC,COURIERITALIC
  };
  enum Colour {BLACK,WHITE,RED,GREEN,BLUE,CYAN,MAGENTA,YELLOW};

 private:
  FrameBase* parent;

  Style style_;
  float width_;

  Font font_;
  int size_;
  Tk_Font tkfont_;

  Colour colour_;
  char* colorName_;
  int color_;

 private:
  void initFont();

 public:
  Attribute(FrameBase*);
  ~Attribute();

  void setStyle(double);
  void setWidth(double);
  void setSize(double);
  void setFont(double);
  void setColour(double);

  Style style() {return style_;}
  float width() {return width_;}
  int size() {return size_;}
  Font font() {return font_;}
  Colour colour() {return colour_;}

  const char* colorName() {return colorName_;}
  Tk_Font tkfont() {return tkfont_;}
  int color() {return color_;}
};

class Grid {
 public:
  enum RenderMode {X11, PS};
  enum RenderBound{WIDGETBB,IMAGEBB,MOSAICBB};
  enum GridType {ANALYSIS,PUBLICATION};

 private:
  FrameBase* parent;
  GridType type_;

  char* option_;
  RenderMode renderMode;
  int psMode;
  CoordSystem system_;
  SkyFrame sky_;
  SkyFormat skyFormat_;
  Attribute* line;
  Attribute* text;

 private:
  int doit(RenderMode, RenderBound);
  int matrixMap(void*, Matrix&, const char*);

  int x11Line(int n, const float* x, const float* y);
  int x11Text(const char*, float, float, const char*, float, float);

  int psLine(int n, const float* x, const float* y);
  int psText(const char*, float, float, const char*, float, float);

  void psColor(Attribute*);
  void psWidth(Attribute*);
  void psStyle(Attribute*);

  void drawString(const char*, Vector, const char*);
  void drawRotString(const char*, Vector, const char*, double);

  void* fits2ast();
  void* wcs2ast(void*);

  void putFitsCard(void*, const char*, const char*);
  void putFitsCard(void*, const char*, double);

  double calcTextAngle(double);

 public:
  Grid(FrameBase*, CoordSystem, SkyFrame, SkyFormat, GridType, const char*);
  virtual ~Grid();

  int render();
  int ps(int);

  CoordSystem system() {return system_;}
  SkyFrame sky() {return sky_;}
  SkyFormat skyFormat() {return skyFormat_;}
  GridType type() {return type_;}
  const char* option() {return option_;}

  int gFlush() {return 1;}
  int gLine(int n, const float* x, const float* y)
    {return (renderMode==X11) ? x11Line(n,x,y) : psLine(n,x,y);}
  int gQch(float*, float*);
  int gMark(int, const float*, const float*, int) {return 1;}
  int gText(const char* txt, float x, float y, 
	     const char* just, float upx, float upy)
    {return (renderMode==X11) ? 
       x11Text(txt,x,y,just,upx,upy) : psText(txt,x,y,just,upx,upy);}
  int gTxExt(const char*, float, float, const char*, 
	      float, float, float*, float*);
  int gAttr(int attr, double value, double* old, int prim);
  int gScales(float *alpha, float *beta);
  int gCap(int, int);
};

#endif
