// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "composite.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Composite::Composite(const Composite& a) : Marker(a) 
{
  members = a.members;
}

Composite::Composite(FrameBase* p, const Vector& ctr, 
		     double ang,
		     const char* clr, int wth, 
		     const char* fnt, const char* txt,
		     unsigned short prop, const char* cmt,
		     const List<Tag>& tg, const List<CallBack>& cb)
  : Marker(p, ctr, ang, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  strcpy(type, "composite");

  handle = new Vector[4];
  numHandle = 4;

  updateBBox();
}

void Composite::updateBBox()
{
  // Handles
  BBox bb(center * bckCanvasMatrix());
  Marker* mk=members.head();
  while (mk) {
    Marker* m = mk->dup();
    m->setComposite(fwdRefMatrix(), angle);
    for(int i=0; i<m->getNumHandle(); i++)
      bb.bound(m->getHandle(i) * bckRefMatrix());
    delete m;
    mk=mk->next();
  }
  bb.expand(3); // a little more room around the edges

  Matrix mm = fwdCanvasMatrix();
  handle[0] = bb.ll   * mm;
  handle[1] = bb.lr() * mm;
  handle[2] = bb.ur   * mm;
  handle[3] = bb.ul() * mm;

  // bbox
  bbox = BBox(center * parent->refToCanvas);
  for (int i=0; i<numHandle; i++)
    bbox.bound(handle[i]);
  bbox.expand(3); // give us room for control handles

  // calculate overall bbox
  calcAllBBox();
}

void Composite::updateCoords(const Matrix& mx)
{
  Marker* mk=members.head();
  while (mk) {
    Vector cc = center;
    mk->setComposite(fwdRefMatrix(), angle);
    mk->updateCoords(mx);
    center = cc*mx;
    mk->setComposite(bckCanvasMatrix(), -angle);
    center = cc;
    mk=mk->next();
  }

  Marker::updateCoords(mx);
}

void Composite::draw(Drawable drawable)
{
  renderHandles(drawable);
  renderText(drawable, parent->refToWidget, SRC);

  Marker* mk=members.head();
  while (mk) {
    Marker* m = mk->dup();
    m->setComposite(fwdRefMatrix(), angle);
    m->setComposite(properties | INCLUDE, colorName, lineWidth, highlited);
    m->draw(drawable);
    delete m;
    mk=mk->next();
  }

  render(drawable, parent->refToWidget, SRC);
}

void Composite::drawXOR()
{
  renderText(Tk_WindowId(parent->tkwin), parent->refToWindow, XOR);

  Marker* mk=members.head();
  while (mk) {
    Marker* m = mk->dup();
    m->setComposite(fwdRefMatrix(), angle);
    m->setComposite(properties | INCLUDE, colorName, lineWidth, highlited);
    m->drawXOR();
    delete m;
    mk=mk->next();
  }

  render(Tk_WindowId(parent->tkwin), parent->refToWindow, XOR);
}

void Composite::drawMagnifier(const Matrix& mx)
{
  renderText(parent->magnifierPixmap, mx, SRC);

  Marker* mk=members.head();
  while (mk) {
    Marker* m = mk->dup();
    m->setComposite(fwdRefMatrix(), angle);
    m->setComposite(properties | INCLUDE, colorName, lineWidth, highlited);
    m->drawMagnifier(mx);
    delete m;
    mk=mk->next();
  }

  render(parent->magnifierPixmap, mx, SRC);
}

void Composite::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  // INCLUDE/EXCLUDE
  setGC(mode);
  if (!(properties & INCLUDE)) {
    Vector ll = (handle[0] * parent->getCanvasToRef() * mx).round();
    Vector ur = (handle[2] * parent->getCanvasToRef() * mx).round();

    if (mode==SRC)
      XSetForeground(display, gc, parent->getRedColor());

    XDRAWLINE(display, drawable, gc, (int)ll[0], (int)ll[1], 
	      (int)ur[0], (int)ur[1]);    
  }
}

void Composite::ps(int mode)
{
  Marker* mk=members.head();
  while (mk) {
    Marker* m = mk->dup();
    m->setComposite(fwdRefMatrix(), angle);
    m->setComposite(properties | INCLUDE, colorName, lineWidth, highlited);
    m->ps(mode);
    delete m;

    mk=mk->next();
  }

  // INCLUDE/EXCLUDE
  Marker::ps(mode);
  if (!(properties & INCLUDE)) {
    psColor(mode, "red");
    ostringstream str;
    str << "newpath " 
	<< handle[0].TkCanvasPs(parent->canvas) << "moveto"
	<< handle[2].TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }
}

int Composite::isIn(const Vector& v)
{
  if (!bbox.isIn(v))
    return 0;

  Marker* mk=members.head();
  while (mk) {
    Marker* m = mk->dup();
    m->setComposite(fwdRefMatrix(), angle);
    if (m->isIn(v)) {
      delete m;
      return 1;
    }
    delete m;

    mk=mk->next();
  }

  return 0;
}

void Composite::append(Marker* m)
{
  m->setComposite(bckCanvasMatrix(), -angle);

  members.append(m);
  updateBBox();
}

Marker* Composite::extract()
{
  Marker* mk=members.head();
  if (mk) {
    members.extractNext(mk);
    mk->setComposite(fwdRefMatrix(), angle);
    updateBBox();
  }
  return mk;
}

// list

void Composite::list(ostream& str, CoordSystem sys, SkyFrame sky, 
		 SkyFormat format, int conj, int strip)
{
  if (!strip) {
    FitsImage* ptr = parent->findFits(center);
    listPre(str, sys, sky, ptr, strip, 1);

    switch (sys) {
    case IMAGE:
    case PHYSICAL:
    case DETECTOR:
    case AMPLIFIER:
      {
	Vector v = ptr->mapFromRef(center,sys);
	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	    << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
      }
      break;
    default:
      if (ptr->hasWCS(sys)) {
	if (ptr->hasWCSEqu(sys)) {
	  switch (format) {
	  case DEGREES:
	    {
	      Vector v = ptr->mapFromRef(center,sys,sky);
	      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] 
		  << ','
		  << radToDeg(parent->mapAngleFromRef(angle,sys,sky)) << ')';
	    }
	    break;
	  case SEXAGESIMAL:
	    {
	      char buf[64];
	      ptr->mapFromRef(center,sys,sky,format,buf,64);
	      char ra[16];
	      char dec[16];
	      string x(buf);
	      istringstream wcs(x);
	      wcs >> ra >> dec;
	      str << type << '(' << ra << ',' << dec << ',' 
		  << radToDeg(parent->mapAngleFromRef(angle,sys,sky)) << ')';
	    }
	    break;
	  }
	}
	else {
	  Vector v = ptr->mapFromRef(center,sys);
	  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	      << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
	}
      }
    }
      
    listPost(str, 1, strip);
  }

  Marker* mk=members.head();
  while (mk) {
    Marker* m = mk->dup();
    mk=mk->next();

    m->setComposite(fwdRefMatrix(), angle);
    m->list(str, sys, sky, format, (mk?1:0), strip);
    delete m;
  }
}

void Composite::listCiao(ostream& str, CoordSystem sys, int conj, int strip)
{
  Marker* mk=members.head();
  while (mk) {
    Marker* m = mk->dup();
    mk=mk->next();

    m->setComposite(fwdRefMatrix(), angle);
    m->listCiao(str, sys, (mk?1:0), strip);
    delete m;
  }
}

void Composite::listPros(ostream& str, CoordSystem sys, SkyFrame sky,
			 SkyFormat format, int strip)
{
  Marker* mk=members.head();
  while (mk) {
    Marker* m = mk->dup();
    m->setComposite(fwdRefMatrix(), angle);
    m->listPros(str, sys, sky, format, strip);
    delete m;

    mk=mk->next();
  }
}

void Composite::listSAOtng(ostream& str, CoordSystem sys, SkyFrame sky,
			   SkyFormat format, int strip)
{
  Marker* mk=members.head();
  while (mk) {
    Marker* m = mk->dup();
    m->setComposite(fwdRefMatrix(), angle);
    m->listSAOtng(str, sys, sky, format, strip);
    delete m;

    mk=mk->next();
  }
}

void Composite::listSAOimage(ostream& str, int strip)
{
  Marker* mk=members.head();
  while (mk) {
    Marker* m = mk->dup();
    m->setComposite(fwdRefMatrix(), angle);
    m->listSAOimage(str, strip);
    delete m;

    mk=mk->next();
  }
}

void Composite::listXY(ostream& str, CoordSystem sys, SkyFrame sky,
		       SkyFormat format, int strip)
{
  Marker* mk=members.head();
  while (mk) {
    Marker* m = mk->dup();
    m->setComposite(fwdRefMatrix(), angle);
    m->listXY(str, sys, sky, format, strip);
    delete m;

    mk=mk->next();
  }
}

