// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "basebox.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

BaseBox::BaseBox(const BaseBox& a) : BaseMarker(a) {}

BaseBox::BaseBox(FrameBase* p, const Vector& ctr, 
		 double ang,
		 const char* clr, int w, const char* f, 
		 const char* t, unsigned short prop, const char* c,
		 const List<Tag>& tag, const List<CallBack>& cb)
  : BaseMarker(p, ctr, ang, 5, clr, w, f, t, prop, c, tag, cb) {}

int BaseBox::isIn(const Vector& v)
{
  // v is in canvas coords
  Vector s = annuli_[numAnnuli_-1];

  // zero radius
  if (!s[0] || !s[1])
    return 0;

  Matrix mm = bckRefMatrix();
  Vector p = v * mm;

  if (p[0]<-s[0]/2 || p[0]>s[0]/2 || p[1]<-s[1]/2 || p[1]>s[1]/2)
    return 0;
  else
    return 1;
}

// protected

void BaseBox::updateHandles()
{
  // handles are in canvas coords
  // we can't garantee that the radii have been sorted yet
  // also, annuli[i] could be negative

  if (handle)
    delete [] handle;
  handle = new Vector[numHandle];

  Vector max;
  for(int i=0; i<numAnnuli_; i++) {
    Vector foo = annuli_[i].abs();
    if (max[0]<foo[0])
      max = foo;
  }
  Vector s = max;

  Matrix mm = fwdCanvasMatrix();

  handle[0] = Vector(-s[0],-s[1])/2 * mm;
  handle[1] = Vector( s[0],-s[1])/2 * mm;
  handle[2] = Vector( s[0], s[1])/2 * mm;
  handle[3] = Vector(-s[0], s[1])/2 * mm;

  // the rest are radii
  for (int i=4; i<numHandle; i++)
    handle[i] = Vector((annuli_[i-4][0])/2.,0) * mm;
}

void BaseBox::newVertices()
{
  if (vertices_)
    deleteVertices();

  if (startAng_ == 0 && stopAng_ == 360)
    newVerticesA();
  else
    newVerticesB();
}

void BaseBox::newVerticesA()
{
  Matrix mm = fwdRefMatrix();

  numPoints_ = 5;
  vertices_ = new Vector*[numAnnuli_+1];
  for (int i=0; i<numAnnuli_; i++)
    vertices_[i]= new Vector[numPoints_];
  vertices_[numAnnuli_] = new Vector[2];

  for (int i=0; i<numAnnuli_; i++) {
    vertices_[i][0] = Vector(-annuli_[i][0]/2.,-annuli_[i][1]/2.) * mm;
    vertices_[i][1] = Vector( annuli_[i][0]/2.,-annuli_[i][1]/2.) * mm;
    vertices_[i][2] = Vector( annuli_[i][0]/2., annuli_[i][1]/2.) * mm;
    vertices_[i][3] = Vector(-annuli_[i][0]/2., annuli_[i][1]/2.) * mm;
    vertices_[i][4] = Vector(-annuli_[i][0]/2.,-annuli_[i][1]/2.) * mm;
  }

  vertices_[numAnnuli_][0] = vertices_[numAnnuli_-1][0];
  vertices_[numAnnuli_][1] = vertices_[numAnnuli_-1][2];
}

void BaseBox::newVerticesB()
{
  numPoints_ = 0; // we will increment later
  vertices_ = new Vector*[numAnnuli_+1];
  for (int ii=0; ii<numAnnuli_; ii++)
    vertices_[ii]= new Vector[7];
  vertices_[numAnnuli_] = new Vector[2];

  double a1 = degToRad(startAng_);
  double a2 = degToRad(stopAng_);

  if (a1>2*M_PI-FLT_EPSILON)
    a1 -= 2*M_PI;
  if (a2>2*M_PI-FLT_EPSILON)
    a2 -= 2*M_PI;
  if (a2<=a1+FLT_EPSILON)
    a2 += 2*M_PI;
  
  int cnt = 0;
  for (int ii=0; ii<numAnnuli_; ii++) {
    Vector rr = (annuli_[ii]).abs();
    if (!(rr[0] == 0 || rr[1] == 0)) {
      double phi = rr.angle();
      int s1 =0;
      int s2 =0;
      cnt = 0;

      vertBTest(&s1, &s2, a1, a2, 0, phi, 0, ii, &cnt);
      vertBTest(&s1, &s2, a1, a2, phi, M_PI-phi, 1, ii, &cnt);
      vertBTest(&s1, &s2, a1, a2, M_PI-phi, M_PI+phi, 2, ii, &cnt);
      vertBTest(&s1, &s2, a1, a2, M_PI+phi, 2*M_PI-phi, 3, ii, &cnt);
      vertBTest(&s1, &s2, a1, a2, 2*M_PI-phi,2*M_PI , 4, ii, &cnt);

      vertBTest(&s1, &s2, a1, a2, 2*M_PI,2*M_PI+phi , 5, ii, &cnt);
      vertBTest(&s1, &s2, a1, a2, 2*M_PI+phi,3*M_PI-phi , 6, ii, &cnt);
      vertBTest(&s1, &s2, a1, a2, 3*M_PI-phi,3*M_PI+phi , 7, ii, &cnt);
      vertBTest(&s1, &s2, a1, a2, 3*M_PI+phi,4*M_PI-phi , 8, ii, &cnt);
      vertBTest(&s1, &s2, a1, a2, 4*M_PI-phi,4*M_PI, 9, ii, &cnt);
    }
  }

  // final number of segments
  numPoints_ = ++cnt;

  // include/exclude
  Matrix mm = fwdRefMatrix();
  vertices_[numAnnuli_][0] = 
    Vector(-annuli_[numAnnuli_-1][0]/2.,-annuli_[numAnnuli_-1][1]/2.) * mm;
  vertices_[numAnnuli_][1] = 
    Vector( annuli_[numAnnuli_-1][0]/2., annuli_[numAnnuli_-1][1]/2.) * mm;
}

void BaseBox::vertBTest(int* s1, int* s2, double a1, double a2, 
			double b1, double b2, int quad, int ii, int* cnt)
{
  if ((!(*s1)) && (a1>=b1-FLT_EPSILON) && (a1<b2+FLT_EPSILON))
    *s1 =1;
  if ((!(*s2)) && (a2>b1-FLT_EPSILON) && (a2<=b2+FLT_EPSILON))
    *s2 =1;

  if (((*s1) && (!(*s2))) || ((*s1) && (*s2)))
    vertBPrep(a1,a2,b1,b2,quad,ii,cnt);

  if (*s1&&*s2)
    *s1=*s2=0;
}

void BaseBox::vertBPrep(double a1, double a2, double ll, double ul, 
			int quad, int ii, int* cnt)
{
  if (!(a1 >= ll && a1 <= ul))
    a1 = ll;
  if (!(a2 >= ll && a2 <= ul))
    a2 = ul;

  if (a1>a2) {
    vertBSeg(ll,a2,quad,ii,cnt);
    vertBSeg(a1,ul,quad,ii,cnt);
  }
  else
    vertBSeg(a1,a2,quad,ii,cnt);
}

void BaseBox::vertBSeg(double ang1, double ang2, int quad, int ii, int* cnt)
{
  Vector rr = (annuli_[ii]/2).abs();
  double phi = rr.angle();
  Matrix mm = fwdRefMatrix();

  if (ang1>2*M_PI-FLT_EPSILON)
    ang1 -= 2*M_PI;
  if (ang2>2*M_PI-FLT_EPSILON)
    ang2 -= 2*M_PI;

  Vector r1,r2;
  switch (quad) {
  case 0:
  case 5:
    {
      double a1 = 0;
      double a2 = phi;
      double b1 = rr[1]*(ang1-a1)/(a2-a1);
      double b2 = rr[1]*(ang2-a1)/(a2-a1);
      r1 = Vector(rr[0],-b1);
      r2 = Vector(rr[0],-b2);
    }
    break;
  case 1:
  case 6:
    {
      double a1 = phi;
      double a2 = M_PI-phi;
      double b1 = rr[0] - (2*rr[0]*(ang1-a1)/(a2-a1));
      double b2 = rr[0] - (2*rr[0]*(ang2-a1)/(a2-a1));
      r1 = Vector(b1,-rr[1]);
      r2 = Vector(b2,-rr[1]);
    }
    break;
  case 2:
  case 7:
    {
      double a1 = M_PI-phi;
      double a2 = M_PI+phi;
      double b1 = rr[1] - (2*rr[1]*(ang1-a1)/(a2-a1));
      double b2 = rr[1] - (2*rr[1]*(ang2-a1)/(a2-a1));
      r1 = Vector(-rr[0],-b1);
      r2 = Vector(-rr[0],-b2);
    }
    break;
  case 3:
  case 8:
    {
      double a1 = M_PI+phi;
      double a2 = 2*M_PI-phi;
      double b1 = rr[0] - (2*rr[0]*(ang1-a1)/(a2-a1));
      double b2 = rr[0] - (2*rr[0]*(ang2-a1)/(a2-a1));
      r1 = Vector(-b1, rr[1]);
      r2 = Vector(-b2, rr[1]);
    }
    break;
  case 4:
  case 9:
    {
      double a1 = 2*M_PI-phi;
      double a2 = 2*M_PI;
      double b1 = rr[1] - (rr[1]*(ang1-a1)/(a2-a1));
      double b2 = rr[1] - (rr[1]*(ang2-a1)/(a2-a1));
      r1 = Vector(rr[0],b1);
      r2 = Vector(rr[0],b2);
    }
    break;
  }

  // set vertices
  vertices_[ii][(*cnt)++] = r1 * mm;
  vertices_[ii][(*cnt)] = r2 * mm;
}
