// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <iostream>
#include <sstream>
#include <iomanip>
using namespace std;

#include "hdu.h"
#include "head.h"
#include "util.h"

FitsHDU::FitsHDU(FitsHead* head)
{
  extname_ = head->getString("EXTNAME");
  extver_ = head->getInteger("EXTVER", 0);

  bitpix_ = head->getInteger("BITPIX", 0);
  naxes_ = head->getInteger("NAXIS", 0);
  if (naxes_>FTY_MAXAXES)
    naxes_ = FTY_MAXAXES;

  for(int i=0; i<naxes_; i++)
    naxis_[i] = head->getInteger(keycat("NAXIS",(i+1)), 0);

  pixnum_ = naxes_>=2 ? naxis_[0] * naxis_[1] : 0;
  pixbytes_ = 0;
  realbytes_ = 0;
  databytes_ = 0;
  datablocks_ = 0;
}

FitsHDU::~FitsHDU()
{
  if (extname_)
    delete [] extname_;
}

char* FitsHDU::keycat(const char* name, int i)
{
  ostringstream str;
  str << name << i << ends;
  memcpy(keybuf,str.str().c_str(),str.str().length());
  return keybuf;
}

void FitsHDU::updateCards(FitsHead* head)
{
  head->setInteger("BITPIX", bitpix_, NULL);
  head->setInteger("NAXIS", naxes_, NULL);

  for (int i=1; i<=naxes_; i++)
      head->setInteger(keycat("NAXIS", i), naxis_[i-1], NULL);
}

FitsImageHDU::FitsImageHDU(FitsHead* head) : FitsHDU(head)
{
  if (naxes_>0) {
    datapixels_ = 1;
    for (int i=0; i<naxes_; i++ )
      datapixels_ *= naxis_[i];
  }
  else
    datapixels_ = 0;

  pixbytes_ = pixnum_ * (abs(bitpix_)/8);
  realbytes_ = datapixels_ * (abs(bitpix_)/8);
  datablocks_ = (realbytes_ + (FTY_BLOCK-1))/FTY_BLOCK;
  databytes_ = datablocks_ * FTY_BLOCK;

  bzero_ = head->getReal("BZERO", 0.0);
  bscale_ = head->getReal("BSCALE", 1.0);
  hasblank_ = head->find("BLANK") ? 1:0;
  blank_ = head->getInteger("BLANK", 0);
}

void FitsImageHDU::updateCards(FitsHead* head)
{
  FitsHDU::updateCards(head);
  if (blank_)
    if (bitpix_ > 0)
      head->setInteger("BLANK", blank_, NULL);

  if (bzero_)
    head->setReal("BZERO", bzero_ , 7, NULL);
  if (bscale_ != 1)
    head->setReal("BSCALE", bscale_, 7, NULL);
}

FitsTableHDU::FitsTableHDU(FitsHead* head) : FitsHDU(head)
{
  tfields_ = head->getInteger("TFIELDS", 0);
  cols_ = NULL;

  realbytes_ = naxis_[0] * naxis_[1]; // number of rows * width of row in bytes
  datablocks_ = (realbytes_+(FTY_BLOCK-1))/FTY_BLOCK;
  databytes_ = datablocks_*FTY_BLOCK;
}

FitsTableHDU::~FitsTableHDU()
{
  if (cols_) {
    for (int i=0; i<tfields_; i++)
      if (cols_[i])
	delete cols_[i];

    delete [] cols_;
  }
}

char* FitsTableHDU::list()
{
  char buf[1024]; // make it big
  buf[0] = '\0';

  ostringstream str;
  for (int i=0; i<tfields_; i++)
    if (cols_[i])
      str << cols_[i]->ttype() << ' ';
  str << ends;
  return dupstr(str.str().c_str());
}

FitsColumn* FitsTableHDU::find(const char* name)
{
  for (int i=0; i<tfields_; i++) {
    if (cols_[i]) {
      char* n = toUpper(name);
      char* t = toUpper(cols_[i]->ttype());

      if (!strcmp(n,t)) {
	delete [] n;
	delete [] t;
	return cols_[i];
      }
      else {
	delete [] n;
	delete [] t;
      }
    }
  }
  return NULL;
}

Vector FitsTableHDU::dimension(const char* name)
{
  FitsColumn* col = find(name);
  return col ? col->dimension() : Vector();
}

FitsAsciiTableHDU::FitsAsciiTableHDU(FitsHead* head) : FitsTableHDU(head)
{
  cols_ = new FitsColumn*[tfields_];

  int offset = 0;
  for (int i=0; i<tfields_; i++) {
    cols_[i] = new FitsAsciiColumn(head, i+1, offset);
    offset += cols_[i]->width();
  }
}

FitsBinTableHDU::FitsBinTableHDU(FitsHead* head) : FitsTableHDU(head)
{
  cols_ = new FitsColumn*[tfields_];

  int offset = 0;
  for (int i=0; i<tfields_; i++) {
    char* tform = head->getString(keycat("TFORM",i+1));
    int repeat;
    char type;
    if (tform) {
      string x(tform);
      istringstream str(x);
      if (isalpha(tform[0]))
	str >> type;
      else
	str >> repeat >> type;
    }

    switch (type) {
    case 'A':
      cols_[i] = new FitsBinColumnA(head, i+1, offset);
      break;
    case 'X':
      cols_[i] = new FitsBinColumnBit(head, i+1, offset);
      break;
    case 'L':
    case 'B':
      cols_[i] = new FitsBinColumnT<unsigned char>(head, i+1, offset);
      break;
    case 'I':
      cols_[i] = new FitsBinColumnT<short>(head, i+1, offset);
      break;
    case 'U':
      cols_[i] = new FitsBinColumnT<unsigned short>(head, i+1, offset);
      break;
    case 'J':
      cols_[i] = new FitsBinColumnT<int>(head, i+1, offset);
      break;
    case 'V':
      cols_[i] = new FitsBinColumnT<unsigned int>(head, i+1, offset);
      break;
    case 'E':
      cols_[i] = new FitsBinColumnT<float>(head, i+1, offset);
      break;
    case 'D':
      cols_[i] = new FitsBinColumnT<double>(head, i+1, offset);
      break;
    default:
      cols_[i] = NULL;
      cerr << "HDU Internal Error: Unsupported Bin Table Type: "
	   << tform << endl;
    }

    delete [] tform;
    if (cols_[i])
      offset += cols_[i]->width();
  }
}

