// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <limits.h>
#include <string.h>

#include "colorbarrgbtrue8.h"
#include "util.h"

// Tk Canvas Widget Function Declarations

int ColorbarRGBTrueColor8CreateProc(Tcl_Interp*, Tk_Canvas, Tk_Item*, int,
				    Tcl_Obj *const []);

// Colorbar Specs

static Tk_CustomOption tagsOption = {
  Tk_CanvasTagsParseProc, Tk_CanvasTagsPrintProc, NULL
};

struct ColorbarRGBTrueColor8Options {
  Tk_Item item;              // required by tk
  int x, y;                  // Coordinates of positioning point on canvas
  int width;                 // widget width
  int height;                // widget height
  Tk_Anchor anchor;          // Where to anchor widget relative to x,y
  char* cmdName;             // Suggested Tcl command name

  Widget* widget;            // pointer to widget class

  int colors;
};

static Tk_ConfigSpec colorbarRGBTrueColor8Specs[] = {
  {TK_CONFIG_STRING, "-command", NULL, NULL, "colorbarrgb",
   Tk_Offset(ColorbarRGBTrueColor8Options, cmdName), 
   TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-x", NULL, NULL, "1",
   Tk_Offset(ColorbarRGBTrueColor8Options, x), 
   TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-y", NULL, NULL, "1",
   Tk_Offset(ColorbarRGBTrueColor8Options, y), 
   TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-width", NULL, NULL, "512",
   Tk_Offset(ColorbarRGBTrueColor8Options, width), 
   TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-height", NULL, NULL, "22",
   Tk_Offset(ColorbarRGBTrueColor8Options, height), 
   TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_ANCHOR, "-anchor", NULL, NULL, "nw",
   Tk_Offset(ColorbarRGBTrueColor8Options, anchor), 0, NULL},
  {TK_CONFIG_CUSTOM, "-tags", NULL, NULL, NULL,
   0, TK_CONFIG_NULL_OK, &tagsOption},

  {TK_CONFIG_INT, "-colors", NULL, NULL, "1024",
   Tk_Offset(ColorbarRGBTrueColor8Options, colors), 0, NULL},

  {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0, NULL},
};

// Tk Static Structure

static Tk_ItemType colorbarRGBTrueColor8Type = {
  "colorbarrgbtruecolor8",        // name
  sizeof(ColorbarRGBTrueColor8Options), // size
  ColorbarRGBTrueColor8CreateProc, // configProc
  colorbarRGBTrueColor8Specs,     // configSpecs
  WidgetConfigProc,             // configProc
  WidgetCoordProc,              // coordProc
  WidgetDeleteProc,             // deleteProc
  WidgetDisplayProc,            // displayProc
  0,                            // alwaysRedraw
  WidgetPointProc,              // pointProc
  WidgetAreaProc,               // areaProc
  WidgetPostscriptProc,         // postscriptProc
  WidgetScaleProc,              // scaleProc
  WidgetTranslateProc,          // translateProc
  (Tk_ItemIndexProc*)NULL,      // indexProc
  (Tk_ItemCursorProc*)NULL,     // icursorProc
  (Tk_ItemSelectionProc*)NULL,  // selectionProc
  (Tk_ItemInsertProc*)NULL,     // insertProc
  (Tk_ItemDCharsProc*)NULL,     // dCharsProc
  (Tk_ItemType*)NULL            // nextPtr
};

// Non-Member Functions

int ColorbarRGBTrueColor8_Init(Tcl_Interp* interp)
{
  Tk_CreateItemType(&colorbarRGBTrueColor8Type);
  return TCL_OK;
}

int ColorbarRGBTrueColor8CreateProc(Tcl_Interp* interp, Tk_Canvas canvas, 
				    Tk_Item* item, int argc, 
				    Tcl_Obj *const argv[])
{
  ColorbarRGBTrueColor8* colorbar = 
    new ColorbarRGBTrueColor8(interp, canvas, item);

  // and set default configuration

  if (colorbar->configure(argc, (const char**)argv, 0) != TCL_OK) {
    delete colorbar;
    Tcl_AppendResult(interp, " error occured while creating colorbar.", NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}

// ColorbarRGBTrueColor8

ColorbarRGBTrueColor8::ColorbarRGBTrueColor8(Tcl_Interp* i, Tk_Canvas c, 
					     Tk_Item* item)
  : ColorbarRGBTrueColor(i, c, item)
{
  configSpecs = colorbarRGBTrueColor8Specs;  // colorbar configure options
}

int ColorbarRGBTrueColor8::initColormap()
{
  colorCount = (((ColorbarRGBTrueColor8Options*)options)->colors);
  return initColormapForReal();
}

void ColorbarRGBTrueColor8::updateColors()
{
  updateColorCells();

  // fill in xmap
  // make sure we have a pixmap

  if (!pixmap || !xmap)
    return;

  int& width = options->width;
  int& height = options->height;
  char* data = xmap->data;
    
  // Fill in colorbar data
  // check for bits/pixel

  unsigned char row[xmap->bytes_per_line];

  int rs,gs,bs;
  unsigned char rm = decodeMask((unsigned char)visual->red_mask, &rs);
  unsigned char gm = decodeMask((unsigned char)visual->green_mask, &gs);
  unsigned char bm = decodeMask((unsigned char)visual->blue_mask, &bs);

   // red
  for (int i=0; i<width; i++) {
    int j = ((int)(double(i)/width*colorCount))*3;
    row[i] = ((colorCells[j] & rm) >> rs);
  }
  for (int j=0; j<(int)(height/3.); j++)
    memcpy(data+(j*xmap->bytes_per_line), row, xmap->bytes_per_line);

  // green
  for (int i=0; i<width; i++) {
    int j = ((int)(double(i)/width*colorCount))*3;
    row[i] = ((colorCells[j+1] & gm) >> gs);
  }
  for (int j=(int)(height/3.); j<(int)(height*2/3.); j++)
    memcpy(data+(j*xmap->bytes_per_line), row, xmap->bytes_per_line);

  // blue
  for (int i=0; i<width; i++) {
    int j = ((int)(double(i)/width*colorCount))*3;
    row[i] = ((colorCells[j+2]   & bm) >> bs);
  }
  for (int j=(int)(height*2/3.); j<height; j++)
    memcpy(data+(j*xmap->bytes_per_line), row, xmap->bytes_per_line);

  // update the pixmap from map image and clean up

  XPutImage(display, pixmap, gc, xmap, 0, 0, 0, 0, width, height);

  // border
  renderBorder();

  redraw();
}

