#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc Bin {bx by} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin factor $bx $by"
	set bin(factor) "[$current(frame) get bin factor]"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc BinAbout {x y} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin about $x $y"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc BinCols {x y z} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) \
	    "$current(frame) bin cols \{$x\} \{$y\} \{$z\}"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc BinFilter {str} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin filter \{\{$str\}\}"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc BinToFit {} {
    global current
    global bin
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin to fit"
	set bin(factor) "[$current(frame) get bin factor]"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc ChangeBinFactor {} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin factor to $bin(factor)"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc ChangeBinDepth {} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin depth $bin(depth)"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc ChangeBinFunction {} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin function $bin(function)"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc ChangeBinBufferSize {} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin buffer size $bin(buffersize)"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc BinDialog {} {
    global bin
    global bindlg
    global ds9
    global menu

    # see if we already have a window visible

    if [winfo exist $bin(top)] {
	raise $bin(top)
	return
    }

    set w $bin(top)
    set title "Binning Parameters"

    # create the window

    toplevel $w -colormap $ds9(main)
    wm title $w $title
    wm iconname $w $title
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW BinDestroyDialog

    $w configure -menu $bin(mb)

    menu $bin(mb) -tearoff 0
    $bin(mb) add cascade -label File -menu $bin(mb).file
    $bin(mb) add cascade -label Edit -menu $bin(mb).edit
    $bin(mb) add cascade -label Method -menu $bin(mb).method
    $bin(mb) add cascade -label Block -menu $bin(mb).block
    $bin(mb) add cascade -label Buffer -menu $bin(mb).buffer

    menu $bin(mb).file -tearoff 0 -selectcolor $menu(selectcolor)
    $bin(mb).file add command -label "Apply" -command BinApplyDialog
    $bin(mb).file add command -label "Clear Filter" \
	-command BinClearFilterDialog
    $bin(mb).file add separator
    $bin(mb).file add command -label "Close" -command BinDestroyDialog

    menu $bin(mb).edit -tearoff 0 -selectcolor $menu(selectcolor)
    $bin(mb).edit add command -label "Cut" -command BinCutDialog
    $bin(mb).edit add command -label "Copy" -command BinCopyDialog
    $bin(mb).edit add command -label "Paste" -command BinPasteDialog
    $bin(mb).edit add command -label "Clear" -command BinClearDialog

    menu $bin(mb).method -tearoff 0 -selectcolor $menu(selectcolor)
    $bin(mb).method add radiobutton -label "Average" \
	-variable bin(function) -value average -command ChangeBinFunction
    $bin(mb).method add radiobutton -label "Sum" \
	-variable bin(function) -value sum -command ChangeBinFunction

    menu $bin(mb).block -tearoff 0 -selectcolor $menu(selectcolor)
    $bin(mb).block add command -label "Block In" -command "Bin .5 .5"
    $bin(mb).block add command -label "Block Out" -command "Bin 2 2"
    $bin(mb).block add separator
    $bin(mb).block add command -label "Block to Fit Frame" \
	-command "BinToFit"
    $bin(mb).block add separator
    $bin(mb).block add radiobutton -label "Block 1" \
	-variable bin(factor) -value { 1 1 } -command ChangeBinFactor
    $bin(mb).block add radiobutton -label "Block 2" \
	-variable bin(factor) -value { 2 2 } -command ChangeBinFactor
    $bin(mb).block add radiobutton -label "Block 4" \
	-variable bin(factor) -value { 4 4 } -command ChangeBinFactor
    $bin(mb).block add radiobutton -label "Block 8" \
	-variable bin(factor) -value { 8 8 } -command ChangeBinFactor
    $bin(mb).block add radiobutton -label "Block 16" \
	-variable bin(factor) -value { 16 16 } -command ChangeBinFactor
    $bin(mb).block add radiobutton -label "Block 32" \
	-variable bin(factor) -value { 32 32 } -command ChangeBinFactor
    $bin(mb).block add radiobutton -label "Block 64" \
	-variable bin(factor) -value { 64 64 } -command ChangeBinFactor
    $bin(mb).block add radiobutton -label "Block 128" \
	-variable bin(factor) -value { 128 128 } -command ChangeBinFactor

    menu $bin(mb).buffer -tearoff 0 -selectcolor $menu(selectcolor)
    $bin(mb).buffer add radiobutton -label "128x128" \
	-variable bin(buffersize) -value 128 -command ChangeBinBufferSize 
    $bin(mb).buffer add radiobutton -label "256x256" \
	-variable bin(buffersize) -value 256 -command ChangeBinBufferSize 
    $bin(mb).buffer add radiobutton -label "512x512" \
	-variable bin(buffersize) -value 512 -command ChangeBinBufferSize 
    $bin(mb).buffer add radiobutton -label "1024x1024" \
	-variable bin(buffersize) -value 1024 -command ChangeBinBufferSize
    $bin(mb).buffer add radiobutton -label "2048x2048" \
	-variable bin(buffersize) -value 2048 -command ChangeBinBufferSize
    $bin(mb).buffer add radiobutton -label "4096x4096" \
	-variable bin(buffersize) -value 4096 -command ChangeBinBufferSize

    frame $w.cols -relief groove -borderwidth 2
    frame $w.center -relief groove -borderwidth 2
    frame $w.filter -relief groove -borderwidth 2
    frame $w.z -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2

    pack $w.cols $w.center $w.filter $w.z -fill x
    pack $w.buttons -fill x -ipadx 4 -ipady 4

    frame $w.cols.f
    frame $w.center.f
    frame $w.filter.f
    frame $w.z.f
    pack $w.cols.f $w.center.f $w.filter.f $w.z.f -ipadx 4 -ipady 4 -anchor w

    # Columns

    label $w.cols.f.title -text "Bin Columns"
    label $w.cols.f.titlefactor -text "Block"
    label $w.cols.f.titlemin -text "Min"
    label $w.cols.f.titlemax -text "Max"
    menubutton $w.cols.f.x -textvariable bindlg(xcol) -menu $w.cols.f.x.m \
	-relief raised -bd 2 -width 10
    entry $w.cols.f.xfactor -textvariable bindlg(factor,x) -width 8
    label $w.cols.f.xmin -textvariable bindlg(xcol,min) -width 12 \
	-relief groove
    label $w.cols.f.xmax -textvariable bindlg(xcol,max) -width 12 \
	-relief groove
    menubutton $w.cols.f.y -textvariable bindlg(ycol) -menu $w.cols.f.y.m \
	-relief raised -bd 2 -width 10
    entry $w.cols.f.yfactor -textvariable bindlg(factor,y) -width 8 
    label $w.cols.f.ymin -textvariable bindlg(ycol,min) -width 12 \
	-relief groove
    label $w.cols.f.ymax -textvariable bindlg(ycol,max) -width 12 \
	-relief groove

    grid rowconfigure $w.cols.f 0 -pad 4
    grid rowconfigure $w.cols.f 1 -pad 4
    grid rowconfigure $w.cols.f 2 -pad 4

    grid $w.cols.f.title $w.cols.f.titlefactor $w.cols.f.titlemin \
	$w.cols.f.titlemax -padx 4 -sticky ew
    grid $w.cols.f.x $w.cols.f.xfactor $w.cols.f.xmin $w.cols.f.xmax \
	-padx 4 -sticky w
    grid $w.cols.f.y $w.cols.f.yfactor $w.cols.f.ymin $w.cols.f.ymax \
	-padx 4 -sticky w

    # Center

    label $w.center.f.title -text "Bin Center"
    entry $w.center.f.x -textvariable bindlg(x) -width 12 	 
    entry $w.center.f.y -textvariable bindlg(y) -width 12 
    checkbutton $w.center.f.auto -text "or center of data" \
	-selectcolor $menu(selectcolor) -variable bindlg(auto)

    grid rowconfigure $w.center.f 0 -pad 4
    grid rowconfigure $w.center.f 1 -pad 4

    grid $w.center.f.title $w.center.f.x $w.center.f.y -padx 4 -sticky w
    grid x $w.center.f.auto - -padx 4 -sticky w

    # Filter

    label $w.filter.f.title -text "Bin Filter"
    entry $w.filter.f.filter -textvariable bindlg(filter) -width 40

    grid rowconfigure $w.filter.f 0 -pad 4
    grid $w.filter.f.title $w.filter.f.filter -padx 4 -sticky w

    # Bin 3rd Column

    label $w.z.f.title -text "Bin 3rd Column"
    label $w.z.f.titledepth -text "Depth"
    label $w.z.f.titlemin -text "Min"
    label $w.z.f.titlemax -text "Max"

    menubutton $w.z.f.z -textvariable bindlg(zcol) \
	-menu $w.z.f.z.m -relief raised -bd 2 -width 10
    entry $w.z.f.depth -textvariable bindlg(depth) -width 8
    entry $w.z.f.min -textvariable bindlg(zcol,min) -width 12 
    entry $w.z.f.max -textvariable bindlg(zcol,max) -width 12 

    grid rowconfigure $w.z.f 0 -pad 4
    grid rowconfigure $w.z.f 1 -pad 4

    grid $w.z.f.title $w.z.f.titledepth $w.z.f.titlemin $w.z.f.titlemax \
	-padx 4 -sticky ew
    grid $w.z.f.z $w.z.f.depth $w.z.f.min $w.z.f.max -padx 4 -sticky w

    button $w.buttons.apply -text "Apply" -command BinApplyDialog
    button $w.buttons.clear -text "Clear Filter" -command BinClearFilterDialog
    button $w.buttons.close -text "Close" -command BinDestroyDialog
    pack $w.buttons.apply $w.buttons.clear $w.buttons.close \
	-side left -padx 10 -expand true

    $w.cols.f.xfactor select range 0 end

    set bindlg(auto) 0
    set bindlg(minmax) 1

    UpdateBinDialog
}

proc PopUp {b m l cmd} {
    destroy $m

    menu $m -tearoff 0
    for {set i 0} {$i<[llength $l]} {incr i} {
	$m add command -label [lindex $l $i] \
	    -command "global bindlg;set $b [lindex $l $i]; $cmd"
    }
}

proc BlankPopUp {m} {
    destroy $m
    menu $m -tearoff 0
}

proc UpdateBinDialog {} {
    global bin
    global bindlg
    global current

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateBinDialog"
    }

    if {![winfo exist $bin(top)]} {
	return
    }
    if {$current(frame) == ""} {
	return
    }

    set w $bin(top)

    if {[$current(frame) has bin]} {
	set bf "[$current(frame) get bin factor]"
	set bindlg(factor,x) [lindex $bf 0]
	set bindlg(factor,y) [lindex $bf 1]

	set cols [$current(frame) get bin cols]
	set colslist "[$current(frame) get bin list]"

	set bindlg(xcol) [lindex $cols 0]
	set bindlg(ycol) [lindex $cols 1]
	PopUp bindlg(xcol) $w.cols.f.x.m $colslist UpdateXCol
	PopUp bindlg(ycol) $w.cols.f.y.m $colslist UpdateYCol

	set mm [$current(frame) get bin cols minmax  \{$bindlg(xcol)\}]
	set bindlg(xcol,min) [lindex $mm 0]
	set bindlg(xcol,max) [lindex $mm 1]

	set mm [$current(frame) get bin cols minmax \{$bindlg(ycol)\}]
	set bindlg(ycol,min) [lindex $mm 0]
	set bindlg(ycol,max) [lindex $mm 1]

	set cursor [$current(frame) get bin cursor]
	set bindlg(x) [lindex $cursor 0]
	set bindlg(y) [lindex $cursor 1]

	set bindlg(filter) [$current(frame) get bin filter]

	set bindlg(depth) [$current(frame) get bin depth]
	set bindlg(zcol) [lindex $cols 2]
	PopUp bindlg(zcol) $w.z.f.z.m $colslist UpdateZCol
	set mm [$current(frame) get bin cols minmax \{$bindlg(zcol)\}]
	set bindlg(zcol,min) [lindex $mm 0]
	set bindlg(zcol,max) [lindex $mm 1]

    } else {
	set bindlg(factor,x) {}
	set bindlg(factor,y) {}
	set bindlg(xcol) {}
	set bindlg(xcol,min) {}
	set bindlg(xcol,max) {}
	set bindlg(ycol) {}
	set bindlg(ycol,min) {}
	set bindlg(ycol,max) {}

	set bindlg(x) {}
	set bindlg(y) {}

	set bindlg(filter) {}

	set bindlg(depth) {}
	set bindlg(zcol) {}
	set bindlg(zcol,min) {}
	set bindlg(zcol,max) {}

	BlankPopUp $w.cols.f.x.m
	BlankPopUp $w.cols.f.y.m
	BlankPopUp $w.z.f.z.m
    }
}

proc UpdateXCol {} {
    global current
    global bindlg

    if {$current(frame) != ""
	&& [$current(frame) has bin]
	&& $bindlg(xcol) != ""} {

	set mm [$current(frame) get bin cols minmax \{$bindlg(xcol)\}]
	set bindlg(xcol,min) [lindex $mm 0]
	set bindlg(xcol,max) [lindex $mm 1]
    } else {
	    set bindlg(xcol,min) {}
	    set bindlg(xcol,max) {}
    }
}

proc UpdateYCol {} {
    global current
    global bindlg

    if {$current(frame) != ""
	&& [$current(frame) has bin]
	&& $bindlg(ycol) != ""} {

	set mm [$current(frame) get bin cols minmax \{$bindlg(ycol)\}]
	set bindlg(ycol,min) [lindex $mm 0]
	set bindlg(ycol,max) [lindex $mm 1]
    } else {
	    set bindlg(ycol,min) {}
	    set bindlg(ycol,max) {}
    }
}

proc UpdateZCol {} {
    global current
    global bindlg

    if {$current(frame) != ""
	&& [$current(frame) has bin]
	&& $bindlg(zcol) != ""} {

	if {$bindlg(minmax)} {
	    set mm [$current(frame) get bin cols minmax \{$bindlg(zcol)\}]
	    set bindlg(zcol,min) [lindex $mm 0]
	    set bindlg(zcol,max) [lindex $mm 1]
	}
    } else {
	    set bindlg(zcol,min) {}
	    set bindlg(zcol,max) {}
    }
}

proc BinApplyDialog {} {
    global current
    global bin
    global bindlg

    if {$current(frame) == {}} {
	# reset
	set bindlg(auto) 0
	return
    }

    # clean up filter if needed
    set bindlg(filter) [string trimleft $bindlg(filter)]
    set bindlg(filter) [string trimright $bindlg(filter)]

    # delete any markers if needed
    if {[$current(frame) has bin]} {
	set foo [$current(frame) get bin cols]
	set xcol [lindex $foo 0]
	set ycol [lindex $foo 1]

	if {$xcol != $bindlg(xcol) || $ycol != $bindlg(ycol)} {
	    $current(frame) marker delete all
	}
    }

    SetWatchCursor

    if {$bindlg(depth)>1} {
	CubeDialog

	if {$bindlg(auto)} {
	    if {$bindlg(factor,x) != {}
		&& $bindlg(factor,y) != {}
		&& $bindlg(depth) != {}
		&& $bindlg(zcol,min) != {}
		&& $bindlg(zcol,max) != {}
		&& $bindlg(xcol) != {}
		&& $bindlg(ycol) != {}
		&& $bindlg(zcol) != {}} {

		$current(frame) bin to \
		    $bindlg(factor,x) $bindlg(factor,y)\
		    $bindlg(depth) \
		    $bindlg(zcol,min) $bindlg(zcol,max) \
		    about center \
		    \{$bindlg(xcol)\} \{$bindlg(ycol)\} \
		    \{$bindlg(zcol)\} \{$bindlg(filter)\}
	    }
	} else {
	    if {$bindlg(factor,x) != {}
		&& $bindlg(factor,y) != {}
		&& $bindlg(depth) != {}
		&& $bindlg(zcol,min) != {}
		&& $bindlg(zcol,max) != {}
		&& $bindlg(x) != {}
		&& $bindlg(y) != {}
		&& $bindlg(xcol) != {}
		&& $bindlg(ycol) != {}
		&& $bindlg(zcol) != {}} {

		$current(frame) bin to \
		    $bindlg(factor,x) $bindlg(factor,y) \
		    $bindlg(depth) \
		    $bindlg(zcol,min) $bindlg(zcol,max) \
		    about $bindlg(x) $bindlg(y) \
		    \{$bindlg(xcol)\} \{$bindlg(ycol)\} \
		    \{$bindlg(zcol)\} \{$bindlg(filter)\}
	    }
	}
    } else {
	if {$bindlg(auto)} {
	    if {$bindlg(factor,x) != {}
		&& $bindlg(factor,y) != {}
		&& $bindlg(xcol) != {}
		&& $bindlg(ycol) != {}} {

		$current(frame) bin to \
		    $bindlg(factor,x) $bindlg(factor,y) \
		    about center \
		    \{$bindlg(xcol)\} \{$bindlg(ycol)\} \
		    \{$bindlg(filter)\}
	    }
	} else {
	    if {$bindlg(factor,x) != {}
		&& $bindlg(factor,y) != {}
		&& $bindlg(x) != {}
		&& $bindlg(y) != {}
		&& $bindlg(xcol) != {}
		&& $bindlg(ycol) != {}} {

		$current(frame) bin to \
		    $bindlg(factor,x) $bindlg(factor,y) \
		    about $bindlg(x) $bindlg(y) \
		    \{$bindlg(xcol)\} \{$bindlg(ycol)\} \
		    \{$bindlg(filter)\}
	    }
	}
    }

    UpdateContourMenu
    UpdateCubeDialog
    UpdateGraphXAxis
    UpdateScaleDialog
    UnsetWatchCursor

    # the bining center will shift
    UpdateBinDialog
    UpdateBinMenu

    # reset
    set bindlg(auto) 0
}

proc BinClearFilterDialog {} {
    global bin

    $bin(top).filter.f.filter delete 0 end
}

proc BinDestroyDialog {} {
    global bin

    if {[winfo exist $bin(top)]} {
	destroy $bin(top)
	destroy $bin(mb)
    }
}

proc BinCutDialog {} {
    global bin

    set w [focus -displayof $bin(top)]

    if {![catch {set data [string range [$w get] [$w index sel.first] [expr {[$w index sel.last] - 1}]]}]} {
        clipboard clear -displayof $w
        clipboard append -displayof $w $data
        $w delete sel.first sel.last
    }
}

proc BinCopyDialog {} {
    global bin

    set w [focus -displayof $bin(top)]

    if {![catch {set data [string range [$w get] [$w index sel.first] [expr {[$w index sel.last] - 1}]]}]} {
        clipboard clear -displayof $w
        clipboard append -displayof $w $data
    }
}

proc BinPasteDialog {} {
    global tcl_platform
    global bin

    set w [focus -displayof $bin(top)]

    catch {
        if {"$tcl_platform(platform)" != "unix"} {
            catch {
                $w delete sel.first sel.last
            }
        }
        $w insert insert [selection get -displayof $w -selection CLIPBOARD]
        tkEntrySeeInsert $w
    }
}

proc BinClearDialog {} {
    global bin

    set w [focus -displayof $bin(top)]

    if {![catch {selection get -displayof $bin(top)}]} {
	$w delete sel.first sel.last
    }
}

proc UpdateBinMenu {} {
    global ds9
    global current
    global bin
    global menu

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateBinMenu"
    }

    if {$current(frame) != ""} {
	if {[$current(frame) has fits]} {
	    if {[$current(frame) has bin]} {
		$ds9(mb) entryconfig $menu(bin) -state normal
		$ds9(buttons).major.bin configure -state normal
	    } else {
		$ds9(mb) entryconfig $menu(bin) -state disabled
		$ds9(buttons).major.bin configure -state disabled
	    }
	} else {
	    $ds9(mb) entryconfig $menu(bin) -state normal
	    $ds9(buttons).major.bin configure -state normal
	}

	set bin(function) [$current(frame) get bin function]
	set bin(factor) "[$current(frame) get bin factor]"
	set bin(depth) [$current(frame) get bin depth]
	set bin(buffersize) [$current(frame) get bin buffer size]
    } else {
	$ds9(mb) entryconfig $menu(bin) -state disabled
	$ds9(buttons).major.bin configure -state disabled
    }
}

proc ProcessBinCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global bin

    switch -- [string tolower [lindex $var $i]] {
	about {
	    BinAbout [lindex $var [expr $i+1]] [lindex $var [expr $i+2]]
	    incr i 2
	}
	buffersize {
	    incr i
	    set bin(buffersize) [lindex $var $i]
	    ChangeBinBufferSize
	}
	cols {
	    BinCols \"[lindex $var [expr $i+1]]\" \"[lindex $var [expr $i+2]]\" \"[lindex $var [expr $i+3]]\"
	    incr i 3
	}
	factor {
	    incr i
	    set bx [lindex $var $i]
	    set by [lindex $var [expr $i+1]]
	    # note: the spaces are needed so that the menus are in sync
	    if {$by != {} && [string is double $by]} {
		set bin(factor) " $bx $by "
		incr i
	    } else {
		set bin(factor) " $bx $bx "
	    }
	    ChangeBinFactor
	}
	depth {
	    incr i
	    set bin(depth) [lindex $var $i]
	    ChangeBinDepth
	}
	filter {
	    incr i
	    BinFilter [lindex $var $i]
	}
	function {
	    incr i
	    set bin(function) [string tolower [lindex $var $i]]
	    ChangeBinFunction
	}
	to {
	    # eat the 'fit'
	    incr i
	    BinToFit
	}
	smooth {
	    incr i
	    uplevel #1 "ProcessSmoothCmd $varname $iname"
	}
    }
}

