// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "frametrue16.h"
#include "colorscaletrue16.h"
#include "fitsimage.h"
#include "nan.h"
#include "ps.h"
#include "util.h"

// Tk Canvas Widget Function Declarations

int FrameTrueColor16CreateProc(Tcl_Interp*, Tk_Canvas, Tk_Item*, int, Tcl_Obj *const []);

// FrameTrueColor16 Specs

static Tk_CustomOption tagsOption = {
  Tk_CanvasTagsParseProc, Tk_CanvasTagsPrintProc, NULL
};

static Tk_ConfigSpec frameTrueColor16Specs[] = {

  {TK_CONFIG_STRING, "-command", NULL, NULL, "frame",
   Tk_Offset(WidgetOptions, cmdName), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-x", NULL, NULL, "1",
   Tk_Offset(WidgetOptions, x), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-y", NULL, NULL, "1",
   Tk_Offset(WidgetOptions, y), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-width", NULL, NULL, "512",
   Tk_Offset(WidgetOptions, width), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-height", NULL, NULL, "512",
   Tk_Offset(WidgetOptions, height), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_ANCHOR, "-anchor", NULL, NULL, "nw",
   Tk_Offset(WidgetOptions, anchor), 0, NULL},
  {TK_CONFIG_CUSTOM, "-tags", NULL, NULL, NULL,
   0, TK_CONFIG_NULL_OK, &tagsOption},

  {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0, NULL},
};

// Tk Static Structure

static Tk_ItemType frameTrueColor16Type = {
  "frametruecolor16",           // name
  sizeof(WidgetOptions),        // item size
  FrameTrueColor16CreateProc,   // configProc
  frameTrueColor16Specs,        // configSpecs
  WidgetConfigProc,             // configProc
  WidgetCoordProc,              // coordProc
  WidgetDeleteProc,             // deleteProc
  WidgetDisplayProc,            // displayProc
  0,                            // alwaysRedraw
  WidgetPointProc,              // pointProc
  WidgetAreaProc,               // areaProc
  WidgetPostscriptProc,         // postscriptProc
  WidgetScaleProc,              // scaleProc
  WidgetTranslateProc,          // translateProc
  (Tk_ItemIndexProc*)NULL,      // indexProc
  WidgetICursorProc,            // icursorProc
  (Tk_ItemSelectionProc*)NULL,  // selectionProc
  (Tk_ItemInsertProc*)NULL,     // insertProc
  (Tk_ItemDCharsProc*)NULL,     // dCharsProc
  (Tk_ItemType*)NULL            // nextPtr
};

// Non-Member Functions

int FrameTrueColor16_Init(Tcl_Interp* interp)
{
  Tk_CreateItemType(&frameTrueColor16Type);
  return TCL_OK;
}

int FrameTrueColor16CreateProc(Tcl_Interp* interp, Tk_Canvas canvas, 
			       Tk_Item* item, int argc, Tcl_Obj *const argv[])
{
  FrameTrueColor16* frame = new FrameTrueColor16(interp, canvas, item);

  // and set default configuration

  if (frame->configure(argc, (const char**)argv, 0) != TCL_OK) {
    delete frame;
    Tcl_AppendResult(interp, " error occured while creating frame.", NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}

// FrameTrueColor16 Member Functions

FrameTrueColor16::FrameTrueColor16(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item)
  : FrameTrueColor(i, c, item)
{
  configSpecs = frameTrueColor16Specs;  // frame configure options
}

FrameTrueColor16::~FrameTrueColor16()
{
  // we must do this at this level, because updateColorScale is called
  unloadAllFits();
}

void FrameTrueColor16::encodeTrueColor(XColor* src, char* dest)
{
  if (!baseXImage)
    return;

  int msb = baseXImage->byte_order;
  int rs,gs,bs;
#ifndef _WIN32
  unsigned short rm = decodeMask((unsigned short)visual->red_mask, &rs);
  unsigned short gm = decodeMask((unsigned short)visual->green_mask, &gs);
  unsigned short bm = decodeMask((unsigned short)visual->blue_mask, &bs);
#else
  unsigned short rm = decodeMask((unsigned short)0x7C00, &rs);
  unsigned short gm = decodeMask((unsigned short)0x03E0, &gs);
  unsigned short bm = decodeMask((unsigned short)0x001F, &bs);
#endif

  // we need to check to byteswap when we have cross platforms

  unsigned short r = (unsigned char)src->red;
  unsigned short g = (unsigned char)src->green;
  unsigned short b = (unsigned char)src->blue;
  unsigned short a = 0;
  a |= rs>0 ? ((r & rm) << rs) : ((r & rm) >> -rs);
  a |= gs>0 ? ((g & gm) << gs) : ((g & gm) >> -gs);
  a |= bs>0 ? ((b & bm) << bs) : ((b & bm) >> -bs);

  if ((!msb && lsb()) || (msb && !lsb()))
    memcpy(dest, &a, 2);
  else {
    unsigned char* rr = (unsigned char*)(&a);
    *(dest) = *(rr+1);
    *(dest+1) = *(rr);
  }
}

void FrameTrueColor16::updateColorScale()
{
  // we need colors before we can construct a scale

  if (!indexCells || !colorCells)
    return;

  // since we need to know about bytes per pixel and byte order, 
  // we may need to wait til we have an XImage available and try again

  if (!baseXImage)
    return;

  encodeTrueColor(bgColor, bgTrueColor_);
  encodeTrueColor(nanColor, nanTrueColor_);

  if (colorScale)
    delete colorScale;

  switch (currentScale->colorScaleType()) {
  case FrScale::LINEARSCALE:
    colorScale =
      new LinearScaleTrueColor16(colorCount, indexCells, colorCells, 
				 colorCount, visual, baseXImage->byte_order);
    break;
  case FrScale::LOGSCALE:
    colorScale =
      new LogScaleTrueColor16(SCALESIZE, indexCells, colorCells, colorCount,
			      visual, baseXImage->byte_order);
    break;
  case FrScale::SQUAREDSCALE:
    colorScale =
      new SquaredScaleTrueColor16(SCALESIZE, indexCells, colorCells, 
				  colorCount, visual, baseXImage->byte_order);
    break;
  case FrScale::SQRTSCALE:
    colorScale = 
      new SqrtScaleTrueColor16(SCALESIZE, indexCells, colorCells, colorCount,
			       visual, baseXImage->byte_order);
    break;
  case FrScale::IISSCALE:
    colorScale =
      new IISScaleTrueColor16(indexCells, colorCells, colorCount, 
			      visual, baseXImage->byte_order);
    break;
  case FrScale::HISTEQUSCALE:
    calcHistEqu();
    colorScale =
      new HistEquScaleTrueColor16(SCALESIZE, indexCells, colorCells, 
				  colorCount, currentScale->histequ(),
				  HISTEQUSIZE, 
				  visual, baseXImage->byte_order);
    break;
  }
}
