#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include <sstream>
#include <iostream>
#include <string>
#include <algorithm>
#include <rumba/manifoldFile.h>
#include <rumba/manifold.h>
#include <rumba/arghandler.h>
#include <qimage.h>

#define NDEBUG

using RUMBA::intPoint;

typedef intPoint (*point_transform_t)(const intPoint&);
intPoint axialToAxial(const intPoint& p) { return p; }


struct ManifoldImageInput
{
	ManifoldImageInput():base(0),overlay(0),mask(0),f(axialToAxial),
   	lower_(0),upper_(0), ptrLower(NULL), ptrUpper(NULL), proportion(false)	{}

	void setLower(double x) { lower_ = x; ptrLower = &x; }
	void setUpper(double x) { upper_ = x; ptrUpper = &x; }

	bool swapped() const
	{
		return ptrLower && ptrUpper && (*ptrLower > *ptrUpper);
	}

	double threshold(double x, double min, double max) const;


	RUMBA::ManifoldFile* base;
	RUMBA::ManifoldFile* overlay;
	RUMBA::Manifold<char>* mask;
	std::string colormapfile;
	std::string overlaymapfile;
	point_transform_t f;
	double lower_, upper_;
	double* ptrLower, *ptrUpper;
	bool proportion;
};


double ManifoldImageInput::threshold(double x, double min, double max) const
{
	double scale = 1;
	if (proportion)
		scale = 1/ (max-min);
	if (swapped())
	{
		if (x< (*ptrUpper)*scale && x> (*ptrLower)*scale )
			return (max-min)/2;
		else
			return x;
	}
	else
	{
		if ( ptrUpper && (x > (*ptrUpper)*scale ) )
			return min;
		if ( ptrLower && (x < (*ptrLower)*scale ) )
			return min;
		return x;
	}
}



class ManifoldImageColorMap
{
public:
	ManifoldImageColorMap(
		const char* bfile,const char* ofile,QRgb background = 0
	)	;

	ManifoldImageColorMap
		(
		 const QImage& base, 
		 const QImage& overlay, 
		 QRgb background
		 );
	QRgb overlayPixel(int) const;
	QRgb basePixel(int) const;

	ManifoldImageColorMap copy() const;

private:
	QImage base;
	QImage overlay;
	QRgb background;
};

class ManifoldImage 
{
public:
	ManifoldImage();
	ManifoldImage(const ManifoldImage&);
	ManifoldImage& operator=(const ManifoldImage&);
	ManifoldImage(
			const RUMBA::Manifold<short>& base, 
			const RUMBA::Manifold<short>& overlay,
			const RUMBA::Manifold<char>& mask,
			const ManifoldImageColorMap& cmap
			);

	virtual ~ManifoldImage();
	QRgb getColor(int x, int y, int z = 0, int t = 0) const;
	QRgb getColor(const intPoint&) const;
	int width() const;
	int height() const;
	int depth() const;
	int timepoints() const;
	ManifoldImage copy() const;
	void loadImage(const ManifoldImageInput& im, int t);
	static ManifoldImageColorMap defaultColorMap();
	static QImage defaultBase();
	static QImage defaultOverlay();

private:
	RUMBA::Manifold<short> base;
	RUMBA::Manifold<short> overlay;
	RUMBA::Manifold<char> mask;
	ManifoldImageColorMap cmap;
};



void debug_cmap(const ManifoldImageColorMap& cmap)
{
	QRgb tmp;
	for (int i = 0; i < 256; ++i )
	{
		tmp = cmap.overlayPixel(i);
		cerr << i << " " << qRed(tmp) << " " << qBlue(tmp) << " " << qGreen(tmp) << std::endl;

	}

	for ( int i = 0; i < 256; ++i )
	{
		tmp = cmap.basePixel(i);
		cerr << i << " " << qRed(tmp) << " " << qBlue(tmp) << " " << qGreen(tmp) << std::endl;
	}

}


ManifoldImageColorMap::ManifoldImageColorMap(
	const char* bfile,const char* ofile,QRgb background
):background(background)
{
	if (strlen(bfile))
	{
		base.load(bfile); 
	}
	else
	{
		base = ManifoldImage::defaultBase();
	}
	if (strlen(ofile))
	{
		overlay.load(ofile);
	}
	else
	{
		overlay = ManifoldImage::defaultOverlay();
	}
}


ManifoldImageColorMap::ManifoldImageColorMap
	(
	 const QImage& base, 
	 const QImage& overlay, 
	 QRgb background
 )
	:base(base),overlay(overlay),background(background)
{
	
}


ManifoldImageColorMap ManifoldImageColorMap::copy() const
{ 
	return ManifoldImageColorMap(base.copy(),overlay.copy(),background);
}

QRgb ManifoldImageColorMap::overlayPixel(int i) const
{
	assert (overlay.valid(i,0));
	return overlay.pixel(i,0);
}

QRgb ManifoldImageColorMap::basePixel(int i) const
{
	assert (base.valid(i,0));
	return base.pixel(i,0);
}

QRgb ManifoldImage::getColor (int x, int y, int z, int t) const
{
	assert (x<overlay.width()&&y<overlay.height()&&z<overlay.depth()&&t<overlay.timepoints());
	assert (x<base.width()&&y<base.height()&&z<base.depth()&&t<base.timepoints());
	if ( mask.getElement(x,y,z,0) )
		return cmap.overlayPixel(overlay.getElement(x,y,z,t));
	else
		return cmap.basePixel(base.getElement(x,y,z,t));
}

int ManifoldImage::width() const
{
	return base.width();
}

int ManifoldImage::height() const
{
	return base.height();
}

int ManifoldImage::depth() const
{
	return base.depth();
}

int ManifoldImage::timepoints() const
{
	return base.depth();
}

ManifoldImage::ManifoldImage() 
	: cmap(defaultColorMap())
{}

ManifoldImage::ManifoldImage(const ManifoldImage& Im)
: base(Im.base), overlay(Im.overlay), mask(Im.mask), cmap(Im.cmap)
{}

ManifoldImage& 
ManifoldImage::operator=(const ManifoldImage& Im)
{
	base = Im.base;
	overlay = Im.overlay;
	mask = Im.mask;
	cmap = Im.cmap;
	return *this;
}

ManifoldImage ManifoldImage::copy() const
{
	ManifoldImage result(base.copy(),overlay.copy(),mask.copy(),cmap.copy());
	return result;
}

QImage ManifoldImage::defaultBase()
{
	QImage baseMap(256,1,8,256);
	for ( int i = 0; i < 256; ++i )
		baseMap.setColor(i,qRgb(i,i,i));
	for ( int i = 0; i<256; ++i )
		baseMap.setPixel(i,0,i);

	return baseMap;
}

QImage ManifoldImage::defaultOverlay()
{
	QImage overlayMap(256,1,8,256);
	for ( int i = 0; i < 256; ++i )
		overlayMap.setColor(i,qRgb(i,i<128?0:2*(i-128),0));
	for ( int i = 0; i<256; ++i )
		overlayMap.setPixel(i,0,i);

	return overlayMap;

}

ManifoldImageColorMap ManifoldImage::defaultColorMap() 
{
	return ManifoldImageColorMap(defaultBase(),defaultOverlay(),qRgb(0,0,0));

}

ManifoldImage::ManifoldImage(
		const RUMBA::Manifold<short>& base, 
		const RUMBA::Manifold<short>& overlay,
		const RUMBA::Manifold<char>& mask,
		const ManifoldImageColorMap& cmap
		):base(base),overlay(overlay),mask(mask),cmap(cmap)
{

}



ManifoldImage::~ManifoldImage()
{}

// x,y,z,t -> x,z,y,t
intPoint axialToCoronal(const intPoint& p) 
{ 
	intPoint q(p); 
	std::swap(q.y(),q.z()); 
	return q; 
}

// x,y,z,t -> z,y,x,t   WANT: x,y,z,t -> y,z,x,t
intPoint axialToSagital(const intPoint& p) 
{ 
	intPoint q(p);
	std::swap(q.x(),q.z()); 
	std::swap(q.y(),q.z()); // ?????
	return q;
}




std::string getname(int i, int j)
{
	std::ostringstream strout;
	strout << "image-" << setw(4) << setfill('0') << i 
		<< "-slice-" << setw(4) << setfill('0') << j << ".png";
	return strout.str();
}

std::string getmosaicname(int i)
{
	std::ostringstream strout;
	strout << "image-" << setw(3) << setfill('0') << i <<  ".png";
	return strout.str();
}

void initColorTable(QImage& im)
{
	for ( int i = 0; i < 256; ++i )
		im.setColor(i,qRgb(i,0,0));

}

void writeslice(
		ManifoldImage in,
		int z,
		int t,
		const std::string& name
		)
{
	QImage im(in.width(),in.height(),32);
	for ( int i = 0; i < in.width(); ++i )
		for ( int j = 0; j <in.height(); ++j )
		{ 
			im.setPixel(i,in.height()-j-1,in.getColor(i,j,z));
		}

	if (!im.save( (name + "-" + getname(z,t) ).c_str(),"PNG"))
		throw RUMBA::Exception("Save failed, bailing out");
	else 
		std::cerr << "Saved: " << (name + "-" + getname(z,t)) << std::endl;
}




void writemosaic(
		ManifoldImage in,
		int t,
		const std::string& name,
		double pad = 0
		)
{
	int gridsize = 0;
	int h_offset = 0, v_offset = 0;
	while ( gridsize*gridsize < in.depth() ) 
		++gridsize;
	QImage im(
		static_cast<int>((in.width()+pad)*gridsize+pad),
		static_cast<int>((in.height()+pad)*gridsize+pad),
		32
	);
	for ( int z = 0; z < in.depth(); ++z )
	{
		h_offset = static_cast<int>(pad+(in.width()+pad) * (z%gridsize));
		v_offset = static_cast<int>(pad+(in.height()+pad) * (z/gridsize));

		for ( int i = 0; i < in.width(); ++i )
			for ( int j = 0; j < in.height(); ++j )
			{
				assert (im.valid(i+h_offset, (in.height()-j-1)+v_offset));
				im.setPixel(
					i+h_offset,
					(in.height()-j-1)+v_offset, 
					in.getColor(i,j,z));

			}
	}

	if (!im.save( (name + "-" + getmosaicname(t)).c_str(),"PNG"))
		throw RUMBA::Exception("Save failed, bailing out");
	else
		std::cerr << "Saved: " << (name + "-" + getmosaicname(t)) << std::endl;

}

void maxmin(RUMBA::ManifoldFile* in, int t, double& Min, double& Max)
{
	double tmp;
	Min = Max = in->getElementDouble(t*in->pixels());
	for (int i = 0; i<in->pixels(); ++i )
	{
		if ( (tmp=(*in)[i+t*in->pixels()])>Max ) 
			Max=tmp;
		if ( (tmp=(*in)[i+t*in->pixels()])<Min ) 
			Min=tmp;
	}
}

void writeSlices( ManifoldImageInput im, const std::string& outfile)
{
	intPoint dims = im.base->extent();
	dims=im.f(dims);
	ManifoldImage X;

	for (int i = 0; i < dims.t(); ++i )
	{
		X.loadImage(im,i);
		for ( int j = 0; j < dims.z(); ++j )
		{
			writeslice(X,j,i,outfile.c_str());
		}
	}

}


void writeMosaics ( ManifoldImageInput im, const std::string& outfile, double pad=0)
{
	intPoint dims = im.base->extent();
	dims = im.f(dims);
	ManifoldImage X;

	for (int t = 0; t < im.base->timepoints(); ++t )
	{
		X.loadImage(im,t);
		writemosaic(X,t,outfile.c_str(),pad);
	}
}

/**
 *  Load and re-orient an image
 */
void ManifoldImage::loadImage ( const ManifoldImageInput& im, int t )
{
	double tmp = 0;
	if (!im.base)
		throw RUMBA::Exception("base is a null image in loadImage()");
	intPoint dims = im.base->extent();
	dims = im.f(dims);

	base = RUMBA::Manifold<short>(dims);
	overlay = RUMBA::Manifold<short>(dims);
	mask = RUMBA::Manifold<char>(dims);


	double bMin,bMax,oMin,oMax;

	if (im.base)
		maxmin(im.base,t,bMin,bMax);
	else
		std::fill(base.begin(),base.end(),0);

	if (im.overlay)
		maxmin(im.overlay,t,oMin,oMax);
	else
		std::fill(overlay.begin(),overlay.end(),0);

	std::cout << bMin << " " << bMax << " " << oMin << " " << oMax << std::endl;
	std::cout << dims.x() << "," << dims.y() << "," << dims.z()  << std::endl;

	if (!im.mask)
		std::fill(mask.begin(),mask.end(),1);

	cmap = ManifoldImageColorMap(
		im.colormapfile.c_str(), im.overlaymapfile.c_str()
	);


	for ( int j = 0; j < dims.z(); ++j )
	{
		for ( int k = 0; k < dims.y(); ++k )
		{
			for ( int m = 0; m < dims.x(); ++m )
			{
				if (im.base)
				{
					tmp = im.base->getElementDouble(im.f(intPoint(m,k,j,t))) 
						-bMin;
					base.setElement 
						(
						 intPoint(m,k,j,0), 
						 static_cast<int>(tmp*256./(bMax+1e-15-bMin) )
						 );
				}

				if (im.overlay)
				{
					tmp = im.overlay->getElementDouble(im.f(intPoint(m,k,j,t)))
					  -oMin;

					tmp = im.threshold(tmp,oMin,oMax);

					if (im.ptrLower && tmp < *im.ptrLower)
						tmp = oMin;

					if (im.ptrUpper && tmp > *im.ptrUpper)
						tmp = oMin;
	
					overlay.setElement
					( 
					 intPoint(m,k,j,0), 
					 static_cast<int> (tmp*256./(oMax+1e-15-oMin))
					 );
				}
	
				if (im.mask) 	
				{
					mask.setElement 
						( 
						 intPoint(m,k,j,0), 
						 im.mask->getElement(im.f(intPoint(m,k,j,0)))
						 );
				}
			}
		}
	}

}

std::string usage ()
{
	return "Usage: writeslices (-i|--infile) infile  (-o|--outfile) outfile \n"
		"[ --overlay overlayfile ]\n"
		"[ --mask maskfile] [--mosaic|-m]  [ (--colormap|-c) colormap1 ] \n"
		"[ --overlaymap colormap2]  [--cellpadding N]\n";
}

RUMBA::Argument myArgs[] = 
{
	RUMBA::Argument("mosaic",RUMBA::FLAG,'m'),
	RUMBA::Argument("orientation",RUMBA::ALPHA,'\0', "" ),
	RUMBA::Argument("overlay",RUMBA::ALPHA,'\0', ""),
	RUMBA::Argument("mask",RUMBA::ALPHA, '\0', "" ),
	RUMBA::Argument("colormap",RUMBA::ALPHA, 'c', "" ),
	RUMBA::Argument("overlaymap",RUMBA::ALPHA, '\0', "" ),
	RUMBA::Argument("cellpadding",RUMBA::NUMERIC,'p', 0 ),
	RUMBA::Argument("lower",RUMBA::NUMERIC,'l' ),
	RUMBA::Argument("upper",RUMBA::NUMERIC,'u' ),
	RUMBA::Argument("proportion", RUMBA::FLAG , '\0'),
	RUMBA::Argument()
};

int main(int argc,char** argv)
{
	std::string infile,outfile,overlayfile,maskfile;
	std::string colormapfile,overlaymapfile;
	std::string orientation;
	ManifoldImageInput im;
	double padding=0;
	bool proportion;
	double lower = 1e-10, upper=1e-10;

	try
	{
		ManifoldImageColorMap cmap(ManifoldImage::defaultColorMap());
		RUMBA::ArgHandler argh(argc,argv,myArgs);
		if (argh.arg("help"))
		{
			std::cerr << usage() << std::endl;
			exit(0);
		}
		argh.arg("infile",infile);
		argh.arg("outfile",outfile);
		argh.arg("orientation",orientation);
		argh.arg("overlay",overlayfile);
		argh.arg("mask",maskfile);
		argh.arg("colormap",colormapfile);
		argh.arg("overlaymap",overlaymapfile);
		argh.arg("cellpadding",padding);
		im.proportion = argh.arg("proportion");

		if (argh.arg("lower"))
		{
			argh.arg("lower",lower);
			im.setLower(lower);
		}
		if (argh.arg("upper"))
		{
			argh.arg("upper",upper);
			im.setUpper(upper);
		}

		if ( orientation == "s" )
			im.f = axialToSagital;
		else if ( orientation == "c" )
			im.f = axialToCoronal;
		else
			if ( orientation != "a" && (! orientation.empty() ) )
				throw RUMBA::Exception("Valid orientations are a,c and s");



		im.base = RUMBA::ManifoldFile::construct(infile.c_str());
		if (!im.base) 
			throw RUMBA::Exception ( string("couldn't open ")+infile);
		im.base->setCacheStrategy(RUMBA::CACHE_VOL);

		if (!overlayfile.empty())
		{
			im.base = RUMBA::ManifoldFile::construct(infile.c_str());
			if (!im.base) 
				throw RUMBA::Exception ( string("couldn't open ")+infile);
			im.base->setCacheStrategy(RUMBA::CACHE_VOL);

			im.overlay = RUMBA::ManifoldFile::construct(overlayfile.c_str());
			if (!im.overlay) 
				throw RUMBA::Exception(string("couldn't open ")+overlayfile);
			im.overlay->setCacheStrategy(RUMBA::CACHE_VOL);
		}
		else
		{
			im.overlay = RUMBA::ManifoldFile::construct(infile.c_str());
			if (!im.overlay) 
				throw RUMBA::Exception ( string("couldn't open ")+infile);
			im.overlay ->setCacheStrategy(RUMBA::CACHE_VOL);
		}

		if (!maskfile.empty())
		{
			im.mask = new RUMBA::Manifold<char>(maskfile.c_str());
		}

		if (!overlaymapfile.empty())
		{
			std::cerr << "Setting overlaymapfile: " << overlaymapfile.c_str()
				<< std::endl;
			im.overlaymapfile = overlaymapfile;
		}

		if (!colormapfile.empty())
		{
			std::cerr << "Setting colormapfile: " << overlaymapfile.c_str()
				<< std::endl;
			im.colormapfile = colormapfile;
			if (overlayfile.empty())
				im.overlaymapfile = colormapfile;
		}

		if (argh.arg("mosaic"))
			writeMosaics(im,outfile,padding);
		else 
			writeSlices(im,outfile);



		delete im.base;
		delete im.overlay;
		delete im.mask;

	}
	catch ( RUMBA::Exception& e )
	{
		std::cerr<<e.error()<<std::endl;
		return 1;
	}
	return 0;

}
