#ifndef RUMBA_SVD_H
#define RUMBA_SVD_H

/**
  \file svd.h
	\author Donovan Rebbechi, Giorgio Grasso, Ben Bly
	Copyright Ben Bly. This file is released under the artistic license.
	See the file COPYING for details.

  */

#include <rumba/manifoldmatrix.h>

namespace RUMBA
{

/**
  * Decompose an mxn matrix X into X = UDV^t where D is a diagonal nxn matrix,
  * V is orthogonal, and nxn, U is mxn. This version saves time and memory
  * by not accumulating U.
  */
void svd 
( 
 const ManifoldMatrix& X, 
 ManifoldMatrix& V, 
 ManifoldMatrix & D 
 );

/**
  * SVD. Decomposes an mxn matrix with m>n, X, as UDV^t where V is an 
  * orthogonal nxn matrix, D is a diagonal nxn matrix, U is an mxn matrix
  * with orthogonal columns.
  */
void svd 
( 
 const ManifoldMatrix& X,  
 ManifoldMatrix& U, 
 ManifoldMatrix& V, 
 ManifoldMatrix & D 
 );

/**
  * Destructive SVD. Decomposes an mxn matrix with m>n, X, as UDV^t 
  * where V is an  orthogonal nxn matrix, D is a diagonal nxn matrix, 
  * U is an mxn matrix with orthogonal columns.
  * The contents of X after this function is called are undefined. U is not 
  * computed explicitly.
  *
  * This function uses less memory than the destructive version, and saves
  * time by not accumulating the value of U.
  */
void destructive_svd
(
 ManifoldMatrix& X,
 ManifoldMatrix& V,
 ManifoldMatrix& D
 );

/**
  * This function overwrites X with garbage, as well as computing the singular
  * value decomposition of X.  Note that you can call this function with 
  * X == U, overwriting the matrix with the orthogonal decomposition of its
  * column space. This function has the advantage that compared to the 
  * non-destructive version, it saves memory.
  */
void destructive_svd
(
 ManifoldMatrix& X,
 ManifoldMatrix& U,
 ManifoldMatrix& V,
 ManifoldMatrix& D
 );

} /* namespace RUMBA */

#endif
