#ifndef RUMBA_GIVENS_H
#define RUMBA_GIVENS_H

#include <rumba/manifoldmatrix.h>


/**
  * \file givens.h
  * \author Donovan Rebbechi, Ben Bly, Giorgio Grasso
  * Copyright Ben Bly
  * This file is released under the artistic license. See the file
  * COPYING for details.
  */



namespace RUMBA
{

class Givens;

/**
  * left multiply A by the transpose of the Givens(Jacobi) 
  * matrix G(p,q,theta(c,s))
  */
void apply_left_givens 
( ManifoldMatrix& A, unsigned int p, unsigned int q, double c, double s);

/**
  * left multiply A by the transpose of the Givens(Jacobi) 
  * matrix G
  */
void apply_left_givens
(ManifoldMatrix& A, const Givens& g);

/**
 * right-multiply the matrix A by the Givens(Jacobi) matrix G(p,q,theta(c,s))
 */
void apply_right_givens 
( ManifoldMatrix& A, unsigned int p, unsigned int q, double c, double s);

/**
 * right-multiply the matrix A by the Givens(Jacobi) matrix G
 */
void apply_right_givens
(ManifoldMatrix& A, const Givens& g);

/** see: Matrix computations, P216, Algorithm 5.1.3: givens
* tranform such that
*   | c s|t |a|  =  |r|
*	 |-s c|  |b|     |0|
*
*	 note that this is the same as:
*
*   [ a b ]| c s|   =  [ r 0 ]
*	        |-s c|      
*
*/
Givens get_givens(double a, double b);

/** A derivative method useful if you want to "zero" the upper or left entry
* eg a transorm such that
*
*   | c s|t |a|  =  |0|
*	 |-s c|  |b|     |r|
*
*  Note that this is the same as requiring:
*
*   [ a b ]| c s|   =  [ 0 r ]
*          |-s c|      
*/

Givens get_givens2(double a, double b);


/**
  * A class for representing givens transforms. Givens transforms are
  * plane rotations. Up to conjugacy by permutation, they look like this:
  *   cos(theta)    sin(theta)
  *   -sin(theta)   cos(theta)
  *
  *  \author Donovan Rebbechi, Ben Bly, Giorgio Grasso.
  *  Copyright Ben Bly. This software is released under 
  *  the artistic license. See the accompanying COPYING file for details.
  */
class Givens
{
public:

	/**
	  * Default constructor. Identity matrix.
	  */
	Givens() : C(1),S(0), i(0),j(1)
	{
	}

	/**
	  *   Create a Givens transform in the plane spanned by e_i, e_j
	  * whose restriction to that plane is of the form
	  *    C S 
	  *   -S C
	  * Note that the caller needs to make sure C^2 + S^2  = 1.
	  */
	Givens(double x, double y, int i=0, int j=0)
	: C(x), S(y), i(i), j(j)
	{
		if ( fabs( (C*C+S*S) - 1 ) > 1e-8 )
			throw RUMBA::Exception("Inconsistent data");
	}

	/**
	  * Set the rotation plane.
	  */
	inline void setCoords(int x, int y){ i=x;j=y; }
		

	//! Accessor 
	int row() const{ return i; }
	//! Accessor
	int col() const{ return j; }
	//! Accessor
	double c() const{ return C; }
	//! Accessor
	double s() const{ return S; }

	//! return the inverse of the transform
	Givens inv () const { return Givens(C,S,j,i); }
	/** adjust the co-ordinate plane. Useful for cycling through co-ordinate
	  * planes.
	  */
	void adjustCoords(int x, int y) { i+=x; j+=y; }

private:
	double C;
	double S;
	int i,j;
};

/**
* pre-multiplies applies a list of givens transforms. Each transform
* goes on the left, ie Gn * ... * G2 * G1 * right  
* If inv is true, apply inverse of each transform Gi instead of the
* transform itself.
*/
void multiply ( 
	const std::list<Givens>& left, ManifoldMatrix& right, bool inv = false
);

/** post-multiplies applies a list of givens transforms. Each transform
* goes on the right, ie  left * G1 * G2 * .....
* If inv is true, apply inverse of each transform Gi instead of the
* transform itself.
*/
void multiply ( 
	ManifoldMatrix& left, const std::list<Givens>& right , bool inv=false
);

}

#endif
