#ifndef RUMBA_IOHANDLER_H
#define RUMBA_IOHANDLER_H

//#ifdef __GNUC__
//#define streampos ios::seek_dir
//#endif

#include <rumba_config.h>
#include <fstream>
#include <strstream>

#include <rumba/iohandler_base.h>
#include <rumba/fileIO.h>
#include <rumba/rumba_system.h>
#include <rumba/log.h>
#include <rumba/point.hpp>
#include <rumba/manifold.h>




using RUMBA::BaseManifold;

namespace {
	template<class T>
	std::string toString(const T& x)
	{
		std::ostrstream out;
		std::string result;
		out << x;
		result=out.str();
		out.freeze(0);
		return result;
	}
}

void STREAM_ASSERT (std::fstream& x, const char* file, int line);


namespace RUMBA 
{ 

// TYPE is the TYPE of the underlying data
template<class TYPE> 
class IOHandler
: virtual public RUMBA::IOHandlerBase // permit multiple inheritence.
{
public:
	IOHandler();
	IOHandler
	(
	  const char* filename, 
	  int Size,
	  std::ios_base::openmode mode = std::ios::in
	);
	/** is the stream writable ? */
	virtual bool writeable() const;

	virtual void grow(int);
	virtual int streamsize();

	virtual void seekg(int x, std::ios::seekdir pos = std::ios::beg )
	{ 
		Stream.seekg (x * sizeof(TYPE), pos); 
		if (!Stream)
			throw RUMBA::BadFile ( 
				std::string("stream in error state")+
				 + ", line " + toString(__LINE__)
			);
	}

	virtual void seekp(int x, std::ios::seekdir pos = std::ios::beg )
	{ 

		Stream.seekp(x * sizeof(TYPE), pos); 
	}


	/**
	*	This one retrieves from the stream data member, not from memory.
	*/
	virtual void iGet ( BaseManifold*, int, int, bool);


	/**
 	 *	Create a Manifold of the same type. Return BaseManifold because
	 * we want to call this method from the base class.
  	*/
	virtual BaseManifold* createManifold(const intPoint&) const;

	/** 
	* retrieve and convert to TYPEa with rumba_cast
	*/
	virtual void cGet (char* dest,  int nelts, bool le)		{ cGet_impl(dest,nelts,le); }
	virtual void cGet (short* dest,  int nelts, bool le)	{ cGet_impl(dest,nelts,le); }
	virtual void cGet (int* dest,  int nelts, bool le)		{ cGet_impl(dest,nelts,le); }
	virtual void cGet (float* dest,  int nelts, bool le)	{ cGet_impl(dest,nelts,le); }
	virtual void cGet (double* dest,  int nelts, bool le)	{ cGet_impl(dest,nelts,le); }

	virtual void cPut (char* src,  int nelts, bool le)		{ cPut_impl(src,nelts,le); }
	virtual void cPut (short* src,  int nelts, bool le)	{ cPut_impl(src,nelts,le); }
	virtual void cPut (int* src,  int nelts, bool le)		{ cPut_impl(src,nelts,le); }
	virtual void cPut (float* src,  int nelts, bool le)	{ cPut_impl(src,nelts,le); }
	virtual void cPut (double* src,  int nelts, bool le)	{ cPut_impl(src,nelts,le); }


	RUMBA::BaseManifold* get
		(
		 const intPoint& org, 
		 const intPoint& dims, 
		 const intPoint& skip, 
		 bool le
		 );

	void get (
		const intPoint& org, const intPoint& dims, 
		const intPoint& skip, bool le, BaseManifold* result
		);

	void put (
		const Manifold<TYPE>& M, const intPoint& org, 
		const intPoint& dims, const intPoint& skip, bool le
		);

	template<class TYPEa>
	void put (
			const Manifold<TYPEa>& M, const intPoint& org, 
			const intPoint& dims, const intPoint& skip, bool le
			);

	void put (
			const BaseManifold*, 
			const intPoint& org, const intPoint& dims, const intPoint& skip, 
			bool le
			);

	/** 
	*	The int parameter is how many elements should be retrieved. 
	*	The TYPE* parameter is a pointer where the elements will be placed.
	* 	The caller is responsible for allocating the TYPE* parameter !
	*/
//	void iGet ( TYPE*, const char*, int, bool);

private:
	/**
	  *	cGet gets dispatched to a template implementation. Can't do it directly
	  * because member templates can't be virtual.
	  *
	  */
	template<class TYPEa> void cGet_impl (TYPEa* dest, int nelts, bool);
	void cGet_impl (TYPE* dest, int nelts, bool);
	template<class TYPEa> void cPut_impl (TYPEa* src, int nelts, bool);
	void cPut_impl (TYPE* src, int nelts, bool);

	std::ios_base::openmode Mode;
	int Size;

};

} // namespace RUMBA



template<class TYPE>
RUMBA::IOHandler<TYPE>
::IOHandler()
: IOHandlerBase()
{
#ifdef DEBUG
	log.logName() << "Calling IOHandler<TYPE>::IOHandler()\n";
#endif
}

template<class TYPE>
RUMBA::IOHandler<TYPE>
::IOHandler(const char* filename, int Size, std::ios_base::openmode mode )
: IOHandlerBase(filename, Size, mode), Mode(mode|std::ios::binary|std::ios::in)
{
#ifdef DEBUG
	log.logName() << "Calling IOHandler<TYPE>::IOHandler(" << filename 
		<< "," << mode << ")" << "\n";
#endif


//	Stream.open(filename);
	if ( !Stream )
	{
#ifdef DEBUG
		log.logName() << "Stream open failed\n";
#endif
		throw RUMBA::BadFile(
			std::string("Attempt to grow input stream " ) + __FILE__ + " : " +
			toString( __LINE__ ) );

	}

	// clear the file appropriately ... 

	if ( mode & std::ios::out )
	{
		grow(size());
	}

	if (!Stream)
		throw RUMBA::BadFile(
			std::string("Attempt to grow input stream " ) + __FILE__ + " : " +
			toString( __LINE__ ) );
	
}




template<class TYPE>
void RUMBA::IOHandler<TYPE>::grow(int)
{
	TYPE x = TYPE();
	if ( Mode & std::ios::out )
	{
		seekp(0);
		int top = size()-streamsize();
		for ( int i = 0; i < top; ++i )
			Stream.write(reinterpret_cast<char*>(&x), sizeof(TYPE));
		if ( Mode & std::ios::in ) 
			seekg(0);
		seekp(0);
	}
	else 
	{
		throw RUMBA::BadFile(
			std::string("Attempt to grow input stream " ) + __FILE__ + " : " +
			toString( __LINE__ ) );
	}
}

template<class TYPE>
int 
RUMBA::IOHandler<TYPE>::streamsize()
{
	return filesize(Stream) / sizeof(TYPE);
}

template<class TYPE>
BaseManifold*
RUMBA::IOHandler<TYPE>
::createManifold(const intPoint& p) const
{
	return new Manifold<TYPE> (p);
}

template<class TYPE>
bool
RUMBA::IOHandler<TYPE>
::writeable() const
{
	return Mode&std::ios::out;
}

template <class TYPE>
void RUMBA::IOHandler<TYPE>
::iGet (BaseManifold* dest, int start, int nelts, bool le)
{
#ifdef DEBUG
	log.logName() << "calling IOHandler<TYPE>::iGet(BaseManifold*, int,int,bool )" << "\n";
#endif
	std::streampos savep=Stream.tellp();
	std::streampos saveg=Stream.tellg();
	char* buf = new char[ nelts * sizeof(TYPE) ];
	const TYPE* ptr;

	// get the data pointer from dest, and offset by start.
	ptr = (dynamic_cast<Manifold<TYPE>*> (dest) )->data() + start;

	Stream.read(buf,nelts*sizeof(TYPE));

	RUMBA::iGet( const_cast<TYPE*>(ptr),buf,nelts,le);
	delete[] buf;
	Stream.seekg(saveg);
	Stream.seekp(savep);
	STREAM_ASSERT(Stream,__FILE__,__LINE__ );

}

template <class TYPE> template<class TYPEa>
void RUMBA::IOHandler<TYPE>
::cGet_impl (TYPEa* dest, int nelts, bool le)
{
#ifdef DEBUG
	log.logName() << "cGet_impl called" << "\n";
#endif
	std::streampos savep=Stream.tellp();
	std::streampos saveg=Stream.tellg();
	TYPE* buf = new TYPE[nelts];
	char* srcbuf; 
	if ( RUMBA::littleEndianHost() != le ) 
	{	// byteswap required
		srcbuf = new char[nelts*sizeof(TYPE)];
		Stream.read(srcbuf,nelts*sizeof(TYPE));
		RUMBA::iGet ( buf, srcbuf,nelts, le);
		delete[] srcbuf;
	}
	else	// no byteswap required. read straight into buf and bypass bs buffer
	{
		Stream.read( (char*)buf, nelts*sizeof(TYPE) );
	}

	TYPE* tmp_buf = buf; // we need buf later
	for ( int i = 0; i < nelts; ++i )
	{
		*(dest++) =  rumba_cast<TYPEa>(*(tmp_buf++));  
	}

	delete[] buf;
	Stream.seekp(savep);
	Stream.seekg(saveg);
	STREAM_ASSERT(Stream,__FILE__,__LINE__ );

}

// partial specialisation, where types match. this is faster than the
// general version because we don't need a conversion buffer
template <class TYPE>
void RUMBA::IOHandler<TYPE>
::cGet_impl (TYPE* dest, int nelts, bool le)
{
	assert(Stream);
	std::streampos savep=Stream.tellp();
	std::streampos saveg=Stream.tellg();
	char* srcbuf; 
	if ( RUMBA::littleEndianHost() != le )
	{
		srcbuf = new char[nelts*sizeof(TYPE)];
		Stream.read(srcbuf,nelts*sizeof(TYPE));
		RUMBA::iGet ( dest, srcbuf,nelts, le);
		delete[] srcbuf;
	}
	else // fast: no byteswap buffer required.
	{
		Stream.read ( (char*)dest, nelts*sizeof(TYPE));
	}
	Stream.seekp(savep);
	Stream.seekg(saveg);
	STREAM_ASSERT(Stream,__FILE__,__LINE__ );

}

template <class TYPE> template<class TYPEa>
void RUMBA::IOHandler<TYPE>
::cPut_impl (TYPEa* src, int nelts, bool le)
{
#ifdef DEBUG
	log.logName() << "cPut_impl( " << (void*)src << "," << nelts 
		<< "," << le << ")\n";
#endif
	std::streampos savep = Stream.tellp();
	std::streampos saveg = Stream.tellg();

	TYPE* buf = new TYPE[nelts]; // conversion buffer
	char* destbuf;  // byteswap buffer

	TYPE* tmp_buf = buf;	// save buf -- we'll need it later
	for ( int i = 0; i < nelts; ++i )
		*tmp_buf++ =  rumba_cast<TYPE>(*src++); // fill conversion buffer

	if ( RUMBA::littleEndianHost() != le )
	{	// byteswap required, so initialise and use byteswap buffer
		destbuf = new char[nelts*sizeof(TYPE)];
		RUMBA::iPut (destbuf,buf,nelts, le);	
		Stream.write(destbuf,nelts*sizeof(TYPE));
		delete[] destbuf;
	}
	else
	{	// no byteswap needed. So we can write from conversion buffer
		Stream.write((char*)buf,nelts*sizeof(TYPE));
	}
	delete[] buf;
	STREAM_ASSERT(Stream,__FILE__,__LINE__ );
	Stream.seekp(savep);
	Stream.seekg(saveg);


}

// partial specialisation
template <class TYPE>
void RUMBA::IOHandler<TYPE>
::cPut_impl (TYPE* src, int nelts, bool le)
{
//	log.logName() << "cPut_impl( " << (void*)src << "," << nelts << "," << le << ")\n";
	std::streampos savep=Stream.tellp();
	std::streampos saveg=Stream.tellg();

	char* destbuf; 
	if ( RUMBA::littleEndianHost() != le ) // byteswap required
	{

		destbuf = new char[nelts*sizeof(TYPE)];
		RUMBA::iPut (destbuf,src,nelts, le);	
		Stream.write(destbuf,nelts*sizeof(TYPE));
		delete[] destbuf;
	}
	else // no byteswap needed -> no temporary byteswap buffer required
	{
		Stream.write( (char*)src,nelts*sizeof(TYPE));
	}
	Stream.seekp(savep);
	Stream.seekg(saveg);
	STREAM_ASSERT(Stream,__FILE__,__LINE__ );

}

template<class TYPE>
RUMBA::BaseManifold*
RUMBA::IOHandler<TYPE>
::get
(const intPoint& org, const intPoint& dims, const intPoint& skip, bool le)
{
	Manifold<TYPE>* result = new Manifold<TYPE>(dims);
	get (org,dims,skip,le,result);
	return result;
}

template<class TYPE>
void
RUMBA::IOHandler<TYPE>
::get
(
 const intPoint& org, 
 const intPoint& dims, 
 const intPoint& skip, 
 bool le, 
 BaseManifold* result
 )
{
	std::streampos savep=Stream.tellp();
	std::streampos saveg=Stream.tellg();
	Manifold<TYPE>* res;
	if (!( res = dynamic_cast<Manifold<TYPE>* >(result)) )
		throw RUMBA::BadConvert("Dynamic cast failed");

//	Manifold<TYPE>* result = new Manifold<TYPE>(dims);
	bool swap = ( RUMBA::littleEndianHost() != le );
	int mSkipY = dims.x(); 
	int mSkipZ = dims.x() * dims.y();
	int mSkipT = dims.x() * dims.y() * dims.z();
	TYPE* buf = 0;
	int pos;
	if ( swap )
		buf = new TYPE[dims.x()];

	for ( int i=0; i<dims.t(); ++i)
		for (int j=0; j<dims.z(); ++j)
			for (int k=0; k<dims.y(); ++k)
			{

				pos = org.x()*skip.x()+ 
					(org.y()+k)*skip.y()+ 
					(org.z()+j)*skip.z()+ 
					(org.t()+i)*skip.t() ;
				seekg(pos,std::ios::beg);
					
				if ( !swap )
				{
					STREAM_ASSERT(Stream,__FILE__,__LINE__ ) ;
					res->readFromStream
					( 
							Stream , 
							k*mSkipY + j*mSkipZ + i*mSkipT, 
							dims.x() 
					);
					STREAM_ASSERT(Stream,__FILE__,__LINE__ );
				}
				else // byteswap
				{
#ifdef DEBUG
					log.logName() << "Byteswap " << "\n";
#endif
					Stream.read ( (char*)buf, dims.x() * sizeof(TYPE) );
					STREAM_ASSERT(Stream,__FILE__,__LINE__ );

					RUMBA::iGet ( buf, (char*)buf, dims.x(), le );
					std::istrstream in ( (char*)buf, dims.x() * sizeof(TYPE) );
					if ( !in)
					{
						throw RUMBA::Exception ( "IOHandler::get(): couldn't create stream in \n" );
					}
					res->readFromStream
					( 
					 	in,
						k*mSkipY + j*mSkipZ + i*mSkipT, 
						dims.x() 
					);

					if (!in) 
					{
						throw RUMBA::Exception ( "IOHandler::get(): stream in failed after reading result\n" );
					}

				}
			}
	if (buf) delete[] buf;
	Stream.seekp(savep);
	Stream.seekg(saveg);

};

template<class TYPE> 
void
RUMBA::IOHandler<TYPE>
::put(const BaseManifold* M, const intPoint & org, const intPoint & dims, 
		const intPoint& skip, bool le)
{
#ifdef DEBUG
	log.logName() << "Calling put(BaseManifold*) " << "\n";
#endif
	const Manifold<TYPE>* N;
	try {
		N = dynamic_cast<const Manifold<TYPE>* >(M);	// should use exception specs for this.
	}
	catch(...)
	{
		log.logName() << "put(" << M << ") failed: dynamic cast unsuccesful"<<"\n";
		return;
	}

	put(*N,org,dims,skip,le);
}	

template<class TYPE> template< class TYPEa>
void
RUMBA::IOHandler<TYPE>
::put(const Manifold<TYPEa> &M, const intPoint& org, const intPoint& dims, 
		const intPoint& skip, bool le)
{
	Manifold<TYPE> N(M);
	put(N,org,dims,skip,le);
}


template<class TYPE>
void
RUMBA::IOHandler<TYPE>
::put(const Manifold<TYPE> &M, const intPoint& org, const intPoint& dims, 
		const intPoint& skip, bool le)
{
	std::streampos savep=Stream.tellp();
	std::streampos saveg=Stream.tellg();
	if (!writeable())
		throw RUMBA::BadFile(
				std::string("Attempt to write to a read only file" )
				+ __FILE__ + " : " + toString(__LINE__) );
#ifdef DEBUG
	log.logName() << "Calling put (Manifold<TYPE> .... )" << "\n";
	log.logName() << "dims.x() is " << dims.x() << "\n";
#endif

	bool swap = ( RUMBA::littleEndianHost() != le );
	int mSkipY = dims.x(); 
	int mSkipZ = dims.x() * dims.y();
	int mSkipT = dims.x() * dims.y() * dims.z();
	TYPE* buf = 0;
	if ( swap )
		buf = new TYPE[dims.x()];

	seekp(org.x()*skip.x() + org.y()*skip.y() + org.z()*skip.z()+org.t()*skip.t());
	for ( int i=0; i<dims.t(); ++i)
		for (int j=0; j<dims.z(); ++j)
			for (int k=0; k<dims.y(); ++k)
			{
				seekp( 
						org.x()*skip.x()+
						(org.y()+k)*skip.y()+
						(org.z()+j)*skip.z()+
						(org.t()+i)*skip.t() 
						);
				if ( !swap )
				{
					M.writeToStream
					( 
							Stream , 
							k*mSkipY + j*mSkipZ + i*mSkipT, 
							dims.x() 
					);
				}
				else // byteswap
				{

					// turn buf into a stream so we can write to it
					std::ostrstream out ( (char*)buf, dims.x() * sizeof(TYPE) );
					M.writeToStream
					( 
					 	out,
						k*mSkipY + j*mSkipZ + i*mSkipT, 
						dims.x() 
					);
					
					// byteswap
					RUMBA::iPut ( (char*)buf, buf, dims.x(), le );
					Stream.write ( (char*)buf, dims.x() * sizeof(TYPE) );
				}
			}
	if (buf) delete[] buf;
	Stream.seekp(savep);
	Stream.seekg(saveg);

	STREAM_ASSERT(Stream,__FILE__,__LINE__ );

};





#endif
