#include <rumba/point.hpp>
#include <rumba/manifold.h>
#include <vector>


/**
  * \file utils.hpp
  * \author Donovan Rebbechi, Ben Bly, Giorgio Grasso
  * Copyright Ben Bly
  * This file is released under the artistic license. See the file
  * COPYING for details.
  */



namespace RUMBA {

Manifold<double> cycle_vector(const std::vector<double> &, bool transpose = false);
double maximum ( const BaseManifold* M  );
double minimum ( const BaseManifold* M );
double mean ( const BaseManifold* M );
double standard_deviation  ( const BaseManifold* M );
Manifold<double> z_normalize ( const BaseManifold* M );
std::vector<int> histogram
	( int n, const BaseManifold* M, double* min, double* max );

std::vector<int> histogram
	( int n, const BaseManifold* M, double lower, double upper);
void downsample(const intPoint&, const BaseManifold* in, BaseManifold* out);


template<class TYPE> void normalize( TYPE low, TYPE high, BaseManifold* M );

template<class TYPE> TYPE maximum ( const Manifold<TYPE>& M  );
template<class TYPE> TYPE minimum ( const Manifold<TYPE>& M );
template<class TYPE> void normalize( TYPE low, TYPE high, Manifold<TYPE>& M );

template<class TYPE> Manifold<TYPE> downsample ( const intPoint&, const Manifold<TYPE> & );


template<class TYPE>
void normalize( TYPE low, TYPE high, BaseManifold* M )
{
	int i;	
	double oldhigh = maximum ( M );
	double oldlow = minimum ( M );
	double stretch;
	int size=M->size();

	if  ( oldhigh == oldlow ) // unlikely, but we want to avoid div by 0.
		stretch = 0;
	else
		stretch = (double)( high - low ) / ( oldhigh - oldlow );

	for ( i = 0; i<size; ++i)
		M->setElementDouble ( i, ( M->getElementDouble(i) - oldlow ) * stretch + low );

}



template<class TYPE>
TYPE maximum ( const Manifold<TYPE>& M  )
{
	TYPE max = M[0];
	int i;
	for ( i = 0; i < M.size(); ++i )
		if ( M[i] > max )
			max = M[i];

	return max;
}

template<class TYPE>
TYPE minimum ( const Manifold<TYPE>& M )
{
	TYPE min = M[0];
	int i;
	for ( i = 0; i < M.size(); ++i )
		if ( M[i] < min )
			min = M[i];

	return min;
}

template<class TYPE>
void normalize( TYPE low, TYPE high, Manifold<TYPE>& M )
{
	int i;	
	TYPE oldhigh = maximum ( M );
	TYPE oldlow = minimum ( M );
	double stretch;
	int size=M.size();

	if  ( oldhigh == oldlow )
		stretch = 0;
	else
		stretch = (double)( high - low ) / ( oldhigh - oldlow );

	for ( i = 0; i<size; ++i)
		M[i] = (TYPE)( ( M[i] - oldlow ) * stretch + low );

}




template<class TYPE> 
Manifold<TYPE> downsample ( const intPoint& step, const Manifold<TYPE> & M)
{
	intPoint ex;
	ex.x = M.width() / (step.x>0 ? step.x : 1 );
	ex.y = M.height() / (step.y>0 ? step.y : 1 );
	ex.z = M.depth() / (step.z>0 ? step.z : 1 );
	ex.t = M.timepoints() / (step.t>0 ? step.t : 1 );

	Manifold<TYPE> result(ex);
	intPoint count;
	count.x = count.y = count.z = count.t = 0;

	for ( count.t = 0; count.t < ex.t; ++count.t )
		for ( count.z = 0; count.z < ex.z; ++ count.z )
			for ( count.y = 0; count.y < ex.y; ++count.y )
				for ( count.x = 0; count.x < ex.x; ++count.x )
					result.setElement ( count, M.getElement 
							( count.x * step.x, count.y * step.y, count.z * step.z, count.t * step.t ) );

	return result;

}


} // namespace RUMBA

