#ifndef RUMBA_ORIENTATION_H
#define RUMBA_ORIENTATION_H
/** 
  * interface for change of co-ordinates callback
  */

#include <rumba/point.hpp>
#include <rumba/boundingbox.h>


namespace RUMBA
{
class PointTransformImpl;
class PointTransform
{
public:
	PointTransform();
	PointTransform(const PointTransform&);
	PointTransform& operator=(const PointTransform&);

	/**
	 * It's virtual,but don't subclass this unless there's a very good reason 
	 * for it. Subclass PointTransform_impl	instead.
	 */
	virtual ~PointTransform(); 

	/**
	  * Evaluate the transform at a given intPoint x, and return the resulting
	  * point
	  */
	intPoint operator() (const intPoint&);

	/**
	  * Evaluate the transform at a given intPoint x, and return by 
	  * reference via y 
	  */
	void operator() (const intPoint& x, intPoint& y);

	/**
	  * Evaluate the transform, but ignore parity flags, applying permutation
	  * only
	  */
	intPoint map_extent (const intPoint&);
	BoundingBox operator() (const BoundingBox&);


	/**
	 * return the composite transform h(x) = right(left(x))
	 */
	static PointTransform 
		compose(const PointTransform& left, const PointTransform& right);
	/**
	 *  Return the inverse of a transform
	 */
	PointTransform invert() const;

	PointTransform(PointTransformImpl*);

	/**
	  * Print debugging information
	  */
	void debug();


private:
	PointTransformImpl* impl;
};



typedef char pdir_t ;

pdir_t patient_left();
pdir_t patient_right();
pdir_t patient_head();
pdir_t patient_foot();
pdir_t patient_back();
pdir_t patient_front();

class OrientationImpl;
class Orientation
{
public:
	/**
	 *  Same as 
	 *  Orientation(patient_right(),patient_front(),patient_foot())
	 */
	Orientation(); 
	/**
	 *	o1: zero of first co-ordinate. Eg if o1 is patient_left(), the 
	 *	first co-ordinate is the axis that points from the patient left to
	 *	the patient right
	 *	o2: zero of second co-ordinate
	 *	o3: zero of third co-ordinate
	 *	xyzt: true if data is laid out in xyzt form, false if it's txyz.
	 */
	Orientation(pdir_t o1,pdir_t o2,pdir_t o3,const intPoint&,bool xyzt=true);
	Orientation(const Orientation&);
	Orientation& operator=(const Orientation&);
	virtual ~Orientation();

	/**
	  * Return character representation of orientation
	  */
	char toChar() const; 

	/**
	  * Return orientation field x.
	  */
	int operator[](int x) const;
	PointTransform transform(const Orientation&) const;

	/**
	  * return the extent
	  */
	RUMBA::intPoint extent() const;

	/**
	  * assign an extent
	  */

	void setExtent(const RUMBA::intPoint&);


private:
	OrientationImpl* impl;
};

/**
  * Returns compose(t2,t1.invert() ) where t1 is a transform from o1 to some
  * reference orientation O, and t2 is a transform from o2 to O.
  *
  * The resulting transform is from o1 to o2.
  */
inline PointTransform transform(const Orientation& o1, const Orientation& o2 )
{
	return o1.transform(o2);
}

/* transform a particular volume from in to out */
void volume_transform 
( 
 const RUMBA::BaseManifold& in, int in_vol, 
 RUMBA::BaseManifold& out, int out_vol
 );

/* transform volumes one at a time */
void volume_transform( const RUMBA::BaseManifold& in, RUMBA::BaseManifold& out);

} // namespace RUMBA



#endif
