#ifndef MANIFOLD_H
#define MANIFOLD_H


/**
  * \file manifold.h
  * \author Donovan Rebbechi, Ben Bly, Giorgio Grasso
  * Copyright Ben Bly
  * This file is released under the artistic license. See the file
  * COPYING for details.
  */



#define USE_EXCEPTIONS
#define USE_RUMBA_CAST

#include <string>
#include <map>
#include <list>

using std::string;

#include <rumba/baseManifold.h>
#include <rumba/orientation.h>





namespace RUMBA
{
	template <class TYPE> class Manifold;
}

	template <class TYPE>
	RUMBA::Manifold<TYPE> operator+
		(const RUMBA::Manifold<TYPE>&, const RUMBA::Manifold<TYPE>&);
	template<class TYPE>
	RUMBA::Manifold<TYPE> operator+
		(const RUMBA::Manifold<TYPE>&, const TYPE );
	template<class TYPE>
	RUMBA::Manifold<TYPE> operator +
		(const TYPE, const RUMBA::Manifold<TYPE>&);

	template<class TYPE>
	RUMBA::Manifold<TYPE> operator-
		(const RUMBA::Manifold<TYPE>&, const RUMBA::Manifold<TYPE>&);
	template<class TYPE>
	RUMBA::Manifold<TYPE> operator-
		(const RUMBA::Manifold<TYPE>&, const TYPE);
	template<class TYPE>
	RUMBA::Manifold<TYPE> operator-
		(const TYPE, const RUMBA::Manifold<TYPE>&);

	template<class TYPE>
	RUMBA::Manifold<TYPE> operator *
		(const RUMBA::Manifold<TYPE>&, const RUMBA::Manifold<TYPE>&);
	template<class TYPE>
	RUMBA::Manifold<TYPE> operator *
		(const RUMBA::Manifold<TYPE>&, const TYPE);
	template<class TYPE>
	RUMBA::Manifold<TYPE> operator *
		(const TYPE, const RUMBA::Manifold<TYPE>&);

	template<class TYPE>
	RUMBA::Manifold<TYPE> operator /
		(const RUMBA::Manifold<TYPE>&, const RUMBA::Manifold<TYPE>&);
	template<class TYPE>
	RUMBA::Manifold<TYPE> operator /
		(const RUMBA::Manifold<TYPE>&, const TYPE);
	template<class TYPE>
	RUMBA::Manifold<TYPE> operator /
		(const TYPE, const RUMBA::Manifold<TYPE>&);


namespace RUMBA {

/**
	Manifold is a class used to represent neuro-imaging data. 
	It also facilitates the loading and saving of data in many 
	different formats.
	The main content of the class is a 4 dimensional data set.

	There are two ways that one might want to access the points of a 
	manifold: the more intuitive way is to use (x,y,z,t) coordinates.
	Another viewpoint is that each point has an index corresponding to
	an ordering on the points -- that is, you can view it as a 1 dimensional
	array, where (x,y,z,t) corresponds to 
	x + y(width) + z(width)(height) + t ( width )(height)(depth)
	Using a single index to reference elements
	makes more sense if you want to iterate over the set of points
	in the manifold.

	To access points of the Manifold by the (x,y,z,t) system, you use set
	and getElement methods. To access via the raw index, use operator[].

	The Manifold class also facilitates sub Manifold extraction / drop-in.
	these operations are supported via set, get and getVolume methods.

	There are also several arithmatic operations on manifolds. These 
	are just pointwise operations.

	This discussion would not be complete without a discussion of the 
	loading and saving which is useful in itself. These operations 
	facilitate loading and saving files from different formats. 
	The load and save methods decide appropriate formats from the filename
	given.	
	
	@short A class to represent neuroimaging data.
	@author Donovan Rebbechi, Giorgio Grasso, Ben Bly
	
*/


template <class TYPE>
class Manifold : public BaseManifold {


public:
	typedef TYPE* iterator;
	typedef const TYPE* const_iterator;

	/* 	Constructors/Destructors	*/
/**
*	Default constructor. 
*/
	Manifold();
/**
*	Constructs a Manifold from the file. The format of the file is determined
*	from the filename ( in particular, the extension of the filename ).
*/
	Manifold(std::string filename);
/**
*	Initialise a Manifold and specify the width/height.
*/
	Manifold(const intPoint& e);
/**
  *  Initialise a manifold with a given orientation. The orientation must 
  *  include extents 
  */
	Manifold(const Orientation& o);

/**
*	Copy constructor. NOTE!!! This does *NOT* copy the data, which is reference
	counted (and *not* copy-on-write). If you want to
	copy the data, use the copy() method.
*/
	Manifold(const Manifold<TYPE>& manifold);

/**
  * Deep copy.
  */
	Manifold copy() const;
/**
*	Conversion constructor
*/
	template<class TYPEa> Manifold(const Manifold<TYPEa>& manifold);
	~Manifold();



	virtual void loadData ( ManifoldFile* f);

	virtual void print(std::ostream& stream = std::cout) const;

	/*------------------------------------------*/
	/*			Accessor Methods 				*/
	/*------------------------------------------*/

/**
*	Return the underlying data array.
*
*/
	const TYPE *data() const;

/**
*	Get a point in the manifold, referenced by coordinates (x,y,z,t) 
*/
	TYPE getElement(int x, int y, int y, int t) const;
/**
*	Get a point in the manifold, referenced by offset
*/
	TYPE getElement(int pos) const;
/**
*	Get a point in the manifold, referenced by point in 4d co-ordinates
*/	
	TYPE getElement(const intPoint&) const;

/**
  * Similar to above, converts value to an integer. Usually this is called
  * indirectly via BaseManifold pointers and references.
  */
	virtual int getElementInt ( int pos ) const;

  //! Similar to above
	virtual int getElementInt ( int x, int y, int z, int t ) const;

  //! Similar to above
	virtual int getElementInt ( const intPoint& p) const;

  //! Similar to above
	virtual double getElementDouble ( int pos ) const;

  //! Similar to above
	virtual double getElementDouble ( int x, int y, int z, int t ) const;

  //! Similar to above
	virtual double getElementDouble ( const intPoint& p ) const;


/**
*	Get a point referenced by index in undrlying array. Note that this
*	returns a reference, so you can use this to assign.
*/
	TYPE& operator[] (int pos);
/**
*	This function gets called if you have a constant Manifold.
*/
	const TYPE& operator[] ( int pos ) const;

/**
*	Get the three dimensional sub Manifold at timepoint pos.
*/
	Manifold<TYPE> getVolume (int pos) const;

/**
*	Retrieve a sub Manifold. The origin is the corner of the Manifold,
*	the "extents" array corresponds to the dimensions of the Manifold
*	you want to retrieve.
*/
	virtual Manifold* 
		get(const intPoint& origin, const intPoint& extents) const;
/**
*	Similar to the above method.
*/
	virtual Manifold* 
		get(int x,int y,int z,int t, int dx, int dy, int dz, int dt) const;

/** 
*	Set coordinate (x,y,z,t ) to val.
*/
	void setElement(int x, int y, int y, int t, TYPE val);

//! Similar to above
	void setElement(int pos, TYPE val);

//! Similar to above
	void setElement(const intPoint&, TYPE val);


	Manifold& fill (const TYPE);


/**
*	This method is primarily for BaseManifolds where you don't know what
*	the underlying data type is.
*/
	virtual void setElementDouble(int pos, double val );

	//! Similar to above
	virtual void setElementDouble(int x,int y, int z, int t, double val );

	//! Similar to above
	virtual void setElementDouble(const intPoint& p, double val);
/**
*	This method is primarily for BaseManifolds where you don't know what
*	the underlying data type is.
*/
	virtual void setElementInt(int pos, int val );

	//! Similar to above
	virtual void setElementInt(int x,int y, int z, int t, int val );

	//! Similar to above
	virtual void setElementInt(const intPoint& p, int val);

/*
*	overwrite points beginning at pos with manifold.
*/
	void set(Manifold<TYPE>& manifold, int pos);
	template<class TYPEa> void set(Manifold<TYPEa>& manifold, int pos);
/*
*	overwrite points beginning at (x,y,z,t) with manifold.
*/
	void set(Manifold<TYPE>& manifold, int x, int y, int z, int t);
	template<class TYPEa> void set(Manifold<TYPEa>& manifold, int x, int y, int z, int t);

/*
 * 	Overwrite at given point
 */
	void set(Manifold<TYPE>& manifold, const intPoint& p);
	template<class TYPEa> void set(Manifold<TYPEa>& manifold, const intPoint& p);

	void threshold(const TYPE val);
	Manifold* crop(const TYPE val);


/*
 *	Random access iterators. 
 *
 */
	iterator begin() { return DataArray; }
	iterator end() { return DataArray+size(); }

	const_iterator begin() const { return DataArray; }
	const_iterator end() const { return DataArray+size(); }

	/*------------------------------------------------------------------*/
	/*			Operators												*/
	/*	These are just pointwise operations.							*/
	/*------------------------------------------------------------------*/
	Manifold& operator = (const Manifold& manifold);
	template<class TYPEa> Manifold& operator = (const Manifold<TYPEa>& manifold);

//	Removed to avoid template ambiguities. Use fill instead.
//	Manifold& operator = (const TYPE);
//	template<class TYPEa> Manifold& operator = (const TYPEa);






	Manifold& operator+= (const Manifold& manifold);
	Manifold& operator+= (const TYPE val);

	Manifold& operator-= (const Manifold& manifold);
	Manifold& operator-= (const TYPE val);

	Manifold& operator*= (const Manifold& manifold);
	Manifold& operator*= (const TYPE val);

	Manifold& operator/= (const Manifold& manifold);
	Manifold& operator/= (const TYPE val);

	/**
	  * Read raw data from a stream, and write to the elements with offsets 
	  * between start and start+nelts-1
	  * Note that the data in the stream is assumed to have the same 
	  * endian-ness as the host.
	  */
	virtual void readFromStream ( std::istream&, int start, int nelts );

	/**
	  * Write raw data to a stream, and write to the elements with offsets 
	  * between 
	  * start and start+nelts-1
	  * Note that the data in the stream is assumed to have the same 
	  * endian-ness as the host.
	  */	
	virtual void writeToStream ( std::ostream&, int start, int nelts ) const;

	/**
	  * Self-check. Increments PASS/FAIL/COUNT appropriately.
	  */
	static void test(int& PASS, int& FAIL, int& COUNT);

protected:
	virtual void allocate(const intPoint& ext);
	

private:

//	int getDataType() const;
	std::string getDataString() const;

	/*--------------------------------------------------*/
	/*				Data Members						*/
	/*--------------------------------------------------*/
	TYPE	*DataArray;
	mutable unsigned long * RefCount;

	void saveData(ManifoldFile* f) const;

	// for the reference count
	void release();


};
} // namespace RUMBA

#include <rumba/manifold.cc>



#endif
