/***************************************************************************
 * RT2400 SourceForge Project - http://rt2400.sourceforge.net              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 *                                                                         *
 *   Licensed under the GNU GPL                                            *
 *   Original code supplied under license from RaLink Inc, 2003.           *
 ***************************************************************************/

 /***************************************************************************
 *      Module Name: sync.c
 *
 *      Abstract:
 *
 *      Revision History:
 *      Who             When            What
 *      --------        -----------     -----------------------------
 *      MarkW           9th  Feb 04     Baseline of code
 *      defekt          13th Feb 04     ATOMIC kmalloc fix
 ***************************************************************************/


#include "rt_config.h"

/*
    ==========================================================================
    Description:
        The sync state machine,
    Parameters:
        Sm - pointer to the state machine
    Note:
        the state machine looks like the following

    Column 1-2
                        SYNC_IDLE                       JOIN_WAIT_BEACON
    MT2_MLME_SCAN_REQ   mlme_scan_req_action            invalid_state_when_scan
    MT2_MLME_JOIN_REQ   mlme_join_req_action            invalid_state_when_join
    MT2_MLME_START_REQ  mlme_start_req_action           invalid_state_when_start
    MT2_PEER_BEACON     peer_beacon                     peer_beacon_at_join_wait_beacon_action
    MT2_PEER_PROBE_RSP  peer_beacon                     drop
    MT2_PEER_ATIM       drop                            drop
    MT2_SCAN_TIMEOUT    Drop                            Drop
    MT2_BEACON_TIMEOUT  Drop                            beacon_timeout_at_join_wait_beacon_action
    MT2_ATIM_TIMEOUT    Drop                            Drop
    MT2_PEER_PROBE_REQ  ????                            drop

    column 3
                         SCAN_LISTEN
    MT2_MLME_SCAN_REQ    invalid_state_when_scan
    MT2_MLME_JOIN_REQ    invalid_state_when_join
    MT2_MLME_START_REQ   invalid_state_when_start
    MT2_PEER_BEACON      peer_beacon_at_scan_action
    MT2_PEER_PROBE_RSP   peer_probe_rsp_at_scan_action
    MT2_PEER_ATIM        drop
    MT2_SCAN_TIMEOUT     scan_timeout_action
    MT2_BEACON_TIMEOUT   Drop
    MT2_ATIM_TIMEOUT     Drop
    MT2_PEER_PROBE_REQ   drop
    ==========================================================================
 */
VOID SyncStateMachineInit(
    IN PRTMP_ADAPTER pAd,
    IN STATE_MACHINE *Sm,
    OUT STATE_MACHINE_FUNC Trans[])
{
    StateMachineInit(Sm, (STATE_MACHINE_FUNC*)Trans, MAX_SYNC_STATE, MAX_SYNC_MSG, (STATE_MACHINE_FUNC)Drop, SYNC_IDLE, SYNC_MACHINE_BASE);

    // column 1
    StateMachineSetAction(Sm, SYNC_IDLE, MT2_MLME_SCAN_REQ, (STATE_MACHINE_FUNC)MlmeScanReqAction);
    StateMachineSetAction(Sm, SYNC_IDLE, MT2_MLME_JOIN_REQ, (STATE_MACHINE_FUNC)MlmeJoinReqAction);
    StateMachineSetAction(Sm, SYNC_IDLE, MT2_MLME_START_REQ, (STATE_MACHINE_FUNC)MlmeStartReqAction);
    StateMachineSetAction(Sm, SYNC_IDLE, MT2_PEER_BEACON, (STATE_MACHINE_FUNC)PeerBeacon);
//  StateMachineSetAction(Sm, SYNC_IDLE, MT2_PEER_PROBE_RSP, (STATE_MACHINE_FUNC)PeerBeacon);
    StateMachineSetAction(Sm, SYNC_IDLE, MT2_PEER_PROBE_REQ, (STATE_MACHINE_FUNC)PeerProbeReqAction);

    //column 2
    StateMachineSetAction(Sm, JOIN_WAIT_BEACON, MT2_MLME_SCAN_REQ, (STATE_MACHINE_FUNC)InvalidStateWhenScan);
    StateMachineSetAction(Sm, JOIN_WAIT_BEACON, MT2_MLME_JOIN_REQ, (STATE_MACHINE_FUNC)InvalidStateWhenJoin);
    StateMachineSetAction(Sm, JOIN_WAIT_BEACON, MT2_MLME_START_REQ, (STATE_MACHINE_FUNC)InvalidStateWhenStart);
    StateMachineSetAction(Sm, JOIN_WAIT_BEACON, MT2_PEER_BEACON, (STATE_MACHINE_FUNC)PeerBeaconAtJoinAction);
    StateMachineSetAction(Sm, JOIN_WAIT_BEACON, MT2_BEACON_TIMEOUT, (STATE_MACHINE_FUNC)BeaconTimeoutAtJoinAction);

    // column 3
    StateMachineSetAction(Sm, SCAN_LISTEN, MT2_MLME_SCAN_REQ, (STATE_MACHINE_FUNC)InvalidStateWhenScan);
    StateMachineSetAction(Sm, SCAN_LISTEN, MT2_MLME_JOIN_REQ, (STATE_MACHINE_FUNC)InvalidStateWhenJoin);
    StateMachineSetAction(Sm, SCAN_LISTEN, MT2_MLME_START_REQ, (STATE_MACHINE_FUNC)InvalidStateWhenStart);
    StateMachineSetAction(Sm, SCAN_LISTEN, MT2_PEER_BEACON, (STATE_MACHINE_FUNC)PeerBeaconAtScanAction);
    StateMachineSetAction(Sm, SCAN_LISTEN, MT2_PEER_PROBE_RSP, (STATE_MACHINE_FUNC)PeerBeaconAtScanAction);
    StateMachineSetAction(Sm, SCAN_LISTEN, MT2_SCAN_TIMEOUT, (STATE_MACHINE_FUNC)ScanTimeoutAction);

    // initialize the timer
    init_timer(&pAd->Mlme.SyncAux.BeaconTimer);
    pAd->Mlme.SyncAux.BeaconTimer.data = (unsigned long)pAd;
    pAd->Mlme.SyncAux.BeaconTimer.function = &BeaconTimeout;

    init_timer(&pAd->Mlme.SyncAux.ScanTimer);
    pAd->Mlme.SyncAux.ScanTimer.data = (unsigned long)pAd;
    pAd->Mlme.SyncAux.ScanTimer.function = &ScanTimeout;
}

/*
    ==========================================================================
    Description:
        Becaon timeout handler, executed in timer thread
    ==========================================================================
 */
VOID BeaconTimeout(
    IN unsigned long data)
{
    RTMP_ADAPTER *pAd = (RTMP_ADAPTER *)data;

    DBGPRINT(RT_DEBUG_TRACE,"SYNC - BeaconTimeout\n");
    MlmeEnqueue(&pAd->Mlme.Queue, SYNC_STATE_MACHINE, MT2_BEACON_TIMEOUT, 0, NULL);
    MlmeHandler(pAd);
}


/*
    ==========================================================================
    Description:
        ATIM timeout handler, executed in timer thread
    ==========================================================================
 */
VOID AtimTimeout(
    IN unsigned long data)
{
    RTMP_ADAPTER *pAd = (RTMP_ADAPTER *)data;

    DBGPRINT(RT_DEBUG_TRACE,"SYNC - AtimTimeout \n");
    MlmeEnqueue(&pAd->Mlme.Queue, SYNC_STATE_MACHINE, MT2_ATIM_TIMEOUT, 0, NULL);
    MlmeHandler(pAd);
}

/*
    ==========================================================================
    Description:
        Scan timeout handler, executed in timer thread
    ==========================================================================
 */
VOID ScanTimeout(
    IN unsigned long data)
{
    RTMP_ADAPTER *pAd = (RTMP_ADAPTER *)data;

    DBGPRINT(RT_DEBUG_INFO,"SYNC - Scan Timeout \n");
    MlmeEnqueue(&pAd->Mlme.Queue, SYNC_STATE_MACHINE, MT2_SCAN_TIMEOUT, 0, NULL);
    MlmeHandler(pAd);
}

/*!
 *  \brief MLME Scan Request Procedure
 *  \param p_frame a pointer to a FrameDesc
 *  \return a pointer to FrameDesc, usually not used
 *  \pre
 *  \post
 */
VOID MlmeScanReqAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    UCHAR          Ssid[MAX_LEN_OF_SSID], SsidLen, ScanType, BssType;
    ULONG		   Now;

    // first check the parameter sanity
    if (ScanReqParmSanity(pAd,
                          Elem->Msg,
                          Elem->MsgLen,
                          &BssType,
                          Ssid,
                          &SsidLen,
                          &ScanType))
    {
        DBGPRINT(RT_DEBUG_TRACE, "SYNC - MlmeScanReqAction\n");
		Now = jiffies;
		pAd->PortCfg.LastScanTime = Now;
        // reset all the timers
        del_timer_sync(&pAd->Mlme.SyncAux.BeaconTimer);
        del_timer_sync(&pAd->Mlme.SyncAux.ScanTimer);

        // record desired BSS parameters
        pAd->Mlme.SyncAux.BssType = BssType;
        pAd->Mlme.SyncAux.ScanType = ScanType;
        pAd->Mlme.SyncAux.SsidLen = SsidLen;
        memcpy(pAd->Mlme.SyncAux.Ssid, Ssid, SsidLen);

        // start from the first channel
        pAd->Mlme.SyncAux.Channel = FirstChannel(pAd);
        ScanNextChannel(pAd);
    }
    else
    {
        KPRINT(KERN_ERR, "SYNC - MlmeScanReqAction() sanity check fail. BUG!!!\n");
        pAd->Mlme.SyncMachine.CurrState = SYNC_IDLE;
        MlmeCntlConfirm(pAd, MT2_SCAN_CONF, MLME_INVALID_FORMAT);
    }
}

/*
    ==========================================================================
    Description:
        Join req state machine procedure
    Parameters:
        Elem - containing mlme_join_req
    ==========================================================================
 */
VOID MlmeJoinReqAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    BSS_ENTRY    *pBss;
    MLME_JOIN_REQ_STRUCT *Info = (MLME_JOIN_REQ_STRUCT *)(Elem->Msg);

    DBGPRINT(RT_DEBUG_TRACE, "SYNC - MlmeJoinReqAction(BSS #%d)\n", Info->BssIdx);

    // reset all the timers
    del_timer_sync(&pAd->Mlme.SyncAux.ScanTimer);
    del_timer_sync(&pAd->Mlme.SyncAux.BeaconTimer);

    pBss = &pAd->Mlme.CntlAux.SsidBssTab.BssEntry[Info->BssIdx];

    // record the desired SSID & BSSID we're waiting for
    COPY_MAC_ADDR(&pAd->Mlme.SyncAux.Bssid, &pBss->Bssid);
    memcpy(pAd->Mlme.SyncAux.Ssid, pBss->Ssid, pBss->SsidLen);
    pAd->Mlme.SyncAux.SsidLen = pBss->SsidLen;

    // switch channel and waiting for beacon timer
    AsicSwitchChannel(pAd, pBss->Channel);
    AsicLockChannel(pAd, pBss->Channel);
    DBGPRINT(RT_DEBUG_TRACE, "SYNC - Switch to channel %d, SSID %s \n", pBss->Channel, pAd->Mlme.SyncAux.Ssid);
    DBGPRINT(RT_DEBUG_TRACE, "SYNC - Wait BEACON from %02x:%02x:%02x:%02x:%02x:%02x ...\n",
        pAd->Mlme.SyncAux.Bssid.Octet[0], pAd->Mlme.SyncAux.Bssid.Octet[1],
        pAd->Mlme.SyncAux.Bssid.Octet[2], pAd->Mlme.SyncAux.Bssid.Octet[3],
        pAd->Mlme.SyncAux.Bssid.Octet[4], pAd->Mlme.SyncAux.Bssid.Octet[5]);
    pAd->Mlme.SyncAux.BeaconTimer.expires = jiffies + (JOIN_TIMEOUT * HZ)/1000;
    add_timer(&pAd->Mlme.SyncAux.BeaconTimer);	// in mSec

    pAd->Mlme.SyncMachine.CurrState = JOIN_WAIT_BEACON;
}

/*!
 *  \brief Start Req state machine procedure, starting an IBSS
 */
VOID MlmeStartReqAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    UCHAR         Ssid[MAX_LEN_OF_SSID], SsidLen;

#ifdef	SINGLE_ADHOC_LINKUP
	ULONG	Bssidx;
	BOOLEAN CfExist = FALSE;
	CF_PARM CfParm;
#endif

    if (StartParmSanity(pAd, Elem->Msg, Elem->MsgLen, Ssid, &SsidLen))
    {
        DBGPRINT(RT_DEBUG_TRACE, "SYNC - MlmeStartReqAction\n");

        // reset all the timers
        del_timer_sync(&pAd->Mlme.SyncAux.ScanTimer);
        del_timer_sync(&pAd->Mlme.SyncAux.BeaconTimer);

        // PortCfg.PrivacyInvoked should have been set via OID_802_11_WEP_STATUS.
        // pAd->PortCfg.PrivacyInvoked = FALSE;

        memcpy(pAd->PortCfg.Ssid, Ssid, SsidLen);
        pAd->PortCfg.SsidLen           = SsidLen;
        pAd->PortCfg.BssType           = BSS_INDEP;
        pAd->PortCfg.CapabilityInfo    = CAP_GENERATE(0,1,0,0,pAd->PortCfg.PrivacyInvoked);
        pAd->PortCfg.BeaconPeriod      = pAd->PortCfg.IbssConfig.BeaconPeriod;
        pAd->PortCfg.AtimWin           = pAd->PortCfg.IbssConfig.AtimWin;
        pAd->PortCfg.Channel           = pAd->PortCfg.IbssConfig.Channel;
        pAd->PortCfg.SupportedRatesLen = pAd->PortCfg.IbssConfig.SupportedRatesLen;
        pAd->PortCfg.SupportedRates[0] = pAd->PortCfg.IbssConfig.SupportedRates[0];
        pAd->PortCfg.SupportedRates[1] = pAd->PortCfg.IbssConfig.SupportedRates[1];
        pAd->PortCfg.SupportedRates[2] = pAd->PortCfg.IbssConfig.SupportedRates[2];
        pAd->PortCfg.SupportedRates[3] = pAd->PortCfg.IbssConfig.SupportedRates[3];

        pAd->PortCfg.Pss = PWR_ACTIVE;

        // generate a radom number as BSSID
        MacAddrRandomBssid(pAd, &pAd->PortCfg.Bssid);
        AsicSetBssid(pAd, &pAd->PortCfg.Bssid);
        AsicSwitchChannel(pAd, pAd->PortCfg.Channel);
        AsicLockChannel(pAd, pAd->PortCfg.Channel);

#ifdef	SINGLE_ADHOC_LINKUP
		// Add itself as the entry within BSS table
		Bssidx = BssTableSearch(&pAd->PortCfg.BssTab, &pAd->PortCfg.Bssid);
		if (Bssidx == BSS_NOT_FOUND)
		{
			Bssidx = BssTableSetEntry(pAd, &pAd->PortCfg.BssTab, &pAd->PortCfg.Bssid,
				Ssid, SsidLen, pAd->PortCfg.BssType, pAd->PortCfg.BeaconPeriod,
				CfExist, &CfParm, pAd->PortCfg.AtimWin, pAd->PortCfg.CapabilityInfo,
				pAd->PortCfg.SupportedRates, pAd->PortCfg.SupportedRatesLen,
				pAd->PortCfg.Channel, Elem->Rssi);
		}
#endif
        pAd->Mlme.SyncMachine.CurrState = SYNC_IDLE;
        MlmeCntlConfirm(pAd, MT2_START_CONF, (USHORT)MLME_SUCCESS);
    }
    else
    {
        KPRINT(KERN_ERR, "SYNC - MlmeStartReqAction() sanity check fail. BUG!!!\n");
        pAd->Mlme.SyncMachine.CurrState = SYNC_IDLE;
        MlmeCntlConfirm(pAd, MT2_START_CONF, MLME_INVALID_FORMAT);
    }
}

/*!
 *  \brief peer sends beacon back when scanning
 */
VOID PeerBeaconAtScanAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    MACADDR         Bssid, Addr2;
    UCHAR           Ssid[MAX_LEN_OF_SSID], BssType, Channel, Rates[MAX_LEN_OF_SUPPORTED_RATES], RatesLen,
                    SsidLen, DtimCount, DtimPeriod, BcastFlag, MessageToMe, Legacy;
    CF_PARM         CfParm;
    USHORT          BeaconPeriod, AtimWin, CapabilityInfo;
    MACFRAME       *Fr;
    LARGE_INTEGER   TimeStamp;
    BOOLEAN         CfExist = FALSE;

    // memset(Ssid, 0x00, MAX_LEN_OF_SSID);
    Fr = (MACFRAME *) Elem->Msg;
    if (BeaconAndProbeRspSanity(pAd,
                                Elem->Msg,
                                Elem->MsgLen,
                                &Addr2,
                                &Bssid, Ssid,
                                &SsidLen,
                                &BssType,
                                &BeaconPeriod,
                                &Channel,
                                &TimeStamp,
                                &CfExist,
                                &CfParm,
                                &AtimWin,
                                &CapabilityInfo,
                                Rates,
                                &RatesLen,
                                &DtimCount,
                                &DtimPeriod,
                                &BcastFlag,
                                &MessageToMe,
                                &Legacy))
    {
        ULONG Idx;
        UCHAR Rssi = 0;

        // This correct im-proper RSSI indication during SITE SURVEY issue.
        // Always report bigger RSSI during SCANNING when receiving multiple BEACONs from the same AP.
        // This case happens because BEACONs come from adjacent channels, so RSSI become weaker as we
        // switch to more far away channels.
        Idx = BssTableSearch(&pAd->PortCfg.BssTab, &Bssid);
        if (Idx != BSS_NOT_FOUND)
            Rssi = pAd->PortCfg.BssTab.BssEntry[Idx].Rssi;
        if (Elem->Rssi > Rssi)
            Rssi = Elem->Rssi;

        DBGPRINT(RT_DEBUG_INFO, "SYNC - PeerBeaconAtScanAction (Subtype=%d, SsidLen=%d, Ssid=%s)\n", Fr->Hdr.SubType, SsidLen,Ssid);

        BssTableSetEntry(pAd, &pAd->PortCfg.BssTab, &Bssid, Ssid, SsidLen, BssType,
                         BeaconPeriod, CfExist, &CfParm, AtimWin, CapabilityInfo, Rates,
                         RatesLen, Channel, Rssi);
    }
    // sanity check fail, ignored
}


/*
    ==========================================================================
    Description:
        When waiting joining the (I)BSS, beacon received from external
    ==========================================================================
 */
VOID PeerBeaconAtJoinAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    MACADDR       Bssid, Addr2;
    UCHAR         Ssid[MAX_LEN_OF_SSID], SsidLen, BssType, Channel, RatesLen, MessageToMe,
                  Rates[MAX_LEN_OF_SUPPORTED_RATES], DtimCount, DtimPeriod, BcastFlag, Legacy;
    LARGE_INTEGER TimeStamp;
    USHORT        BeaconPeriod, AtimWin, CapabilityInfo;
    CF_PARM       Cf;
    BOOLEAN       CfExist = FALSE;

    if (BeaconAndProbeRspSanity(pAd,
                                Elem->Msg,
                                Elem->MsgLen,
                                &Addr2,
                                &Bssid,
                                Ssid,
                                &SsidLen,
                                &BssType,
                                &BeaconPeriod,
                                &Channel,
                                &TimeStamp,
                                &CfExist,
                                &Cf,
                                &AtimWin,
                                &CapabilityInfo,
                                Rates,
                                &RatesLen,
                                &DtimCount,
                                &DtimPeriod,
                                &BcastFlag,
                                &MessageToMe,
                                &Legacy))
    {
        //if ((MAC_ADDR_EQUAL(&pAd->Mlme.SyncAux.Bssid, &Bssid) &&
        //	(pAd->Mlme.SyncAux.SsidLen == SsidLen) &&
        //  RTMPEqualMemory(pAd->Mlme.SyncAux.Ssid, Ssid, (ULONG) SsidLen)))
		if (MAC_ADDR_EQUAL(&pAd->Mlme.SyncAux.Bssid, &Bssid))
        {
            DBGPRINT(RT_DEBUG_TRACE, "SYNC - receive desired BEACON at JoinWaitBeacon...\n");
            del_timer_sync(&pAd->Mlme.SyncAux.BeaconTimer);

            // Update RSSI to prevent No signal display when cards first initialized
            pAd->PortCfg.LastRssi = Elem->Rssi;

			if (pAd->Mlme.CntlAux.SsidLen > 0)
			{
				memcpy(pAd->PortCfg.Ssid, pAd->Mlme.CntlAux.Ssid, pAd->Mlme.CntlAux.SsidLen);
				pAd->PortCfg.SsidLen = pAd->Mlme.CntlAux.SsidLen;
			}
			else
			{
            memcpy(pAd->PortCfg.Ssid, Ssid, SsidLen);
            pAd->PortCfg.SsidLen = SsidLen;
			}

            COPY_MAC_ADDR(&pAd->PortCfg.Bssid, &Bssid);
            AsicSetBssid(pAd, &pAd->PortCfg.Bssid);

            pAd->PortCfg.BssType = BssType;
            pAd->PortCfg.BeaconPeriod = BeaconPeriod;
            pAd->PortCfg.Channel = Channel;
            //pAd->PortCfg.PrivacyInvoked = CAP_IS_PRIVACY_ON(CapabilityInfo);
            pAd->PortCfg.SupportedRatesLen = RatesLen;
            memcpy(pAd->PortCfg.SupportedRates, Rates, RatesLen);

            // Check for 802.11g information, if 802.11 b /g mixed mode.
            // We can't support its short preamble for now.
           	pAd->PortCfg.CapabilityInfo = CapabilityInfo;

            if ((BssType == BSS_INDEP) && (CAP_IS_IBSS_ON(CapabilityInfo)))
            {
                pAd->PortCfg.AtimWin = AtimWin;
            }
            else if (BssType == BSS_INFRA)
            {
                pAd->PortCfg.CfpPeriod = Cf.CfpPeriod;
                pAd->PortCfg.CfpMaxDuration = Cf.CfpMaxDuration;
                pAd->PortCfg.CfpDurRemain = Cf.CfpDurRemaining;
                pAd->PortCfg.DtimCount = DtimCount;
                pAd->PortCfg.DtimPeriod = DtimPeriod;
                pAd->PortCfg.CfpCount = Cf.CfpCount;
                pAd->PortCfg.CfpPeriod = Cf.CfpPeriod;

                AsicEnableBssSync(pAd);
            }

            pAd->Mlme.SyncMachine.CurrState = SYNC_IDLE;
            MlmeCntlConfirm(pAd, MT2_JOIN_CONF, MLME_SUCCESS);
        }
        // not to me BEACON, ignored
    }
    // sanity check fail, ignore this frame
}

/*
    ==========================================================================
    Description:
        receive BEACON from peer
    Parameters:
        Elem -
    ==========================================================================
 */
VOID PeerBeacon(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    MACADDR       Bssid, Addr2;
    CHAR          Ssid[MAX_LEN_OF_SSID];
    CF_PARM       CfParm;
    UCHAR         SsidLen, MessageToMe=0, BssType, Channel, Rates[MAX_LEN_OF_SUPPORTED_RATES];
    UCHAR         RatesLen, DtimCount=0, DtimPeriod=0, BcastFlag=0, Legacy;
    USHORT        CapabilityInfo, AtimWin, BeaconPeriod;
    LARGE_INTEGER TimeStamp;
    BOOLEAN       CfExist = FALSE;
    USHORT        TbttNumToNextWakeUp;

    if (!INFRA_ON(pAd) && !ADHOC_ON(pAd))
        return;

    if (BeaconAndProbeRspSanity(pAd,
                                Elem->Msg,
                                Elem->MsgLen,
                                &Addr2,
                                &Bssid,
                                Ssid,
                                &SsidLen,
                                &BssType,
                                &BeaconPeriod,
                                &Channel,
                                &TimeStamp,
                                &CfExist,
                                &CfParm,
                                &AtimWin,
                                &CapabilityInfo,
                                Rates,
                                &RatesLen,
                                &DtimCount,
                                &DtimPeriod,
                                &BcastFlag,
                                &MessageToMe,
                                &Legacy))
    {
        BOOLEAN is_my_bssid, is_my_ssid;
        ULONG   Bssidx, Now;
        BSS_ENTRY *pBss;

        is_my_bssid = (MAC_ADDR_EQUAL(&Bssid, &pAd->PortCfg.Bssid) ? TRUE : FALSE);
        is_my_ssid = (((pAd->PortCfg.SsidLen == SsidLen) && RTMPEqualMemory(Ssid, pAd->PortCfg.Ssid, (ULONG) SsidLen)) ? TRUE : FALSE);

        // ignore BEACON not for my SSID
        if ((! is_my_ssid) && (! is_my_bssid))
            return;

        DBGPRINT(RT_DEBUG_INFO, "SYNC - PeerBeacon from %02x:%02x:%02x:%02x:%02x:%02x - Dtim=%d/%d, Rssi=%02x\n",
            Bssid.Octet[0], Bssid.Octet[1], Bssid.Octet[2],
            Bssid.Octet[3], Bssid.Octet[4], Bssid.Octet[5],
            DtimCount, DtimPeriod, Elem->Rssi);

        //
        // Housekeeping "SsidBssTab" table for later-on ROAMing usage.
        //
        Bssidx = BssTableSearch(&pAd->Mlme.CntlAux.SsidBssTab, &Bssid);
        if (Bssidx == BSS_NOT_FOUND)
        {
        	// Return immediately when in transition process when changing association
        	// Found this bug when doing WHQL ad-hoc test case
        	if (pAd->PortCfg.SsidLen != pAd->Mlme.CntlAux.SsidLen)
        		return;
        	if (!RTMPEqualMemory(pAd->PortCfg.Ssid, pAd->Mlme.CntlAux.Ssid, pAd->PortCfg.SsidLen))
        		return;

            // discover new AP of this network, create BSS entry
            Bssidx = BssTableSetEntry(pAd, &pAd->Mlme.CntlAux.SsidBssTab, &Bssid, Ssid, SsidLen,
                        BssType, BeaconPeriod, CfExist, &CfParm, AtimWin, CapabilityInfo,
                        Rates, RatesLen, Channel, Elem->Rssi);

            if (Bssidx == BSS_NOT_FOUND) // return if BSS table full
                return;

            DBGPRINT(RT_DEBUG_TRACE, "SYNC - New AP added to SsidBssTab[%d], RSSI=%d, MAC=%02x:%02x:%02x:%02x:%02x:%02x\n",
                Bssidx, Elem->Rssi, Bssid.Octet[0], Bssid.Octet[1], Bssid.Octet[2],
                Bssid.Octet[3], Bssid.Octet[4], Bssid.Octet[5]);
        }

        Now = jiffies;
        pBss = &pAd->Mlme.CntlAux.SsidBssTab.BssEntry[Bssidx];
        pBss->Rssi = Elem->Rssi;       // lastest RSSI
        pBss->LastBeaconRxTime = Now;   // last RX timestamp

        //
        // BEACON from my BSSID - either IBSS or INFRA network
        //
        if (is_my_bssid)
        {
            // 2002/12/06 - patch Abocom AP bug, which forgets to set "Privacy" bit in
            // AssocRsp even though this bit is ON in Beacon. So we update according
            // to following Beacon frame.
            // pAd->PortCfg.PrivacyInvoked = CAP_IS_PRIVACY_ON(CapabilityInfo);

            // check if RSSI reaches threshold
            pAd->PortCfg.LastBeaconRxTime = Now;
            pAd->PortCfg.LastRssi = (pAd->PortCfg.LastRssi + Elem->Rssi) / 2;
//          pAd->PortCfg.LastRssi = Elem->Rssi;
            if ((pAd->PortCfg.RssiTriggerMode == RSSI_TRIGGERED_UPON_BELOW_THRESHOLD) &&
                (pAd->PortCfg.LastRssi < pAd->PortCfg.RssiTrigger))
            {
                DBGPRINT(RT_DEBUG_TRACE, "SYNC - NdisMIndicateStatus *** RSSI %d dBm, less than threshold %d dBm\n",
                    pAd->PortCfg.LastRssi - RSSI_TO_DBM_OFFSET, pAd->PortCfg.RssiTrigger - RSSI_TO_DBM_OFFSET);
            }
            else if ((pAd->PortCfg.RssiTriggerMode == RSSI_TRIGGERED_UPON_EXCCEED_THRESHOLD) &&
                (pAd->PortCfg.LastRssi > pAd->PortCfg.RssiTrigger))
            {
                DBGPRINT(RT_DEBUG_TRACE, "SYNC - NdisMIndicateStatus *** RSSI %d dBm, greater than threshold %d dBm\n",
                    pAd->PortCfg.LastRssi - RSSI_TO_DBM_OFFSET, pAd->PortCfg.RssiTrigger - RSSI_TO_DBM_OFFSET);
            }

            // only INFRASTRUCTURE mode support power-saving feature
            if (INFRA_ON(pAd) && (pAd->PortCfg.Psm == PWR_SAVE))
            {
                //  1. AP has backlogged unicast-to-me frame, stay AWAKE, send PSPOLL
                //  2. AP has backlogged broadcast/multicast frame and we want those frames, stay AWAKE
                //  3. we have outgoing frames in TxRing or PrioRing, better stay AWAKE
                //  4. Psm change to PWR_SAVE, but AP not been informed yet, we better stay AWAKE
                //  5. otherwise, put PHY back to sleep to save battery.
                if (MessageToMe)
                {
                    DBGPRINT(RT_DEBUG_TRACE, "SYNC - AP backlog unicast-to-me, stay AWAKE, send PSPOLL\n");
                    EnqueuePsPoll(pAd);
                }
                else if (BcastFlag && (DtimCount == 0) && pAd->PortCfg.RecvDtim)
                {
                    DBGPRINT(RT_DEBUG_TRACE, "SYNC - AP backlog broadcast/multicast, stay AWAKE\n");
                }
                else if ((RTMPFreeDescriptorRequest(pAd, TX_RING, TX_RING_SIZE) != NDIS_STATUS_SUCCESS) ||
                    (RTMPFreeDescriptorRequest(pAd, PRIO_RING, PRIO_RING_SIZE) != NDIS_STATUS_SUCCESS))
                {
                    DBGPRINT(RT_DEBUG_TRACE, "SYNC - outgoing frame in TxRing/PrioRing, stay AWAKE\n");
                }
                else
                {
                    USHORT NextDtim = DtimCount;

                    if (NextDtim == 0)
                        NextDtim = DtimPeriod;

                    TbttNumToNextWakeUp = pAd->PortCfg.DefaultListenCount;
                    if (pAd->PortCfg.RecvDtim && (TbttNumToNextWakeUp > NextDtim))
                        TbttNumToNextWakeUp = NextDtim;

                    DBGPRINT(RT_DEBUG_TRACE, "SYNC - PHY sleeps for %d Tbcn\n", TbttNumToNextWakeUp);
                    AsicSleepThenAutoWakeup(pAd, TbttNumToNextWakeUp);
                }
            }

#ifndef	SINGLE_ADHOC_LINKUP
            // At least another peer in this IBSS, declare MediaState as CONNECTED
            if (ADHOC_ON(pAd) && (pAd->MediaState == NdisMediaStateDisconnected))
            {
                pAd->MediaState = NdisMediaStateConnected;

                // 2003/03/12 - john
                // Make sure this entry in "PortCfg.BssTab" table, thus complies to Microsoft's policy that
                // "site survey" result should always include the current connected network.
                //
                Bssidx = BssTableSearch(&pAd->PortCfg.BssTab, &Bssid);
                if (Bssidx == BSS_NOT_FOUND)
                {
                    Bssidx = BssTableSetEntry(pAd, &pAd->PortCfg.BssTab, &Bssid, Ssid, SsidLen,
                                BssType, BeaconPeriod, CfExist, &CfParm, AtimWin, CapabilityInfo,
                                Rates, RatesLen, Channel, Elem->Rssi);
                }
            }
#endif
        }
        // not my BSSID, ignore it
    }
    // sanity check fail, ignore this frame
}

VOID PeerProbeReqAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    MACADDR       Addr2;
    CHAR          Ssid[MAX_LEN_OF_SSID];
    UCHAR         SsidLen, Rates[MAX_LEN_OF_SUPPORTED_RATES], RatesLen;
    MACHDR        ProbeRspHdr;
    UCHAR         *OutBuffer = NULL;
    ULONG         FrameLen = 0;
    LARGE_INTEGER FakeTimestamp;
    UCHAR         SsidIe = IE_SSID, DsIe = IE_DS_PARM, IbssIe = IE_IBSS_PARM, SuppIe = IE_SUPP_RATES,
                  DsLen = 1, IbssLen = 2;

    if (! ADHOC_ON(pAd))
        return;

    if (PeerProbeReqSanity(pAd, Elem->Msg, Elem->MsgLen, &Addr2, Ssid, &SsidLen, Rates, &RatesLen))
    {
        if ((SsidLen == 0) || RTMPEqualMemory(Ssid, pAd->PortCfg.Ssid, (ULONG) SsidLen))
        {
            CSR15_STRUC Csr15;

            // we should respond a ProbeRsp only when we're the last BEACON transmitter
            // in this ADHOC network.
            RTMP_IO_READ32(pAd, CSR15, &Csr15.word);
            if (Csr15.field.BeaconSent == 0)
            {
                DBGPRINT(RT_DEBUG_INFO, "SYNC - NOT last BEACON sender, no PROBE_RSP to %02x:%02x:%02x:%02x:%02x:%02x...\n",
                    Addr2.Octet[0],Addr2.Octet[1],Addr2.Octet[2],Addr2.Octet[3],Addr2.Octet[4],Addr2.Octet[5] );
                return;
            }

            // allocate and send out ProbeRsp frame
            OutBuffer = kmalloc(MAX_LEN_OF_MLME_BUFFER, GFP_ATOMIC);
            if(OutBuffer == NULL)
                return;

            DBGPRINT(RT_DEBUG_INFO, "SYNC - Send PROBE_RSP to %02x:%02x:%02x:%02x:%02x:%02x...\n",
                Addr2.Octet[0],Addr2.Octet[1],Addr2.Octet[2],Addr2.Octet[3],Addr2.Octet[4],Addr2.Octet[5] );
            MgtMacHeaderInit(pAd, &ProbeRspHdr, SUBTYPE_PROBE_RSP, 0, &Addr2, &pAd->PortCfg.Bssid);
            MakeOutgoingFrame(OutBuffer,                        &FrameLen,
                              MAC_HDR_LEN,                      (CHAR *)&ProbeRspHdr,
                              TIMESTAMP_LEN,                    &FakeTimestamp,
                              2,                                &pAd->PortCfg.BeaconPeriod,
                              2,                                &pAd->PortCfg.CapabilityInfo,
                              1,                                &SsidIe,
                              1,                                &pAd->PortCfg.SsidLen,
                              pAd->PortCfg.SsidLen,             pAd->PortCfg.Ssid,
                              1,                                &SuppIe,
                              1,                                &pAd->PortCfg.SupportedRatesLen,
                              pAd->PortCfg.SupportedRatesLen,   pAd->PortCfg.SupportedRates,
                              1,                                &DsIe,
                              1,                                &DsLen,
                              1,                                &pAd->PortCfg.Channel,
                              1,                                &IbssIe,
                              1,                                &IbssLen,
                              2,                                &pAd->PortCfg.AtimWin,
                              NULL);
            MiniportMMRequest(pAd, OutBuffer, FrameLen);
        }
    }
}

VOID BeaconTimeoutAtJoinAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    DBGPRINT(RT_DEBUG_TRACE, "SYNC - BeaconTimeoutAtJoinAction\n");
    pAd->Mlme.SyncMachine.CurrState = SYNC_IDLE;
    MlmeCntlConfirm(pAd, MT2_JOIN_CONF, MLME_REJ_TIMEOUT);
}

/*!
 *  \brief Scan timeout procedure
 *  \note basically add channel index by 1 and rescan
 */
VOID ScanTimeoutAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    pAd->Mlme.SyncAux.Channel = NextChannel(pAd, pAd->Mlme.SyncAux.Channel);
    ScanNextChannel(pAd);
}

/*!
 *  \brief Scan next channel
 *  \param none, all use global variables
 *  \return a pointer to a FrameDesc, for debug only, usually not used
 *  \pre
 *  \post
 */
VOID ScanNextChannel(
    IN PRTMP_ADAPTER pAd)
{
    MACHDR          Hdr;
    UCHAR           SsidIe = IE_SSID, SuppRateIe = IE_SUPP_RATES;
    VOID           *OutBuffer = NULL;
	VOID           *OutBuffer2 = NULL;
    ULONG           FrameLen = 0;
    UCHAR           SsidLen = 0;

    if (pAd->Mlme.SyncAux.Channel == 0)
    {
        DBGPRINT(RT_DEBUG_TRACE, "SYNC - End of SCAN, restore to channel %d\n",pAd->PortCfg.Channel);
        AsicSwitchChannel(pAd, pAd->PortCfg.Channel);
        AsicLockChannel(pAd, pAd->PortCfg.Channel);

        pAd->Mlme.SyncMachine.CurrState = SYNC_IDLE;
        MlmeCntlConfirm(pAd, MT2_SCAN_CONF, MLME_SUCCESS);
    }
    else
    {
        AsicSwitchChannel(pAd, pAd->Mlme.SyncAux.Channel);

        // If no connection currently, then we wait for a little bit longer
        // on each channel. but total SCAN time still limits within 3 sec (DDK
        // constraint). If connection is active, we spend a smaller time on each
        // channel to minimize the out-of-service time.
        // TODO: We need more intelligent rules here to further improve out-of-service issue.
        // e.g. temporary stop copying NDIS packet to TxRing until SCAN complete
        if (INFRA_ON(pAd) || ADHOC_ON(pAd))
        {
        	pAd->Mlme.SyncAux.ScanTimer.expires = jiffies + (pAd->PortCfg.MinChnlTime * HZ)/1000;
        	add_timer(&pAd->Mlme.SyncAux.ScanTimer);	// in mSec
        }
        else
        {
        	pAd->Mlme.SyncAux.ScanTimer.expires = jiffies + (pAd->PortCfg.MaxChnlTime * HZ)/1000;
        	add_timer(&pAd->Mlme.SyncAux.ScanTimer);	// in mSec
        }

        if (pAd->Mlme.SyncAux.ScanType == SCAN_ACTIVE)
        {
            OutBuffer = kmalloc(MAX_LEN_OF_MLME_BUFFER, GFP_ATOMIC);
            if(OutBuffer == NULL)
            {
                DBGPRINT(RT_DEBUG_TRACE, "SYNC - ScanNextChannel() allocate memory fail\n");
                pAd->Mlme.SyncMachine.CurrState = SYNC_IDLE;
                MlmeCntlConfirm(pAd, MT2_SCAN_CONF, MLME_FAIL_NO_RESOURCE);
                return;
            }

			// Allocate another for probe scan with SSID
			OutBuffer2 = kmalloc(MAX_LEN_OF_MLME_BUFFER, GFP_ATOMIC);
			if(OutBuffer2 == NULL)
			{
                DBGPRINT(RT_DEBUG_TRACE, "SYNC - ScanNextChannel() allocate memory fail\n");
                pAd->Mlme.SyncMachine.CurrState = SYNC_IDLE;
                MlmeCntlConfirm(pAd, MT2_SCAN_CONF, MLME_FAIL_NO_RESOURCE);
                return;
            }

            DBGPRINT(RT_DEBUG_INFO, "SYNC - send ProbeReq @ channel=%d...\n", pAd->Mlme.SyncAux.Channel);
            MgtMacHeaderInit(pAd, &Hdr, SUBTYPE_PROBE_REQ, 0, &pAd->PortCfg.Broadcast, &pAd->PortCfg.Broadcast);
            MakeOutgoingFrame(OutBuffer,        &FrameLen,
                              sizeof(MACHDR),   (UCHAR*)&Hdr,
                              1,                &SsidIe,
                              1,                &SsidLen,
                              1,                &SuppRateIe,
                              1,                &pAd->PortCfg.SupportedRatesLen,
                              pAd->PortCfg.SupportedRatesLen, pAd->PortCfg.SupportedRates,
                              NULL);
            MiniportMMRequest(pAd, OutBuffer, FrameLen);
			// make another probe scan with SSID from mlme.cntlaux.ssid
			SsidLen = pAd->Mlme.CntlAux.SsidLen;
			MakeOutgoingFrame(OutBuffer2,        &FrameLen,
							  sizeof(MACHDR),   (UCHAR*)&Hdr,
							  1,                &SsidIe,
							  1,                &SsidLen,
							  SsidLen,			pAd->Mlme.CntlAux.Ssid,
							  1,                &SuppRateIe,
							  1,                &pAd->PortCfg.SupportedRatesLen,
							  pAd->PortCfg.SupportedRatesLen, pAd->PortCfg.SupportedRates,
							  NULL);
			MiniportMMRequest(pAd, OutBuffer2, FrameLen);
        }

        pAd->Mlme.SyncMachine.CurrState = SCAN_LISTEN;
    }
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID InvalidStateWhenScan(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    DBGPRINT(RT_DEBUG_TRACE, "AYNC - InvalidStateWhenScan(state=%d). Reset SYNC machine\n", pAd->Mlme.SyncMachine.CurrState);
    pAd->Mlme.SyncMachine.CurrState = SYNC_IDLE;
    MlmeCntlConfirm(pAd, MT2_SCAN_CONF, MLME_STATE_MACHINE_REJECT);
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID InvalidStateWhenJoin(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    DBGPRINT(RT_DEBUG_TRACE, "InvalidStateWhenJoin(state=%d). Reset SYNC machine\n", pAd->Mlme.SyncMachine.CurrState);
    pAd->Mlme.SyncMachine.CurrState = SYNC_IDLE;
    MlmeCntlConfirm(pAd, MT2_JOIN_CONF, MLME_STATE_MACHINE_REJECT);
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID InvalidStateWhenStart(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    DBGPRINT(RT_DEBUG_TRACE, "InvalidStateWhenStart(state=%d). Reset SYNC machine\n", pAd->Mlme.SyncMachine.CurrState);
    pAd->Mlme.SyncMachine.CurrState = SYNC_IDLE;
    MlmeCntlConfirm(pAd, MT2_START_CONF, MLME_STATE_MACHINE_REJECT);
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID EnqueuePsPoll(
    IN PRTMP_ADAPTER pAd)
{
    PSPOLL_FRAME  *PsFr;

    DBGPRINT(RT_DEBUG_TRACE, "SYNC - send PsPoll ...\n");

    PsFr = kmalloc(MAX_LEN_OF_MLME_BUFFER, GFP_ATOMIC);
    if(PsFr != NULL)
    {
        memcpy((VOID *)PsFr, (VOID *)&pAd->Mlme.PsFr, sizeof(PSPOLL_FRAME));
        MiniportMMRequest(pAd, (VOID *)PsFr, sizeof(PSPOLL_FRAME));
    }

}

VOID EnqueueNullFrame(
    IN PRTMP_ADAPTER pAd)
{
     MACHDR         *NullFr;

    NullFr = kmalloc(MAX_LEN_OF_MLME_BUFFER, GFP_ATOMIC);
    if(NullFr != NULL)
    {
        memcpy((VOID *)NullFr, (VOID *)&pAd->Mlme.NullFr, sizeof(MACHDR));
        RTMPSendNullFrame(pAd, (VOID *)NullFr, sizeof(MACHDR));
    }

}

UCHAR FirstChannel(
    IN PRTMP_ADAPTER pAd)
{
	switch (pAd->PortCfg.CountryRegion)
	{
		case REGION_FCC:	// 1 - 11
			return 1;

		case REGION_IC:		// 1 -11
			return 1;

		case REGION_ETSI:	// 1 - 13
			return 1;

		case REGION_SPAIN:	// 10 - 11
			return 10;

		case REGION_FRANCE:	// 10 -13
			return 10;

		case REGION_MKK:	// 14
			return 14;

		case REGION_MKK1:	// 1 - 14
			return 1;

		case REGION_ISRAEL:	// 3 - 9
			return 3;

		default:			// Error
			return 0;
	}
}

// return 0 if no more next channel
UCHAR NextChannel(
    IN PRTMP_ADAPTER pAd,
    IN UCHAR channel)
{
	switch (pAd->PortCfg.CountryRegion)
	{
		case REGION_FCC:	// 1 - 11
			if ((channel > 0) && (channel < 11))
				return (++channel);
			break;

		case REGION_IC:		// 1 -11
			if ((channel > 0) && (channel < 11))
				return (++channel);
			break;

		case REGION_ETSI:	// 1 - 13
			if ((channel > 0) && (channel < 13))
				return (++channel);
			break;

		case REGION_SPAIN:	// 10 - 11
			if ((channel > 9) && (channel < 11))
				return (++channel);
			break;

		case REGION_FRANCE:	// 10 -13
			if ((channel > 9) && (channel < 13))
				return (++channel);
			break;

		case REGION_MKK:	// 14
			break;

		case REGION_MKK1:	// 1 - 14
			if ((channel > 0) && (channel < 14))
				return (++channel);
			break;

		case REGION_ISRAEL:	// 3 - 9
			if ((channel > 2) && (channel < 9))
				return (++channel);
			break;

		default:			// Error
			return 0;
	}
    return (0);
}

