
(define-class <path-graphic> (<leaf-object>)
  (subpaths type: <vector> init-value: '#())
  (insideness type: <symbol> init-value: 'non-zero))

(define-class <subpath> (<object>)
  (in-path type: <path-graphic>)
  (closed? type: <boolean> init-value: #f)
  (path-points type: <vector> init-value: '#()))

(define-class <path-point> (<object>)
  (in-subpath type: <subpath>)
  (constraint type: <symbol> init-value: 'corner)   ; in (smooth curve corner)
  (position type: <point>)
  (in-handle type: <size> init-value: $zero-size)   ; relative to posn
  (out-handle type: <size> init-value: $zero-size)) ; relative to posn

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   Painting
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (zero-size? s)
  (and (zero? (dx s)) (zero? (dy s))))

(define (pick-on-subpath (self <subpath>) owner pt ctm)
  (pick-on-path owner pt ctm
		(vector->list
		 (vector-map position (path-points self)))))

(define-method pick-list* ((self <path-graphic>) pt ctm)
  (let ((r '()))
    (vector-for-each
     (lambda (sp)
       (let ((psp (pick-on-subpath sp self pt ctm)))
	 (if (not (null? psp))
	     (set! r (append psp r)))))
     (subpaths self))
    r))

(define-method paint-artwork* ((self <path-graphic>) dev)
  (build-path-on-device self dev)
  (stroke dev))

(define-method paint-object* ((self <path-graphic>) dev)
  (with-gstate-saved
   dev
   (lambda ()
     (build-path-on-device self dev)
     ;
     (let ((fc (get-property self 'fill-color #f))
	   (sc (get-property self 'stroke-color #f)))
       (if fc
	   (begin
	     (closepath dev)
	     (setcolor dev (device-color dev fc))
	     (fill dev)))
       (if (not (eq? sc 'none))
	   (begin
	     (if sc
		 (setcolor dev (device-color dev sc)))
	     (stroke dev)))))))

(define (build-path-on-device (self <path-graphic>) dev)
  (vector-for-each
   (lambda ((sp <subpath>))
     (let (((ppv <vector>) (path-points sp)))
       (moveto dev (position (vector-ref ppv 0)))
       (let loop ((i 1)
		  (prev-pp (vector-ref ppv 0)))
	 (if (< i (vector-length ppv))
	     (let ((pp (vector-ref ppv i)))
	       (if (or (not (zero-size? (out-handle prev-pp)))
		       (not (zero-size? (in-handle pp))))
		   (curveto dev
			    (point+ (position prev-pp)
				    (out-handle prev-pp))
			    (point+ (position pp)
				    (in-handle pp))
			    (position pp))
		   (lineto dev (position pp)))
	       (loop (+ i 1) pp))
	     (begin
	       ;; all done with subpath... close it if needed
	       (if (closed? sp)
		   (closepath dev)))))))
   (subpaths self)))

;;; handle IDs are structured as follows:
;;;
;;;                         <--12 bits--> 2 bits
;;;      +-----------------+-------------+----+
;;;      |  subpath #      | path point #|grip|
;;;      +-----------------+-------------+----+
;;;
;;;  where grip = 00 for main point, 01 for in-handle, 10 for out-handle
;;;
 
(define-method accum-handles ((self <path-graphic>) accum)
  (let ((subpath-id 0))
    (vector-for-each
     (lambda (sp)
       (let ((pathpoint-id 0))
	 (vector-for-each
	  (lambda (pp)
	    (accum self (position pp) pathpoint-id)
	    (set! pathpoint-id (+ pathpoint-id #b100)))
	  (path-points sp))
	 (set! subpath-id (+ subpath-id #b100000000000))))
     (subpaths self))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   Dragging the whole and the parts
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-method start-active-drag ((self <path-graphic>)
                                  (in-view <open-view>)
                                  (initial-pt <point>))
  (wm "path-graphic: start-active-drag not implemented")
  (values))

(define-method start-active-drag-handle ((self <path-graphic>)
                                         (in-view <open-view>)
                                         handle-id
                                         (initial-pt <point>))
  (wm "path-graphic: start-active-drag-handle not implemented")
  (values))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   Placement Tool
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (drawpen-button-press (view <open-view>)
			      (at <point>)
			      modifier-state)
  ;; there are two basic cases, depending on whether or not there
  ;; is already an open point selected.
  ;; (the modal interaction is quite similar, except that if we
  ;; are extending an open point, we also draw the line segment
  ;; being created)
  (let ((op (open-point-selected view))
	(win (content-window view))
	(gc (transient-gc-nodash (content-window view)))
	(h-was at))
    ;
    (define (draw (h <point>))
      (set! h (make-point (inexact->exact (x h))
			  (inexact->exact (y h))))
      ; draw the center point
      (draw-rectangle win gc (- (x at) 1) (- (y at) 1) 3 3 #t)
      ; draw the out-handle
      (draw-rectangle win gc (- (x h) 1) (- (y h) 1) 3 3 #t)
      ; draw the in-handle
      (let* ((d (point- at h))
	     (i (point+ at d)))
	(draw-rectangle win gc (- (x i) 1) (- (y i) 1) 3 3 #t)
	(draw-line win gc (x i) (y i) (x h) (y h))))
    ;;
    (draw at)
    ;;
    (set-active-drag-proc!
     view
     (vector
      ; mouse-motion handler
      (lambda ((now-at <point>) state)
	(draw h-was)
	; doing this in device space...?
	(if (shift-state? state)
	    (set! now-at (shift-constraint-filter now-at at)))
	(draw now-at)
	(set! h-was now-at)
	(flush-client))
      ; mouse-up handler
      (lambda ((finally-at <point>) state)
	(draw h-was)
	; doing this in device space...?
	(if (shift-state? state)
	    (set! finally-at (shift-constraint-filter finally-at at)))
	(bind ((u->d (view-ctm (underlying-object view)))
	       (d->u (invert-transform u->d))
	       (posn (transform at d->u))
	       (h (point- finally-at at))
	       (gr sp (open-point-parents view op))
	       (pp (if (and (zero? (dx h)) (zero? (dy h)))
		       (make <path-point>
			     in-subpath: sp
			     constraint: 'corner
			     position: posn
			     in-handle: $zero-size
			     out-handle: $zero-size)
		       (make <path-point>
			     in-subpath: sp
			     constraint: 'smooth
			     position: posn
			     in-handle: (size- $zero-size (transform h d->u))
			     out-handle: (transform h d->u)))))
	  ;; if there is an open path, add it, otherwise, create one
	  (let ((id (append-path-point! sp pp)))
	    (dm "new path point has id: ~s" id)
	    (clear-all-areas (in-document view))
	    (do-select view gr id)
	    (update-handles view))))))))

;;; add a new path-point to an open subpath
;;; returns the id of the new handle

(define (append-path-point! (sp <subpath>) (pp <path-point>))
  (let ((subpath-id (logical-shift-left (vmemq sp (subpaths (in-path sp))) 14))
	(ppv (path-points sp)))
    (set-path-points! sp (vector-append ppv (vector pp)))
    (+ subpath-id (logical-shift-left (vector-length ppv) 2))))

;;; if there is no path-point, create the parents

(define (open-point-parents view pp)
  (if pp
      (values (in-path (in-subpath pp)) (in-subpath pp))
      (let* ((par (page-contents (view-page (underlying-object view))))
	     (path (make <path-graphic>
			 in-document: (in-document par)
			 parent-object: par
			 origin: $zero-point
			 graphic-bounding-box: (make-rect 0 0 0 0)
			 subpaths: '#()))
	     (sp (make <subpath>
		       in-path: path)))
	(set-subpaths! path (vector sp))
	(values path sp))))

;;;  determine if an open point is selected, and return it if so
;;;  otherwise, return #f
;;;  (an open point is a <path-point> in an unclosed <sub-path>
;;;  that is at one (either) end of of the sub-path)

(define (open-point-selected (view <open-view>))
  ;; XXX temporary implementation, it's an approximation
  (let* ((sel (key-sequence (current-selection view)))
	 (sel1 (if (pair? sel) (car sel) #f)))
    (if (instance? sel1 <path-graphic>)
	(vector-last (path-points (vector-last (subpaths sel1))))
	#f)))

(define (vector-last (v <vector>))
  (vector-ref v (sub1 (vector-length v))))

;;;
;;; hack for testing...
;;;

(add-major-mode!
 (make <major-mode>
       name: 'pen
       button-press-proc: drawpen-button-press))

(define-interactive (pen-mode view)
  (interactive (owner))
  (set-major-mode! view (get-major-mode 'pen)))

(graphic-set-key #\M-4 pen-mode)

;;;

(define (transient-gc-nodash win)
  (get-property win 'transient-gc-nodash (create-transient-gc-nodash win)))

(define (create-transient-gc-nodash win)
  (let* ((scrn (drawable-screen win))
	 (gc (create-gcontext drawable: win
			      function: 'boole-xor
			      ; this hilight technique is taken from
			      ; X11 (Xlib - vol.1) (O'Reilly) p.205
			      foreground: (bitwise-xor
					   (screen-black-pixel scrn)
					   (screen-white-pixel scrn))
			      line-width: 0
			      background: (screen-white-pixel scrn))))
    (set-property! win 'transient-gc gc)
    gc))

;;;

;;; need to support all the <path-point> options...

(define (path-point-from-extern (in <subpath>) 
				#key x y
				     (in-dx default: 0)
				     (in-dy default: 0)
				     (out-dx default: 0)
				     (out-dy default: 0))
  (make <path-point>
	in-subpath: in
	position: (make-point x y)
	in-handle: (if (and (eq? in-dx 0) (eq? in-dy 0))
		       $zero-size
		       (make-size in-dx in-dy))
	out-handle: (if (and (eq? out-dx 0) (eq? out-dy 0))
			$zero-size
			(make-size out-dx out-dy))))

(define (subpath-from-extern (in <path-graphic>) 
			     #key points
			          (closed? default: #f))
  (let ((sp (make <subpath>
		  closed?: closed?
		  in-path: in)))
    (set-path-points! 
     sp
     (list->vector
      (map (lambda (pp)
	     (case (car pp)
	       ((path-point)
		(apply path-point-from-extern sp (cdr pp)))
	       (else
		(em "not a path-point: ~s" pp))))
	   points)))
    sp))

(define (path-from-extern in-group 
			  #key subpaths
			       (stroke-color default: #f)
			       (fill-color default: #f))
  (let ((p (make <path-graphic>
		 in-document: (in-document in-group)
		 parent-object: in-group
		 graphic-bounding-box: (make-rect 0 0 0 0))))
    (set-subpaths! 
     p
     (list->vector
      (map (lambda (sp)
	     (case (car sp)
	       ((subpath)
		(apply subpath-from-extern p (cdr sp)))
	       (else
		(em "not a subpath: ~s" sp))))
	   subpaths)))
    (if stroke-color (set-property! p 'stroke-color stroke-color))
    (if fill-color (set-property! p 'fill-color fill-color))
    (recompute-graphic-bounding-box! p)
    p))

(define (paste-path-from-extern extern group offset)
  (apply path-from-extern group (cdr extern)))

;;;

(define-method recompute-graphic-bounding-box! ((self <path-graphic>))
  (let ((xmin 0)
	(xmax 0)
	(ymin 0)
	(ymax 0)
	(first? #t))
    (vector-for-each
     (lambda ((sp <subpath>))
       (vector-for-each
	(lambda ((pp <path-point>))
	  ;; NEED TO TAKE INTO ACCOUNT in/out HANDLES
	  (let ((x (x (position pp)))
		(y (y (position pp))))
	    (if first?
		(begin
		  (set! xmin x)
		  (set! xmax x)
		  (set! ymin y)
		  (set! ymax y)
		  (set! first? #f))
		(begin
		  (set! xmin (min xmin x))
		  (set! xmax (max xmax x))
		  (set! ymin (min ymin y))
		  (set! ymax (max ymax y))))))
	(path-points sp)))
     (subpaths self))
    (let ((r (bbox-rect xmin ymin xmax ymax)))
      (set-graphic-bounding-box! self r)
      r)))

