/*
 * robwxutil.cpp
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// Implements utilities for wxWidgets

#include <wx/wx.h>
#include <wx/listctrl.h>
#include <wx/config.h>
#include <wx/mimetype.h>

#include <rtstring.h>
#include <rtmath.h>

#include "robwxutil.h"

#ifdef __WXMSW__ // (WINDOWS)
#  include <windows.h>
#  include <wx/msw/winundef.h>

int Wx::Startup()
{
	char* dummy_argv = "dummy";
	return wxEntry(GetModuleHandle(0), 0, dummy_argv, SW_SHOW);
}

#else // (UNIX-LIKE)

int Wx::Startup()
{
	char* dummy_argv[] = { "dummy", 0 };
	return wxEntry(1, dummy_argv);
}

#endif // (OS SELECTION)

wxString Wx::Conv(const lrt::String& str)
{
	return wxString(str.cStr());
}

lrt::String Wx::Conv(const wxString& str)
{
	return lrt::String(str.c_str());
}


void Wx::StoreArray(const lrt::Array<int>& arr, const wxString& cfgKey)
{
	wxString val;
	for(int c = 0; c < arr.length(); c++) {
		val << arr[c] << ",";
	}
	val.Truncate(val.Len() - 1); // stop last comma
	wxConfigBase* cfg = wxConfig::Get();
	cfg->Write(cfgKey, val);
}

lrt::Array<int> Wx::ReadArray(int def, const wxString& cfgKey)
{
	wxConfigBase* cfg = wxConfig::Get();
	lrt::String str = Wx::Conv(cfg->Read(cfgKey, ""));
	lrt::Array<lrt::String> arr = str.split(",", "");
	lrt::Array<int> ret(arr.length());
	for(int i = 0; i < arr.length(); i++)
		ret[i] = arr[i].intValue(def);
	return ret; 
}

// FIXME: for toplevel windows (frames dialogs)
// * handle "MAXIMIZED" case specially
void Wx::StoreRect(wxWindow* win, const wxString& cfgKey)
{
	wxString val; 
	wxRect rect = win->GetRect();
	wxConfigBase* cfg = wxConfig::Get();

	if(wxDynamicCast(win, wxTopLevelWindow)) // top level windows can be iconized or maximized which must be treated separately
	{
		// In iconized or maximized state, the GetRect() function return values are not really helpful. 
		wxTopLevelWindow* twin = wxDynamicCast(win, wxTopLevelWindow);
		if(twin->IsIconized()) // we don't want to save this state. It would be confusing. 
			return; 
		else if(twin->IsMaximized())
		{
			// just re-use previous state
			val = cfg->Read(cfgKey, "");
			if(val.AfterLast(',') != "max") // was not already saved max'ed => append ",max"
				val += ",max"; 
			cfg->Write(cfgKey, val);
			return;
		}
	}
	// window is no toplevel, or is not in a special state
	val = wxString::Format("%d,%d,%d,%d", rect.x, rect.y, rect.width, rect.height);
	cfg->Write(cfgKey, val);
}

// FIXME: for toplevel windows (frames dialogs)
// * handle "MAXIMIZED" case specially
// * make sure that the window is not positioned offscreen due to resolution changes
void Wx::RestoreRect(wxWindow* win, const wxString& cfgKey)
{
	wxConfigBase* cfg = wxConfig::Get();
	lrt::String str = Wx::Conv(cfg->Read(cfgKey, ""));
	lrt::Array<lrt::String> arr = str.split(",", "");
	if(arr.length() < 4 || arr.length() > 5) // invalid str
		return;
	// validate sizes (-1 is OK and does nothing)
	int x = arr[0].intValue(-1), y = arr[1].intValue(-1), w = arr[2].intValue(-1), h = arr[3].intValue(-1);
	wxSize screenSize = wxSize(wxSystemSettings::GetMetric(wxSYS_SCREEN_X), wxSystemSettings::GetMetric(wxSYS_SCREEN_Y));
	if(x < -100 || x > screenSize.x - 100) x = -1;
	if(y < -10  || y > screenSize.y - 100) y = -1;
	if(w < 1    || w > screenSize.x) w = -1;
	if(h < 1    || h > screenSize.y) h = -1; 

	win->SetSize(x, y, w, h);

	// handle maximization
	if(arr.length() == 5 && arr[4] == "max")
	{
		wxTopLevelWindow* twin = wxDynamicCast(win, wxTopLevelWindow);
		if(twin) twin->Maximize(true);
	}
		
}

void Wx::StoreColumns(wxListCtrl* ctrl, const wxString& cfgKey)
{
	int colCount = ctrl->GetColumnCount();
	wxString val;
	for(int c = 0; c < colCount; c++) {
		val << ctrl->GetColumnWidth(c) << ",";
	}
	wxConfigBase* cfg = wxConfig::Get();
	cfg->Write(cfgKey, val);
}

void Wx::RestoreColumns(wxListCtrl* ctrl, const wxString& cfgKey)
{
	int colCount = ctrl->GetColumnCount();
	wxConfigBase* cfg = wxConfig::Get();
	lrt::String str = Wx::Conv(cfg->Read(cfgKey, ""));
	lrt::Array<lrt::String> arr = str.split(",", "");
	int term = arr.length(); if(colCount < term) term = colCount;
	for(int c = 0; c < term; c++)
		ctrl->SetColumnWidth(c, arr[c].intValue(-1));
}

void Wx::ShowInBrowser(const wxString& htmlFile)
{
	wxFileType* ftHtml = wxTheMimeTypesManager->GetFileTypeFromMimeType("text/html");
	wxString browserCmd; 
	if(ftHtml) browserCmd = ftHtml->GetOpenCommand(htmlFile); 
	delete ftHtml; ftHtml = 0; 

	if(browserCmd.IsEmpty()) { // resort to asking the user... 
		browserCmd = ReadString("", "/BrowserCmd"); 
		if(browserCmd.IsEmpty())
			browserCmd = wxGetTextFromUser("Enter the command to start your favourite browser:", "Browser selection", "netscape"); 
		if(browserCmd.IsEmpty()) // user doesn't want to enter browser
			return; 
		else {
			StoreString(browserCmd, "/BrowserCmd"); 
			browserCmd += " " + htmlFile;
		}
	}
	if(!wxExecute(browserCmd)) {
		wxLogError("Cannot start browser for showing HTML file. Please open %s yourself.", htmlFile.c_str()); 
		StoreString("", "/BrowserCmd"); // browser does not work, ask again next time
	}
	delete ftHtml; 
}

wxString Wx::ReadString(const wxString& def, const wxString& key)
{
	wxConfigBase* cfg = wxConfig::Get();
	return cfg->Read(key, def);
}

void Wx::StoreString(const wxString& val, const wxString& key)
{
	wxConfigBase* cfg = wxConfig::Get();
	cfg->Write(key, val);
}
