//roardmx.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2009-2014
 *
 *  This file is part of libroardsp a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroardsp is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "libroarlight.h"

// base(ic) check
#define BCHK(x) if ( (x) == NULL ) { roar_err_set(ROAR_ERROR_FAULT); return -1; }

// database access:
static const struct {
 const int id;
 const char * const name;
} __eventnames[] = {
//$ grep '^#define ROAR_ROARDMX_EVENT_' roardmx.h | cut -d ' ' -f2 | while read line; do name=`echo $line | cut -d_ -f4 | tr A-Z a-z`; for suffix in beat off on hold; do printf " {%-56s \"%s\"},\n" "$line|ROAR_ROARDMX_ETYPE_`tr a-z A-Z <<<$suffix`," $name-$suffix; done; printf " {%-56s \"%s\"},\n" $line, $name; done
 {ROAR_ROARDMX_EVENT_NONE|ROAR_ROARDMX_ETYPE_BEAT,         "none-beat"},
 {ROAR_ROARDMX_EVENT_NONE|ROAR_ROARDMX_ETYPE_OFF,          "none-off"},
 {ROAR_ROARDMX_EVENT_NONE|ROAR_ROARDMX_ETYPE_ON,           "none-on"},
 {ROAR_ROARDMX_EVENT_NONE|ROAR_ROARDMX_ETYPE_HOLD,         "none-hold"},
 {ROAR_ROARDMX_EVENT_NONE,                                 "none"},
 {ROAR_ROARDMX_EVENT_STEP|ROAR_ROARDMX_ETYPE_BEAT,         "step-beat"},
 {ROAR_ROARDMX_EVENT_STEP|ROAR_ROARDMX_ETYPE_OFF,          "step-off"},
 {ROAR_ROARDMX_EVENT_STEP|ROAR_ROARDMX_ETYPE_ON,           "step-on"},
 {ROAR_ROARDMX_EVENT_STEP|ROAR_ROARDMX_ETYPE_HOLD,         "step-hold"},
 {ROAR_ROARDMX_EVENT_STEP,                                 "step"},
 {ROAR_ROARDMX_EVENT_TAP|ROAR_ROARDMX_ETYPE_BEAT,          "tap-beat"},
 {ROAR_ROARDMX_EVENT_TAP|ROAR_ROARDMX_ETYPE_OFF,           "tap-off"},
 {ROAR_ROARDMX_EVENT_TAP|ROAR_ROARDMX_ETYPE_ON,            "tap-on"},
 {ROAR_ROARDMX_EVENT_TAP|ROAR_ROARDMX_ETYPE_HOLD,          "tap-hold"},
 {ROAR_ROARDMX_EVENT_TAP,                                  "tap"},
 {ROAR_ROARDMX_EVENT_BEAT|ROAR_ROARDMX_ETYPE_BEAT,         "beat-beat"},
 {ROAR_ROARDMX_EVENT_BEAT|ROAR_ROARDMX_ETYPE_OFF,          "beat-off"},
 {ROAR_ROARDMX_EVENT_BEAT|ROAR_ROARDMX_ETYPE_ON,           "beat-on"},
 {ROAR_ROARDMX_EVENT_BEAT|ROAR_ROARDMX_ETYPE_HOLD,         "beat-hold"},
 {ROAR_ROARDMX_EVENT_BEAT,                                 "beat"},
 {ROAR_ROARDMX_EVENT_BLACKOUT|ROAR_ROARDMX_ETYPE_BEAT,     "blackout-beat"},
 {ROAR_ROARDMX_EVENT_BLACKOUT|ROAR_ROARDMX_ETYPE_OFF,      "blackout-off"},
 {ROAR_ROARDMX_EVENT_BLACKOUT|ROAR_ROARDMX_ETYPE_ON,       "blackout-on"},
 {ROAR_ROARDMX_EVENT_BLACKOUT|ROAR_ROARDMX_ETYPE_HOLD,     "blackout-hold"},
 {ROAR_ROARDMX_EVENT_BLACKOUT,                             "blackout"},
 {ROAR_ROARDMX_EVENT_FULLON|ROAR_ROARDMX_ETYPE_BEAT,       "fullon-beat"},
 {ROAR_ROARDMX_EVENT_FULLON|ROAR_ROARDMX_ETYPE_OFF,        "fullon-off"},
 {ROAR_ROARDMX_EVENT_FULLON|ROAR_ROARDMX_ETYPE_ON,         "fullon-on"},
 {ROAR_ROARDMX_EVENT_FULLON|ROAR_ROARDMX_ETYPE_HOLD,       "fullon-hold"},
 {ROAR_ROARDMX_EVENT_FULLON,                               "fullon"},
 {ROAR_ROARDMX_EVENT_FLASH|ROAR_ROARDMX_ETYPE_BEAT,        "flash-beat"},
 {ROAR_ROARDMX_EVENT_FLASH|ROAR_ROARDMX_ETYPE_OFF,         "flash-off"},
 {ROAR_ROARDMX_EVENT_FLASH|ROAR_ROARDMX_ETYPE_ON,          "flash-on"},
 {ROAR_ROARDMX_EVENT_FLASH|ROAR_ROARDMX_ETYPE_HOLD,        "flash-hold"},
 {ROAR_ROARDMX_EVENT_FLASH,                                "flash"},
 {ROAR_ROARDMX_EVENT_STROBE|ROAR_ROARDMX_ETYPE_BEAT,       "strobe-beat"},
 {ROAR_ROARDMX_EVENT_STROBE|ROAR_ROARDMX_ETYPE_OFF,        "strobe-off"},
 {ROAR_ROARDMX_EVENT_STROBE|ROAR_ROARDMX_ETYPE_ON,         "strobe-on"},
 {ROAR_ROARDMX_EVENT_STROBE|ROAR_ROARDMX_ETYPE_HOLD,       "strobe-hold"},
 {ROAR_ROARDMX_EVENT_STROBE,                               "strobe"},
 {ROAR_ROARDMX_EVENT_STROBEREADY|ROAR_ROARDMX_ETYPE_BEAT,  "strobeready-beat"},
 {ROAR_ROARDMX_EVENT_STROBEREADY|ROAR_ROARDMX_ETYPE_OFF,   "strobeready-off"},
 {ROAR_ROARDMX_EVENT_STROBEREADY|ROAR_ROARDMX_ETYPE_ON,    "strobeready-on"},
 {ROAR_ROARDMX_EVENT_STROBEREADY|ROAR_ROARDMX_ETYPE_HOLD,  "strobeready-hold"},
 {ROAR_ROARDMX_EVENT_STROBEREADY,                          "strobeready"},
 {ROAR_ROARDMX_EVENT_STROBELOAD|ROAR_ROARDMX_ETYPE_BEAT,   "strobeload-beat"},
 {ROAR_ROARDMX_EVENT_STROBELOAD|ROAR_ROARDMX_ETYPE_OFF,    "strobeload-off"},
 {ROAR_ROARDMX_EVENT_STROBELOAD|ROAR_ROARDMX_ETYPE_ON,     "strobeload-on"},
 {ROAR_ROARDMX_EVENT_STROBELOAD|ROAR_ROARDMX_ETYPE_HOLD,   "strobeload-hold"},
 {ROAR_ROARDMX_EVENT_STROBELOAD,                           "strobeload"},
 {ROAR_ROARDMX_EVENT_FOG|ROAR_ROARDMX_ETYPE_BEAT,          "fog-beat"},
 {ROAR_ROARDMX_EVENT_FOG|ROAR_ROARDMX_ETYPE_OFF,           "fog-off"},
 {ROAR_ROARDMX_EVENT_FOG|ROAR_ROARDMX_ETYPE_ON,            "fog-on"},
 {ROAR_ROARDMX_EVENT_FOG|ROAR_ROARDMX_ETYPE_HOLD,          "fog-hold"},
 {ROAR_ROARDMX_EVENT_FOG,                                  "fog"},
 {ROAR_ROARDMX_EVENT_FOGREADY|ROAR_ROARDMX_ETYPE_BEAT,     "fogready-beat"},
 {ROAR_ROARDMX_EVENT_FOGREADY|ROAR_ROARDMX_ETYPE_OFF,      "fogready-off"},
 {ROAR_ROARDMX_EVENT_FOGREADY|ROAR_ROARDMX_ETYPE_ON,       "fogready-on"},
 {ROAR_ROARDMX_EVENT_FOGREADY|ROAR_ROARDMX_ETYPE_HOLD,     "fogready-hold"},
 {ROAR_ROARDMX_EVENT_FOGREADY,                             "fogready"},
 {ROAR_ROARDMX_EVENT_FOGHEAT|ROAR_ROARDMX_ETYPE_BEAT,      "fogheat-beat"},
 {ROAR_ROARDMX_EVENT_FOGHEAT|ROAR_ROARDMX_ETYPE_OFF,       "fogheat-off"},
 {ROAR_ROARDMX_EVENT_FOGHEAT|ROAR_ROARDMX_ETYPE_ON,        "fogheat-on"},
 {ROAR_ROARDMX_EVENT_FOGHEAT|ROAR_ROARDMX_ETYPE_HOLD,      "fogheat-hold"},
 {ROAR_ROARDMX_EVENT_FOGHEAT,                              "fogheat"}
};

int roar_roardmx_str2event(const char * event) {
 size_t i;

 if ( event == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 for (i = 0; i < (sizeof(__eventnames)/sizeof(*__eventnames)); i++)
  if ( !strcasecmp(event, __eventnames[i].name) )
   return __eventnames[i].id;

 roar_err_set(ROAR_ERROR_NOENT);
 return -1;
}

const char * roar_roardmx_event2str(const int event) {
 size_t i;

 for (i = 0; i < (sizeof(__eventnames)/sizeof(*__eventnames)); i++)
  if ( __eventnames[i].id == event )
   return __eventnames[i].name;

 roar_err_set(ROAR_ERROR_NOENT);
 return NULL;
}

// generic things:
int roar_roardmx_message_new (struct roar_roardmx_message * mes) {
 BCHK(mes);

 memset(mes, 0, sizeof(struct roar_roardmx_message));

 mes->version = ROAR_ROARDMX_VERSION;

 return 0;
}

// low level:
//int roar_roardmx_message_set_flag(struct roar_roardmx_message * mes, unsigned char   flag);
//int roar_roardmx_message_set_len (struct roar_roardmx_message * mes, size_t          type);
//int roar_roardmx_message_get_data(struct roar_roardmx_message * mes, unsigned char ** data);

// medium level:
int roar_roardmx_message_set_type(struct roar_roardmx_message * mes, unsigned char   type) {
 BCHK(mes);

 // check if type contains non-type bits.
 if ( (type | ROAR_ROARDMX_MASK_TYPE) - ROAR_ROARDMX_MASK_TYPE ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 }

 mes->type = type;

 return 0;
}

int roar_roardmx_message_get_flag(struct roar_roardmx_message * mes, unsigned char * flag) {
 BCHK(mes);

 *flag = mes->flags;

 return 0;
}

int roar_roardmx_message_get_type(struct roar_roardmx_message * mes, unsigned char * type) {
 BCHK(mes);

 *type = mes->type;

 return 0;
}

int roar_roardmx_message_get_len (struct roar_roardmx_message * mes, size_t        * length) {
 BCHK(mes);

 *length = mes->length;

 return 0;
}


// IO:
int roar_roardmx_message_send(struct roar_roardmx_message * mes, struct roar_vio_calls * vio) {
 BCHK(mes);
 BCHK(vio);

 if ( mes->length > ROAR_ROARDMX_DATA_LENGTH ) { // this is very fatal!
  roar_panic(ROAR_FATAL_ERROR_MEMORY_CORRUPTION, NULL);
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 mes->data[0] =  mes->version;
 mes->data[1] = (mes->flags & ROAR_ROARDMX_MASK_FLAGS) |
                (mes->type  & ROAR_ROARDMX_MASK_TYPE ) ;

 mes->data[2] = mes->length;

 return roar_vio_write(vio, mes->data, mes->length + 3) == (ssize_t)(mes->length + 3) ? 0 : -1;
}

int roar_roardmx_message_recv(struct roar_roardmx_message * mes, struct roar_vio_calls * vio) {
 BCHK(mes);
 BCHK(vio);

 if ( roar_roardmx_message_new(mes) == -1 )
  return -1;

 if ( roar_vio_read(vio, mes->data, 3) != 3 )
  return -1;

 mes->version = mes->data[0];

 if ( mes->version != ROAR_ROARDMX_VERSION ) {
  roar_err_set(ROAR_ERROR_NSVERSION);
  return -1;
 }

 mes->flags  = mes->data[1] & ROAR_ROARDMX_MASK_FLAGS;
 mes->type   = mes->data[1] & ROAR_ROARDMX_MASK_TYPE;

 mes->length = mes->data[2];

 if ( roar_vio_read(vio, &(mes->data[3]), mes->length) != (ssize_t)mes->length )
  return -1;

 return 0;
}

// Data/high level:
// * *:
int roar_roardmx_message_add_chanval(struct roar_roardmx_message * mes, uint16_t channel, unsigned char val) {
 register uint16_t * chan;

 BCHK(mes);

 switch (mes->type) {
  case ROAR_ROARDMX_TYPE_SSET:
  case ROAR_ROARDMX_TYPE_INC8S:
   break;
  default:
    roar_err_set(ROAR_ERROR_TYPEMM);
    return -1;
   break;
 }

 if ( (mes->length + 3) > ROAR_ROARDMX_DATA_LENGTH ) { // message would be to long
  roar_err_set(ROAR_ERROR_NOSPC);
  return -1;
 }

 chan = (uint16_t *) &(mes->data[mes->length + 3]);

 *chan = ROAR_HOST2NET16(channel);

 mes->data[mes->length + 2 + 3] = val;

 mes->length += 3;

 return 0;
}

int roar_roardmx_message_get_chanval(struct roar_roardmx_message * mes, uint16_t * channel, unsigned char * val, int index) {
 register uint16_t * chan;

 BCHK(mes);

 if ( index < 0 ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 }

 if ( mes->version != ROAR_ROARDMX_VERSION ) {
  roar_err_set(ROAR_ERROR_NSVERSION);
  return -1;
 }

 switch (mes->type) {
  case ROAR_ROARDMX_TYPE_SSET:
  case ROAR_ROARDMX_TYPE_INC8S:
    if ( index >= (ROAR_ROARDMX_DATA_LENGTH/3) )
     return -1;

    *val     = mes->data[3 * index + 2 + 3];
    chan     = (uint16_t *) &(mes->data[3 + 3 * index]);
    *channel = ROAR_NET2HOST16(*chan);
    return 0;
   break;
 }

 roar_err_set(ROAR_ERROR_NSTYPE);
 return -1;
}

int roar_roardmx_message_numchannels(struct roar_roardmx_message * mes) {
 BCHK(mes);

 if ( mes->version != ROAR_ROARDMX_VERSION ) {
  roar_err_set(ROAR_ERROR_NSVERSION);
  return -1;
 }

 switch (mes->type) {
  case ROAR_ROARDMX_TYPE_SSET:
  case ROAR_ROARDMX_TYPE_INC8S:
    return mes->length /  3;
   break;
  case ROAR_ROARDMX_TYPE_IPO1:
    return mes->length /  6;
   break;
  case ROAR_ROARDMX_TYPE_IPO4:
    return mes->length / 12;
   break;
  case ROAR_ROARDMX_TYPE_RANGESET:
    return mes->length /  5;
   break;
  case ROAR_ROARDMX_TYPE_EVENT:
  case ROAR_ROARDMX_TYPE_CONTROL:
    return 0;
   break;
 }

 roar_err_set(ROAR_ERROR_NSTYPE);
 return -1;
}

// * SSET:
int roar_roardmx_message_new_sset   (struct roar_roardmx_message * mes) {
 BCHK(mes);

 if ( roar_roardmx_message_new(mes) == -1 )
  return -1;

 mes->type = ROAR_ROARDMX_TYPE_SSET;

 return 0;
}

// * IPO1:
// Not yet supported.
// * IPO4:
// Not yet supported.
// * INC8S:
// Not yet supported.
// * RANGESET:
// Not yet supported.
// * EVENT:
int roar_roardmx_message_new_event(struct roar_roardmx_message * mes) {
 BCHK(mes);

 if ( roar_roardmx_message_new(mes) == -1 )
  return -1;

 mes->type = ROAR_ROARDMX_TYPE_EVENT;

 return 0;
}

int roar_roardmx_message_add_events(struct roar_roardmx_message * mes, const uint8_t * events, size_t len) {
 BCHK(mes);

 if ( mes->type != ROAR_ROARDMX_TYPE_EVENT ) {
  roar_err_set(ROAR_ERROR_TYPEMM);
  return -1;
 }

 if ( (mes->length + len) > ROAR_ROARDMX_DATA_LENGTH ) { // message would be to long
  roar_err_set(ROAR_ERROR_NOSPC);
  return -1;
 }

 memcpy(mes->data + 3 + mes->length, events, len);
 mes->length += len;

 return 0;
}

int roar_roardmx_message_get_events(struct roar_roardmx_message * mes, const uint8_t ** events, size_t * len) {
 BCHK(mes);

 if ( mes->type != ROAR_ROARDMX_TYPE_EVENT ) {
  roar_err_set(ROAR_ERROR_TYPEMM);
  return -1;
 }

 *events = (const uint8_t *)(mes->data + 3);
 *len    = mes->length;

 return 0;
}

// * CONTROL:
// Not yet supported.

//ll
