///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/field_element.h"
#include "rheolef/field_evaluate.h"
namespace rheolef {

// cstor by specifying the quadrature formulae
template<class T, class M>
void
field_element<T,M>::initialize (const space_basic<T,M>& X, const quadrature_option_type& qopt) const
{
  _X = X;
  if (qopt.get_order() != std::numeric_limits<quadrature_option_type::size_type>::max()) {
    _quad.set_order (qopt.get_order());
  } else {
    size_type k = X.get_numbering().degree();
    _quad.set_order (2*k-1); // see Raviart & Thomas, Masson, 1988, page 130, theorem 5.3.1
  }
  _quad.set_family (qopt.get_family());

  trace_macro ("quadrature : " << _quad.get_family_name() << " order " << _quad.get_order());
  check_macro (get_geo().coordinate_system() == space_constant::cartesian,
	"riesz: unsupported non cartesian coordinate system");
  _basis_on_quad.set (_quad, get_basis());
  _piola_on_quad.set (_quad, get_piola_basis());
  _initialized = true;
}
// ==========================================================================
// scalar valued argument
// ==========================================================================
// compute lk(i) such that lk(i) = sum_q phi_i(xq)*f(xq)*wq
// where phi_i, i=[0:ni[ is the local FEM basis 
template<class T, class M>
void
field_element<T,M>::eval (
    const geo_element&    K,
    const std::vector<T>& fq,
    std::vector<T>&       lk) const
{
  const geo_basic<T,M>& omega = get_geo();
  size_type d     = omega.dimension();
  size_type map_d = K.dimension();
  std::vector<size_type> dis_inod_K;
  std::vector<size_type> dis_inod_L;
  omega.dis_inod (K, dis_inod_K);
  reference_element hat_K   = K.variant();
  reference_element tilde_L = hat_K;
  if (is_on_band()) {
    size_type first_dis_ie = omega.sizes().ownership_by_dimension [K.dimension()].first_index();
    size_type K_ie = K.dis_ie() - first_dis_ie;
    size_type L_ie = get_band().sid_ie2bnd_ie (K_ie);
    const geo_element& L = get_band().band() [L_ie];
    get_band().band().dis_inod (L, dis_inod_L);
    tilde_L = L.variant();
  }
  size_type ni = get_basis().size(tilde_L);
  lk.resize (ni);
  fill (lk.begin(), lk.end(), T(0));
  tensor_basic<T> DF;
  typename quadrature<T>::const_iterator
    first_quad = _quad.begin(hat_K),
    last_quad  = _quad.end  (hat_K);
  for (size_type q = 0; first_quad != last_quad; first_quad++, q++) {
    jacobian_piola_transformation (omega, _piola_on_quad, K, dis_inod_K, q, DF);
    T det_DF = det_jacobian_piola_transformation (DF, d, map_d);
    if (!is_on_band()) {
      _basis_on_quad.evaluate (hat_K, q);
    } else {
      point_basic<T> xq = rheolef::piola_transformation (omega, _piola_on_quad, hat_K, dis_inod_K, q);
      point_basic<T> tilde_xq = rheolef::inverse_piola_transformation (get_band().band(), tilde_L, dis_inod_L, xq);
      _basis_on_quad.evaluate (tilde_L, tilde_xq);
    }
    T wq = 1;
    wq *= weight_coordinate_system (omega, _piola_on_quad, K, dis_inod_K, q);
    wq *= det_DF;
    wq *= (*first_quad).w;
    T fwq = fq[q]*wq;
    size_type loc_idof = 0;
    for (typename basis_on_pointset<T>::const_iterator
	    first_phi = _basis_on_quad.begin(),
            last_phi  = _basis_on_quad.end(); 
            first_phi != last_phi; first_phi++, loc_idof++) {
      lk [loc_idof] += (*first_phi)*fwq;
    }
  }
}
// ==========================================================================
// vector-valued argument
// ==========================================================================
template<class T, class M>
void
field_element<T,M>::eval (
    const geo_element&                  K,
    const std::vector<point_basic<T> >& fq,
    std::vector<T>&                     lk) const
{
  check_macro (!is_on_band(), "riesz: unsupported vector-valued argument on a band");
  const geo_basic<T,M>& omega = get_geo();
  std::vector<size_type> dis_inod;
  omega.dis_inod (K, dis_inod);
  size_type d = omega.dimension();
  size_type map_d = K.dimension();
  size_t ni = get_basis().size(K);
  size_type n_comp = d;
  lk.resize (n_comp*ni);
  fill (lk.begin(), lk.end(), T(0.0));
  typename quadrature<T>::const_iterator
    first_quad = _quad.begin(K),
    last_quad  = _quad.end  (K);
  tensor_basic<T> DF;
  for (size_type q = 0; first_quad != last_quad; first_quad++, q++) {
    jacobian_piola_transformation (omega, _piola_on_quad, K, dis_inod, q, DF);
    T det_DF = det_jacobian_piola_transformation (DF, d, map_d);
    T wq = 1;
    wq *= weight_coordinate_system (omega, _piola_on_quad, K, dis_inod, q);
    wq *= det_DF;
    wq *= (*first_quad).w; 
    point_basic<T> fwq = fq[q]*wq;
    typename basis_on_pointset<T>::const_iterator
	first_phi = _basis_on_quad.begin (K, q),
        last_phi  = _basis_on_quad.end   (K, q);
    for (size_t loc_comp_idof = 0; first_phi != last_phi; first_phi++, loc_comp_idof++) {
      for (size_t i_comp = 0; i_comp < n_comp; i_comp++) {
        size_type loc_idof = i_comp*ni + loc_comp_idof;
        lk [loc_idof] += (*first_phi)*fwq[i_comp];
      }
    }
  }
}
// ==========================================================================
// field argument
// ==========================================================================
template<class T, class M>
void 
field_element<T,M>::operator() (
        const geo_element&      K,
	const field_basic<T,M>& fh,
	std::vector<T>&         lk) const
{
  check_macro (!is_on_band(), "riesz: unsupported field argument on a band");
#ifdef TODO
  switch (fh.get_valued_type()) {
    case fem_helper::scalar: {
#endif // TODO
      // 1) evaluate f at all integration points xq ; fem basis is evaluated at xq one time for all
      std::vector<T> fq(_quad.size(K));
      std::vector<size_type> dis_idof;
      const numbering<T,M>& fem = get_space().get_numbering();
      fem.dis_idof (get_geo().sizes(), K, dis_idof);
      for (size_type q = 0, nq = _quad.size(K); q < nq; q++) {
        fq[q] = field_evaluate (fh, _basis_on_quad, K, dis_idof, q);
      }
      // 2) integrate f using quadrature formulae (xq,wq)
      eval (K, fq, lk);
#ifdef TODO
      break;
    }
    case fem_helper::vectorial: {
      size_t d = fh.dimension();
      std::vector<point_basic<T> > fq(_quad.size(K));
      typename quadrature<T>::const_iterator
        first = _quad.begin(K),
        last  = _quad.end  (K);
      for (size_type q = 0; first != last; first++, q++) {
        meshpoint xq (K.index(), (*first).x);
	for (size_t i = 0; i < d; i++) {
          fq[q][i] = fh.evaluate (xq, i);
        }
      }
      eval (K, fq, lk);
      break;
    }
    default:
      error_macro ("unsupported yet");
  }
#endif // TODO
}
// ==========================================================================
// instanciation in library
// ==========================================================================
template class field_element<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class field_element<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

}// namespace rheolef
