/*  RetroArch - A frontend for libretro.
 *  Copyright (C) 2011-2016 - Daniel De Matteis
 *
 *  RetroArch is free software: you can redistribute it and/or modify it under the terms
 *  of the GNU General Public License as published by the Free Software Found-
 *  ation, either version 3 of the License, or (at your option) any later version.
 *
 *  RetroArch is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 *  without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE.  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along with RetroArch.
 *  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdint.h>
#include <string.h>

#include <retro_assert.h>
#include <compat/strl.h>

#include "../msg_hash.h"
#include "../configuration.h"

#ifdef __clang__
#pragma clang diagnostic ignored "-Winvalid-source-encoding"
#endif

 /* IMPORTANT:
  * For non-english characters to work without proper unicode support,
  * we need this file to be encoded in ISO 8859-1 (Latin1), not UTF-8.
  * If you save this file as UTF-8, you'll break non-english characters
  * (e.g. German "Umlauts" and Portugese diacritics).
 */
/* DO NOT REMOVE THIS. If it causes build failure, it's because you saved the file as UTF-8. Read the above comment. */
extern const char force_iso_8859_1[sizeof("")==7+1 ? 1 : -1];

int menu_hash_get_help_es(uint32_t hash, char *s, size_t len)
{
   uint32_t driver_hash = 0;
   settings_t      *settings = config_get_ptr();

   /* If this one throws errors, stop sledgehammering square pegs into round holes and */
   /* READ THE COMMENTS at the top of the file. */
   (void)sizeof(force_iso_8859_1);

   switch (hash)
   {
      case MENU_LABEL_VALUE_HELP_AUDIO_VIDEO_TROUBLESHOOTING_DESC:
         {
            /* Work around C89 limitations */
            char u[501];
            char t[501];

            strlcpy(t,
                  "RetroArch utiliza un formato nico para \n"
                  "sincronizar vdeo y sonido que necesita \n"
                  "calibrarse con la frecuencia de \n"
                  "actualizacin de tu monitor para obtener \n"
                  "el mejor rendimiento. \n"
                  " \n"
                  "Si notas cortes de sonido o en la imagen,\n"
                  "lo normal es que necesites calibrar estos\n"
                  "ajustes. Aqu van algunas opciones:\n"
                  " \n", sizeof(t));
            snprintf(u, sizeof(u),
                  "a) Ve a '%s' -> '%s' y activa\n"
                  "'Vdeo por hilos'. En este modo la tasa\n"
                  "de refresco es irrelevante, habr ms fps,\n"
                  "'Vdeo multincleo'. En este modo la \n"
                  "frecuencia es irrelevante, habr ms fps,\n"
                  "pero la imagen podra ser menos fluida.\n"
                  "b) Ve a '%s' -> '%s' y busca\n"
                  "'%s'. Deja que se ejecute durante\n"
                  "2048 fotogramas y selecciona Aceptar.",
                  msg_hash_to_str(MENU_ENUM_LABEL_VALUE_SETTINGS),
                  msg_hash_to_str(MENU_ENUM_LABEL_VALUE_VIDEO_SETTINGS),
                  msg_hash_to_str(MENU_ENUM_LABEL_VALUE_SETTINGS),
                  msg_hash_to_str(MENU_ENUM_LABEL_VALUE_VIDEO_SETTINGS),
                  msg_hash_to_str(MENU_ENUM_LABEL_VALUE_VIDEO_REFRESH_RATE_AUTO)
                     );
            strlcat(s, t, len);
            strlcat(s, u, len);
         }
         break;
      case MENU_LABEL_VALUE_HELP_SCANNING_CONTENT_DESC:
         snprintf(s, len,
               "Para escanear contenidos ve a '%s' \n"
               "y selecciona '%s' o \n"
               "'%s'.\n"
               " \n"
               "Esto comparar los archivos con las entradas en \n"
               "la base de datos. Si hay una coincidencia, \n"
               "aadir una entrada en una coleccin.\n"
               " \n"
               "Entonces podrs acceder fcilmente al contenido\n"
               "si vas a '%s' ->\n"
               "'%s'\n"
               "en vez de tener que pasar por el navegador \n"
               "de archivos constantemente.\n"
               " \n"
               "NOTA: El contenido de algunos ncleos podra\n"
               "no ser localizable. Entre los ejemplos estn\n"
               "PlayStation, MAME, FBA, y puede que otros."
               ,
               msg_hash_to_str(MENU_ENUM_LABEL_VALUE_ADD_CONTENT_LIST),
               msg_hash_to_str(MENU_ENUM_LABEL_VALUE_SCAN_DIRECTORY),
               msg_hash_to_str(MENU_ENUM_LABEL_VALUE_SCAN_FILE),
               msg_hash_to_str(MENU_ENUM_LABEL_VALUE_LOAD_CONTENT_LIST),
               msg_hash_to_str(MENU_ENUM_LABEL_VALUE_CONTENT_COLLECTION_LIST)
               );
         break;
      case MENU_LABEL_VALUE_MENU_CONTROLS_PROLOG:
         snprintf(s, len,
               "Puedes usar estos controles en tu mando\n"
               "o teclado para controlar el men: \n"
               " \n"
               );
         break;
      case MENU_LABEL_VALUE_EXTRACTING_PLEASE_WAIT:
         strlcpy(s, "Extrayendo, espera, por favor...\n", len);
         break;
      case MENU_LABEL_WELCOME_TO_RETROARCH:
         snprintf(s, len,
               "Bienvenido a RetroArch\n"
               "\n"
               "Para ms informacin ve al men \n"
			   "de Ayuda.\n"
               );
         break;
      case MENU_LABEL_INPUT_DRIVER:
         if (settings)
            driver_hash = msg_hash_calculate(settings->input.driver);

         switch (driver_hash)
         {
            case MENU_LABEL_INPUT_DRIVER_UDEV:
               {
                  /* Work around C89 limitations */
                  char u[501];
                  char t[501];

                  strlcpy(t,
                        "Controlador de entrada udev. \n"
                        " \n"
                        "Este controlador puede funcionar sin X. \n"
                        " \n"
                        "Utiliza la API ms reciente para joypads \n"
                        "evdec para dar compatibilidad con joysticks. \n"
                        "Permite conexin en caliente y force \n"
                        "feedback (si lo admite el dispositivo). \n",
                        sizeof(t));
                  strlcpy(u,
                        " \n"
                        "El controlador lee los eventos evdev para \n"
                        "dar compatibilidad con teclados. Tambin \n"
                        "es compatible con retrollamadas de teclado, \n"
                        "ratones y pantallas tctiles. \n"
                        " \n"
                        "La mayora de las distros tienen los nodos \n"
                        "/dev/input en modo root-only (modo 600). \n"
                        "Puedes configurar una regla udev que los haga \n"
                        "accesibles fuera de la raz.", sizeof(u)
                        );

                  strlcat(s, t, len);
                  strlcat(s, u, len);
               }
               break;
            case MENU_LABEL_INPUT_DRIVER_LINUXRAW:
               snprintf(s, len,
                     "Controlador de entrada linuxraw. \n"
                     " \n"
                     "Este controlador necesita de un TTY activo. \n"
                     "Los eventos de teclado se leen directamente \n"
                     "desde el TTY, lo que es ms simple pero no tan \n"
                     "flexible como udev. \n"
                     "No es compatible con ratones, etc. \n"
                     " \n"
                     "Este controlador utiliza la antigua API de \n"
                     "joysticks (/dev/input/js*).");
               break;
            default:
               snprintf(s, len,
                     "Controlador de entrada.\n"
                     " \n"
                     "El controlador de vdeo podra forzar \n"
                     "el uso de un controlador de entrada \n"
                     "distinto.");
               break;
         }
         break;
      case MENU_LABEL_LOAD_CONTENT:
         snprintf(s, len,
               "Cargar contenido. \n"
               "Buscar contenido. \n"
               " \n"
               "Para cargar contenidos necesitas \n"
               "un 'ncleo' y un archivo de contenido.\n"
               " \n"
               "Para controlar el lugar donde el men \n"
               "empieza a buscar contenidos, cambia \n"
               "la opcin 'Carpeta del navegador de \n"
               "archivos'. En caso de que no est \n"
               "configurada, empezar desde la raz.\n"
               " \n"
               "El navegador filtrar las extensiones \n"
               "del ltimo ncleo seleccionado en \n"
               "'Cargar ncleo' y lo utilizar al \n"
               "cargar un contenido."
               );
         break;
      case MENU_LABEL_CORE_LIST:
         snprintf(s, len,
               "Cargar ncleo. \n"
               " \n"
               "Busca una implementacin de ncleo \n"
               "para libretro. El navegador empezar \n"
               "desde la ruta de tu carpeta de ncleos.\n"
               "Si est en blanco, empezar desde \n"
               "la raz.\n"
               " \n"
               "Si la carpeta de ncleos es una carpeta,\n"
               "el men la utilizar como carpeta \n"
               "base. Si la carpeta de ncleos es una \n"
               "ruta completa, empezar en la carpeta \n"
               "donde se encuentre el archivo.");
         break;
      case MENU_LABEL_LOAD_CONTENT_HISTORY:
         snprintf(s, len,
               "Cargar contenido del historial. \n"
               " \n"
               "Cuando se cargan contenidos, estos y \n"
               "las combinaciones de ncleos de libretro \n"
               "se guardan en el historial. \n"
               " \n"
               "El historial se guarda en un archivo en la \n"
               "misma carpeta que el archivo de configura- \n"
               "cin de RetroArch. Si no se ha cargado un \n"
               "archivo de configuracin al iniciar, no se \n"
               "guardar ni cargar el historial, y la \n"
               "opcin no existir en el men principal."
               );
         break;
      case MENU_LABEL_VIDEO_DRIVER:
         if (settings)
            driver_hash = msg_hash_calculate(settings->video.driver);

         switch (driver_hash)
         {
            case MENU_LABEL_VIDEO_DRIVER_GL:
               snprintf(s, len,
                     "Controlador de vdeo OpenGL. \n"
                     " \n"
                     "Este controlador permite que los ncleos \n"
                     "libretro GL se utilicen, adems de las \n"
                     "implementaciones renderizadas por\n"
                     "software del ncleo.\n"
                     " \n"
                     "El rendimiento de las implementaciones \n"
                     "por software y libretro GL dependen \n"
                     "del controlador GL que tenga tu \n"
                     "tarjeta grfica.");
               break;
            case MENU_LABEL_VIDEO_DRIVER_SDL2:
               snprintf(s, len,
                     "Controlador de vdeo SDL 2.\n"
                     " \n"
                     "Este es un controlador de vdeo por \n"
                     "software SDL 2.\n"
                     " \n"
                     "El rendimiento para las implementaciones \n"
                     "libretro por software depende de la \n"
                     "implementacin SDL de tu plataforma.");
               break;
            case MENU_LABEL_VIDEO_DRIVER_SDL1:
               snprintf(s, len,
                     "Controlador de vdeo SDL.\n"
                     " \n"
                     "Este es un controlador de vdeo por \n"
                     "software SDL 1.2.\n"
                     " \n"
                     "Su rendimiento es considerado inferior \n"
                     "a lo ptimo. Utilzalo nicamente como \n"
                     "ltimo recurso.");
               break;
            case MENU_LABEL_VIDEO_DRIVER_D3D:
               snprintf(s, len,
                     "Controlador de vdeo Direct3D. \n"
                     " \n"
                     "El rendimiento de los ncleos que \n"
                     "rendericen por software depender \n"
                     "del controlador D3D de tu tarjeta \n"
                     "grfica.");
               break;
            case MENU_LABEL_VIDEO_DRIVER_EXYNOS:
               snprintf(s, len,
                     "Controlador de vdeo Exynos-G2D. \n"
                     " \n"
                     "Este es un controlador de vdeo Exynos \n"
                     "de bajo nivel. Utiliza el bloque G2D \n"
                     "del SoC Exynos de Samsung para las \n"
                     "operaciones de blit. \n"
                     " \n"
                     "El rendimiento de los ncleos \n"
                     "renderizados por software debera \n"
                     "ser ptimo.");
               break;
            case MENU_LABEL_VIDEO_DRIVER_DRM:
               snprintf(s, len,
                     "Controlador de vdeo de DRM simple. \n"
                     " \n"
                     "Este es un controlador de vdeo que \n"
                     "usa libdrm para escalado por hardware \n"
                     "mediante los overlays de la GPU. \n"
                     " \n"
                     "El blitting se hace por software.");
               break;
            case MENU_LABEL_VIDEO_DRIVER_SUNXI:
               snprintf(s, len,
                     "Controlador de vdeo Sunxi-G2D. \n"
                     " \n"
                     "Este es un controlador de vdeo Sunxi \n"
                     "de bajo nivel. Utiliza el bloque G2D \n"
                     "de todos los SoC Allwinner.");
               break;
            default:
               snprintf(s, len,
                     "Controlador de vdeo actual.");
               break;
         }
         break;
      case MENU_LABEL_AUDIO_DSP_PLUGIN:
         snprintf(s, len,
               "Plugin de sonido DSP.\n"
               "Procesa el sonido antes de enviarlo \n"
               "al controlador."
               );
         break;
      case MENU_LABEL_AUDIO_RESAMPLER_DRIVER:
         if (settings)
            driver_hash = msg_hash_calculate(settings->audio.resampler);

         switch (driver_hash)
         {
            case MENU_LABEL_AUDIO_RESAMPLER_DRIVER_SINC:
               snprintf(s, len,
                     "Implementacin SINC en ventana.");
               break;
            case MENU_LABEL_AUDIO_RESAMPLER_DRIVER_CC:
               snprintf(s, len,
                     "Implementacin de cosenos complejos.");
               break;
         }
         break;
      case MENU_LABEL_VIDEO_SHADER_PRESET:
         snprintf(s, len,
               "Cargar preajustes de shaders. \n"
               " \n"
               " Carga directamente un preajuste "
#ifdef HAVE_CG
               "Cg"
#endif
#ifdef HAVE_GLSL
#ifdef HAVE_CG
               "/"
#endif
               "GLSL"
#endif
#ifdef HAVE_HLSL
#if defined(HAVE_CG) || defined(HAVE_HLSL)
               "/"
#endif
               "HLSL"
#endif
               ". \n"
               "El men de shaders se actualizar. \n"
               " \n"
               "Si el CGP utiliza mtodos de escalado \n"
               "complejos (por ejemplo, escalado de \n"
               "origen, el mismo factor para X/Y), podra \n"
               "no mostrar un factor de escalado correcto \n"
               "en el men."
               );
         break;
      case MENU_LABEL_VIDEO_SHADER_SCALE_PASS:
         snprintf(s, len,
               "La escala de esta pasada. \n"
               " \n"
               "El factor de escala es acumulativo, \n"
               "por ejemplo: 2x para el primer pase y \n"
               "2x para el segundo dar una escala \n"
               "total de 4x. \n"
               " \n"
               "Si el ltimo pase tiene un factor de \n"
               "escala, el resultado se estirar por \n"
               "toda la pantalla con el filtro espe- \n"
               "cificado en 'Filtro predeterminado'. \n"
               " \n"
               "Si has seleccionado 'No importa', se \n"
               "utilizar o bien la escala 1x o se \n"
               "estirar a pantalla completa en funcin \n"
               "de si es el ltimo pase o no."
               );
         break;
      case MENU_LABEL_VIDEO_SHADER_NUM_PASSES:
         snprintf(s, len,
               "Pases de shaders. \n"
               " \n"
               "RetroArch permite mezclar diversos shaders \n"
               "con pasadas arbitrarias, filtros persona- \n"
               "lizados de hardware y factores de escala. \n"
               " \n"
               "Esta opcin especifica la cantidad de pasadas \n"
               "de shaders a utilizar. Si seleccionas 0 y \n"
               "luego 'Aplicar cambios en shaders', \n"
               "utilizars un shader 'en blanco'. \n"
               " \n"
               "La opcin filtro predeterminado afectar \n"
               "al filtro de estiramiento.");
         break;
      case MENU_LABEL_VIDEO_SHADER_PARAMETERS:
         snprintf(s, len,
               "Parmetros de shaders. \n"
               " \n"
               "Modifica directamente el shader actual. \n"
               "No se guardar en el preajuste CGP/GLSLP.");
         break;
      case MENU_LABEL_VIDEO_SHADER_PRESET_PARAMETERS:
         snprintf(s, len,
               "Parmetros del preajuste de shaders. \n"
               " \n"
               "Modifica el preajuste de shaders que \n"
               "se encuentra actualmente en el men."
               );
         break;
      case MENU_LABEL_VIDEO_SHADER_PASS:
         snprintf(s, len,
               "Ruta al shader. \n"
               " \n"
               "Todos los shaders deben ser del \n"
               "mismo tipo (por ejemplo: CG, GLSL \n"
               "o HLSL). \n"
               " \n"
               "Modifica la carpeta de shaders para \n"
               "indicar la ruta en la que el navegador \n"
               "comenzar a buscar shaders."
               );
         break;
      case MENU_LABEL_CONFIG_SAVE_ON_EXIT:
         snprintf(s, len,
               "Guarda la configuracin al salir.\n"
               "Es til para el men, ya que los \n"
               "ajustes pueden ser modificados. \n"
               "Sobrescribir la configuracin. \n"
               " \n"
               "No se preservarn los #include \n"
               "y los comentarios. \n"
               " \n"
               "El archivo de configuracin se \n"
               "considera inmutable por diseo \n"
               "ya que es probable que el usuario \n"
               "lo controle, y no debera ser \n"
               "sobrescrito sin que el usuario \n"
               "no se entere."
#if defined(RARCH_CONSOLE) || defined(RARCH_MOBILE)
               "\nSin embargo, no suele ser el caso \n"
               "en consolas, donde no es posible \n"
               "examinar el archivo de configuracin \n"
               "de forma manual."
#endif
               );
         break;
      case MENU_LABEL_VIDEO_SHADER_FILTER_PASS:
         snprintf(s, len,
               "Filtro de hardware para esta pasada. \n"
               " \n"
               "Si se ha seleccionado 'No importa', \n"
               "se utilizar el filtro predeterminado."
               );
         break;
      case MENU_LABEL_AUTOSAVE_INTERVAL:
         snprintf(s, len,
               "Guarda automticamente la SRAM \n"
               "no voltil en intervalos regulares.\n"
               " \n"
               "Esta opcin est desactivada por \n"
               "defecto a menos que se haya configurado \n"
               "lo contrario. El intervalo se mide \n"
               "en segundos. \n"
               " \n"
               "Si utlizas 0, desactivars el \n"
               "guardado automtico.");
         break;
      case MENU_LABEL_INPUT_BIND_DEVICE_TYPE:
         snprintf(s, len,
               "Tipo de dispositivo de entrada. \n"
               " \n"
               "Selecciona el tipo de dispositivo a \n"
               "utilizar. Es relevante para el ncleo \n"
               "de libretro."
               );
         break;
      case MENU_LABEL_LIBRETRO_LOG_LEVEL:
         snprintf(s, len,
               "Ajusta el nivel de registro de los \n"
               "ncleos de libretro (GET_LOG_INTERFACE). \n"
               " \n"
               " Si un nivel de registro indicado por \n"
               " un ncleo de libretro est por debajo \n"
               " del nivel indicado en libretro_log, \n"
               " ser ignorado.\n"
               " \n"
               " Los registros DEBUG siempre son ignorados \n"
               " a menos que est activado el modo \n"
               " de verbosidad (--verbose).\n"
               " \n"
               " DEBUG = 0\n"
               " INFO  = 1\n"
               " WARN  = 2\n"
               " ERROR = 3"
               );
         break;
      case MENU_LABEL_STATE_SLOT_INCREASE:
      case MENU_LABEL_STATE_SLOT_DECREASE:
         snprintf(s, len,
               "Ranuras de guardados rpidos.\n"
               " \n"
               " Si se selecciona la ranura 0, el nombre \n"
               " del guardado rpido es *.state \n"
               " (o lo que est definido en la lnea \n"
               "de comandos).\n"
               "Si la ranura es != 0, la ruta ser (path)(d), \n"
               "siendo (d) el nmero de ranura.");
         break;
      case MENU_LABEL_SHADER_APPLY_CHANGES:
         snprintf(s, len,
               "Aplicar cambios en shaders. \n"
               " \n"
               "Despus de cambiar los ajustes de shaders, \n"
               "utiliza esta opcin para aplicar los \n"
               "cambios. \n"
               " \n"
               "Los cambios en los shaders es una \n"
               "operacin que ocupa bastante memoria, \n"
               "as que han de hacerse de forma \n"
               "explcita. \n"
               " \n"
               "Al aplicar shaders, se guarda su configuracin \n"
               "a un archivo temporal (menu.cgp ormenu.glslp) \n"
               "y se cargan. El archivo persistir al \n"
               "abandonar RetroArch. El archivo se guarda en \n"
               "la carpeta Shader."
               );
         break;
      case MENU_LABEL_INPUT_BIND_DEVICE_ID:
         snprintf(s, len,
               "Dispositivo de entrada. \n"
               " \n"
               "Selecciona el mando a utilizar por \n"
               "el usuario N. \n"
               "El nombre del mando est disponible."
               );
         break;
      case MENU_LABEL_MENU_TOGGLE:
         snprintf(s, len,
               "Muestra u oculta el men.");
         break;
      case MENU_LABEL_GRAB_MOUSE_TOGGLE:
         snprintf(s, len,
               "Permite o no capturar el ratn.\n"
               " \n"
               "Al capturar el ratn, RetroArch lo oculta \n"
               "y mantiene el puntero del ratn dentro de \n"
               "la ventana para que la entrada relativa \n"
               "del ratn funcione mejor.");
         break;
      case MENU_LABEL_DISK_NEXT:
         snprintf(s, len,
               "Rota entre las imgenes de disco. \n"
               "Utiliza esta opcin despus de \n"
               "expulsar un disco. \n"
               " \n"
               " Termina la operacin volviendo \n"
               " a conmutar el botn de expulsin.");
         break;
      case MENU_LABEL_VIDEO_FILTER:
#ifdef HAVE_FILTERS_BUILTIN
         snprintf(s, len,
               "Filtro de vdeo basado en la CPU.");
#else
         snprintf(s, len,
               "Filtro de vdeo basado en la CPU.\n"
               " \n"
               "Ruta a una librera dinmica.");
#endif
         break;
      case MENU_LABEL_AUDIO_DEVICE:
         snprintf(s, len,
               "Anula el dispositivo de sonido \n"
               "predeterminado que utiliza el \n"
               "controlador de sonido.\n"
               "Esta opcin depende del contro- \n"
               "lador. Por ejemplo:\n"
#ifdef HAVE_ALSA
               " \n"
               "ALSA necesita un dispositivo PCM."
#endif
#ifdef HAVE_OSS
               " \n"
               "OSS necesita una ruta (por ejemplo: \n"
               "/dev/dsp)."
#endif
#ifdef HAVE_JACK
               " \n"
               "JACK necesita nombres de puertos \n"
               "(por ejemplo: system:playback1\n"
               ",system:playback_2)."
#endif
#ifdef HAVE_RSOUND
               " \n"
               "RSound necesita una direccin IP \n"
               "de un servidor RSound."
#endif
               );
         break;
      case MENU_LABEL_DISK_EJECT_TOGGLE:
         snprintf(s, len,
               "Alterna el botn de expulsin \n"
               "de discos.\n"
               " \n"
               "Se utiliza para contenidos \n"
               "que utilicen varios discos.");
         break;
      case MENU_LABEL_ENABLE_HOTKEY:
         snprintf(s, len,
               "Activa otras teclas rpidas.\n"
               " \n"
               "Si esta tecla rpida est asignada a un \n"
               "teclado, un botn o un eje de un joystick, \n"
               "el resto de teclas rpidas se desactivarn \n"
               "a menos que esta tecla se mantenga pulsada \n"
               "al mismo tiempo. \n"
               " \n"
               "Esto sirve para implementaciones centradas \n"
               "en RETRO_KEYBOARD, que ocupan una gran \n"
               "porcin del teclado y no es una buena idea \n"
               "que las teclas rpidas interfieran con \n"
               "el funcionamiento normal.");
         break;
      case MENU_LABEL_REWIND_ENABLE:
         snprintf(s, len,
               "Activa el rebobinado.\n"
               " \n"
               "Esto consume rendimiento, as que \n"
               "est desactivado por defecto.");
         break;
      case MENU_LABEL_LIBRETRO_DIR_PATH:
         snprintf(s, len,
               "Carpeta de ncleos. \n"
               " \n"
               "Una carpeta donde se buscarn las \n"
               "implementaciones de ncleos libretro.");
         break;
      case MENU_LABEL_VIDEO_REFRESH_RATE_AUTO:
         {
            /* Work around C89 limitations */
            char u[501];
            char t[501];

            strlcpy(t,
                  "Frecuencia de actualizacin automtica.\n"
                  " \n"
                  "La frecuencia de actualizacin precisa del \n"
                  "monitor (en Hz). Se utiliza para calcular \n"
                  "la frecuencia de entrada de sonido con esta \n"
                  "frmula: \n"
                  " \n"
                  "audio_input_rate = veloc. de entrada de juego \n"
                  "* frecuencia de actualizacin de pantalla / \n"
                  "frecuencia de actualizacin de juego \n"
                  " \n", sizeof(t));
            strlcpy(u,
                  "Si la implementacin no indica un valor, se \n"
                  "asumir de forma predeterminada el sistema \n"
                  "NTSC por compatibilidad.\n"
                  " \n"
                  "Este valor debera ser lo ms similar a 60Hz \n"
                  "para evitar cambios en el tono. Si tu \n"
                  "monitor no funciona a 60Hz o similar, \n"
                  "desactiva la sincrona vertical y deja \n"
                  "esta opcin en su valor predeterminado.", sizeof(u));
            strlcat(s, t, len);
            strlcat(s, u, len);
         }
         break;
      case MENU_LABEL_VIDEO_ROTATION:
         snprintf(s, len,
               "Fuerza una rotacin concreta \n"
               "de la pantalla.\n"
               " \n"
               "La rotacin se aade a las rotaciones \n"
               "que indique el ncleo libretro (ver \n"
               "Permitir rotacin de vdeo).");
         break;
      case MENU_LABEL_VIDEO_SCALE:
         snprintf(s, len,
               "Resolucin en pantalla completa.\n"
               " \n"
               "Si se indica 0, se utilizar la \n"
               "resolucin del entorno.\n");
         break;
      case MENU_LABEL_FASTFORWARD_RATIO:
         snprintf(s, len,
               "Proporcin de aceleracin."
               " \n"
               "Indica la velocidad mxima a la que \n"
               "se ejecutar el contenido mientras \n"
               "es acelerado.\n"
               " \n"
               " (Por ejemplo, 5.0 para un contenido \n"
               "a 60 FPS => Mximo de 300 FPS).\n"
               " \n"
               "RetroArch pasar a segundo plano \n"
               "para asegurarse de que no se supere \n"
               "la velocidad mxima. Este lmite \n"
               "no es completamente preciso.");
         break;
      case MENU_LABEL_VIDEO_MONITOR_INDEX:
         snprintf(s, len,
               "Monitor preferido.\n"
               " \n"
               "0 (predeterminado) significa que no hay \n"
               "preferencia por un monitor en concreto, \n"
               "y de 1 en adelante (siendo 1 el primer \n"
               "monitor) sugiere a RetroArch que \n"
               "utilice ese monitor especfico.");
         break;
      case MENU_LABEL_VIDEO_CROP_OVERSCAN:
         snprintf(s, len,
               "Obliga a recortar los fotogramas \n"
               "con overscan.\n"
               " \n"
               "El comportamiento exacto de esta \n"
               "opcin depende de la implementacin \n"
               "del ncleo.");
         break;
      case MENU_LABEL_VIDEO_SCALE_INTEGER:
         snprintf(s, len,
               "Solo escala el vdeo con nmeros \n"
               "enteros.\n"
               " \n"
               "El tamao base depende de la geometra \n"
               "que indique el sistema y de la \n"
               "proporcin de aspecto.\n"
               " \n"
               "Si no se ha configurado la opcin \n"
               "Forzar proporcin, los valores X/Y \n"
               "se escalarn de forma independiente \n"
               "segn nmeros enteros.");
         break;
      case MENU_LABEL_AUDIO_VOLUME:
         snprintf(s, len,
               "Volumen de sonido expresado en dB.\n"
               " \n"
               "0 dB es el volumen normal, sin ganancia \n"
               "aplicada. La ganancia se puede controlar \n"
               "en tiempo real con los botones de \n"
               "Subir volumen / Bajar volumen.");
         break;
      case MENU_LABEL_AUDIO_RATE_CONTROL_DELTA:
         snprintf(s, len,
               "Control de la frecuencia de sonido.\n"
               " \n"
               "Si seleccionas 0, desactivars el control \n"
               "de la frecuencia. Cualquier otro valor \n"
               "cambiar el delta de control de la \n"
               "frecuencia de sonido.\n"
               " \n"
               "Define cunta frecuencia de entrada puede \n"
               "ajustarse de forma dinmica.\n"
               " \n"
               " La frecuencia de entrada se define como: \n"
               " frecuencia de entrada * (1.0 +/- \n"
               "(delta de control de frecuencia))");
         break;
      case MENU_LABEL_AUDIO_MAX_TIMING_SKEW:
         snprintf(s, len,
               "Variacin mxima en la sincrona de \n"
               "sonido.\n"
               " \n"
               "Define la variacin mxima de la \n"
               "frecuencia de entrada. Podras aumentar \n"
               "el valor para cambiar la sincrona, por \n"
               "ejemplo, si ejecutas ncleos PAL en \n"
               "monitores NTSC, a cambio de tener un \n"
               "tono de sonido impreciso.\n"
               " \n"
               " La frecuencia de entrada se define como: \n"
               " frecuencia de entrada * (1.0 +/- \n"
               "(variacin mxima de sincrona))");
         break;
      case MENU_LABEL_OVERLAY_NEXT:
         snprintf(s, len,
               "Cambia a la siguiente superposicin.\n"
               " \n"
               "Se expande alrededor.");
         break;
      case MENU_LABEL_LOG_VERBOSITY:
         snprintf(s, len,
               "Activa o desactiva el nivel de \n"
               "verbosidad del frontend.");
         break;
      case MENU_LABEL_VOLUME_UP:
         snprintf(s, len,
               "Aumenta el volumen del sonido.");
         break;
      case MENU_LABEL_VOLUME_DOWN:
         snprintf(s, len,
               "Disminuye el volumen del sonido.");
         break;
      case MENU_LABEL_VIDEO_DISABLE_COMPOSITION:
         snprintf(s, len,
               "Desactiva por la fuerza la composicin.\n"
               "Actualmente solo funciona en \n"
               "Windows Vista/7.");
         break;
      case MENU_LABEL_PERFCNT_ENABLE:
         snprintf(s, len,
               "Activa o desactiva los contadores \n"
               "de rendimiento del frontend.");
         break;
      case MENU_LABEL_SYSTEM_DIRECTORY:
         snprintf(s, len,
               "Carpeta de sistemas. \n"
               " \n"
               "Asigna la carpeta 'system'.\n"
               "Los ncleos pueden buscar dentro \n"
               "de ella para cargar BIOS, \n"
               "configuraciones especficas para \n"
               "un sistema, etc.");
         break;
      case MENU_LABEL_SAVESTATE_AUTO_SAVE:
         snprintf(s, len,
               "Guarda automticamente un guardado rpido \n"
               "al terminar la ejecucin de RetroArch.\n"
               " \n"
               "RetroArch cargar automticamente cualquier \n"
               "guardado rpido que se encuentre en esa ruta \n"
               "al arrancar si la opcin 'Cargar guardado \n"
               "rpido automticamente' est activada.");
         break;
      case MENU_LABEL_VIDEO_THREADED:
         snprintf(s, len,
               "Utilizar un controlador de vdeo \n"
               "por hilos.\n"
               " \n"
               "Esta opcin podra mejorar el rendimiento \n"
               "a costa de ganar retraso y perder fluidez \n"
               "de vdeo.");
         break;
      case MENU_LABEL_VIDEO_VSYNC:
         snprintf(s, len,
               "Sincrona vertical para vdeo.\n");
         break;
      case MENU_LABEL_VIDEO_HARD_SYNC:
         snprintf(s, len,
               "Intenta sincronizar la CPU y la GPU \n"
               "de una forma ms estricta.\n"
               " \n"
               "Puede reducir el retraso a costa \n"
               "de perder rendimiento.");
         break;
      case MENU_LABEL_REWIND_GRANULARITY:
         snprintf(s, len,
               "Nivel de detalle del rebobinado.\n"
               " \n"
               "Cuando se rebobina un nmero determinado \n"
               "de fotogramas, puedes rebobinar varios \n"
               "fotogramas a la vez, aumentando la \n"
               "velocidad de rebobinado.");
         break;
      case MENU_LABEL_SCREENSHOT:
         snprintf(s, len,
               "Capturar la pantalla.");
         break;
      case MENU_LABEL_VIDEO_FRAME_DELAY:
         snprintf(s, len,
               "Ajusta la cantidad de retraso en \n"
               "milisegundos despus de la sincrona \n"
               "vertical antes de ejecutar el ncleo.\n"
               "\n"
               "Puede reducir el retraso a costa de \n"
               "aumentar la probabilidad de perder \n"
               "fluidez.\n"
               " \n"
               "El valor mximo es 15.");
         break;
      case MENU_LABEL_VIDEO_HARD_SYNC_FRAMES:
         snprintf(s, len,
               "Ajusta la cantidad de fotogramas \n"
               "que puede adelantar la CPU a la GPU \n"
               "al utilizar 'Sincrona estricta \n"
               "de CPU'.\n"
               " \n"
               "El valor mximo es 3.\n"
               " \n"
               " 0: Sincroniza inmediatamente a GPU.\n"
               " 1: Sincroniza con el fotograma anterior.\n"
               " 2: Etc...");
         break;
      case MENU_LABEL_VIDEO_BLACK_FRAME_INSERTION:
         snprintf(s, len,
               "Introduce un fotograma negro \n"
               "entre fotogramas.\n"
               " \n"
               "Es til para los monitores que \n"
               "funcionan a 120 Hz y quieren jugar \n"
               "a material a 60 Hz eliminando el \n"
               "ghosting.\n"
               " \n"
               "La frecuencia de actualizacin de vdeo \n"
               "an debe configurarse como si utilizaras \n"
               "un monitor de 60 Hz (divide la frecuencia \n"
               "de actualizacin entre 2).");
         break;
      case MENU_LABEL_RGUI_SHOW_START_SCREEN:
         snprintf(s, len,
               "Muestra la pantalla de inicio en el men.\n"
               "Se desactiva automticamente cuando se \n"
               "ha visto por primera vez.\n"
               " \n"
               "Esta opcin solo se actualiza dentro \n"
               "de la configuracin si 'Guardar \n"
               "configuracin al salir' est activado.\n");
         break;
      case MENU_LABEL_CORE_SPECIFIC_CONFIG:
         snprintf(s, len,
               "Carga un archivo de configuracin \n"
               "concreto en funcin del ncleo \n"
               "que se vaya a utilizar.\n");
         break;
      case MENU_LABEL_VIDEO_FULLSCREEN:
         snprintf(s, len, 
                "Activa o desactiva el modo \n"
                "en pantalla completa.");
         break;
      case MENU_LABEL_BLOCK_SRAM_OVERWRITE:
         snprintf(s, len,
               "Impide que la SRAM sea sobrescrita \n"
               "mientras se carga un guardado rpido.\n"
               " \n"
               "Podra provocar defectos en los \n"
               "juegos.");
         break;
      case MENU_LABEL_PAUSE_NONACTIVE:
         snprintf(s, len,
               "Pausa el juego cuando la ventana \n"
               "pasa a estar en segundo plano.");
         break;
      case MENU_LABEL_VIDEO_GPU_SCREENSHOT:
         snprintf(s, len,
               "Si es posible, las capturas de \n"
               "pantalla se realizan del material \n"
               "pos-shaders de GPU.");
         break;
      case MENU_LABEL_SCREENSHOT_DIRECTORY:
         snprintf(s, len,
               "Carpeta de capturas de pantalla. \n"
               " \n"
               "En esta carpeta se guardarn \n"
               "las capturas de pantalla."
               );
         break;
      case MENU_LABEL_VIDEO_SWAP_INTERVAL:
         snprintf(s, len,
               "Intervalo de cambio de la \n"
               "sincrona vertical.\n"
               " \n"
               "Utiliza un intervalo personalizado para \n"
               "la sincrona vertical. Utiliza este \n"
               "parmetro para reducir a la mitad \n"
               "la frecuencia de actualizacin \n"
               "del monitor.");
         break;
      case MENU_LABEL_SAVEFILE_DIRECTORY:
         snprintf(s, len,
               "Carpeta de partidas guardadas. \n"
               " \n"
               "Almacena todas las partidas guardadas \n"
               "(*.srm) en esta carpeta. Incluye los \n"
               "archivos relacionados, como .bsv, .rt, \n"
               ".psrm, etctera.\n"
               " \n"
               "Esta carpeta ser ignorada si se \n"
               "utilizan explcitamente las opciones por \n"
               "lnea de comandos.");
         break;
      case MENU_LABEL_SAVESTATE_DIRECTORY:
         snprintf(s, len,
               "Carpeta de guardados rpidos. \n"
               " \n"
               "Almacena todos los guardados rpidos \n"
               "(*.state) en esta carpeta.\n"
               " \n"
               "Esta carpeta ser ignorada si se \n"
               "utilizan explcitamente las opciones por \n"
               "lnea de comandos.");
         break;
      case MENU_LABEL_ASSETS_DIRECTORY:
         snprintf(s, len,
               "Carpeta de recursos. \n"
               " \n"
               "Esta ubicacin es donde las interfaces \n"
               "de mens buscan de forma predeterminada \n"
               "los recursos a cargar, etc.");
         break;
      case MENU_LABEL_DYNAMIC_WALLPAPERS_DIRECTORY:
         snprintf(s, len,
               "Carpeta de fondos dinmicos de pantalla. \n"
               " \n"
               "Aqu se almacenan los fondos de pantalla \n"
               "que el men cargar de forma dinmica \n"
               "segn el contexto.");
         break;
      case MENU_LABEL_SLOWMOTION_RATIO:
         snprintf(s, len,
               "Cantidad de velocidad reducida."
               " \n"
               "Al reducir la velocidad, el contenido \n"
               "se ralentizar segn este factor.");
         break;
      case MENU_LABEL_INPUT_AXIS_THRESHOLD:
         snprintf(s, len,
               "Define el margen de los ejes.\n"
               " \n"
               "Indica la distancia mnima que debe \n"
               "recorrer un eje para que provoque \n"
               "una pulsacin del botn.\n"
               "Los valores posibles son [0.0, 1.0].");
         break;
      case MENU_LABEL_INPUT_TURBO_PERIOD:
         snprintf(s, len, 
               "Perodo de turbo.\n"
               " \n"
               "Describe la velocidad con la que se \n"
               "pulsan los botones en el modo turbo."
               );
         break;
      case MENU_LABEL_INPUT_AUTODETECT_ENABLE:
         snprintf(s, len,
               "Activa la autoconfiguracin de \n"
               "entrada. \n"
               " \n"
               "Intentar configurar de forma \n"
               "automtica los mandos que se \n"
               "conecten en cualquier momento, \n"
               "como el sistema Plug-and-play.");
         break;
      case MENU_LABEL_CAMERA_ALLOW:
         snprintf(s, len,
               "Permite o no el acceso a la cmara \n"
               "por parte de los ncleos.");
         break;
      case MENU_LABEL_LOCATION_ALLOW:
         snprintf(s, len,
               "Permite o no los servicios de \n"
               "localizacin por parte de los ncleos.");
         break;
      case MENU_LABEL_TURBO:
         snprintf(s, len,
               "Permite usar el turbo.\n"
               " \n"
               "Si mantienes pulsado el botn de turbo \n"
               "mientras pulsas otro botn, este ltimo \n"
               "entrar en un modo turbo en el que el \n"
               "estado del botn se modula mediante una \n"
               "seal peridica. \n"
               " \n"
               "La modulacin se detiene cuando se deja \n"
               "de pulsar el botn original (no el botn \n"
               "de turbo).");
         break;
      case MENU_LABEL_OSK_ENABLE:
         snprintf(s, len,
               "Activa o desactiva el teclado \n"
               "en pantalla.");
         break;
      case MENU_LABEL_AUDIO_MUTE:
         snprintf(s, len,
               "Silencia o no el sonido.");
         break;
      case MENU_LABEL_REWIND:
         snprintf(s, len,
               "Mantn pulsado este botn para rebobinar.\n"
               " \n"
               "Para que este botn funcione, debes tener \n"
               "activada la opcin de rebobinar.");
         break;
      case MENU_LABEL_EXIT_EMULATOR:
         snprintf(s, len,
               "Asigna una tecla para abandonar \n"
               "RetroArch limpiamente."
#if !defined(RARCH_MOBILE) && !defined(RARCH_CONSOLE)
               "\nSi cierras el programa con \n"
               "cualquier forma brusca (SIGKILL, \n"
               "etc.) no se guardar la RAM, etc. \n"
               "En sistemas Unix, SIGINT/SIGTERM \n"
               "permite un cierre limpio."
#endif
               );
         break;
      case MENU_LABEL_LOAD_STATE:
         snprintf(s, len,
               "Carga el guardado rpido.");
         break;
      case MENU_LABEL_SAVE_STATE:
         snprintf(s, len,
               "Guarda rpidamente la partida.");
         break;
      case MENU_LABEL_NETPLAY_FLIP_PLAYERS:
         snprintf(s, len,
               "Cambia los usuarios en red.");
         break;
      case MENU_LABEL_CHEAT_INDEX_PLUS:
         snprintf(s, len,
               "Aumenta el ndice de trucos.\n");
         break;
      case MENU_LABEL_CHEAT_INDEX_MINUS:
         snprintf(s, len,
               "Disminuye el ndice de trucos.\n");
         break;
      case MENU_LABEL_SHADER_PREV:
         snprintf(s, len,
               "Aplica el shader anterior de la carpeta.");
         break;
      case MENU_LABEL_SHADER_NEXT:
         snprintf(s, len,
               "Aplica el siguiente shader de la carpeta.");
         break;
      case MENU_LABEL_RESET:
         snprintf(s, len,
               "Reinicia el contenido.\n");
         break;
      case MENU_LABEL_PAUSE_TOGGLE:
         snprintf(s, len,
               "Alterna el estado en pausa.");
         break;
      case MENU_LABEL_CHEAT_TOGGLE:
         snprintf(s, len,
               "Alterna el ndice de trucos.\n");
         break;
      case MENU_LABEL_HOLD_FAST_FORWARD:
         snprintf(s, len,
               "Mantn pulsado este botn para avanzar \n"
               "rpidamente. Sultalo para desactivar \n"
               "esta funcin.");
         break;
      case MENU_LABEL_SLOWMOTION:
         snprintf(s, len,
               "Mantn pulsado este botn para \n"
               "ir a cmara lenta.");
         break;
      case MENU_LABEL_FRAME_ADVANCE:
         snprintf(s, len,
               "Avanza un fotograma mientras \n"
               "el contenido est en pausa.");
         break;
      case MENU_LABEL_MOVIE_RECORD_TOGGLE:
         snprintf(s, len,
               "Activa o desactiva la grabacin.");
         break;
      case MENU_LABEL_L_X_PLUS:
      case MENU_LABEL_L_X_MINUS:
      case MENU_LABEL_L_Y_PLUS:
      case MENU_LABEL_L_Y_MINUS:
      case MENU_LABEL_R_X_PLUS:
      case MENU_LABEL_R_X_MINUS:
      case MENU_LABEL_R_Y_PLUS:
      case MENU_LABEL_R_Y_MINUS:
         snprintf(s, len,
               "El eje de un joystick analgico \n"
               "(estilo DualShock).\n"
               " \n"
               "Se asigna como siempre, sin embargo, si se \n"
               "asigna un eje analgico real, puede leerse \n"
               "de forma analgica.\n"
               " \n"
               "El eje X positivo es hacia la derecha. \n"
               "El eje Y positivo es hacia abajo.");
         break;
      case MENU_LABEL_VALUE_WHAT_IS_A_CORE_DESC:
         snprintf(s, len,
               "RetroArch, por si solo, no hace nada. \n"
               " \n"
               "Para que haga algo necesitas cargar \n"
               "un programa en l. \n"
               "\n"
               "Llamamos a estos programas 'ncleos \n"
               "libretro', o 'ncleos' para abreviar. \n"
               " \n"
               "Para cargar un ncleo, selecciona uno \n"
               "en 'Cargar ncleo'. \n"
               " \n"
#ifdef HAVE_NETWORKING
               "Puedes conseguir ncleos de varias formas: \n"
               "* Descrgalos en:\n"
               "'%s' -> '%s'.\n"
               "* Cpialos manualmente a:\n"
               "'%s'.",
               msg_hash_to_str(MENU_ENUM_LABEL_VALUE_ONLINE_UPDATER),
               msg_hash_to_str(MENU_ENUM_LABEL_VALUE_CORE_UPDATER_LIST),
               msg_hash_to_str(MENU_ENUM_LABEL_VALUE_LIBRETRO_DIR_PATH)
#else
               "Puedes conseguir ncleos si los\n"
               "trasladas a mano en la carpeta\n"
               "'%s'.",
               msg_hash_to_str(MENU_ENUM_LABEL_VALUE_LIBRETRO_DIR_PATH)
#endif
               );
         break;
      case MENU_LABEL_VALUE_HELP_CHANGE_VIRTUAL_GAMEPAD_DESC:
         snprintf(s, len,
               "Puedes cambiar la superposicin del \n"
               "mando virtual si vas a '%s' \n"
               "-> '%s'."
               " \n"
               "Desde ah puedes cambiar la superposicin, \n"
               "el tamao y opacidad de sus botones, etc.\n"
               " \n"
               "NOTA: Las superposiciones de mandos \n"
               "virtuales estn ocultas de forma \n"
               "predeterminada si ests dentro del men. \n"
               "Si quieres cambiar este comportamiento, \n"
               "cambia '%s' a Desactivado/false.",
               msg_hash_to_str(MENU_ENUM_LABEL_VALUE_SETTINGS),
               msg_hash_to_str(MENU_ENUM_LABEL_VALUE_OVERLAY_SETTINGS),
               msg_hash_to_str(MENU_ENUM_LABEL_VALUE_INPUT_OVERLAY_HIDE_IN_MENU)
               );
         break;
      default:
         if (s[0] == '\0')
            strlcpy(s, msg_hash_to_str(MENU_ENUM_LABEL_VALUE_NO_INFORMATION_AVAILABLE), len);
         return -1;
   }

   return 0;
}

const char *msg_hash_to_str_es(enum msg_hash_enums msg)
{
   switch (msg)
   {
      case MSG_PROGRAM:
         return "RetroArch";
      case MSG_MOVIE_RECORD_STOPPED:
         return "Deteniendo grabacin de vdeo.";
      case MSG_MOVIE_PLAYBACK_ENDED:
         return "Reproduccin detenida.";
      case MSG_AUTOSAVE_FAILED:
         return "No se ha podido iniciar el autoguardado.";
      case MSG_NETPLAY_FAILED_MOVIE_PLAYBACK_HAS_STARTED:
         return "Se ha iniciado una reproduccin. No se puede ejecutar el juego en red.";
      case MSG_NETPLAY_FAILED:
         return "Error al iniciar el juego en red.";
      case MSG_LIBRETRO_ABI_BREAK:
         return "se ha compilado con una versin distinta a esta implementacin de libretro.";
      case MSG_REWIND_INIT_FAILED_NO_SAVESTATES:
         return "Esta implementacin no admite guardados rpidos. No se puede utilizar el rebobinado.";
      case MSG_REWIND_INIT_FAILED_THREADED_AUDIO:
         return "La implementacin utiliza sonido multincleo. No se puede utilizar el rebobinado.";
      case MSG_REWIND_INIT_FAILED:
         return "Error al iniciar el bfer de rebobinado. Rebobinado desactivado.";
      case MSG_REWIND_INIT:
         return "Iniciando bfer de rebobinado, tamao:";
      case MSG_CUSTOM_TIMING_GIVEN:
         return "Se ha indicado un ritmo personalizado";
      case MSG_VIEWPORT_SIZE_CALCULATION_FAILED:
         return "Error al calcular el tamao de ventana! Se utilizarn datos en bruto. Probablemente esto no acabe bien...";
      case MSG_HW_RENDERED_MUST_USE_POSTSHADED_RECORDING:
         return "El ncleo Libretro se renderiza por hardware. Es necesario utilizar la grabacin post-shaders.";
      case MSG_RECORDING_TO:
         return "Grabando a";
      case MSG_DETECTED_VIEWPORT_OF:
         return "Ventana detectada:";
      case MSG_TAKING_SCREENSHOT:
         return "Capturando pantalla.";
      case MSG_FAILED_TO_TAKE_SCREENSHOT:
         return "Error al capturar pantalla.";
      case MSG_FAILED_TO_START_RECORDING:
         return "Error al comenzar a grabar.";
      case MSG_RECORDING_TERMINATED_DUE_TO_RESIZE:
         return "Grabacin terminada por cambio de tamao.";
      case MSG_USING_LIBRETRO_DUMMY_CORE_RECORDING_SKIPPED:
         return "Se est utilizando el ncleo dummy de libretro. Anulando grabacin.";
      case MSG_UNKNOWN:
         return "Desconocido";
      case MSG_LOADING_CONTENT_FILE:
         return "Cargando archivo de contenido";
      case MSG_RECEIVED:
         return "recibido";
      case MSG_UNRECOGNIZED_COMMAND:
         return "Comando no reconocido";
      case MSG_SENDING_COMMAND:
         return "Enviando comando";
      case MSG_GOT_INVALID_DISK_INDEX:
         return "Se ha obtenido un ndice de disco no vlido.";
      case MSG_FAILED_TO_REMOVE_DISK_FROM_TRAY:
         return "Error al extraer el disco de la bandeja.";
      case MSG_REMOVED_DISK_FROM_TRAY:
         return "Se ha retirado el disco de la bandeja.";
      case MSG_VIRTUAL_DISK_TRAY:
         return "bandeja de disco virtual.";
      case MSG_FAILED_TO:
         return "Error:";
      case MSG_TO:
         return "a";
      case MSG_SAVING_RAM_TYPE:
         return "Guardando tipo de RAM";
      case MSG_SAVING_STATE:
         return "Guardando rpidamente";
      case MSG_LOADING_STATE:
         return "Cargando rpidamente";
      case MSG_FAILED_TO_LOAD_MOVIE_FILE:
         return "Error al cargar el archivo de pelcula";
      case MSG_FAILED_TO_LOAD_CONTENT:
         return "Error al cargar el contenido";
      case MSG_COULD_NOT_READ_CONTENT_FILE:
         return "No se ha podido leer el archivo de contenido";
      case MSG_GRAB_MOUSE_STATE:
         return "Capturar estado de ratn";
      case MSG_PAUSED:
         return "En pausa.";
      case MSG_UNPAUSED:
         return "Sin pausa.";
      case MSG_FAILED_TO_LOAD_OVERLAY:
         return "Error al cargar sobreimposicin.";
      case MSG_FAILED_TO_UNMUTE_AUDIO:
         return "Error al recuperar el sonido.";
      case MSG_AUDIO_MUTED:
         return "Sonido silenciado.";
      case MSG_AUDIO_UNMUTED:
         return "Sonido recuperado.";
      case MSG_RESET:
         return "Reiniciar";
      case MSG_FAILED_TO_LOAD_STATE:
         return "Error al cargar rpidamente desde";
      case MSG_FAILED_TO_SAVE_STATE_TO:
         return "Error al guardar rpidamente a";
      case MSG_FAILED_TO_SAVE_SRAM:
         return "Error al guardar la SRAM";
      case MSG_STATE_SIZE:
         return "Tamao de guardado rpido";
      case MSG_FOUND_SHADER:
         return "Shader encontrado";
      case MSG_SRAM_WILL_NOT_BE_SAVED:
         return "No se guardar la SRAM.";
      case MSG_BLOCKING_SRAM_OVERWRITE:
         return "Bloqueando sobrescritura de SRAM";
      case MSG_CORE_DOES_NOT_SUPPORT_SAVESTATES:
         return "El ncleo no es compatible con los guardados rpidos.";
      case MSG_SAVED_STATE_TO_SLOT:
         return "Guardando rpidamente a la ranura";
      case MSG_SAVED_SUCCESSFULLY_TO:
         return "Partida guardada en";
      case MSG_BYTES:
         return "bytes";
      case MSG_CONFIG_DIRECTORY_NOT_SET:
         return "No se ha asignado la carpeta de configuracin. No se puede guardar la nueva configuracin.";
      case MSG_SKIPPING_SRAM_LOAD:
         return "Omitiendo carga de SRAM.";
      case MSG_APPENDED_DISK:
         return "Disco incorporado";
      case MSG_STARTING_MOVIE_PLAYBACK:
         return "Iniciando reproduccin.";
      case MSG_FAILED_TO_REMOVE_TEMPORARY_FILE:
         return "Error al borrar el archivo temporal";
      case MSG_REMOVING_TEMPORARY_CONTENT_FILE:
         return "Borrando archivo temporal de contenido";
      case MSG_LOADED_STATE_FROM_SLOT:
         return "Carga rpida desde la ranura";
      case MSG_COULD_NOT_PROCESS_ZIP_FILE:
         return "No se ha podido procesar el archivo ZIP.";
      case MSG_SCANNING_OF_DIRECTORY_FINISHED:
         return "Escaneado de carpetas terminado";
      case MSG_SCANNING:
         return "Escaneando";
      case MSG_REDIRECTING_CHEATFILE_TO:
         return "Redirigiendo archivo de trucos a";
      case MSG_REDIRECTING_SAVEFILE_TO:
         return "Redirigiendo partida guardada a";
      case MSG_REDIRECTING_SAVESTATE_TO:
         return "Redirigiendo guardado rpido a";
      case MSG_SHADER:
         return "Shader";
      case MSG_APPLYING_SHADER:
         return "Aplicando shader";
      case MSG_FAILED_TO_APPLY_SHADER:
         return "Error al aplicar shader.";
      case MSG_STARTING_MOVIE_RECORD_TO:
         return "Iniciando grabacin en";
      case MSG_FAILED_TO_START_MOVIE_RECORD:
         return "Error al iniciar la grabacin.";
      case MSG_STATE_SLOT:
         return "Ranura de guardado rpido";
      case MSG_RESTARTING_RECORDING_DUE_TO_DRIVER_REINIT:
         return "Reiniciando grabacin debido al reinicio de un controlador.";
      case MSG_SLOW_MOTION:
         return "Cmara lenta.";
      case MSG_SLOW_MOTION_REWIND:
         return "Rebobinar cmara lenta.";
      case MSG_REWINDING:
         return "Rebobinando.";
      case MSG_REWIND_REACHED_END:
         return "Se ha llegado al final del bfer de rebobinado.";
      case MSG_CHEEVOS_HARDCORE_MODE_ENABLE:
         return "Modo Extremo Activado: guardar estado y reboninar se han desactivado.";
      case MSG_TASK_FAILED:
        return "Error";
      case MSG_DOWNLOADING:
        return "Descargando";
      case MSG_EXTRACTING:
        return "Extrayendo";
      case MENU_ENUM_LABEL_VALUE_MENU_LINEAR_FILTER:
         return "Filtro lineal del men";
      case MENU_ENUM_LABEL_MENU_THROTTLE_FRAMERATE:
         return "Acelerar velocidad del men";
      case MENU_ENUM_LABEL_VALUE_CHEEVOS_TEST_UNOFFICIAL:
         return "Probar versin no oficial";
      case MENU_ENUM_LABEL_VALUE_CHEEVOS_SETTINGS:
         return "Retrologros";
      case MENU_ENUM_LABEL_VALUE_INPUT_ICADE_ENABLE:
         return "Activar asignar mando al teclado";
      case MENU_ENUM_LABEL_VALUE_INPUT_KEYBOARD_GAMEPAD_MAPPING_TYPE:
         return "Tipo de asignacin de mando para teclado";
      case MENU_ENUM_LABEL_VALUE_INPUT_SMALL_KEYBOARD_ENABLE:
         return "Activar miniteclado";
      case MENU_ENUM_LABEL_VALUE_SAVE_CURRENT_CONFIG:
         return "Guardar configuracin actual";
      case MENU_ENUM_LABEL_VALUE_STATE_SLOT:
         return "Ranura de guardado";
      case MENU_ENUM_LABEL_VALUE_CHEEVOS_HARDCORE_MODE_ENABLE:
         return "Modo Extremo";
      case MENU_ENUM_LABEL_VALUE_ACCOUNTS_CHEEVOS_SETTINGS:
         return "Cuenta Cheevos";
      case MENU_ENUM_LABEL_VALUE_ACCOUNTS_CHEEVOS_USERNAME:
         return "Usuario";
      case MENU_ENUM_LABEL_VALUE_ACCOUNTS_CHEEVOS_PASSWORD:
         return "Contrasea";
      case MENU_ENUM_LABEL_VALUE_ACCOUNTS_RETRO_ACHIEVEMENTS:
         return "Retrologros";
      case MENU_ENUM_LABEL_VALUE_ACCOUNTS_LIST:
         return "Cuentas";
      case MENU_ENUM_LABEL_VALUE_ACCOUNTS_LIST_END:
         return "Enlace a lista de cuentas";
      case MENU_ENUM_LABEL_VALUE_DEBUG_PANEL_ENABLE:
         return "Activar panel de depuracin";
      case MENU_ENUM_LABEL_VALUE_HELP_SCANNING_CONTENT:
         return "Buscar contenido";
      case MENU_ENUM_LABEL_VALUE_CHEEVOS_DESCRIPTION:
         return "Descripcin";
      case MENU_ENUM_LABEL_VALUE_HELP_AUDIO_VIDEO_TROUBLESHOOTING:
         return "Solucionar problemas de vdeo/sonido";
      case MENU_ENUM_LABEL_VALUE_HELP_CHANGE_VIRTUAL_GAMEPAD:
         return "Cambiar el mando virtual superpuesto";
      case MENU_ENUM_LABEL_VALUE_HELP_WHAT_IS_A_CORE:
         return "Qu es un ncleo?";
      case MENU_ENUM_LABEL_VALUE_HELP_LOADING_CONTENT:
         return "Cargando contenidos";
      case MENU_ENUM_LABEL_VALUE_HELP_LIST:
         return "Ayuda";
      case MENU_ENUM_LABEL_VALUE_HELP_CONTROLS:
         return "Controles bsicos del men";
      case MENU_ENUM_LABEL_VALUE_BASIC_MENU_CONTROLS:
         return "Controles bsicos del men";
      case MENU_ENUM_LABEL_VALUE_BASIC_MENU_CONTROLS_SCROLL_UP:
         return "Desplazar hacia arriba";
      case MENU_ENUM_LABEL_VALUE_BASIC_MENU_CONTROLS_SCROLL_DOWN:
         return "Desplazar hacia abajo";
      case MENU_ENUM_LABEL_VALUE_BASIC_MENU_CONTROLS_CONFIRM:
         return "Confirmar/Aceptar";
      case MENU_ENUM_LABEL_VALUE_BASIC_MENU_CONTROLS_BACK:
         return "Retroceder";
      case MENU_ENUM_LABEL_VALUE_BASIC_MENU_CONTROLS_START:
         return "Valores predeterminados";
      case MENU_ENUM_LABEL_VALUE_BASIC_MENU_CONTROLS_INFO:
         return "Informacin";
      case MENU_ENUM_LABEL_VALUE_BASIC_MENU_CONTROLS_TOGGLE_MENU:
         return "Alternar men";
      case MENU_ENUM_LABEL_VALUE_BASIC_MENU_CONTROLS_QUIT:
         return "Abandonar";
      case MENU_ENUM_LABEL_VALUE_BASIC_MENU_CONTROLS_TOGGLE_KEYBOARD:
         return "Alternar teclado";
      case MENU_ENUM_LABEL_VALUE_OPEN_ARCHIVE:
         return "Abrir archivo como una carpeta";
      case MENU_ENUM_LABEL_VALUE_LOAD_ARCHIVE:
         return "Cargar archivo con un ncleo";
      case MENU_ENUM_LABEL_VALUE_INPUT_BACK_AS_MENU_ENUM_TOGGLE_ENABLE:
         return "Permitir alternar Back como men";
      case MENU_ENUM_LABEL_INPUT_MENU_ENUM_TOGGLE_GAMEPAD_COMBO:
         return "Combo para alternar mando con men";
      case MENU_ENUM_LABEL_VALUE_INPUT_OVERLAY_HIDE_IN_MENU:
         return "Ocultar superposicin en el men";
      case MENU_ENUM_LABEL_VALUE_LANG_POLISH:
         return "Polaco";
      case MENU_ENUM_LABEL_VALUE_OVERLAY_AUTOLOAD_PREFERRED:
         return "Cargar superposicin preferida automticamente";
      case MENU_ENUM_LABEL_VALUE_UPDATE_CORE_INFO_FILES:
         return "Actualizar archivos de informacin de ncleos";
      case MENU_ENUM_LABEL_VALUE_DOWNLOAD_CORE_CONTENT:
         return "Descargar contenido";
      case MENU_ENUM_LABEL_VALUE_SCAN_THIS_DIRECTORY:
         return "(Escanear esta carpeta)";
      case MENU_ENUM_LABEL_VALUE_SCAN_FILE:
         return "Escanear archivo";
      case MENU_ENUM_LABEL_VALUE_SCAN_DIRECTORY:
         return "Escanear carpeta";
      case MENU_ENUM_LABEL_VALUE_ADD_CONTENT_LIST:
         return "Aadir contenido";
      case MENU_ENUM_LABEL_VALUE_INFORMATION_LIST:
         return "Informacin";
      case MENU_ENUM_LABEL_VALUE_USE_BUILTIN_PLAYER:
         return "Usar reproductor de medios integrado";
      case MENU_ENUM_LABEL_VALUE_CONTENT_SETTINGS:
         return "Men rpido";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_CRC32:
         return "CRC32";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_MD5:
         return "MD5";
      case MENU_ENUM_LABEL_VALUE_LOAD_CONTENT_LIST:
         return "Cargar contenido";
      case MENU_ENUM_LABEL_VALUE_ASK_ARCHIVE:
         return "Preguntar";
      case MENU_ENUM_LABEL_VALUE_PRIVACY_SETTINGS:
         return "Privacidad";
#if 0
      case MENU_ENUM_LABEL_VALUE_HORIZONTAL_MENU: /* Don't change. Breaks everything. (Would be: "Men horizontal") */
         return "Horizontal Menu";
         /* Don't change these yet. Breaks everything. */
      case MENU_ENUM_LABEL_VALUE_SETTINGS_TAB:
         return "Pestaa de ajustes";
      case MENU_ENUM_LABEL_VALUE_HISTORY_TAB:
         return "Pestaa de historial";
      case MENU_ENUM_LABEL_VALUE_ADD_TAB:
         return "Aadir pestaa";
      case MENU_ENUM_LABEL_VALUE_PLAYLISTS_TAB:
         return "Pestaa de listas de reproduccin";
#endif
      case MENU_ENUM_LABEL_VALUE_NO_SETTINGS_FOUND:
         return "No se ha encontrado una configuracin.";
      case MENU_ENUM_LABEL_VALUE_NO_PERFORMANCE_COUNTERS:
         return "No hay contadores de rendimiento.";
      case MENU_ENUM_LABEL_VALUE_DRIVER_SETTINGS:
         return "Controlador";
      case MENU_ENUM_LABEL_VALUE_CONFIGURATION_SETTINGS:
         return "Configuracin";
      case MENU_ENUM_LABEL_VALUE_CORE_SETTINGS:
         return "Ncleo";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SETTINGS:
         return "Vdeo";
      case MENU_ENUM_LABEL_VALUE_LOGGING_SETTINGS:
         return "Registros";
      case MENU_ENUM_LABEL_VALUE_SAVING_SETTINGS:
         return "Guardado";
      case MENU_ENUM_LABEL_VALUE_REWIND_SETTINGS:
         return "Rebobinado";
      case MENU_ENUM_LABEL_VALUE_SHADER:
         return "Shader";
      case MENU_ENUM_LABEL_VALUE_CHEAT:
         return "Truco";
      case MENU_ENUM_LABEL_VALUE_USER:
         return "Usuario";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_BGM_ENABLE:
         return "Activar msica del sistema";
      case MENU_ENUM_LABEL_VALUE_RETROPAD:
         return "RetroPad";
      case MENU_ENUM_LABEL_VALUE_RETROKEYBOARD:
         return "RetroKeyboard";
      case MENU_ENUM_LABEL_VALUE_AUDIO_BLOCK_FRAMES:
         return "Bloquear fotogramas";
      case MENU_ENUM_LABEL_VALUE_INPUT_BIND_MODE:
         return "Modo de asignacin";
      case MENU_ENUM_LABEL_VALUE_INPUT_DESCRIPTOR_LABEL_SHOW:
         return "Mostrar etiquetas de descripcin de la entrada";
      case MENU_ENUM_LABEL_VALUE_INPUT_DESCRIPTOR_HIDE_UNBOUND:
         return "Ocultar descripciones sin asignar de la entrada del ncleo";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FONT_ENABLE:
         return "Mostrar mensajes en pantalla";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FONT_PATH:
         return "Fuente de mensajes en pantalla";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FONT_SIZE:
         return "Tamao de mensajes en pantalla";
      case MENU_ENUM_LABEL_VALUE_VIDEO_MESSAGE_POS_X:
         return "Posicin X de mensajes en pantalla";
      case MENU_ENUM_LABEL_VALUE_VIDEO_MESSAGE_POS_Y:
         return "Posicin Y de mensajes en pantalla";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SOFT_FILTER:
         return "Activar filtros por software";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FILTER_FLICKER:
         return "Filtro de parpadeo";
      case MENU_ENUM_LABEL_VALUE_DIRECTORY_CONTENT:
         return "(Carpeta de contenido)";
      case MENU_ENUM_LABEL_VALUE_UNKNOWN:
         return "Desconocido";
      case MENU_ENUM_LABEL_VALUE_DONT_CARE:
         return "No importa";
      case MENU_ENUM_LABEL_VALUE_LINEAR:
         return "Lineal";
      case MENU_ENUM_LABEL_VALUE_NEAREST:
         return "Ms cercano";
      case MENU_ENUM_LABEL_VALUE_DIRECTORY_DEFAULT:
         return "(Predeterminada)";
      case MENU_ENUM_LABEL_VALUE_DIRECTORY_NONE:
         return "(Ninguna)";
      case MENU_ENUM_LABEL_VALUE_NOT_AVAILABLE:
         return "No disponible";
      case MENU_ENUM_LABEL_VALUE_INPUT_REMAPPING_DIRECTORY:
         return "Carpeta de reasignacin de entrada";
      case MENU_ENUM_LABEL_VALUE_JOYPAD_AUTOCONFIG_DIR:
         return "Carpeta de autoconfiguracin de dispositivo de entrada";
      case MENU_ENUM_LABEL_VALUE_RECORDING_CONFIG_DIRECTORY:
         return "Carpeta de configuracin de grabacin";
      case MENU_ENUM_LABEL_VALUE_RECORDING_OUTPUT_DIRECTORY:
         return "Carpeta de salida de grabacin";
      case MENU_ENUM_LABEL_VALUE_SCREENSHOT_DIRECTORY:
         return "Carpeta de capturas de pantalla";
      case MENU_ENUM_LABEL_VALUE_PLAYLIST_DIRECTORY:
         return "Carpeta de listas de reproduccin";
      case MENU_ENUM_LABEL_VALUE_SAVEFILE_DIRECTORY:
         return "Carpeta de partidas guardadas";
      case MENU_ENUM_LABEL_VALUE_SAVESTATE_DIRECTORY:
         return "Carpeta de guardados rpidos";
      case MENU_ENUM_LABEL_VALUE_STDIN_CMD_ENABLE:
         return "Comandos stdin";
      case MENU_ENUM_LABEL_VALUE_NETWORK_REMOTE_ENABLE:
         return "Mando en red";
      case MENU_ENUM_LABEL_VALUE_VIDEO_DRIVER:
         return "Controlador de vdeo";
      case MENU_ENUM_LABEL_VALUE_RECORD_ENABLE:
         return "Activar grabacin";
      case MENU_ENUM_LABEL_VALUE_VIDEO_GPU_RECORD:
         return "Activar grabacin de GPU";
      case MENU_ENUM_LABEL_VALUE_RECORD_PATH:
         return "Carpeta de salida";
      case MENU_ENUM_LABEL_VALUE_RECORD_USE_OUTPUT_DIRECTORY:
         return "Usar carpeta de salida";
      case MENU_ENUM_LABEL_VALUE_RECORD_CONFIG:
         return "Configuracin de grabacin";
      case MENU_ENUM_LABEL_VALUE_VIDEO_POST_FILTER_RECORD:
         return "Activar grabacin con filtros";
      case MENU_ENUM_LABEL_VALUE_CORE_ASSETS_DIRECTORY:
         return "Carpeta de descargas";
      case MENU_ENUM_LABEL_VALUE_ASSETS_DIRECTORY:
         return "Carpeta de recursos";
      case MENU_ENUM_LABEL_VALUE_DYNAMIC_WALLPAPERS_DIRECTORY:
         return "Carpeta de fondos de pantalla dinmicos";
      case MENU_ENUM_LABEL_VALUE_RGUI_BROWSER_DIRECTORY:
         return "Carpeta del navegador de archivos";
      case MENU_ENUM_LABEL_VALUE_RGUI_CONFIG_DIRECTORY:
         return "Carpeta de configuracin";
      case MENU_ENUM_LABEL_VALUE_LIBRETRO_INFO_PATH:
         return "Carpeta de informacin del ncleo";
      case MENU_ENUM_LABEL_VALUE_LIBRETRO_DIR_PATH:
         return "Carpeta de ncleos";
      case MENU_ENUM_LABEL_VALUE_CURSOR_DIRECTORY:
         return "Carpeta de cursores";
      case MENU_ENUM_LABEL_VALUE_CONTENT_DATABASE_DIRECTORY:
         return "Carpeta de bases de datos de contenidos";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_DIRECTORY:
         return "Carpeta de sistema/BIOS";
      case MENU_ENUM_LABEL_VALUE_CHEAT_DATABASE_PATH:
         return "Carpeta de archivos de trucos";
      case MENU_ENUM_LABEL_VALUE_CACHE_DIRECTORY:
         return "Carpeta de cach";
      case MENU_ENUM_LABEL_VALUE_AUDIO_FILTER_DIR:
         return "Carpeta de filtros de sonido";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SHADER_DIR:
         return "Carpeta de shaders de vdeo";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FILTER_DIR:
         return "Carpeta de filtros de vdeo";
      case MENU_ENUM_LABEL_VALUE_OVERLAY_DIRECTORY:
         return "Carpeta de superposiciones";
      case MENU_ENUM_LABEL_VALUE_OSK_OVERLAY_DIRECTORY:
         return "Carpeta de teclados superpuestos";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_CLIENT_SWAP_INPUT:
         return "Intercambiar entrada en red";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_SPECTATOR_MODE_ENABLE:
         return "Permitir espectadores en red";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_IP_ADDRESS:
         return "Direccin IP";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_TCP_UDP_PORT:
         return "Puerto TCP/UDP para juego en red";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_ENABLE:
         return "Activar juego en red";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_DELAY_FRAMES:
         return "Retraso de fotogramas en red";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_MODE:
         return "Activar cliente en red";
      case MENU_ENUM_LABEL_VALUE_RGUI_SHOW_START_SCREEN:
         return "Mostrar pantalla de inicio";
      case MENU_ENUM_LABEL_VALUE_TITLE_COLOR:
         return "Color de ttulos del men";
      case MENU_ENUM_LABEL_VALUE_ENTRY_HOVER_COLOR:
         return "Color de entrada resaltada del men";
      case MENU_ENUM_LABEL_VALUE_TIMEDATE_ENABLE:
         return "Mostrar fecha y hora";
      case MENU_ENUM_LABEL_VALUE_THREADED_DATA_RUNLOOP_ENABLE:
         return "Buclar datos hilados";
      case MENU_ENUM_LABEL_VALUE_ENTRY_NORMAL_COLOR:
         return "Color de entrada normal del men";
      case MENU_ENUM_LABEL_VALUE_SHOW_ADVANCED_SETTINGS:
         return "Mostrar ajustes avanzados";
      case MENU_ENUM_LABEL_VALUE_MOUSE_ENABLE:
         return "Soporte para teclado";
      case MENU_ENUM_LABEL_VALUE_POINTER_ENABLE:
         return "Soporte tctil";
      case MENU_ENUM_LABEL_VALUE_CORE_ENABLE:
         return "Mostrar nombre del ncleo";
      case MENU_ENUM_LABEL_VALUE_DPI_OVERRIDE_ENABLE:
         return "Activar anulacin de PPP";
      case MENU_ENUM_LABEL_VALUE_DPI_OVERRIDE_VALUE:
         return "Anular PPP";
      case MENU_ENUM_LABEL_VALUE_XMB_SCALE_FACTOR:
         return "Escala del XMB";
      case MENU_ENUM_LABEL_VALUE_XMB_ALPHA_FACTOR:
         return "Transparencia del XMB";
      case MENU_ENUM_LABEL_VALUE_XMB_FONT:
         return "Fuente del XMB";
      case MENU_ENUM_LABEL_VALUE_SUSPEND_SCREENSAVER_ENABLE:
         return "Suspender salvapantallas";
      case MENU_ENUM_LABEL_VALUE_VIDEO_DISABLE_COMPOSITION:
         return "Desactivar composicin de escritorio";
      case MENU_ENUM_LABEL_VALUE_PAUSE_NONACTIVE:
         return "Detenerse en segundo plano";
      case MENU_ENUM_LABEL_VALUE_UI_COMPANION_START_ON_BOOT:
         return "Ejecutar al inicio la IU ayudante";
      case MENU_ENUM_LABEL_VALUE_UI_COMPANION_ENABLE:
         return "Activar IU ayudante";
      case MENU_ENUM_LABEL_VALUE_UI_MENUBAR_ENABLE:
         return "Barra de mens";
      case MENU_ENUM_LABEL_VALUE_ARCHIVE_MODE:
         return "Accin para asociar tipos de archivo";
      case MENU_ENUM_LABEL_VALUE_NETWORK_CMD_ENABLE:
         return "Comandos de red";
      case MENU_ENUM_LABEL_VALUE_NETWORK_CMD_PORT:
         return "Puerto de comandos de red";
      case MENU_ENUM_LABEL_VALUE_HISTORY_LIST_ENABLE:
         return "Activar historial";
      case MENU_ENUM_LABEL_VALUE_CONTENT_HISTORY_SIZE:
         return "Tamao del historial";
      case MENU_ENUM_LABEL_VALUE_VIDEO_REFRESH_RATE_AUTO:
         return "Frecuencia estimada del monitor";
      case MENU_ENUM_LABEL_VALUE_DUMMY_ON_CORE_SHUTDOWN:
         return "Anular al cerrar ncleo";
      case MENU_ENUM_LABEL_VALUE_CORE_SET_SUPPORTS_NO_CONTENT_ENABLE: /* TODO/FIXME */
         return "No ejecutar automticamente";
      case MENU_ENUM_LABEL_VALUE_FRAME_THROTTLE_ENABLE:
         return "Limitar velocidad mxima de ejecucin";
      case MENU_ENUM_LABEL_VALUE_FASTFORWARD_RATIO:
         return "Velocidad mxima de ejecucin";
      case MENU_ENUM_LABEL_VALUE_AUTO_REMAPS_ENABLE:
         return "Cargar autom. archivos de reasignacin";
      case MENU_ENUM_LABEL_VALUE_SLOWMOTION_RATIO:
         return "Cantidad de velocidad reducida";
      case MENU_ENUM_LABEL_VALUE_CORE_SPECIFIC_CONFIG:
         return "Configuracin por ncleo";
      case MENU_ENUM_LABEL_VALUE_GAME_SPECIFIC_OPTIONS:
         return "Usar opciones de ncleo para cada juego si existen";
      case MENU_ENUM_LABEL_VALUE_GAME_SPECIFIC_OPTIONS_CREATE:
         return "Crear archivo de opciones del juego";
      case MENU_ENUM_LABEL_VALUE_GAME_SPECIFIC_OPTIONS_IN_USE:
         return "Archivo de opciones del juego";
      case MENU_ENUM_LABEL_VALUE_AUTO_OVERRIDES_ENABLE:
         return "Cargar autom. archivos de anulacin";
      case MENU_ENUM_LABEL_VALUE_CONFIG_SAVE_ON_EXIT:
         return "Guardar configuracin al salir";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SMOOTH:
         return "Filtrado bilineal por hardware";
      case MENU_ENUM_LABEL_VALUE_VIDEO_GAMMA:
         return "Gamma de vdeo";
      case MENU_ENUM_LABEL_VALUE_VIDEO_ALLOW_ROTATE:
         return "Permitir rotacin";
      case MENU_ENUM_LABEL_VALUE_VIDEO_HARD_SYNC:
         return "Sincrona estricta de GPU";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SWAP_INTERVAL:
         return "Intervalo de alternado de VSync";
      case MENU_ENUM_LABEL_VALUE_VIDEO_VSYNC:
         return "VSync/Sincrona vertical";
      case MENU_ENUM_LABEL_VALUE_VIDEO_THREADED:
         return "Vdeo por hilos";
      case MENU_ENUM_LABEL_VALUE_VIDEO_ROTATION:
         return "Rotacin";
      case MENU_ENUM_LABEL_VALUE_VIDEO_GPU_SCREENSHOT:
         return "Permitir capturas de pantalla de GPU";
      case MENU_ENUM_LABEL_VALUE_VIDEO_CROP_OVERSCAN:
         return "Recortar Overscan (Reinicio)";
      case MENU_ENUM_LABEL_VALUE_VIDEO_ASPECT_RATIO_INDEX:
         return "ndice de proporcin de aspecto";
      case MENU_ENUM_LABEL_VALUE_VIDEO_ASPECT_RATIO_AUTO:
         return "Proporcin de aspecto automtica";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FORCE_ASPECT:
         return "Forzar proporcin de aspecto";
      case MENU_ENUM_LABEL_VALUE_VIDEO_REFRESH_RATE:
         return "Frecuencia de actualizacin";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FORCE_SRGB_DISABLE:
         return "Forzar anulacin del FBO sRGB";
      case MENU_ENUM_LABEL_VALUE_VIDEO_WINDOWED_FULLSCREEN:
         return "Pantalla completa en ventana";
      case MENU_ENUM_LABEL_VALUE_PAL60_ENABLE:
         return "Usar modo PAL60";
      case MENU_ENUM_LABEL_VALUE_VIDEO_VFILTER:
         return "Filtro contra parpadeos";
      case MENU_ENUM_LABEL_VALUE_VIDEO_VI_WIDTH:
         return "Asignar ancho de interfaz visual";
      case MENU_ENUM_LABEL_VALUE_VIDEO_BLACK_FRAME_INSERTION:
         return "Insertar fotogramas negros";
      case MENU_ENUM_LABEL_VALUE_VIDEO_HARD_SYNC_FRAMES:
         return "Fotogramas a sincronizar estrictamente";
      case MENU_ENUM_LABEL_VALUE_SORT_SAVEFILES_ENABLE:
         return "Ordenar partidas guardadas por carpetas";
      case MENU_ENUM_LABEL_VALUE_SORT_SAVESTATES_ENABLE:
         return "Ordenar guardados rpidos por carpetas";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FULLSCREEN:
         return "Pantalla completa";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SCALE:
         return "Escala de ventana";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SCALE_INTEGER:
         return "Escala integral";
      case MENU_ENUM_LABEL_VALUE_PERFCNT_ENABLE:
         return "Contadores de rendimiento";
      case MENU_ENUM_LABEL_VALUE_LIBRETRO_LOG_LEVEL:
         return "Nivel de registro del ncleo";
      case MENU_ENUM_LABEL_VALUE_LOG_VERBOSITY:
         return "Verbosidad del registro";
      case MENU_ENUM_LABEL_VALUE_SAVESTATE_AUTO_LOAD:
         return "Cargar guardado rpido automticamente";
      case MENU_ENUM_LABEL_VALUE_SAVESTATE_AUTO_INDEX:
         return "Indizar automticamente guardados rpidos";
      case MENU_ENUM_LABEL_VALUE_SAVESTATE_AUTO_SAVE:
         return "Guardado rpido automtico";
      case MENU_ENUM_LABEL_VALUE_AUTOSAVE_INTERVAL:
         return "Intervalo de autoguardados de SaveRAM";
      case MENU_ENUM_LABEL_VALUE_BLOCK_SRAM_OVERWRITE:
         return "No sobrescribir SaveRAM al cargar un guardado rpido";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SHARED_CONTEXT:
         return "Activar contexto compartido por HW";
      case MENU_ENUM_LABEL_VALUE_RESTART_RETROARCH:
         return "Reiniciar RetroArch";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_NICKNAME:
         return "Nombre de usuario";
      case MENU_ENUM_LABEL_VALUE_USER_LANGUAGE:
         return "Idioma";
      case MENU_ENUM_LABEL_VALUE_CAMERA_ALLOW:
         return "Permitir cmara";
      case MENU_ENUM_LABEL_VALUE_LOCATION_ALLOW:
         return "Permitir ubicacin";
      case MENU_ENUM_LABEL_VALUE_PAUSE_LIBRETRO:
         return "Pausar al activar el men";
      case MENU_ENUM_LABEL_VALUE_INPUT_OSK_OVERLAY_ENABLE:
         return "Mostrar teclado superpuesto";
      case MENU_ENUM_LABEL_VALUE_INPUT_OVERLAY_ENABLE:
         return "Mostrar superposicin";
      case MENU_ENUM_LABEL_VALUE_VIDEO_MONITOR_INDEX:
         return "ndice del monitor";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FRAME_DELAY:
         return "Retraso de fotogramas";
      case MENU_ENUM_LABEL_VALUE_INPUT_DUTY_CYCLE:
         return "Ciclo de trabajo";
      case MENU_ENUM_LABEL_VALUE_INPUT_TURBO_PERIOD:
         return "Perodo de turbo";
      case MENU_ENUM_LABEL_VALUE_INPUT_AXIS_THRESHOLD:
         return "Margen de ejes de entrada";
      case MENU_ENUM_LABEL_VALUE_INPUT_REMAP_BINDS_ENABLE:
         return "Permitir reasignar controles";
      case MENU_ENUM_LABEL_VALUE_INPUT_MAX_USERS:
         return "Nmero mximo de usuarios";
      case MENU_ENUM_LABEL_VALUE_INPUT_AUTODETECT_ENABLE:
         return "Activar autoconfiguracin";
      case MENU_ENUM_LABEL_VALUE_AUDIO_OUTPUT_RATE:
         return "Frecuencia de sonido (KHz)";
      case MENU_ENUM_LABEL_VALUE_AUDIO_MAX_TIMING_SKEW:
         return "Variacin mxima de sincrona de sonido";
      case MENU_ENUM_LABEL_VALUE_CHEAT_NUM_PASSES:
         return "Pasadas de trucos";
      case MENU_ENUM_LABEL_VALUE_REMAP_FILE_SAVE_CORE:
         return "Guardar archivo de reasignacin del ncleo";
      case MENU_ENUM_LABEL_VALUE_REMAP_FILE_SAVE_GAME:
         return "Guardar archivo de reasignacin del juego";
      case MENU_ENUM_LABEL_VALUE_CHEAT_APPLY_CHANGES:
         return "Aplicar cambios en trucos";
      case MENU_ENUM_LABEL_VALUE_SHADER_APPLY_CHANGES:
         return "Aplicar cambios en shaders";
      case MENU_ENUM_LABEL_VALUE_REWIND_ENABLE:
         return "Activar rebobinado";
      case MENU_ENUM_LABEL_VALUE_CONTENT_COLLECTION_LIST:
         return "Seleccionar de una coleccin";
      case MENU_ENUM_LABEL_VALUE_DETECT_CORE_LIST:
         return "Seleccionar archivo y detectar ncleo";
      case MENU_ENUM_LABEL_VALUE_DOWNLOADED_FILE_DETECT_CORE_LIST:
         return "Seleccionar archivo descargado y detectar ncleo";
      case MENU_ENUM_LABEL_VALUE_LOAD_CONTENT_HISTORY:
         return "Cargar archivos recientes";
      case MENU_ENUM_LABEL_VALUE_AUDIO_ENABLE:
         return "Activar sonido";
      case MENU_ENUM_LABEL_VALUE_FPS_SHOW:
         return "Mostrar velocidad de fotogramas";
      case MENU_ENUM_LABEL_VALUE_AUDIO_MUTE:
         return "Silenciar sonido";
      case MENU_ENUM_LABEL_VALUE_AUDIO_VOLUME:
         return "Volumen de sonido (dB)";
      case MENU_ENUM_LABEL_VALUE_AUDIO_SYNC:
         return "Activar sincrona de sonido";
      case MENU_ENUM_LABEL_VALUE_AUDIO_RATE_CONTROL_DELTA:
         return "Delta de control de frecuencia de sonido";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SHADER_NUM_PASSES:
         return "Pasadas del shader";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_SHA1:
         return "SHA1";
      case MENU_ENUM_LABEL_VALUE_CONFIGURATIONS:
         return "Cargar configuracin";
      case MENU_ENUM_LABEL_VALUE_REWIND_GRANULARITY:
         return "Nivel de detalle del rebobinado";
      case MENU_ENUM_LABEL_VALUE_REMAP_FILE_LOAD:
         return "Cargar archivo de reasignacin";
      case MENU_ENUM_LABEL_VALUE_CUSTOM_RATIO:
         return "Proporcin personalizada";
      case MENU_ENUM_LABEL_VALUE_USE_THIS_DIRECTORY:
         return "(Usar esta carpeta)";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_START_CONTENT:
         return "Ejecutar contenido";
      case MENU_ENUM_LABEL_VALUE_DISK_OPTIONS: /* UPDATE/FIXME */
         return "Opciones del disco del ncleo";
      case MENU_ENUM_LABEL_VALUE_CORE_OPTIONS:
         return "Opciones";
      case MENU_ENUM_LABEL_VALUE_CORE_CHEAT_OPTIONS: /* UPDATE/FIXME */
         return "Opciones de trucos del ncleo";
      case MENU_ENUM_LABEL_VALUE_CHEAT_FILE_LOAD:
         return "Cargar archivo de trucos";
      case MENU_ENUM_LABEL_VALUE_CHEAT_FILE_SAVE_AS:
         return "Guardar archivo de trucos como...";
      case MENU_ENUM_LABEL_VALUE_CORE_COUNTERS:
         return "Contadores del ncleo";
      case MENU_ENUM_LABEL_VALUE_TAKE_SCREENSHOT:
         return "Capturar pantalla";
      case MENU_ENUM_LABEL_VALUE_RESUME:
         return "Reanudar";
      case MENU_ENUM_LABEL_VALUE_DISK_INDEX:
         return "ndice del disco";
      case MENU_ENUM_LABEL_VALUE_FRONTEND_COUNTERS:
         return "Contadores del frontend";
      case MENU_ENUM_LABEL_VALUE_DISK_IMAGE_APPEND:
         return "Asignar imagen de disco";
      case MENU_ENUM_LABEL_VALUE_DISK_CYCLE_TRAY_STATUS:
         return "Estado de la bandeja del disco";
      case MENU_ENUM_LABEL_VALUE_NO_PLAYLIST_ENTRIES_AVAILABLE:
         return "No hay listas de reproduccin.";
      case MENU_ENUM_LABEL_VALUE_NO_CORE_INFORMATION_AVAILABLE:
         return "No hay informacin del ncleo.";
      case MENU_ENUM_LABEL_VALUE_NO_CORE_OPTIONS_AVAILABLE:
         return "No hay opciones del ncleo.";
      case MENU_ENUM_LABEL_VALUE_NO_CORES_AVAILABLE:
         return "No hay ncleos.";
      case MENU_ENUM_LABEL_VALUE_NO_CORE:
         return "Sin ncleo";
      case MENU_ENUM_LABEL_VALUE_DATABASE_MANAGER:
         return "Gestor de bases de datos";
      case MENU_ENUM_LABEL_VALUE_CURSOR_MANAGER:
         return "Gestor de cursores";
      case MENU_ENUM_LABEL_VALUE_MAIN_MENU:
         return "Men principal"; 
      case MENU_ENUM_LABEL_VALUE_SETTINGS:
         return "Ajustes";
      case MENU_ENUM_LABEL_VALUE_QUIT_RETROARCH:
         return "Abandonar RetroArch";
      case MENU_ENUM_LABEL_VALUE_SHUTDOWN:
         return "Apagar";
      case MENU_ENUM_LABEL_VALUE_REBOOT:
         return "Reiniciar";
      case MENU_ENUM_LABEL_VALUE_HELP:
         return "Ayuda";
      case MENU_ENUM_LABEL_VALUE_SAVE_NEW_CONFIG:
         return "Guardar configuracin nueva";
      case MENU_ENUM_LABEL_VALUE_RESTART_CONTENT:
         return "Reiniciar";
      case MENU_ENUM_LABEL_VALUE_CORE_UPDATER_LIST:
         return "Actualizador de ncleos";
      case MENU_ENUM_LABEL_VALUE_CORE_UPDATER_BUILDBOT_URL:
         return "URL de ncleos de Buildbot";
      case MENU_ENUM_LABEL_VALUE_BUILDBOT_ASSETS_URL:
         return "URL de recursos de Buildbot";
      case MENU_ENUM_LABEL_VALUE_NAVIGATION_WRAPAROUND:
         return "Seguir navegacin";
      case MENU_ENUM_LABEL_VALUE_NAVIGATION_BROWSER_FILTER_SUPPORTED_EXTENSIONS_ENABLE:
         return "Filtrar por extensiones compatibles";
      case MENU_ENUM_LABEL_VALUE_CORE_UPDATER_AUTO_EXTRACT_ARCHIVE:
         return "Extraer automticamente el archivo descargado";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFORMATION:
         return "Informacin del sistema";
      case MENU_ENUM_LABEL_VALUE_DEBUG_INFORMATION:
         return "Informacin de depuracin";
      case MENU_ENUM_LABEL_VALUE_ACHIEVEMENT_LIST:
         return "Lista de logros";
      case MENU_ENUM_LABEL_VALUE_ONLINE_UPDATER:
         return "Actualizador en lnea";
      case MENU_ENUM_LABEL_VALUE_CORE_INFORMATION:
         return "Informacin del ncleo";
      case MENU_ENUM_LABEL_VALUE_DIRECTORY_NOT_FOUND:
         return "No se ha encontrado la carpeta.";
      case MENU_ENUM_LABEL_VALUE_NO_ITEMS:
         return "No hay elementos.";
      case MENU_ENUM_LABEL_VALUE_CORE_LIST:
         return "Cargar ncleo";
      case MENU_ENUM_LABEL_VALUE_LOAD_CONTENT:
         return "Seleccionar archivo";
      case MENU_ENUM_LABEL_VALUE_CLOSE_CONTENT:
         return "Cerrar";
      case MENU_ENUM_LABEL_VALUE_MANAGEMENT:
         return "Ajustes de bases de datos";
      case MENU_ENUM_LABEL_VALUE_SAVE_STATE:
         return "Guardado rpido";
      case MENU_ENUM_LABEL_VALUE_LOAD_STATE:
         return "Carga rpida";
      case MENU_ENUM_LABEL_VALUE_RESUME_CONTENT:
         return "Reanudar";
      case MENU_ENUM_LABEL_VALUE_INPUT_DRIVER:
         return "Controlador de entrada";
      case MENU_ENUM_LABEL_VALUE_AUDIO_DRIVER:
         return "Controlador de sonido";
      case MENU_ENUM_LABEL_VALUE_JOYPAD_DRIVER:
         return "Controlador de joypad";
      case MENU_ENUM_LABEL_VALUE_AUDIO_RESAMPLER_DRIVER:
         return "Controlador de muestreo de sonido";
      case MENU_ENUM_LABEL_VALUE_RECORD_DRIVER:
         return "Controlador de grabacin";
      case MENU_ENUM_LABEL_VALUE_MENU_DRIVER:
         return "Controlador de men";
      case MENU_ENUM_LABEL_VALUE_CAMERA_DRIVER:
         return "Controlador de cmara";
      case MENU_ENUM_LABEL_VALUE_LOCATION_DRIVER:
         return "Controlador de ubicacin";
      case MENU_ENUM_LABEL_VALUE_UNABLE_TO_READ_COMPRESSED_FILE:
         return "No se ha podido leer el archivo comprimido.";
      case MENU_ENUM_LABEL_VALUE_OVERLAY_SCALE:
         return "Escala de superposicin";
      case MENU_ENUM_LABEL_VALUE_OVERLAY_PRESET:
         return "Preajuste de superposicin";
      case MENU_ENUM_LABEL_VALUE_AUDIO_LATENCY:
         return "Retraso de sonido (ms)";
      case MENU_ENUM_LABEL_VALUE_AUDIO_DEVICE:
         return "Controlador de sonido";
      case MENU_ENUM_LABEL_VALUE_KEYBOARD_OVERLAY_PRESET:
         return "Preajuste de teclado superpuesto";
      case MENU_ENUM_LABEL_VALUE_OVERLAY_OPACITY:
         return "Opacidad de la superposicin";
      case MENU_ENUM_LABEL_VALUE_MENU_WALLPAPER:
         return "Fondo del men";
      case MENU_ENUM_LABEL_VALUE_DYNAMIC_WALLPAPER:
         return "Fondo de pantalla dinmico";
      case MENU_ENUM_LABEL_VALUE_CORE_INPUT_REMAPPING_OPTIONS:
         return "Opciones de entrada del ncleo";
      case MENU_ENUM_LABEL_VALUE_SHADER_OPTIONS:
         return "Opciones de shaders";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SHADER_PARAMETERS:
         return "Previsualizar parmetros de shaders";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SHADER_PRESET_PARAMETERS:
         return "Parmetros de shaders del men";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SHADER_PRESET_SAVE_AS:
         return "Guardar preajuste de shaders como...";
      case MENU_ENUM_LABEL_VALUE_NO_SHADER_PARAMETERS:
         return "No hay parmetros de shaders.";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SHADER_PRESET:
         return "Cargar preajuste de shaders";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FILTER:
         return "Filtro de vdeo";
      case MENU_ENUM_LABEL_VALUE_AUDIO_DSP_PLUGIN:
         return "Plugin DSP de sonido";
      case MENU_ENUM_LABEL_VALUE_STARTING_DOWNLOAD:
         return "Iniciando descarga: ";
      case MENU_ENUM_LABEL_VALUE_SECONDS:
         return "segundos";
      case MENU_ENUM_LABEL_VALUE_OFF: /* Not changed. Would be "S" */
         return "OFF";
      case MENU_ENUM_LABEL_VALUE_ON: /* Not changed. Would be "NO" */
         return "ON";
      case MENU_ENUM_LABEL_VALUE_UPDATE_ASSETS:
         return "Actualizar recursos";
      case MENU_ENUM_LABEL_VALUE_UPDATE_LAKKA:
         return "Actualizar Lakka";
      case MENU_ENUM_LABEL_VALUE_UPDATE_CHEATS:
         return "Actualizar trucos";
      case MENU_ENUM_LABEL_VALUE_UPDATE_AUTOCONFIG_PROFILES:
         return "Actualizar perfiles de autoconfiguracin";
      case MENU_ENUM_LABEL_VALUE_UPDATE_AUTOCONFIG_PROFILES_HID:
         return "Actualizar perfiles de autoconfiguracin (HID)";
      case MENU_ENUM_LABEL_VALUE_UPDATE_DATABASES:
         return "Actualizar bases de datos";
      case MENU_ENUM_LABEL_VALUE_UPDATE_OVERLAYS:
         return "Actualizar superposiciones";
      case MENU_ENUM_LABEL_VALUE_UPDATE_CG_SHADERS:
         return "Actualizar shaders Cg";
      case MENU_ENUM_LABEL_VALUE_UPDATE_GLSL_SHADERS:
         return "Actualizar shaders GLSL";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_CORE_NAME:
         return "Nombre del ncleo";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_CORE_LABEL:
         return "Etiqueta del ncleo";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_SYSTEM_NAME:
         return "Nombre del sistema";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_SYSTEM_MANUFACTURER:
         return "Fabricante del sistema";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_CATEGORIES:
         return "Categoras";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_AUTHORS:
         return "Autores";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_PERMISSIONS:
         return "Permisos";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_LICENSES:
         return "Licencia(s)";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_SUPPORTED_EXTENSIONS:
         return "Extensiones compatibles";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_FIRMWARE:
         return "Firmware";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_CORE_NOTES:
         return "Notas del ncleo";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_BUILD_DATE:
         return "Fecha de compilacin";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_GIT_VERSION:
         return "Versin de Git";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_CPU_FEATURES:
         return "Caractersticas de CPU";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_FRONTEND_IDENTIFIER:
         return "Identificador del frontend";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_FRONTEND_NAME:
         return "Nombre del frontend";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_FRONTEND_OS:
         return "S.O. del frontend";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_RETRORATING_LEVEL:
         return "Nivel de RetroRating";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_POWER_SOURCE:
         return "Fuente de alimentacin";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_POWER_SOURCE_NO_SOURCE:
         return "No hay una fuente";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_POWER_SOURCE_CHARGING:
         return "Cargando";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_POWER_SOURCE_CHARGED:
         return "Cargada";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_POWER_SOURCE_DISCHARGING:
         return "Descargando";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_VIDEO_CONTEXT_DRIVER:
         return "Controlador de contexto de vdeo";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_DISPLAY_METRIC_MM_WIDTH:
         return "Mostrar ancho mtrico (mm)";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_DISPLAY_METRIC_MM_HEIGHT:
         return "Mostrar alto mtrico (mm)";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_DISPLAY_METRIC_DPI:
         return "Mostrar PPP mtricos";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_LIBRETRODB_SUPPORT:
         return "Soporte de LibretroDB";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_OVERLAY_SUPPORT:
         return "Soporte de superposiciones";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_COMMAND_IFACE_SUPPORT:
         return "Soporte de interfaz de comandos";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_NETWORK_REMOTE_SUPPORT:
         return "Soporte de mando en red";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_NETWORK_COMMAND_IFACE_SUPPORT:
         return "Soporte de interfaz de comandos en red";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_COCOA_SUPPORT:
         return "Soporte de Cocoa";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_RPNG_SUPPORT:
         return "Soporte de PNG (RPNG)";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_SDL_SUPPORT:
         return "Soporte de SDL1.2";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_SDL2_SUPPORT:
         return "Soporte de SDL2";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_VULKAN_SUPPORT:
         return "Soporte Vulkan";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_OPENGL_SUPPORT:
         return "Soporte de OpenGL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_OPENGLES_SUPPORT:
         return "Soporte de OpenGL ES";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_THREADING_SUPPORT:
         return "Soporte de hilos";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_KMS_SUPPORT:
         return "Soporte de KMS/EGL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_UDEV_SUPPORT:
         return "Soporte de Udev";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_OPENVG_SUPPORT:
         return "Soporte de OpenVG";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_EGL_SUPPORT:
         return "Soporte de EGL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_X11_SUPPORT:
         return "Soporte de X11";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_WAYLAND_SUPPORT:
         return "Soporte de Wayland";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_XVIDEO_SUPPORT:
         return "Soporte de XVideo";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_ALSA_SUPPORT:
         return "Soporte de ALSA";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_OSS_SUPPORT:
         return "Soporte de OSS";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_OPENAL_SUPPORT:
         return "Soporte de OpenAL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_OPENSL_SUPPORT:
         return "Soporte de OpenSL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_RSOUND_SUPPORT:
         return "Soporte de RSound";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_ROARAUDIO_SUPPORT:
         return "Soporte de RoarAudio";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_JACK_SUPPORT:
         return "Soporte de JACK";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_PULSEAUDIO_SUPPORT:
         return "Soporte de PulseAudio";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_DSOUND_SUPPORT:
         return "Soporte de DirectSound";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_XAUDIO2_SUPPORT:
         return "Soporte de XAudio2";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_ZLIB_SUPPORT:
         return "Soporte de Zlib";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_7ZIP_SUPPORT:
         return "Soporte de 7zip";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_DYLIB_SUPPORT:
         return "Soporte de libreras dinmicas";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_CG_SUPPORT:
         return "Soporte de Cg";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_GLSL_SUPPORT:
         return "Soporte de GLSL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_HLSL_SUPPORT:
         return "Soporte de HLSL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_LIBXML2_SUPPORT:
         return "Soporte de parseo XML libxml2";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_SDL_IMAGE_SUPPORT:
         return "Soporte de imgenes SDL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_FBO_SUPPORT:
         return "Soporte de render-to-texture OpenGL/Direct3D (shaders multipasos)";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_DYNAMIC_SUPPORT:
         return "Carga dinmica en tiempo real de librera libretro";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_FFMPEG_SUPPORT:
         return "Soporte de FFmpeg";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_CORETEXT_SUPPORT:
         return "Soporte de CoreText";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_FREETYPE_SUPPORT:
         return "Soporte de FreeType";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_NETPLAY_SUPPORT:
         return "Soporte de juego en red (peer-to-peer)";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_PYTHON_SUPPORT:
         return "Soporte de Python (soporte de scripts para shaders)";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_V4L2_SUPPORT:
         return "Soporte de Video4Linux2";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_LIBUSB_SUPPORT:
         return "Soporte de Libusb";
      case MENU_ENUM_LABEL_VALUE_YES:
         return "S";
      case MENU_ENUM_LABEL_VALUE_NO:
         return "No";
      case MENU_ENUM_LABEL_VALUE_BACK:
         return "BACK";
      case MENU_ENUM_LABEL_VALUE_SCREEN_RESOLUTION:
         return "Resolucin de pantalla";
      case MENU_ENUM_LABEL_VALUE_DISABLED:
         return "Desactivado";
      case MENU_ENUM_LABEL_VALUE_PORT:
         return "Puerto";
      case MENU_ENUM_LABEL_VALUE_NONE:
         return "Ninguno";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_DEVELOPER:
         return "Desarrollador";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_PUBLISHER:
         return "Distribuidora";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_DESCRIPTION:
         return "Descripcin";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_GENRE:
         return "Gnero";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_NAME:
         return "Nombre";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_ORIGIN:
         return "Origen";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_FRANCHISE:
         return "Franquicia";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_RELEASE_MONTH:
         return "Mes de lanzamiento";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_RELEASE_YEAR:
         return "Ao de lanzamiento";
      case MENU_ENUM_LABEL_VALUE_TRUE:
         return "Activado";
      case MENU_ENUM_LABEL_VALUE_FALSE:
         return "Desactivado";
      case MENU_ENUM_LABEL_VALUE_MISSING:
         return "Desaparecido";
      case MENU_ENUM_LABEL_VALUE_PRESENT:
         return "Presente";
      case MENU_ENUM_LABEL_VALUE_OPTIONAL:
         return "Opcional";
      case MENU_ENUM_LABEL_VALUE_REQUIRED:
         return "Necesario";
      case MENU_ENUM_LABEL_VALUE_STATUS:
         return "Estado";
      case MENU_ENUM_LABEL_VALUE_AUDIO_SETTINGS:
         return "Sonido";
      case MENU_ENUM_LABEL_VALUE_INPUT_SETTINGS:
         return "Entrada";
      case MENU_ENUM_LABEL_VALUE_ONSCREEN_DISPLAY_SETTINGS:
         return "Textos en pantalla (OSD)";
      case MENU_ENUM_LABEL_VALUE_OVERLAY_SETTINGS:
         return "Superposicin";
      case MENU_ENUM_LABEL_VALUE_MENU_SETTINGS:
         return "Men";
      case MENU_ENUM_LABEL_VALUE_MULTIMEDIA_SETTINGS:
         return "Multimedia";
      case MENU_ENUM_LABEL_VALUE_USER_INTERFACE_SETTINGS:
         return "Interfaz de usuario";
      case MENU_ENUM_LABEL_VALUE_MENU_FILE_BROWSER_SETTINGS:
         return "Navegador de archivos del men";
      case MENU_ENUM_LABEL_VALUE_CORE_UPDATER_SETTINGS:
         return "Actualizador";
      case MENU_ENUM_LABEL_VALUE_NETWORK_SETTINGS:
         return "Red";
      case MENU_ENUM_LABEL_VALUE_PLAYLIST_SETTINGS:
         return "Lista de reproduccin";
      case MENU_ENUM_LABEL_VALUE_USER_SETTINGS:
         return "Usuario";
      case MENU_ENUM_LABEL_VALUE_DIRECTORY_SETTINGS:
         return "Carpeta";
      case MENU_ENUM_LABEL_VALUE_RECORDING_SETTINGS:
         return "Grabacin";
      case MENU_ENUM_LABEL_VALUE_NO_INFORMATION_AVAILABLE:
         return "No hay informacin disponible.";
      case MENU_ENUM_LABEL_VALUE_INPUT_USER_BINDS:
         return "Asignaciones de entrada del usuario %u";
      case MENU_ENUM_LABEL_VALUE_LANG_ENGLISH:
         return "Ingls";
      case MENU_ENUM_LABEL_VALUE_LANG_JAPANESE:
         return "Japons";
      case MENU_ENUM_LABEL_VALUE_LANG_FRENCH:
         return "Francs";
      case MENU_ENUM_LABEL_VALUE_LANG_SPANISH:
         return "Espaol";
      case MENU_ENUM_LABEL_VALUE_LANG_GERMAN:
         return "Alemn";
      case MENU_ENUM_LABEL_VALUE_LANG_ITALIAN:
         return "Italiano";
      case MENU_ENUM_LABEL_VALUE_LANG_DUTCH:
         return "Holands";
      case MENU_ENUM_LABEL_VALUE_LANG_PORTUGUESE:
         return "Portugus";
      case MENU_ENUM_LABEL_VALUE_LANG_RUSSIAN:
         return "Ruso";
      case MENU_ENUM_LABEL_VALUE_LANG_KOREAN:
         return "Coreano";
      case MENU_ENUM_LABEL_VALUE_LANG_CHINESE_TRADITIONAL:
         return "Chino (Tradicional)";
      case MENU_ENUM_LABEL_VALUE_LANG_CHINESE_SIMPLIFIED:
         return "Chino (Simplificado)";
      case MENU_ENUM_LABEL_VALUE_LANG_ESPERANTO:
         return "Esperanto";
      case MENU_ENUM_LABEL_VALUE_LEFT_ANALOG:
         return "Analgico izquierdo";
      case MENU_ENUM_LABEL_VALUE_RIGHT_ANALOG:
         return "Analgico derecho";
      case MENU_ENUM_LABEL_VALUE_INPUT_HOTKEY_BINDS:
         return "Asignaciones de teclas rpidas";
      case MENU_ENUM_LABEL_VALUE_FRAME_THROTTLE_SETTINGS:
         return "Aumento de fotogramas";
      case MENU_ENUM_LABEL_VALUE_SEARCH:
         return "Buscar:";
      case MENU_ENUM_LABEL_VALUE_USE_BUILTIN_IMAGE_VIEWER:
         return "Usar visualizador de imgenes integrado";
      case MENU_ENUM_LABEL_VALUE_ENABLE:
         return "Activar";
      case MENU_ENUM_LABEL_VALUE_START_CORE:
         return "Iniciar ncleo";
      case MENU_ENUM_LABEL_VALUE_INPUT_POLL_TYPE_BEHAVIOR:
         return "Tipo de retardo";
      default:
         break;
   }

   return "null";
}
