/******************************************************************************
*		       							      *
* command.c (part of rCalc)					       	      *
* Copyright 1999, 2000 Gary Benson <rat@spunge.org>		       	      *
*								       	      *
* This program is free software; you can redistribute it and/or modify 	      *
* it under the terms of the GNU General Public License as published by 	      *
* the Free Software Foundation; either version 2 of the License, or    	      *
* (at your option) any later version.				       	      *
*								       	      *
* This program is distributed in the hope that it will be useful,      	      *
* but WITHOUT ANY WARRANTY; without even the implied warranty of       	      *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	       	      *
* GNU General Public License for more details.			       	      *
*								       	      *
* You should have received a copy of the GNU General Public License    	      *
* along with this program; if not, write to the Free Software	       	      *
* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.	       	      *
*								       	      *
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <gnome.h>

#include "console.h"
#include "config.h"
#include "engine.h"

/*****************************************************************************/

/* Table of commands.
*/
static int cmdExit	( int argc, char **argv );
static int cmdHelp	( int argc, char **argv );
static int cmdList	( int argc, char **argv );
/*  static int cmdWorkspace	( int argc, char **argv ); */
static int cmdRemove	( int argc, char **argv );
static int cmdMode	( int argc, char **argv );
static int cmdDegRad	( int argc, char **argv );

typedef struct
{
	char		*name;
	int		(*func)(int argc, char **argv);
	HELPTOPIC	helpTopic;
}
COMMAND;

#define NUM_COMMANDS	13

static COMMAND Command[NUM_COMMANDS]=
{
	/* Engine | Commands | Name of command */
	{ N_("exit"),		cmdExit,	helpExit	},

	/* Engine | Commands | Name of command */
	{ N_("quit"),		cmdExit,	helpExit	},	

	/* Engine | Commands | Name of command */
	{ N_("help"),		cmdHelp,	helpHelp	},

	/* Engine | Commands | Name of command */
	{ N_("helpwin"),	cmdHelp,	helpHelp	},

	/* Engine | Commands | Name of command */
	{ N_("?"),		cmdHelp,	helpHelp	},

	/* Engine | Commands | Name of command */
	{ N_("man"),		cmdHelp,	helpHelp	},

	/* Engine | Commands | Name of command */
	{ N_("ls"),		cmdList,	helpLs		},

	/* Engine | Commands | Name of command */
	{ N_("who"),		cmdList,	helpLs		},

	/* Engine | Commands | Name of command */
	{ N_("rm"),		cmdRemove,	helpRm		},

	/* Engine | Commands | Name of command */
	{ N_("clear"),		cmdRemove,	helpRm		},

	/* Engine | Commands | Name of command */
	{ N_("mode"),		cmdMode,	helpMode	},

	/* Engine | Commands | Name of command */
	{ N_("deg"),		cmdDegRad,	helpMode	},

	/* Engine | Commands | Name of command */
	{ N_("rad"),		cmdDegRad,	helpMode	},

	/* Engine | Commands | Name of command */
/*  	{ N_("workspace"),	cmdWorkspace,	helpWs		}, */
};

/*****************************************************************************/

/* Observer functions for the table of commands
*/
int rCalc_engine_GetNumCommands( void )
{
	return NUM_COMMANDS;
}
char *rCalc_engine_GetCommandName( int i )
{
	return _(Command[i].name);
}
HELPTOPIC rCalc_engine_GetCommandHelp( int i )
{
	return Command[i].helpTopic;
}

/*****************************************************************************/

/* The commands themselves
*/
static int cmdExit( int argc, char **argv )
{
	if( argc != 1 )
	{
		/* Engine | Commands | Prefix for format error messages */
		/* ie typing `exit foo' generates this, since exit takes no arguments */
		Error( "%s %s", _("format:"), _(argv[0]) );
		return 0;
	}
	return 1;
}

static int cmdHelp( int argc, char **argv )
{
	int num,i;
	
	if( argc == 1 )
	{
		if( !strcasecmp( argv[0], _("man") ) )
		{
			Error( "%s %s <%s>",
			       _("format:"), argv[0],
			     /* Engine | Commands | Argument for man command */
			       _("topic") );
			return 0;
		}
		cfgCalc->helpTopic = helpIntro;
		return 0;
	}
	
	num = rCalc_engine_GetNumFunctions();
	for( i=0; i<num; i++ )
	{
		if( strcasecmp( argv[1], rCalc_engine_GetFunctionName(i) ) )
			continue;

		cfgCalc->helpTopic = rCalc_engine_GetFunctionHelp(i);
		return 0;
	}

	num = rCalc_engine_GetNumConstants();
	for( i=0; i<num; i++ )
	{
		if( strcasecmp( argv[1], rCalc_engine_GetConstantName(i) ) )
			continue;

		cfgCalc->helpTopic = rCalc_engine_GetConstantHelp(i);
		return 0;
	}

	num = rCalc_engine_GetNumCommands();
	for( i=0; i<num; i++ )
	{
		if( strcasecmp( argv[1], rCalc_engine_GetCommandName(i) ) )
			continue;

		cfgCalc->helpTopic = rCalc_engine_GetCommandHelp(i);
		return 0;
	}

	/* Engine | Commands | Error message for man and help */
	Error( _("no help page available for `%s'."), argv[1] );
	return 0;
}

static int cmdList( int argc, char **argv )
{
	int num,a,i;
	
	/* Handle the default 'ls'='ls variables'
	** and the alias who='ls variables'
	*/
	if( argc==1 )
	{
		char *argv[2];
		argv[0] = _("ls");
		argv[1] = _("variables");
		return cmdList( 2, argv );
	}

	/* 'who' takes no arguments.
	*/
	if( !strcasecmp( argv[0], _("who") ) )
	{
		Error( "%s %s", _("format:"), argv[0] );
		return 0;
	}

	/* List each item in turn.
	*/
	for( a=1; a<argc; a++ )
	{
		/* Engine | Commands | Argument to ls */
		if( !strcasecmp( argv[a], _("variables") ) )
		{
			if( argc>2 && a!=1 ) Print("\n");
			if( argc>2 ) Print("%s:\n", argv[a] );
			
			num = rCalc_engine_GetNumVariables();
			for( i=0; i<num; i++ )
			{
				Print( "\t%-8s = %.16g\n",
				       rCalc_engine_GetVariableName( i ),
				       rCalc_engine_GetVariableValue( i ) );
			}
		}
		/* Engine | Commands | Argument to ls */
		else if( !strcasecmp( argv[a], _("functions") ) )
		{
			if( argc>2 && a!=1 ) Print("\n");
			if( argc>2 ) Print("%s:\n", argv[a] );
			
			num = rCalc_engine_GetNumFunctions();
			for( i=0; i<num; i++ )
			{
				Print( "\t%s\n",
				       rCalc_engine_GetFunctionName( i ) );
			}
		}
		/* Engine | Commands | Argument to ls */
		else if( !strcasecmp( argv[a], _("constants") ) )
		{
			if( argc>2 && a!=1 ) Print("\n");
			if( argc>2 ) Print("%s:\n", argv[a] );
			
			num = rCalc_engine_GetNumConstants();
			for( i=0; i<num; i++ )
			{
				Print( "\t%-8s = %.16g\n",
				       rCalc_engine_GetConstantName( i ),
				       rCalc_engine_GetConstantValue( i ) );
			}
		}
		/* Engine | Commands | Argument to ls */
		else if( !strcasecmp( argv[a], _("commands") ) )
		{
			if( argc>2 && a!=1 ) Print("\n");
			if( argc>2 ) Print("%s:\n", argv[a] );
			
			num = rCalc_engine_GetNumCommands();
			for( i=0; i<num; i++ )
			{
				Print( "\t%s\n",
				       rCalc_engine_GetCommandName( i ) );
			}
		}
		/* Engine | Commands | Error message for ls */
		else Error( _("%s: unknown item."), argv[a] );
	}
	return 0;
}

/*  static int cmdWorkspace( int argc, char **argv ) */
/*  { */
/*  	cfgCalc->showBrowser_command = TRUE; */
/*  	return 0; */
/*  } */

static int cmdRemove( int argc, char **argv )
{
	VARIABLE *Variable = cfgCalc->variables;
	int a,i,j;

	/* Handle `clear' on its own
	*/
	if( argc == 1 )
	{
		if( strcasecmp( argv[0], _("clear") ) )
		{
			Error( "%s %s %s1 [%s2 ...]",
			       _("format:"), argv[0],
			       /* Engine | Commands | Argument to rm */
			       _("variable"), _("variable") );
			return 0;
		}
		cfgCalc->nVariables = 0;
		cfgCalc->variables_mod = TRUE;

		return 0;
	}
	
	for( a=1; a<argc; a++ )
	{
		boolean success=FALSE;
		
		for( i=0; i<cfgCalc->nVariables; i++ )
		{
			if( strcasecmp( Variable[i].name, argv[a] ) )
				continue;

			cfgCalc->nVariables--;
			for( j=i; j<cfgCalc->nVariables; j++)
				Variable[j]=Variable[j+1];

			cfgCalc->variables_mod = TRUE;
			
			success = TRUE;
			break;
		}

		if( !success ) Error( _("%s: undefined variable."), argv[a] );
	}

	return 0;
}

static int cmdMode( int argc, char **argv )
{
	if( argc == 1 )
	{
		Print( "\t%s\n",
		       cfgCalc->angleMode == angleDegrees ?
		       /* Engine | Commands | Argument to mode */
		       _("degrees") :
		       /* Engine | Commands | Argument to mode */
		       _("radians") );

		return 0;
	}
	else if( argc == 2 )
	{
		if( !strcasecmp( argv[1], _("deg") ) ||
		    !strcasecmp( argv[1], _("rad") ) )
		{
			cmdDegRad( 1, argv+1 );
			return 0;
		}
	}

	Error( "%s %s [%s|%s]",
	       _("format:"), argv[0], _("deg"), _("rad") );
	return 0;
}

static int cmdDegRad( int argc, char **argv )
{
	ANGLE newstate = cfgCalc->angleMode;
	
	if( argc != 1 )
	{
		Error( "%s %s", _("format:"), argv[0] );
		return 0;
	}

	newstate = strcasecmp( argv[0], _("deg") )
		   ? angleRadians : angleDegrees;
	if( cfgCalc->angleMode != newstate )
	{
		cfgCalc->angleMode = newstate;
		cfgCalc->angleMode_mod = TRUE;
	}
	return 0;
}

/*****************************************************************************/

/* Return a pointer to the command structure if the input is a valid command.
*/
static COMMAND *getCommand( char *command )
{
	int i;

	for( i=0; i<NUM_COMMANDS; i++ )
	{
		int len = strlen( _(Command[i].name) );
		char c;

		if( strlen(command) < len ) continue;
		if( strncasecmp( _(Command[i].name), command, len ) ) continue;
		c = command[len];
		if( c && !isspace(c) ) continue;

		return &Command[i];
	}
	return NULL;
}

/* Execute a command, returning non-zero if we are to exit the program.
*/
static int executeCommand( COMMAND *c, char *command )
{
	int argc; char **argv;
	char *line;
	int i;
	
	/* Count the number of arguments (including the command)
	*/
	for( line=command, argc=0; *line; argc++ )
	{
		while( *line &&  isspace(*line) ) line++;
		if( ! *line ) break;
		while( *line && !isspace(*line) ) line++;
	}
	
	/* Build the array of arguments.
	*/
	if(!( argv = calloc( argc, sizeof(char *) ) ))
	{
		/* Engine | Commands | Can't allocate any memory */
		Error( _("not enough memory.") );
		return 0;
	}
	for( line=command, i=0; i<argc; i++ )
	{
		while( *line &&  isspace(*line) ) line++;
		argv[i] = line;
		while( *line && !isspace(*line) ) line++;
		*line++ = 0;
	}

	/* Execute, free and return.
	*/
	i = c->func( argc, argv );
	free( argv );
	return i;
}

/*****************************************************************************/

/* Process a line of text as either a command or an expression, or
** a bash-style semicolon-separated list of any mixture of the two.
*/
static int doCommandOrExpression( char *command )
{
	char *semi;
	COMMAND *c;
	int return_flag;

	if(( semi = strchr( command, ';' ) ))
	{
		/* Handle bash-style semicolon-separated lists of commands.
		*/
		*semi++ = 0;
		if(!( return_flag = doCommandOrExpression( command ) ))
		{
			return_flag = doCommandOrExpression( semi );
		}
	}
	else
	{
		/* Strip leading space from the command.
		*/
		while( *command && isspace(*command) ) command++;
		if( ! *command ) return 0;

		if(( c=getCommand( command ) ))
		{
			/* It is a command, so execute it...
			*/
			return_flag = executeCommand( c, command );
		}
		else
		{
			/* ...otherwise pass it on to the evaluator.
			*/
			rCalc_engine_Evaluate( command );
			return_flag = 0;
		}
	}
	return return_flag;
}

/* Wrapper for the above, so that stage changes only occur
** after each line has been processed, rather than each
** command/expression on each line.
*/
int rCalc_engine_Execute( char *command )
{
	int return_flag;
	
	rCalc_config_Lock();
	return_flag = doCommandOrExpression( command );
	rCalc_config_Unlock();
	rCalc_config_NotifyChange();
	
	return return_flag;
}

/*** end of command.c ********************************************************/



