\name{kalord}
\title{Repeated Measurements Models for Ordinal Variables with
  Frailty or Serial Dependence}
\usage{
kalord(response,times=NULL,distribution="multinomial",
       depend="independence",mu=NULL,ccov=NULL,tvcov=NULL,torder=0,
       interaction=NULL,preg=NULL,ptvc=NULL,pinitial=1,pdepend=NULL,
       envir=sys.frame(sys.parent()),optimize=T,print.level=0,
       ndigit=10,gradtol=0.00001,steptol=0.00001,fscale=1,
       iterlim=100,typsiz=abs(p),stepmax=10*sqrt(p%*%p))
}
\alias{print.kalordinal}
\alias{deviance.kalordinal}
\alias{fitted.kalordinal}
\description{
  \code{kalord} is designed to handle repeated measurements models with
  time-varying covariates. The distributions have two extra parameters
  as compared to the parameterization of the logistic distribution
  specified by \code{distribution}. Dependence among observations on a
  unit can be through gamma frailties (a type of random effect) or
  serial dependence over time.

  Nonlinear regression models can be supplied as formulae where
  parameters are unknowns in which case factor variables cannot be used
  and parameters must be scalars. (See \code{\link{finterp}}.)

  Marginal, individual and predicted profiles can be plotted using
  \code{\link{moprofile}}, \code{\link{ioprofile}} and
  \code{\link{poprofile}}.

  If the responses on a unit are clustered, not longitudinal, use the
  frailty dependence.
}
\arguments{
  \item{response}{A list of two column matrices with responses and
    corresponding times for each individual, one matrix or dataframe of
    response values, or an object of class, \code{response} (created by
    \code{\link{restovec}}) or \code{repeated} (created by
    \code{\link{rmna}} or \code{\link{lvna}}). If the \code{repeated}
    data object contains more than one response variable, give that
    object in \code{envir} and give the name of the response variable to
    be used here.}
  \item{times}{When response is a matrix, a vector of possibly unequally
    spaced times when they are the same for all individuals or a matrix
    of times. Not necessary if equally spaced. Ignored if response has
    class, \code{response} or \code{repeated}.}
  \item{distribution}{Specifies the parameterization of the logistic
    distribution to put in the Pareto distribution. Choices are binary,
    multinomial, continuation-ratio, and proportional-odds.}
  \item{depend}{Type of dependence. Choices are \code{independence},
    \code{Markov}, \code{serial}, and \code{frailty}.}
  \item{mu}{A regression function for the location parameter or a
    formula beginning with ~, specifying either a linear regression
    function in the Wilkinson and Rogers notation or a general function
    with named unknown parameters. The regression function must not
    contain intercepts. Give the initial estimates in \code{preg} or in
    \code{ptvc}.}
  \item{ccov}{A vector or matrix containing time-constant baseline
    covariates with one row per individual, a model formula using
    vectors of the same size, or an object of class, \code{tccov}
    (created by \code{\link{tcctomat}}). If response has class,
    \code{repeated}, the covariates must be supplied as a Wilkinson and
    Rogers formula unless none are to be used or \code{mu} is given.}
  \item{tvcov}{A list of matrices with time-varying covariate values,
    observed at the event times in \code{response}, for each individual
    (one column per variable), one matrix or dataframe of such covariate
    values, or an object of class, \code{tvcov} (created by
    \code{\link{tvctomat}}). If a time-varying covariate is observed at
    arbitrary time, \code{\link{gettvc}} can be used to find the most
    recent values for each response and create a suitable list. If
    response has class, \code{repeated}, the covariates must be supplied
    as a Wilkinson and Rogers formula unless none are to be used or
    \code{mu} is given.}
  \item{torder}{The order of the polynomial in time to be fitted.}
  \item{interaction}{Vector of length equal to the number of
    time-constant covariates, giving the levels of interactions between
    them and the polynomial in time in the \code{linear model}.}
  \item{preg}{Initial parameter estimates for the regression model:
    intercept, one for each covariate in \code{ccov}, and \code{torder}
    plus sum(\code{interaction}). If \code{mu} is a formula with unknown
    parameters, their estimates must be supplied either in their order
    of appearance in the expression or in a named list.}
  \item{ptvc}{Initial parameter estimates for the coefficients of the
    time-varying covariates, as many as in \code{tvcov}.}
  \item{pinitial}{An initial estimate for the initial parameter, if set
    to \code{NULL} this parameter will be fixed at zero. (With
    \code{frailty} dependence, this is the frailty parameter.)}
  \item{pdepend}{An initial estimate for the serial dependence
    parameter.}
  \item{envir}{Environment in which model formulae are to be interpreted
    or a data object of class, \code{repeated}, \code{tccov}, or
    \code{tvcov}; the name of the response variable should be given in
    \code{response}. If \code{response} has class \code{repeated}, it is
    used as the environment.}
  \item{optimize}{If set to \code{TRUE} then \code{\link{nlm}} is used
    to perform the numerical optimization of the likelihood function,
    otherwise if set to \code{FALSE} no optimization is performed.}
  \item{others}{Arguments controlling \code{\link{nlm}}.}
}
\value{
  A list of classes \code{kalordinal} and \code{recursive} is returned.
}
\author{P.J. Lindsey}
\seealso{
  \code{\link{finterp}}, \code{\link{gettvc}}, \code{\link{ioprofile}},
  \code{\link{lvna}}, \code{\link{moprofile}},
  \code{\link{plot.ordinal}}, \code{\link{poprofile}},
  \code{\link{restovec}}, \code{\link{rmna}}, \code{\link{tcctomat}},
  \code{\link{tvctomat}}.
}
\examples{
library(ordinal)

#
# Binary data
#
data(cardiac.indiv)

y <- restovec(cardiac.indiv[,1:4],type="ordinal")

cov <- tcctomat(as.matrix(cardiac.indiv[,5:10]))

w <- rmna(y,ccov=cov)

rm(cardiac.indiv,y,cov)

# Time-constant covariate.
kalord(w,distribution="binary",ccov=~age,preg=c(3.9507,-0.0308),pinit=NULL)

# Time-varying covariate.
kalord(w,distribution="binary",tvcov=~times,preg=1.832,ptvc=0.0573,pinit=NULL)

# Time-constant and time-varying covariate.
kalord(w,distribution="binary",mu=~age+ren+cop+dia+sex+pmi+times,
       ptvc=c(3.888,-0.0289,-0.642,-0.366,-0.314,-0.154,-0.114,0.057),pinit=NULL)

# Time-constant and time-varying covariate with a frailty dependence.
kalord(w,distribution="binary",mu=~age+ren+cop+dia+sex+pmi+times,
       ptvc=c(4.43391,-0.03128,-0.62439,-0.37596,-0.33064,-0.17095,-0.12216,-0.09096),
       pinit=0.1196,dep="frailty")

rm(w)

#
# Ordinal data
#
data(tmi2)

y <- restovec(tmi2[,1:4],type="ordinal")

cov <- tcctomat(tmi2[,5],name="distance")

w <- rmna(y,ccov=cov)

rm(tmi2,y,cov)

# Continuation-ratio model with time-constant covariate with a serial dependence.
kalord(w,distribution="continuation-ratio",ccov=~distance,preg=c(-1.907,7.7,-0.162),
       pinit=2.55,pdep=0.328,dep="serial")

# Proportional-odds model with time-constant covariate with a Markov dependence.
kalord(w,distribution="proportional-odds",ccov=~distance,preg=c(-1.89,11.652,-0.199),
	pinit=3.111,pdep=0.217,dep="Markov")

rm(w)
}
\keyword{models}
