/*
 * Mike Glover
 * mpg4@duluoz.net
 *
 * Johan Ekenberg
 * johan@ekenberg.nu
 *
 * quota.c
 * middle layer to talk to quotactl
 */

#include <config.h>

#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "output.h"
#include "system.h"
#include "quota.h"
#include "quotatool.h"

quota_t *quota_new (int q_type, int id, char *fs_spec)
{
  quota_t *myquota;
  fs_t *fs;
  char *qfile;

  q_type--;			/* see defs in quota.h */
  if ( q_type >= MAXQUOTAS ) {
    output_error ("Unknown quota type: %d", q_type);
    return 0;
  }

  myquota = (quota_t *) malloc (sizeof(quota_t));
  if ( ! myquota ) {
    output_error ("Insufficient memory");
    exit (ERR_MEM);
  }

  fs = system_getfs (fs_spec);
  if ( ! fs ) {
    return NULL;
  }

  qfile = strdup (fs->device);  

  myquota->_id = id;
  myquota->_id_type = q_type;
  myquota->_qfile = qfile;

  free (fs);
  return myquota;
}



inline void quota_delete (quota_t *myquota) {

  free (myquota->_qfile);
  free (myquota);

}



int quota_get (quota_t *myquota)
{
#ifdef LINUX_2_4
  struct mem_dqblk sysquota;
#else
  struct dqblk sysquota;
#endif
  int retval;

  output_debug ("fetching quotas: device='%s',id='%d'", myquota->_qfile,
		myquota->_id);
  retval = quotactl(QCMD(Q_GETQUOTA,myquota->_id_type), myquota->_qfile,
		    myquota->_id, (caddr_t) &sysquota);
  if ( retval < 0 ) {
    output_error ("Failed fetching quotas: %s", strerror(errno));
    return 0;
  }

  /* copy the linux-formatted quota info into our struct */
  memcpy (myquota, &sysquota, sizeof(sysquota));
  
  /* it worked! */
  return 1;
}



int quota_set (quota_t *myquota){
#ifdef LINUX_2_4
  struct mem_dqblk sysquota;
#else
  struct dqblk sysquota;
#endif
  int retval;

  if ( geteuid() != 0 ) {
    output_error ("Only root can set quotas");
    return 0;
  }

  /* copy our data into the linux dqblk */
  memcpy (&sysquota, myquota, sizeof(sysquota));

  /* make the syscall */
  retval = quotactl (QCMD(Q_SETQUOTA,myquota->_id_type),myquota->_qfile,
		     myquota->_id, (caddr_t) &sysquota);
  if ( retval < 0 ) {
    output_error ("Failed setting quota: %s", strerror(errno));
    return 0;
  }

  retval = quotactl (QCMD(Q_SYNC,myquota->_id_type), myquota->_qfile, 
	0, NULL);
  if ( retval < 0 ) {
    output_error ("Failed syncing quotas on %s: %s", myquota->_qfile,
		  strerror(errno));
    return 0;
  }

  return 1;
}



/* int quota_on (int q_type, char *device);
 * int quota_off (int q_type, char *device);
 */


