/***************************************************************
LanceMan's quickplot --- a fast interactive 2D plotter

Copyright (C) 1998, 1999  Lance Arsenault

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; version 2
of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Or look at http://www.gnu.org/copyleft/gpl.html .

**********************************************************************/
/* $Id: plot.c,v 1.12 1999/02/28 02:10:13 lance Exp $ */
#include <stdio.h>
#include <math.h>
#include <X11/Xlib.h> 
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include "data.h"
#include "xwin.h"

/* check for Xevents while drawing plots every NUM_LOOPS_CHECK_XEVENTS */
#define NUM_LOOPS_CHECK_XEVENTS 400

extern int get_window_size(Display *display, Window window, 
                            unsigned int *width, 
                            unsigned int *height);
#if(1)
extern void draw_axis(PlotXwins *win, Plot *plot, Drawable drawable, 
		      unsigned int width, unsigned int height);
#endif

static double maxint= 0.0;


void draw_plot(PlotXwins *win,Plot *plot);

extern void unset_show_pt_vals_undraw(void);
extern void unset_show_fn_vals_undraw(void);
extern unset_show_true_fn_vals_undraw(void);
extern void do_plotbuttomPress(PlotXwins *win, Plot *plot, XEvent *report);
extern void scale_plot(PlotXwins *win,Plot *plot,unsigned int *w,unsigned int *h);

static int out_of_range(double x1,double x2)
{

  if(x1>maxint||x2>maxint||x1<-maxint||x2<-maxint)
    {
      fprintf(stderr,"quickplot WARNING: Zoomed out of range. Values %g and/or %g are to big to interpolate points\n",x1,x2);
      return 1;
    }
  return 0;
}

/*
 * Find the two points that intersect the window x1,y1  x2,y2
 * Given the two points xx1,yy1  xx2,yy2
 * on failure return 0 or 1   on success return 2
 *
 * This Function is now fixed. Roundoff would cause this to
 * fail at the corners, if not for overlap at the corners.
 * I over lap the bounding lines of the plot window and
 * throw out points if I get more then 2 points.
 */
static int get_xsection(double xx1,double yy1,double xx2,double yy2,
			 int *x1, int *y1, int *x2, int *y2,
			double w1, double h1,double w, double h)
{
  double m,a,x,y;
  int xo[4],yo[4];
  int count;/* number of points found */
  
  count = 0;
  m = (yy1 - yy2)/(xx1 - xx2);
  a = yy1 - m*xx1;
  
  if(out_of_range(m,a))
    return 0;

  x = -a/m;
  y = 0.0;
  if(out_of_range(x,y))
    return 0;
  if(x>-1.0 && x<w)/* is x on y=0 line plus one pixel to ether side */
    {
      xo[count]=x;
      yo[count++]=y;
    }
  y = h1;
  x += h1/m; /* x  = (h1 - a)/m */
  if(out_of_range(x,y))
    return 0;
  if(x>-1.0 && x<w)/* is x on y=h1 line plus one pixel to ether side */
    {
      xo[count]=x;
      yo[count++]=y;
    }
  x = 0.0;
  y = a;
  if(y>-1.0 && y<h)/* is y on x=0 line plus one pixel to ether side */
    {
      xo[count]=x;
      yo[count++]=y;
    }
  x = w1;
  y += m*x; /* y = m*x + a */
  if(out_of_range(x,y))
    return 0;
  if(y>-1.0 && y<h)/* is y on x=w1 line plus one pixel to ether side */
    {
      xo[count]=x;
      yo[count++]=y;
    }
  if(count == 2)
    {
     *x1 = xo[0];
     *y1 = yo[0];
     *x2 = xo[1];
     *y2 = yo[1];
     return 2;
    }
  if(count > 2)/* find the far two points */
    {
      int i,j,k,max=0,maxi=0,maxj=0;
      for(i=0;i<count-1;i++)
	for(j=i+1;j<count;j++)
	  if(max < (k = (xo[i]-xo[j])*(xo[i]-xo[j]) 
		      + (yo[i]-yo[j])*(yo[i]-yo[j])))
	    {
	      max = k;
	      maxi = i;
	      maxj = j;
	    }
      *x1 = xo[maxi];
      *y1 = yo[maxi];
      *x2 = xo[maxj];
      *y2 = yo[maxj];
      return 2;
    }
  else if(count == 0)
    return 0;
  /* count == 1 */
  *x2 = *x1 = xo[0];
  *y2 = *y1 = yo[0];
  return 1;

}

/* Looks for the case when xx1 is close enough to xx2
 * so that it appears virtical when viewed in the resolution
 * of the screen. This function's perfect.
 */
static int singular_case(double xx1, int *x1,
			 double xx2, int *x2,
			 double yy1, int *y1,
			 double yy2, int *y2,
			 double h1)
{
  double dy,dx;

  dy = (yy2 > yy1)?(yy2 - yy1):(yy1 - yy2);
  dx = (xx2 > xx1)?(xx2 - xx1):(xx1 - xx2);
  if(dy > dx*h1)
    {
      dx = (xx2 - xx1)/(yy2-yy1);
      *x1 = *x2 = xx2 + (h1/2.0-yy2)*dx;
      if(yy1 > 0.0 && yy1 < h1)
	*y1 = yy1;
      else if(yy1 <= 0.0)
	*y1 = 0;
      else
	*y1 = h1 + 0.5;
      if(yy2 > 0.0 && yy2 < h1)
	*y2 = yy2;
      else if(yy2 <= 0.0)
	*y2 = 0;
      else
	*y2 = h1 + 0.5;
      /*  printf("%g %g\n",yy1,yy2);*/
      return 1;
    }
  return 0;/* failed to be this case */
}

/* This function is necessary for
 * X-servers that barf when you try to 
 * plot points out side of the windows.
 * Why waste time trying to plot values out a range anyway?
 * Because: there can be a line in the plot window from
 * points outside the window.
 * It puts values that are outside of the plot window
 * on the edges of the plot window if
 * they will make lines that are visible in the plot window
 * Returns 1 when x1,y1,x2,y2 are set to plot in
 * the window,
 * returns 0 otherwise.
 */
static int check_range(Plot *plot, 
		  double xx1, double yy1,/* point 1 in */
		  double xx2, double yy2,/* point 2 in */
		  double w,double h,  /* plot window width and height */
		  double w1,double h1,/* plot window width-1 and height-1 */
   /* return ->*/ int *x1, int *y1, int *x2, int *y2)
{
  if((!plot->zoom_count) || /* the first zoom is in the plot window */
     (xx1>-1.0 && xx2>-1.0 && yy1>-1.0 && yy2>-1.0 &&
      xx1<w    && xx2<w    && yy1<h    && yy2<h ))
    {
      *x1 = xx1;  *y1 = yy1;
      *x2 = xx2;  *y2 = yy2;
      return 1; /* check succeded all points are in the plot window */
    }

  if((xx1<0.0 && xx2<0.0) || (yy1<0.0 && yy2<0.0) || 
     (xx1>w1 && xx2>w1) || (yy1>h1 && yy2>h1) )
    return 0; /* check failed, no points or lines are in the plot window */

  /* singular case  x = const or when rounded to a
   * pixel value it's const
   */
  if(singular_case(xx1,x1,xx2,x2,yy1,y1,yy2,y2,h1))
    return 1;
  
  /* singular case  y = const or when rounded to a
   * pixel value it's const
   */
  if(singular_case(yy1,y1,yy2,y2,xx1,x1,xx2,x2,w1))
    return 1;

  /* case: zoomed out of range.
   *  I give up on scaled points that are far away
   * but there could be line in the plot window from them
   * But I'm not going to try to find it.
   * You would have to zoom-in at least 10^8 times the distance between
   * two ajoining points.  So no useful information is lost any way.
   */
  if(xx1>maxint||xx2>maxint||yy1>maxint||yy2>maxint||
     xx1<-maxint||xx2<-maxint||yy1<-maxint||yy2<-maxint)
    {
      fprintf(stderr,
	      "quickplot WARNING: Zoomed out of range: max value= %g  can't interpolate points (%g,%g) (%g,%g) to the window\n",
	     maxint,xx1,yy1,xx2,yy2);
      return 0;
    }

  /* at this point all horizonal or vertial line cases have
   * been checked and so have point singluar cases .... and 
   * so i can calculate slopes with no fear of division by zero
   */
  if(xx1>0.0 && yy1>0.0 && xx1<w1 && yy1<h1)/* point 1 is in window */
    {
      if(get_xsection(xx1,yy1,xx2,yy2,x1,y1,x2,y2,w1,h1,w,h) != 2)
	{
	  fprintf(stderr,
		  "quickplot WARNING: get_xsection() failed in %s line %d\n",
		  __FILE__,__LINE__ -4);
	  return 0;
	}
      if(xx1<xx2)
	{
	  if(*x1<*x2)
	    {
	      *x1=xx1;
	      *y1=yy1;
	    }
	  else
	    {
	      *x2=xx1;
	      *y2=yy1;
	    }
	}
      else
	{
	  if(*x1<*x2)
	    {
	      *x2=xx1;
	      *y2=yy1;
	    }
	  else
	    {
	      *x1=xx1;
	      *y1=yy1;
	    }
	}
      return 1;
    }
  if(xx2>0.0 && yy2>0.0 && xx2<w1 && yy2<h1)/* point 2 is in window */
    {
      if(get_xsection(xx1,yy1,xx2,yy2,x1,y1,x2,y2,w1,h1,w,h) != 2)
	{
	  fprintf(stderr,"quickplot WARNING: get_xsection() failed in %s line %d\n"
		  ,__FILE__,__LINE__ -1);
	  return 0;
	}
      if(xx2<xx1)
	{
	  if(*x2<*x1)
	    {
	      *x2=xx2;
	      *y2=yy2;
	    }
	  else
	    {
	      *x1=xx2;
	      *y1=yy2;
	    }
	}
      else
	{
	  if(*x2<*x1)
	    {
	      *x1=xx2;
	      *y1=yy2;
	    }
	  else
	    {
	      *x2=xx2;
	      *y2=yy2;
	    }
	}
      return 1;
    }

  /* if it made it to here no points are in the window
   * but there may still be a line in the window
   */
  if(get_xsection(xx1,yy1,xx2,yy2,x1,y1,x2,y2,w1,h1,w,h) == 2)
    return 1;

  return 0;
}

/* returns 1 if it makes it through drawing all points
 * returns 0 if an event pulls it out before drawing all points
 */
int Draw_plot(PlotXwins *win, Plot *plot, Drawable drawable, 
		      unsigned int width, unsigned int height)
{
  int i,x1,x2,y1,y2,count;
  double xx1, xx2, yy1, yy2, dheight, dwidth,h1,w1;
  struct Data *dx, *dy;
  XtInputMask mask;
  XEvent report;
  int redraw_set;

  /* maxint is used to compare to the scaled values to
   * see if they are 
   */
  if(maxint == 0.0)
    maxint = pow(256.0,xx1=sizeof(int))/3.0;

  dwidth = width;
  dheight = height;
  w1 = dwidth-1.0;
  h1 = dheight-1.0;
  /* valid plotting ranges are from x=0 to x=width-1 and y=0 to y=height-1 */

  /* set flages so I don't undraw rubber band lines 
   * or else they redraw the old and the new lines 
   * on the next call to show_function_values(),
   * unset_show_true_fn_vals_undraw() 
   * and show_point_values()
   */
  count = 0;
  redraw_set = 0;

#if(1)    
  if(plot->flag&SAME_SCALE && !(plot->flag&NO_AXES))
    draw_axis(win,plot,drawable,width,height);
#endif

  for(i=0;i<plot->num_plots;i++)
    {  
      dx = plot->start[plot->zoom_count][i][X];
      dy = plot->start[plot->zoom_count][i][Y];
      xx1 = ( plot->scale[i][X]*dx->val + plot->shift[i][X] )*
		 plot->z_scale[plot->zoom_count][X] +
		 plot->z_shift[plot->zoom_count][X];
      yy1 = ( plot->scale[i][Y]*dy->val + plot->shift[i][Y] )*
		 plot->z_scale[plot->zoom_count][Y] +
		 plot->z_shift[plot->zoom_count][Y];
      while(dx != plot->end[plot->zoom_count][i][X])
	{
	  xx2 = ( plot->scale[i][X]*(dx->next)->val +
		       plot->shift[i][X] )*
		     plot->z_scale[plot->zoom_count][X] +
		     plot->z_shift[plot->zoom_count][X];
	  yy2 = ( plot->scale[i][Y]*(dy->next)->val +
		       plot->shift[i][Y] )*
		     plot->z_scale[plot->zoom_count][Y] +
		     plot->z_shift[plot->zoom_count][Y];
	  if( !(plot->plot_opt[i] & NO_LINES) && 
	      check_range(plot,xx1,yy1,xx2,yy2,dwidth,dheight,
			  w1,h1,&x1,&y1,&x2,&y2))
	    { 
	      XSetForeground(win->display,win->gc,plot->line_color[i]);
	      XDrawLine(win->display,drawable,win->gc,x1,y1,x2,y2);
	      /*  printf("x1 = %d  y1 = %d x2 = %d  y2 = %d w %d h %d\n",
		     x1,y1,x2,y2,width,height);*/
	    }
	  if( !(plot->plot_opt[i] & NO_POINTS) && 
	      xx1<dwidth && xx1>-1.0 && yy1<dheight && yy1>-1.0 )
	    {
	      x1 = xx1; y1 = yy1;
	      XSetForeground(win->display,win->gc,plot->point_color[i]);

	      XDrawPoint(win->display,drawable,win->gc,x1+1,y1);
	      XDrawPoint(win->display,drawable,win->gc,x1-1,y1);
	      XDrawPoint(win->display,drawable,win->gc,x1,y1+1);
	      XDrawPoint(win->display,drawable,win->gc,x1,y1-1);
	      XDrawPoint(win->display,drawable,win->gc,x1+1,y1-1);
	      XDrawPoint(win->display,drawable,win->gc,x1-1,y1+1);
	      XDrawPoint(win->display,drawable,win->gc,x1-1,y1-1);
	      XDrawPoint(win->display,drawable,win->gc,x1+1,y1+1);
	    }
	  dx = dx->next;
	  dy = dy->next;
	  xx1 = xx2;
	  yy1 = yy2;
	  
	  /* check for Xevents while drawing plots every NUM_LOOPS_CHECK_XEVENTS */
	  if(count > NUM_LOOPS_CHECK_XEVENTS)
	    {
	      if(XCheckWindowEvent(win->display, win->plotwin,ExposureMask 
				   | ButtonPressMask | StructureNotifyMask ,
				   &report))
		{
		  switch(report.type)
		    {
		    case Expose:
		      while(XCheckWindowEvent(win->display,win->plotwin,
					      ExposureMask| StructureNotifyMask,&report));
		      draw_plot(win, plot);
		      return 0;/* didn't make it throught the draw */
		      break;
		    case ConfigureNotify: /* look for window resize */
		      if(get_window_size(win->display,win->plotwin,&width,&height))
			{
			  while(XCheckWindowEvent(win->display,win->plotwin,
				      ExposureMask| StructureNotifyMask,&report));
			  draw_plot(win, plot);
			  return 0;/* didn't make it throught the draw */
			}
		      break;
		    case ButtonPress:
		      if(report.xbutton.button == Button2)
			{
			  XPutBackEvent(win->display,&report);
			  return 0;/* didn't make it throught the draw */
			}
		      break;
		    }
		}
	      if((mask = XtAppPending(win->context)))
		{
		  XFlush(win->display);
		  XtAppProcessEvent(win->context, mask);
		}
	      count = 0;
	    }
	  count++;
	}
      if( !(plot->plot_opt[i] & NO_POINTS) && 
	  xx1<dwidth && xx1>-1.0 && yy1<dheight && yy1>-1.0 )
	{
	  x1 = xx1; y1 = yy1;
	  XSetForeground(win->display,win->gc,plot->point_color[i]);

	  XDrawPoint(win->display,drawable,win->gc,x1+1,y1);
	  XDrawPoint(win->display,drawable,win->gc,x1-1,y1);
	  XDrawPoint(win->display,drawable,win->gc,x1,y1+1);
	  XDrawPoint(win->display,drawable,win->gc,x1,y1-1);
	  XDrawPoint(win->display,drawable,win->gc,x1+1,y1-1);
	  XDrawPoint(win->display,drawable,win->gc,x1-1,y1+1);
	  XDrawPoint(win->display,drawable,win->gc,x1-1,y1-1);
	  XDrawPoint(win->display,drawable,win->gc,x1+1,y1+1);
	}
      XFlush(win->display);
    }
  unset_show_pt_vals_undraw();
  unset_show_fn_vals_undraw();
  unset_show_true_fn_vals_undraw();

  return 1; /* made it throught the draw */
}

/* make a new Pixmap if the width and height have changed
 */
static int remake_plot_pixmap0(PlotXwins *win,
			       Plot *plot, unsigned int width,unsigned int height)
{
  static int init = 0;

  if(init)
    {
      XFreePixmap(win->display, win->plot0pix);
      init = 1;
    }
  win->plot0pix = XCreatePixmap(win->display,DefaultRootWindow(win->display),
				width, height,
				DefaultDepth(win->display,win->screen_number));
  if((win->plot0pix == BadAlloc) || (win->plot0pix ==  BadValue))
    {
      fprintf(stderr,"quickplot Info: Can't create pixmap in file: %s\n",
	      __FILE__);
      fprintf(stderr,"Use the -X option to shut off this pixmap stuff.\n");
      plot->flag &= ~NO_PIXMAP0;
      return 0;
    }
  return 1;
}

/* The logic in this function is not so straight forward.
 * The Draw_plot() called in here can be interupted by Xevents
 * and not finish drawing.
 */
void draw_plot(PlotXwins *win,Plot *plot)
{
  static unsigned int w = 0, h = 0;
  unsigned int width, height;

  (void) scale_plot(win,plot,&width, &height); /*returns 1 if scale changed zero if not */
  if(plot->zoom_count /* the first zoom level (0) is not in the window */
     ||  
     (plot->flag&NO_PIXMAP0)
     || 
     (!(plot->flag&NO_PIXMAP0) && (w != width || h != height))
     )
    {
      XClearWindow(win->display,win->plotwin);
      Draw_plot(win,plot,win->plotwin,width,height);
    }
  if( !(plot->flag&NO_PIXMAP0) && !plot->zoom_count)
    {
      scale_plot(win,plot,&width, &height);
      if(w == width && h == height)
	{
	  XCopyArea(win->display,win->plot0pix,win->plotwin,win->gc,
		    0,0, width,height,0,0);
	  XFlush(win->display);
	  unset_show_pt_vals_undraw();
	  unset_show_fn_vals_undraw();
	  unset_show_true_fn_vals_undraw();
	}
      if(w != width || h != height)
	{
	  if(remake_plot_pixmap0(win,plot,width,height))
	    {
	      XtVaSetValues(win->redrawWig,XtNsensitive,
			    False,XtNlabel, "makPix",NULL);
	      /* Clear all pixels in PixMap to background */
	      XSetForeground(win->display,win->gc,win->BWbackground_pix);
	      XFillRectangle(win->display,win->plot0pix,win->gc,0,0,
			     (int) width-1,(int) height-1);
	      if(Draw_plot(win,plot,win->plot0pix,width,height))
		{
		  w = width;
		  h = height;
		}
	      XtVaSetValues(win->redrawWig,XtNsensitive,
			    True,XtNlabel, "Redraw",NULL);
	    }
	}
    }
}

