/* Copyright (C) 2006 P.L. Lucas
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA. 
 */

#include "autocomplete.h"
#include <QString>
#include <QStringList>
#include <QFile>
#include <QTextStream>
#include <QEvent>
#include <QKeyEvent>
#include <QLineEdit>
#include <iostream>

using namespace std;

Autocomplete::Autocomplete(QWidget *parent): QLineEdit(parent), search_string(""),
		tab_flag(false)
{
	current_match = first_match = word_list.end();
	actual_command_entered=-1;
	
	popup_options=new WordList;
	//popup_options->setWindowFlags(Qt::FramelessWindowHint|Qt::WindowStaysOnTopHint);
	popup_options->setWindowFlags(Qt::Popup);
	//popup_options->setWindowOpacity(0.5);
	popup_options->setLineEdit(this);
}

void Autocomplete::load_from_file(const char *file)
{
	QFile input(file);

	if(input.open(QIODevice::ReadOnly | QIODevice::Text))
	{
		QTextStream in_stream(&input);
		QString word;

		while(!in_stream.atEnd())
		{
			in_stream >> word;
			word_list.push_back(word);
		}

		word_list.sort();
	}else
		cerr << "No se ha podido cargar " << file << endl;
}

void Autocomplete::add(QString word)
{
	word_list.push_back(word);
}

void Autocomplete::remove(QString word)
{
	int i = word_list.indexOf(QRegExp(word, Qt::CaseInsensitive));
	word_list.removeAt(i);
}

void Autocomplete::clear()
{
	word_list.clear();
	current_match = first_match = word_list.end();
}

QString Autocomplete::search(QString start)
{
	QStringList::iterator i;
	QString word;

	search_string = start;

	for(i = word_list.begin(); i != word_list.end(); i++)
	{
		if(i->startsWith(start))
		{
			current_match = first_match = i;
			return *i;
		}
	}

	current_match = first_match = word_list.end();
	return QString("");
}

QString Autocomplete::get_next()
{
	if(first_match != word_list.end())
	{
		current_match++;

		if(!current_match->startsWith(search_string, Qt::CaseInsensitive))
			current_match = first_match;
		
		return *current_match;
	}else
		return QString("");
}


static bool is_variable(QChar ch)
{
	if(ch.isLetterOrNumber() || ch=='_' ) return true;
	return false;
}

/*Search last word entered*/
static QString last_word(QString _text, int pos=-1)
{
	QString start;

	int i;
	
	if(pos<0) i = _text.size() - 1;
	else i=pos;

	//Get the last word being writted
	for(; i >= 0 && is_variable(_text[i]); i--);

	if(i == -1)
		start = _text.left(pos+1);
	else
		start = _text.mid(i+1, pos-i);
	
	return start;
}

/* event function reimplemented */
bool Autocomplete::event(QEvent *event)
{
	if (event->type() == QEvent::KeyPress)
	{
		QKeyEvent *key_event = (QKeyEvent*)event;
		if(key_event->key() == Qt::Key_Tab)
		{
			
			
			// Autocomplete
			if(!tab_flag)
			{
				int x_pos=0, y_pos=0;
				QWidget *w=this;
				while(w->parent()!=NULL)
				{
					x_pos+=((QWidget*)(w->parent()))->x();
					y_pos+=((QWidget*)(w->parent()))->y();
					w=((QWidget*)(w->parent()));
				}
				popup_options->move(x_pos+x(),y_pos+y()+height());
				popup_options->clear();
				popup_options->show();
				
				QString start/*, result*/;
				QString _text = /*lineEdit()->*/text();
				// int i;

				//Get the last word being writted
				// for(i = _text.size() - 1; i >= 0 && _text[i].isLetterOrNumber(); i--);

				// if(i == -1)
				//	start = _text;
				// else
				//	start = _text.right(_text.size() - i - 1);
					
				
				start=last_word(_text, cursorPosition()-1 );
				
				//printf("Posicion del cursor %d\n", cursorPosition());
				
				QString command=
				"function qtoctave_completion_matches()\n"

				"l=completion_matches(\""+start+"\");\n"
				"[nrows,ncols]=size(l);\n"

				"for k=1:nrows\n"
				"fprintf(stderr,\"~~Completion:%s\\n\",l(k,:));\n"
				"endfor\n"

				"endfunction\n"

				"qtoctave_completion_matches()\n"
				;
				
				octave_connection->command_enter(command,false);

				// result = search(start);
				// if(!result.isEmpty())
				//	/*lineEdit()->*/setText(_text.left(i + 1) + search(start));

			}
			// else if(current_match != word_list.end())
			// {
			// 	QString _text = /*lineEdit()->*/text();
			// 	int len = _text.size() - current_match->size();
			// 
			// 	/*lineEdit()->*/setText(_text.left(len) + get_next());
			// }

			tab_flag = true;
			// Event accepted
			key_event->accept();
			return TRUE;
		}
		else
		{
			tab_flag = false;
			popup_options->hide();
		}
	}

	return QWidget::event(event);
}


void Autocomplete::keyPressEvent ( QKeyEvent * event )
{
	if(!commands_entered.isEmpty() && Qt::Key_Up==event->key())
	{
		setText(commands_entered[actual_command_entered]);
		--actual_command_entered;
		if(actual_command_entered<=0)
		{
			actual_command_entered=0;
		}
	}
	else if(!commands_entered.isEmpty() && Qt::Key_Down==event->key())
	{
		actual_command_entered++;
		if(actual_command_entered>=commands_entered.size())
		{
			actual_command_entered=commands_entered.size()-1;
			setText("");
		}
		else setText(commands_entered[actual_command_entered]);
	}
	else if(Qt::Key_Enter==event->key()||Qt::Key_Return==event->key())
	{
		if( !commands_entered.isEmpty() && (commands_entered.last()==text()) ) QLineEdit::keyPressEvent ( event );
		else
		{
			commands_entered << text();
			while(commands_entered.size()>127)
			{
				commands_entered.removeFirst();
			}
			
			emit new_command_entered(commands_entered);
			
			QLineEdit::keyPressEvent ( event );
		}
		actual_command_entered=commands_entered.size()-1;
	}
	else QLineEdit::keyPressEvent ( event );
}

QStringList Autocomplete::commands()
{
	return commands_entered;
}

void Autocomplete::set_octave_connection(OctaveConnection *oc)
{
	octave_connection=oc;
	connect(octave_connection,SIGNAL(line_ready(QString )), this, SLOT(add_completion_match(QString)) );
}

void Autocomplete::add_completion_match(QString line)
{
	if(line.startsWith("~~Completion:"))
	{
		QString word=line.trimmed();
		word=word.mid(13);
		popup_options->addItem(word);
	}
}

WordList::WordList(QWidget *parent):QListWidget(parent)
{
	connect(this,SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(wordSelected(QListWidgetItem*)));
}

bool WordList::event(QEvent *e)
{
	if (e->type() == QEvent::KeyPress)
	{
		/*Only some move keys are process*/
		QKeyEvent *key_event = (QKeyEvent*)e;
		switch (key_event->key())
		{
			case Qt::Key_Tab:
			case Qt::Key_Enter:
			case Qt::Key_Return:
			case Qt::Key_Up:
			case Qt::Key_Down:
				break;
			case Qt::Key_Escape:
			default:
				hide();
		}
	}
	
	return QListWidget::event(e);
}

void WordList::wordSelected(QListWidgetItem *item)
{
	QString _text=line_edit->text();
	QString word=last_word(_text,line_edit->cursorPosition()-1);
	QString word_selected=item->text();
	
	int pos=line_edit->cursorPosition();
	
	_text.replace(pos-word.size(),word.size(), word_selected);
	
	line_edit->setText(_text);
	
	line_edit->setCursorPosition( pos-word.size()+word_selected.size() );
	
	hide();
}

void WordList::setLineEdit(QLineEdit *le)
{
	line_edit=le;
}

