/***************************************************************************
                          qtextconfig.cpp  -  description
                             -------------------
    begin                : Sat Jul 22 2000
    copyright            : (C) 2000 by Michael Goffioul
    email                : gofioul@emic.ucl.ac.be
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "qtextconfig.h"
#include "qclineedit.h"

#include <qpixmap.h>
#include <qapplication.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qlabel.h>
#include <qcheckbox.h>
#include <qcombobox.h>

#include <stdlib.h>

#include "pretty.xpm"
#include "nup1.xpm"

#define	ON_ID	0
#define	OFF_ID	1

#define	UNIT_CM	0
#define	UNIT_IN	1
#define	UNIT_PIXEL	2

#define	DEFAULT_PAGESIZE	5
static ppd_size_t	default_pagesize[] = {
	{0,"Letter",612,792,18,36,594,756},
	{0,"Legal",612,1008,18,36,594,972},
	{0,"A4",595,842,18,36,577,806},
	{0,"COM10",297,684,18,36,279,648},
	{0,"DL",312,624,18,36,294,588}
};

static ppd_size_t* findDefaultFromStr(const char *str) {
	for (int i=0;i<DEFAULT_PAGESIZE;i++)
		if (strcasecmp(str,default_pagesize[i].name) == 0) return default_pagesize+i;
	return NULL;
}
static ppd_size_t* findDefault(const char *mediastr) {
	char	*s = new char[strlen(mediastr)+2], *c;
	ppd_size_t	*sz(0);
	strcpy(s,mediastr);
	strcat(s,",");
	c = strtok(s,",");
	while (c && !sz) {
		sz = findDefaultFromStr(c);
		c = strtok(NULL,",");
	}
	delete [] s;
	return sz;
}

float calcDim(int pixels, int unit) {
	switch (unit) {
		case 0: // cm
			return ((pixels / 72.0f) * 2.54f);
		case 1: // inches
			return (pixels / 72.0f);
		default:
			return (float)(pixels);
	}
}

void calcDimStr(int pixels, int unit, QString& str)
{
	float	value = calcDim(pixels,unit);
	if (unit == UNIT_PIXEL) str.setNum(pixels);
	else str.setNum(value,'f',3);
}

int uncalcDim(float dim, int unit) {
	switch (unit) {
		case 0: // cm
			return (int)((dim / 2.54f) * 72.0f);
		case 1: // inches
			return (int)(dim * 72.0f);
		default:
			return (int)(dim);
	}
}

void convertValue(QLineEdit *e, int old_units, int new_units)
{
	QString	value;
	calcDimStr(uncalcDim(QString(e->text()).toFloat(),old_units),new_units,value);
	e->setText(value.data());
}

QTextConfig::QTextConfig(QWidget *parent, const char *name)
: QTextConfigBase(parent,name), QConfigPage()
{
	ppd_ = 0;
	orientation_ = 3;	// default to portrait
	pagesize_ = 0;

	initDialog();
}

QTextConfig::~QTextConfig(){
}

void QTextConfig::initDialog()
{
	prettyoff_->setChecked(true);
	prettyChanged(OFF_ID);

	customMarginClicked();

	currentunits_ = UNIT_CM;
}

void QTextConfig::prettyChanged(int ID)
{
	if (ID == ON_ID) prettypix_->setPixmap(QPixmap(pretty));
	else prettypix_->setPixmap(QPixmap(nup1));
}

void QTextConfig::customMarginClicked()
{
	bool	value(margincustom_->isChecked());
	top_->setEnabled(value);
	bottom_->setEnabled(value);
	left_->setEnabled(value);
	right_->setEnabled(value);
	units_->setEnabled(value);
	if (!value) updateDefaultMargins();
}

void QTextConfig::initOptions(int num_options, cups_option_t *options, global_ppd_file_t *ppd, ipp_t*)
{
	ppd_ = ppd;

	const char	*str(0);
	// orientation
	if ((str=cupsGetOption("orientation-requested",num_options,options)) != NULL)
		orientation_ = atoi(str);

	// pretty print option
	str = cupsGetOption("prettyprint",num_options,options);
	if (str) {
		prettyon_->setChecked(true);
		prettyChanged(ON_ID);
	}
	// cpi and lpi
	if ((str=cupsGetOption("cpi",num_options,options)) != NULL)
		cpi_->setValue(atoi(str));
	if ((str=cupsGetOption("lpi",num_options,options)) != NULL)
		lpi_->setValue(atoi(str));
	// columns
	if ((str=cupsGetOption("columns",num_options,options)) != NULL)
		column_->setValue(atoi(str));

	// margins
	QString	value;
	// first retrieve page size
	if (ppd && ppd->ppd) {
		ppd_choice_t	*ch = ppdFindMarkedChoice(ppd->ppd,"PageSize");
		str = (ch ? ch->choice : 0);
		pagesize_ = ppdPageSize(ppd->ppd,(str ? str : "A4"));
		// check if PPD file contains enough infos, otherwise use default page sizes
		if (str && (!pagesize_ || pagesize_->width == 0 || pagesize_->length == 0))
			pagesize_ = findDefault(str);
	}
	else {
		if ((str=cupsGetOption("media",num_options,options)) != NULL)
			pagesize_ = findDefault(str);
		else
			// default -> take first default (Letter)
			pagesize_ = default_pagesize;
	}
	// ... and initialize margin values
	if (!pagesize_) marginpreview_->hide();
	else {
		marginpreview_->show();
		updatePagesize();
	}
	updateDefaultMargins();
	// then set custom margins
	bool	marginset(false);
	if ((str=cupsGetOption("page-top",num_options,options)) != NULL) {
		marginset = true;
		calcDimStr(atoi(str),currentunits_,value);
		top_->setText(value.data());
	}
	if ((str=cupsGetOption("page-bottom",num_options,options)) != NULL) {
		marginset = true;
		calcDimStr(atoi(str),currentunits_,value);
		bottom_->setText(value.data());
	}
	if ((str=cupsGetOption("page-left",num_options,options)) != NULL) {
		marginset = true;
		calcDimStr(atoi(str),currentunits_,value);
		left_->setText(value.data());
	}
	if ((str=cupsGetOption("page-right",num_options,options)) != NULL) {
		marginset = true;
		calcDimStr(atoi(str),currentunits_,value);
		right_->setText(value.data());
	}
	if ((str=cupsGetOption("margin-flag",num_options,options)) != NULL)
		marginset = (strcmp(str,"on") == 0);
	if (marginset) {
		margincustom_->setChecked(true);
		customMarginClicked();
		updateMargins();
	}
}

void QTextConfig::getCupsOptions(int& num_options, cups_option_t **options, bool incldef, ipp_t*)
{
	const char	*str(0);

	// pretty print
	if (prettyon_->isChecked())
		num_options = cupsAddOption("prettyprint","",num_options,options);
	else if ((str=cupsGetOption("prettyprint",num_options, *options)) != NULL)
	{ // we need to remove the "prettyprint" option -> copy all options in a new structure
	  // except the "prettyprint" option
		cups_option_t	*o = 0;
		int		n = 0;
		for (int i=0;i<num_options;i++)
			if (strcasecmp((*options)[i].name, "prettyprint") != 0)
				n = cupsAddOption((*options)[i].name, (*options)[i].value, n, &o);
		cupsFreeOptions(num_options, *options);
		num_options = n;
		*options = o;
	}
	// cpi, lpi	and columns
	if (incldef || cpi_->value() != 10)
		num_options = cupsAddOption("cpi",cpi_->text(),num_options,options);
	if (incldef || lpi_->value() != 6)
		num_options = cupsAddOption("lpi",lpi_->text(),num_options,options);
	if (incldef || column_->value() != 1)
		num_options = cupsAddOption("columns",column_->text(),num_options,options);
	// margins
	if (margincustom_->isChecked()) {
		QString	value;
		value.setNum(uncalcDim(QString(top_->text()).toFloat(),currentunits_));
		num_options = cupsAddOption("page-top",value.data(),num_options,options);
		value.setNum(uncalcDim(QString(bottom_->text()).toFloat(),currentunits_));
		num_options = cupsAddOption("page-bottom",value.data(),num_options,options);
		value.setNum(uncalcDim(QString(left_->text()).toFloat(),currentunits_));
		num_options = cupsAddOption("page-left",value.data(),num_options,options);
		value.setNum(uncalcDim(QString(right_->text()).toFloat(),currentunits_));
		num_options = cupsAddOption("page-right",value.data(),num_options,options);
	}
	if (incldef) num_options = cupsAddOption("margin-flag",(margincustom_->isChecked() ? "on" : "off"),num_options,options);
}

void QTextConfig::updateMargins()
{
	bool	ok;
	int	t(-1),b(-1),l(-1),r(-1);
	float	v;
	
	v = QString(top_->text()).toFloat(&ok);
	if (!ok || (t = uncalcDim(v,currentunits_)) < 0) goto end;
	v = QString(bottom_->text()).toFloat(&ok);
	if (!ok || (b = uncalcDim(v,currentunits_)) < 0) goto end;
	v = QString(left_->text()).toFloat(&ok);
	if (!ok || (l = uncalcDim(v,currentunits_)) < 0) goto end;
	v = QString(right_->text()).toFloat(&ok);
	if (!ok || (r = uncalcDim(v,currentunits_)) < 0) goto end;
	marginpreview_->setMargins(t,b,l,r);
	return;
end:
	QApplication::beep();
	if (t < 0) top_->selectAll();
	else if (b < 0) bottom_->selectAll();
	else if (l < 0) left_->selectAll();
	else if (r < 0) right_->selectAll();
	return;
}

void QTextConfig::unitsChanged(int u)
{
	if (currentunits_ != u) {
		convertValue(top_,currentunits_,u);
		convertValue(bottom_,currentunits_,u);
		convertValue(left_,currentunits_,u);
		convertValue(right_,currentunits_,u);
		currentunits_ = u;
	}
}

void QTextConfig::updatePagesize()
{
	if (!pagesize_) return;
	if (orientation_ == 4 || orientation_ == 5)
		marginpreview_->setPageSize(pagesize_->length,pagesize_->width);
	else
		marginpreview_->setPageSize(pagesize_->width,pagesize_->length);
}

void QTextConfig::updateDefaultMargins()
{
	if (!pagesize_) return;
	int	v[4] = {(int)pagesize_->left, (int)pagesize_->bottom, (int)(pagesize_->width-pagesize_->right), (int)(pagesize_->length-pagesize_->top)};
	int	i = orientation_-3;
	if (i > 1) i ^= 1;
	int	l(v[i%4]), b(v[(i+1)%4]), r(v[(i+2)%4]), t(v[(i+3)%4]);
	marginpreview_->setMargins(t,b,l,r);

	QString	value;
	calcDimStr(t,currentunits_,value);
	top_->setText(value.data());
	calcDimStr(b,currentunits_,value);
	bottom_->setText(value.data());
	calcDimStr(l,currentunits_,value);
	left_->setText(value.data());
	calcDimStr(r,currentunits_,value);
	right_->setText(value.data());
}
