/***************************************************************************
                          qcupsprintdialog.cpp  -  description
                             -------------------
    begin                : Thu Jun 15 2000
    copyright            : (C) 2000 by Michael Goffioul
    email                : gofioul@emic.ucl.ac.be
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "qcupsprintdialog.h"

#include <qpixmap.h>
#include <qmessagebox.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qlineedit.h>
#include <qcombobox.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qcheckbox.h>

#include <stdlib.h>

#include <cups/ipp.h>
#include "cupshelper.h"
#include "qcupsconfigdialog.h"
#include "qcupsprinterconfigdialog.h"
#include "ppd-util.h"

#include "printer.xpm"
#include "printer_stopped.xpm"
#include "printer_remote.xpm"
#include "printer_remote_stopped.xpm"
#include "printer_defect.xpm"
#include "printer_class.xpm"
#include "printer_class_stopped.xpm"
#include "collate.xpm"
#include "uncollate.xpm"
#include "collate_rev.xpm"
#include "uncollate_rev.xpm"

void showOptions(int n, cups_option_t *opts)
{
debug("******************************");
debug("Number of options: %d",n);
	for (int i=0;i<n;i++)
	{
		cups_option_t	*o = opts+i;
		debug("%s = %s",o->name,o->value);
	}
debug("******************************");
}

QPixmap getIcon(int type, int state)
{
	if (type <= 0) return QPixmap(printer_defect);
	else
	{
		if (type & CUPS_PRINTER_CLASS)
		{
			if (state == IPP_PRINTER_STOPPED) return QPixmap(printer_class_stopped);
			else return QPixmap(printer_class);
		}
		else
		{ // note: implicit classes are shown again as printers
			if (type & CUPS_PRINTER_REMOTE)
			{
				if (state == IPP_PRINTER_STOPPED) return QPixmap(printer_remote_stopped);
				else return QPixmap(printer_remote);
			}
			else
			{
				if (state == IPP_PRINTER_STOPPED) return QPixmap(printer_stopped);
				else return QPixmap(printer);
			}
		}
	}
}

//********************************************************************************

QCupsPrintDialog::QCupsPrintDialog(QWidget *parent, const char *name, bool useInst)
: QCupsPrintDialogBase(parent,name,true)
{
	useInstances_ = useInst;
	destinations_ = 0;
	dests_edited_ = 0;
	destnum_ = 0;

	propType_ = PrinterAll;
	initialize();
	initCups();

	resize(100,100);
}

QCupsPrintDialog::~QCupsPrintDialog()
{
	// free everything
	closeCups();
}

void QCupsPrintDialog::initialize()
{
	all_->setChecked(true);
	current_->setEnabled(false);
	copies_->setValue(1);

	collate_->setChecked(true);
	collateClicked();

	setCaption(tr("Print"));
}

void QCupsPrintDialog::collateClicked()
{
	QPixmap	pix(collate_->isChecked() ? (reverse_->isChecked() ? collate_rev : collate) : (reverse_->isChecked() ? uncollate_rev : uncollate));
	copiespix_->setPixmap(pix);
}

void QCupsPrintDialog::rangeEntered()
{
	if (!range_->isChecked()) range_->setChecked(true);
}

void QCupsPrintDialog::initCups()
{
	// get destinations and copy them
	if (destinations_ || dests_edited_) closeCups();
	destnum_ = cupsGetDests(&destinations_);
	for (int i=0;i<destnum_;i++) {
		cups_dest_t	*d = destinations_+i;
		cupsAddDest(d->name,d->instance,i,&dests_edited_);
		dests_edited_[i].is_default = d->is_default;
	}
	// parse destinations and fill dest_ combobox
	dest_->clear();
	type_->setText("");
	location_->setText("");
	comment_->setText("");
	state_->setText("");
	// check server infos first
	if (!CupsHelper::get()->checkHost())
	{
		QMessageBox::critical(this,tr("Error"),tr("Unable to connect to CUPS server, check options."),QMessageBox::Ok|QMessageBox::Default,0);
		return;
	}
	// check all destination for printer existence
	int	default_dest = -1;
	for (int i=0;i<destnum_;i++)
	{
		QString	str;
		/* second possibility : change icon for non available printers */
		int	printerState;
		int	printerType = CupsHelper::get()->printerType(destinations_[i].name,&printerState);
		QPixmap	icon = getIcon(printerType, printerState);
		if (destinations_[i].instance)
			str = tr("%1 on %2").arg(destinations_[i].instance).arg(destinations_[i].name);
		else
			str = destinations_[i].name;
		dest_->insertItem(icon,str);
		if (destinations_[i].is_default) default_dest = i;
	}
	if (default_dest == -1 && destnum_ > 0) default_dest = 0;
	dest_->setCurrentItem(default_dest);
	if (default_dest != -1) printerSelected(default_dest);
}

void QCupsPrintDialog::closeCups()
{
	if (destinations_) cupsFreeDests(destnum_,destinations_);
	destinations_ = 0;
	if (dests_edited_) cupsFreeDests(destnum_,dests_edited_);
	dests_edited_ = 0;
	destnum_ = 0;
}

void QCupsPrintDialog::printerSelected(int index)
{
	// do nothing if index out of range
	if (index < 0 || index >= destnum_)
	{
debug("QCupsPrintDialog::printerSelected() : printer index out of range");
		return;
	}
	// get printer attributes
	cups_dest_t	*printer_ = destinations_+index;
	ipp_t	*request = CupsHelper::get()->newIppRequest();
	QString	str;
	str.sprintf("ipp://%s:%d/printers/%s",CupsHelper::get()->host().data(),CupsHelper::get()->port(),printer_->name);
	ippAddString(request,IPP_TAG_OPERATION,IPP_TAG_URI,"printer-uri",NULL,str.data());
	request->request.op.operation_id = IPP_GET_PRINTER_ATTRIBUTES;
	request = CupsHelper::get()->processRequest(request,"/printers/");
	if (!request || request->curtag != IPP_TAG_PRINTER)
	{
		state_->setText(tr("Unable to retrieve printer infos"));
		type_->setText("");
		location_->setText("");
		comment_->setText("");
		propbutton_->setEnabled(false);
		setdefault_->setEnabled(false);
	}
	else
	{
		ipp_attribute_t	*attr = ippFindAttribute(request,"printer-state",IPP_TAG_ENUM);
		if (attr)
			switch (attr->values[0].integer)
			{
				case IPP_PRINTER_IDLE:
					state_->setText(tr("Printer idle"));
					break;
				case IPP_PRINTER_PROCESSING:
					state_->setText(tr("Processing..."));
					break;
				case IPP_PRINTER_STOPPED:
					state_->setText(tr("Stopped"));
					break;
			}
		else
			state_->setText(tr("Unknown"));
		attr = ippFindAttribute(request,"printer-info",IPP_TAG_TEXT);
		if (attr)
			type_->setText(attr->values[0].string.text);
		else
			type_->setText("");
		attr = ippFindAttribute(request,"printer-location",IPP_TAG_TEXT);
		if (attr)
			location_->setText(attr->values[0].string.text);
		else
			location_->setText("");
		attr = ippFindAttribute(request,"printer-make-and-model",IPP_TAG_TEXT);
		if (attr)
			comment_->setText(attr->values[0].string.text);
		else
			comment_->setText("");
		propbutton_->setEnabled(true);
		setdefault_->setEnabled(true);
	}
}
/**  */
void QCupsPrintDialog::optionsClicked()
{
	QCupsConfigDialog	*dlg = new QCupsConfigDialog(this);
	dlg->setServerInfo(CupsHelper::get()->host(),CupsHelper::get()->port());
	dlg->setLoginInfo(CupsHelper::get()->login(),CupsHelper::get()->password());
	if (dlg->exec())
	{
		CupsHelper::get()->setHostInfo(dlg->server(),dlg->port());
		CupsHelper::get()->setLoginInfo(dlg->login(),dlg->password());
		initCups();
	}
	delete dlg;
}
/**  */
void QCupsPrintDialog::propertyClicked()
{
	configdlg_ = new QCupsPrinterConfigDialog(propType_,this);
	connect(configdlg_,SIGNAL(defaultButtonPressed()),SLOT(propertiesSaveClicked()));
	int	printerIndex = dest_->currentItem();
	if (printerIndex >= 0 && printerIndex < destnum_)
	{
		cups_dest_t	*d(destinations_+printerIndex), *d_ed(dests_edited_+printerIndex);
		// initialize with edited destination if it has been edited (num_options > 0), otherwise
		// use normal destination.
		if (!configdlg_->initialize((d_ed->num_options > 0 ? d_ed : d)))
		{
			QString	msg = tr("Unable to retrieve attributes for %1 !").arg(d->name);
			QMessageBox::critical(this,tr("Print error"),msg.data(),QMessageBox::Ok|QMessageBox::Default,0);
		}
		else if (configdlg_->exec())
		{ // printer edited, fill edited destination with cups options (after re-initializing them)
			// free current options
			cupsFreeOptions(d_ed->num_options,d_ed->options);
			d_ed->num_options = 0;
			d_ed->options = 0;
			// get new options for future use
			configdlg_->getCupsOptions(d_ed->num_options,&(d_ed->options));
		}
	}
	delete configdlg_;
	configdlg_ = 0;
}

void QCupsPrintDialog::propertiesSaveClicked()
{
	// first check if conflict, and do nothing if yes
	if (configdlg_->hasConflict()) return;
	cups_dest_t	*currentdest = destinations_+dest_->currentItem(), *d_ed(dests_edited_+dest_->currentItem());
	// free options
	cupsFreeOptions(currentdest->num_options,currentdest->options);
	currentdest->num_options = 0;
	currentdest->options = 0;
	// get new options
	configdlg_->getCupsOptions(currentdest->num_options,&(currentdest->options));
	// save options
	cupsSetDests(destnum_,destinations_);
	// as options has been saved in normal destination, we can free possible edited destination
	if (d_ed->num_options > 0)
	{
		cupsFreeOptions(d_ed->num_options,d_ed->options);
		d_ed->num_options = 0;
		d_ed->options = 0;
	}
}

void QCupsPrintDialog::getAllOptions(int& num_options, cups_option_t **options)
{
	cups_option_t	*opt;
	// first add destinations options, do not add "empty" options if properties have been
	// edited : this option will then be added by "cups_options_".
	cups_dest_t	*currentdest = destinations_+dest_->currentItem(), *d_ed(dests_edited_+dest_->currentItem());
	for (int i=0;i<currentdest->num_options;i++)
	{
		opt = currentdest->options+i;
		if (strlen(opt->value) != 0 || d_ed->num_options == 0) num_options = cupsAddOption(opt->name,opt->value,num_options,options);
	}
	// then add possible changed options with "properties" button, stored in edited destinations.
	// If printer has not been edited or options has been saved, edited options will be empty
	for (int i=0;i<d_ed->num_options;i++)
	{
		opt = d_ed->options+i;
		num_options = cupsAddOption(opt->name,opt->value,num_options,options);
	}

	/*
	// add preview
	if (previewcheck_->isChecked())
		num_options = cupsAddOption("preview","",num_options,options);
	*/

	num_options = cupsAddOption("copies",copies_->text(),num_options,options);
	num_options = cupsAddOption("multiple-document-handling",(collate_->isChecked() ? "separate-documents-collated-copies" : "separate-documents-uncollated-copies"),num_options,options);
	if (reverse_->isChecked())
		num_options = cupsAddOption("OutputOrder","Reverse",num_options,options);
	if (pagecombo_->currentItem() > 0)
		num_options = cupsAddOption("page-set",(pagecombo_->currentItem() == 1 ? "odd" : "even"),num_options,options);
	if (range_->isChecked())
		num_options = cupsAddOption("page-ranges",rangeedit_->text(),num_options,options);
	else if (current_->isChecked())
		;
}

const char* QCupsPrintDialog::printerName()
{
	cups_dest_t	*currentdest = destinations_+dest_->currentItem();
	return currentdest->name;
}

void QCupsPrintDialog::setPropertyType(int type)
{
	propType_ = type;
}

void QCupsPrintDialog::allowCurrentPage(bool on)
{
	current_->setEnabled(on);
}

void QCupsPrintDialog::setPrinter(const char *pd)
{
	if (!pd) return;
	char	*printer((char*)pd);
	char	*instance = strchr(pd,'/');
	int	index(0);
	if (instance) instance++;
	for (index=0;index<destnum_;index++) {
		cups_dest_t	*d(destinations_+index);
		if (strcmp(d->name,printer) == 0 && (!instance || strcmp(instance,d->instance) == 0))
			break;
	}
	if (index < destnum_) dest_->setCurrentItem(index);
}

void QCupsPrintDialog::setCupsOptions(int num_options, cups_option_t *options)
{
	// new options are not added to normal destinations, because if we want to save options
	// for one printer, all other printer will also have these options saved (bad side effect).
	// We use edited destinations for this : when property will be clicked, they will be used, and
	// if we want to save, only the edited printer will have the new options saved.
	for (int i=0;i<destnum_;i++) {
		cups_dest_t	*d(destinations_+i), *d_ed(dests_edited_+i);
		// first copy all normal options to d_ed
		for (int k=0;k<d->num_options;k++) {
			cups_option_t	*opt(d->options+k);
			d_ed->num_options = cupsAddOption(opt->name,opt->value,d_ed->num_options,&(d_ed->options));
		}
		// then add new options to d_ed
		for (int k=0;k<num_options;k++) {
			cups_option_t	*opt(options+k);
			d_ed->num_options = cupsAddOption(opt->name,opt->value,d_ed->num_options,&(d_ed->options));
		}
	}
}

void QCupsPrintDialog::done(int result)
{
	bool	flag(true);
	if (result == Accepted) {	// check for possible conflicts
		cups_dest_t	*dest = destinations_+dest_->currentItem();
		QString	ppdFileName = cupsGetPPD(dest->name);
		ppd_file_t	*ppd = ppdOpenFile(ppdFileName.data());
                if (ppd) {
			int	n(0);
			cups_option_t	*opt;
			getAllOptions(n,&opt);
                        ppdMarkDefaults(ppd);
			cupsMarkOptions(ppd,n,opt);
			if (ppdConflicts(ppd) > 0) {
				flag = false;
				QString	msg = ppdConflictErrorMsg(ppd);
				msg.append(tr("\n(Edit printer's properties and select \"Advanced\" tab)"));
				QMessageBox::critical(this,tr("Configuration error"),msg.data(),QMessageBox::Ok|QMessageBox::Default,0);
			}
			ppdClose(ppd);
		}
		unlink(ppdFileName.data());
		if (!flag) return;
	}
	QDialog::done(result);
}

void QCupsPrintDialog::setDefaultClicked()
{
	int	printerIndex = dest_->currentItem();
	for (int i=0;i<destnum_;i++)
	{
		dests_edited_[i].is_default = 0;
		destinations_[i].is_default = 0;
	}
	destinations_[printerIndex].is_default = 1;
	dests_edited_[printerIndex].is_default = 1;
	cupsSetDests(destnum_,destinations_);
}
