/*
	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution

*/
#ifndef COMMON_H
#define COMMON_H

#include <qstringlist.h>
#include <qvaluevector.h>
#include <qdict.h>
#include <qmap.h>
#include <qevent.h>
#include <qcolor.h>

// some debug macros
#define dbg(x)    qDebug(#x " is <%s>", (x).latin1())
#define dbi(x)    qDebug(#x " is <%i>", (x))
#define dbp(s, x) qDebug(QString(s).arg(QString(x)).latin1())
#define dbs(x)    qDebug(QString(x).latin1())
#define db1       qDebug(QString("Mark Nr. 1").latin1())
#define db2       qDebug(QString("Mark Nr. 2").latin1())
#define db3       qDebug(QString("Mark Nr. 3").latin1())
#define db4       qDebug(QString("Mark Nr. 4").latin1())

// some synctactic sugar
#define loop(type, x, y) type::const_iterator x(y.constBegin()); \
                         for ( ; x != y.constEnd(); ++x)

#define loopList(x, y) QStringList::const_iterator x(y.constBegin()); \
                       for ( ; x != y.constEnd(); ++x)

// type shortcuts
typedef const QString&        SCRef;
typedef QStringList&          SList;
typedef const QStringList&    SCList;
typedef QValueVector<QString> StrVect;

namespace QGit {

	// minimum git version required
	const QString GIT_VERSION = "1.4.0";

	// key bindings
	enum KeyType {
		KEY_UP,
		KEY_DOWN,
		SHIFT_KEY_UP,
		SHIFT_KEY_DOWN,
		KEY_LEFT,
		KEY_RIGHT,
		CTRL_PLUS,
		CTRL_MINUS,
		KEY_U,
		KEY_D,
		KEY_DELETE,
		KEY_B,
		KEY_BCKSPC,
		KEY_SPACE,
		KEY_R,
		KEY_P,
		KEY_F
	};

	// tab pages
	enum TabType {
		TAB_REV,
		TAB_PATCH,
		TAB_FILE
	};

	// graph elements
	enum LaneType {
		EMPTY,
		ACTIVE,
		NOT_ACTIVE,
		MERGE_FORK,
		MERGE_FORK_R,
		MERGE_FORK_L,
		JOIN,
		JOIN_R,
		JOIN_L,
		HEAD,
		HEAD_R,
		HEAD_L,
		TAIL,
		TAIL_R,
		TAIL_L,
		CROSS,
		CROSS_EMPTY,
		INITIAL,
		BRANCH,
		UNAPPLIED,
		APPLIED,
		BOUNDARY,
		BOUNDARY_C, // corresponds to MERGE_FORK
		BOUNDARY_R, // corresponds to MERGE_FORK_R
		BOUNDARY_L, // corresponds to MERGE_FORK_L

		LANE_TYPES_NUM
	};
	const int COLORS_NUM	= 8;

	// graph helpers
	inline bool isHead(int x){ return (x == HEAD || x == HEAD_R || x == HEAD_L); }
	inline bool isTail(int x){ return (x == TAIL || x == TAIL_R || x == TAIL_L); }
	inline bool isJoin(int x){ return (x == JOIN || x == JOIN_R || x == JOIN_L); }
	inline bool isFreeLane(int x){ return (x == NOT_ACTIVE || x == CROSS || isJoin(x)); }
	inline bool isBoundary(int x){ return (x == BOUNDARY || x == BOUNDARY_C ||
	                                       x == BOUNDARY_R || x == BOUNDARY_L); }
	inline bool isMerge(int x){ return (x == MERGE_FORK || x == MERGE_FORK_R ||
	                                    x == MERGE_FORK_L || isBoundary(x)); }

	// custom events
	enum EventType {
		ERROR_EV      = 65432,
		POPUP_LIST_EV = 65433,
		POPUP_FILE_EV = 65434,
		POPUP_TREE_EV = 65435,
		MSG_EV        = 65436,
		ANN_PRG_EV    = 65437,
		UPD_DM_EV     = 65438,
		UPD_DM_MST_EV = 65439
	};

	// list views columns
	enum ColumnType {
		GRAPH_COL   = 0,
		ANN_ID_COL  = 1,
		LOG_COL     = 2,
		AUTH_COL    = 3,
		TIME_COL    = 4,
		DUMMY_COL   = 4, // used in main view to have a valid COMMIT_COL
		COMMIT_COL  = 5,
		LOG_MSG_COL = 98,// dummy col used for log messages searching
		SHA_MAP_COL = 99 // dummy col used when filter output is a set of matching sha
	};

	inline bool isInfoCol(int x) { return (x == TIME_COL || x == LOG_COL || x == AUTH_COL); }

	// default list view widths
	const int DEF_GRAPH_COL_WIDTH = 80;
	const int DEF_LOG_COL_WIDTH   = 450;
	const int DEF_AUTH_COL_WIDTH  = 200;
	const int DEF_TIME_COL_WIDTH  = 100;

	// colors
	const QColor BROWN       = QColor(150, 75, 0);
	const QColor ORANGE      = QColor(255, 160, 50);
	const QColor DARK_ORANGE = QColor(216, 144, 0);
	const QColor LIGHT_BLUE  = QColor(85, 255, 255);
	const QColor PURPLE      = QColor(221, 221, 255);
	const QColor DARK_GREEN  = QColor(0, 205, 0);

	// colors initialized at startup according to system wide settings
	extern QColor ODD_LINE_COL;
	extern QColor EVEN_LINE_COL;

	// patches drag and drop
	const QString PATCHES_DIR  = "/.qgit_patches_copy";
	const QString PATCHES_NAME = "qgit_import";

	// files status
	const QChar MODIFIED = 'M';
	const QChar DELETED  = 'D';
	const QChar NEW      = 'A';
	const QChar RENAMED  = 'R';
	const QChar COPIED   = 'C';
	const QChar UNKNOWN  = '?';

	// git index parameters
	const QString ZERO_SHA        = "0000000000000000000000000000000000000000";
	const QString CUSTOM_SHA      = "CUSTOM";
	const QChar IN_INDEX          = 'I';
	const QChar NOT_IN_INDEX      = 'N';
	const QString ALL_MERGE_FILES = "ALL_MERGE_FILES";

	// settings keys
	const QString APP_KEY         = "/qgit/";
	const QString FP_DIR_KEY      = "format_patch_last_dir";
	const QString FPATCH_ARGS_KEY = "format_patch_args";
	const QString FLAGS_KEY       = "patch_flags";
	const QString CMT_GEOM_KEY    = "commit_viewer_geometry";
	const QString CMT_SPLIT_KEY   = "commit_viewer_splitter_sizes";
	const QString CMT_TEMPL_KEY   = "commit_template_file_path";
	const QString CMT_ARGS_KEY    = "commit_args";
	const QString EX_KEY          = "exclude_file_path";
	const QString EX_PER_DIR_KEY  = "exclude_per_directory_file_name";
	const QString EXT_DIFF_KEY    = "external_diff_viewer";
	const QString REC_REP_KEY     = "recent_open_repos";
	const QString MCR_NAME_KEY    = "macro_name";
	const QString MCR_TEXT_KEY    = "commands";
	const QString MCR_LIST_KEY    = "macro_list";

	// settings default values
	const QString CMT_GEOM_DEF    = "290,140,495,540";
	const QString CMT_SPLIT_DEF   = "155,342";
	const QString CMT_TEMPL_DEF   = ".git/commit-template";
	const QString EX_DEF          = ".git/info/exclude";
	const QString EX_PER_DIR_DEF  = ".gitignore";
	const QString EXT_DIFF_DEF    = "kompare";
	const QString MCR_NAME_DEF    = "New macro";

	// settings booleans
	enum FlagType {
		// removed obsolete option
		MCR_REFRESH_F   = 2,
		NUMBERS_F       = 4,
		// removed obsolete option
		MCR_CMD_LINE_F  = 16,
		DIFF_INDEX_F    = 32,
		SIGN_PATCH_F    = 64,
		SIGN_CMT_F      = 128,
		VERIFY_CMT_F    = 256,
		// removed obsolete option
		REL_DATE_F      = 1024,
		ALL_BRANCHES_F  = 2048,
		WHOLE_HISTORY_F = 4096,
		RANGE_SELECT_F  = 8192
	};
	const int FLAGS_DEF = 8512;

	// settings helpers
	uint flags(SCRef group = "");
	bool testFlag(uint f, SCRef group = "");
	void setFlag(uint f, bool b, SCRef group = "");
	void writeSetting(SCRef key, SCRef value, SCRef group = "");

	// cache file
	const uint C_MAGIC  = 0xA0B0C0D0;
	const int C_VERSION = 12;

	const QString BAK_EXT          = ".bak";
	const QString C_DAT_FILE       = "/qgit_cache.dat";
	const QString C_OLD_DAT_FILE   = "/.qgit_cache.dat";
	const QString C_OLD_DAT_FILE_Z = "/.qgit_cache.dat.z";

	// misc
	const int MAX_DICT_SIZE    = 100003; // must be a prime number see QDict docs
	const int MAX_MENU_ENTRIES = 15;
	const int MAX_RECENT_REPOS = 5;
	const QString QUOTE_CHAR   = "$";
}

class Rev {
public:
	Rev() {}
	Rev(const QString& data, int idx) : orderIdx(idx), d(data), indexed(false) {
		isTag = isBranch = isRef = isCurrentBranch = false;
		isDiffCache = isApplied = isUnApplied = false,
		descRefsMaster = ancRefsMaster = descBrnMaster = -1;

		// these fields must be valid from beginning
		isBoundary = (data[0] == '-'); // public field
		parentsCnt = (data.find('\n', 39) - (int)isBoundary) / 41;
	}
	const QString parent(int idx) const;
	const QStringList parents() const;
	const QString sha() const { return isBoundary ? d.mid(1, 40) : d.left(40); }
	uint parentsCount() const { return parentsCnt; }
	const QString author() const { init(); return d.mid(autStart, autLen); }
	const QString authorDate() const { init(); return d.mid(autDateStart, autDateLen); }
	const QString shortLog() const { init(); return d.mid(sLogStart, sLogLen); }
	const QString longLog() const { init(); return d.right(lLogLen); }

	QValueVector<int> lanes, childs;
	QString tag, branch, ref, tagMsg;
	bool isTag, isBranch, isRef, isCurrentBranch;
	bool isDiffCache, isApplied, isUnApplied, isBoundary;
	int orderIdx;
	QValueVector<int> descRefs;     // list of descendant refs index, normally tags
	QValueVector<int> ancRefs;      // list of ancestor refs index, normally tags
	QValueVector<int> descBranches; // list of descendant branches index
	int descRefsMaster; // in case of many Rev have the same descRefs, ancRefs or
	int ancRefsMaster;  // descBranches these are stored only once in a Rev pointed
	int descBrnMaster;  // by corresponding index xxxMaster

private:
	void init() const { if (!indexed) indexData(); }
	void indexData() const;

	const QString d;
	uint parentsCnt;
	mutable bool indexed;
	mutable int autStart, autLen, autDateStart;
	mutable int autDateLen, sLogStart, sLogLen, lLogLen;
};
typedef QDict<Rev> RevMap; // faster then a map

class RevFile {

	friend class Cache; // to directly load status
	friend class Git;

	// 'status' is one ore more comma separated fields. First field
	// is always the status returned by 'git diff-tree' without -C option.
	// If file is renamed or copied a second field is added with the status
	// returned by 'git diff-tree -C' plus source and destination files info.
	// In case of a working dir file a third field is added with info about
	// git index status of the file (CACHED_FILE or NO_CACHED_FILE)
	QValueVector<QString> status;

public:
	RevFile() {}

	QValueVector<int> dirs; // index of a string vector
	QValueVector<int> names;
	QValueVector<int> mergeParent;

	// status helper functions
	QChar getStatus(int idx) const { return status[idx].at(0); }
	bool statusCmp(int idx, const QChar& st) const { return status[idx].at(0) == st; }
	bool isExtendedStatus(int idx) const { return status[idx].length() != 1; }
	bool isInIndex(int idx) const {

		return (isExtendedStatus(idx) && status[idx].section(',', 2, 2) == QGit::IN_INDEX);
	}
	QString getExtendedStatus(int idx) const {

		return (isExtendedStatus(idx) ? status[idx].section(',', 1, 1) : "");
	}
};
typedef QDict<RevFile> RevFileMap;

class FileAnnotation {
public:
	explicit FileAnnotation(int id) { annId = id; isValid = false; }
	FileAnnotation() { isValid = false; }
	QStringList lines;
	bool isValid;
	int annId;
	QString fileSha;
};

class BaseEvent: public QCustomEvent {
public:
	BaseEvent(SCRef d, int id) : QCustomEvent(id), payLoad(d) {}
	const QString data() const { return payLoad; }
private:
	const QString payLoad; // passed by copy
};

#define DEF_EVENT(X, T) class X : public BaseEvent { public: \
                        X (SCRef d) : BaseEvent(d, T) {} }

DEF_EVENT(MessageEvent, QGit::MSG_EV);
DEF_EVENT(AnnotateProgressEvent, QGit::ANN_PRG_EV);

class DeferredPopupEvent : public BaseEvent {
public:
	DeferredPopupEvent(SCRef msg, int type) : BaseEvent(msg, type) {}
};

class MainExecErrorEvent : public BaseEvent {
public:
	MainExecErrorEvent(SCRef c, SCRef e) : BaseEvent("", QGit::ERROR_EV), cmd(c), err(e) {}
	const QString command() const { return cmd; }
	const QString report() const { return err; }
private:
	const QString cmd, err;
};

#endif
