/***************************************************************************
                          rpreferencesdialog.cpp  -  description
                             -------------------
    begin                : Tue Nov 23 1999
    copyright            : (C) 1999 by Andrew Mustun
    email                : andrew@qcad.org
 ***************************************************************************/

#include "rpreferencesdialog.h"

#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qheader.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qstringlist.h>
#include <qwidget.h>
#include <qwidgetstack.h>

#include "rappwin.h"
#include "rcombobox.h"
#include "rconfig.h"
#include "rgraphic.h"
#include "rlineedit.h"
#include "rlistview.h"
#include "rprgdef.h"
#include "rstatuspanel.h"
#include "rstring.h"

#define WDG_OPTIONS     5
#define WDG_LANGUAGE    10
#define WDG_APPEARANCE  20
#define WDG_FONTS       21
#define WDG_CAD         30
#define WDG_GRID        31
#define WDG_DIMENSIONS  32
#define WDG_PAPER       40
#define WDG_CAM         50
#define WDG_PRINTING    60
#define WDG_PATHS       70


/*! \class RPreferencesDialog
    \author Andrew Mustun
    \brief Provides the preferences dialog called from "Options" -> "Preferences" menu.
*/


/** Constructor
    \param _drawingOnly Show only options for the current drawing not for the application
*/
RPreferencesDialog::RPreferencesDialog(QWidget * _parent, bool _drawingOnly, const char* _name, bool _modal, WFlags _f)
 : QDialog( _parent, _name, _modal, _f )
{
  setFixedSize(575, 480);
  drawingOnly = _drawingOnly;

  if(!drawingOnly) setCaption( QString(DEF_APPNAME) + " " + tr("Preferences") );
  else             setCaption( tr("Drawing preferences") );

  QBoxLayout *bl;
  QWidget* wdg;
  RComboBox* cb;
  RLineEdit* le;
  QGroupBox* gb;
  //QButtonGroup* bg;
  int border=6;


  RGraphic *gr=0;
  if( RAppWin::getRAppWin()->currentDoc() ) {
    gr = RAppWin::getRAppWin()->currentDoc();
  }
  else {
    if(drawingOnly) {
      return;
    }
  }

  bool lMetric;
  if(!drawingOnly) {
    lMetric = mtIsUnitMetric( RCONFIG->getSettingUnit("CAD:Unit") );
  }
  else {
    lMetric = mtIsUnitMetric( gr->getUnit() );
  }

  // create list view
  lvCategory = new RListView( this );
  lvCategory->setMultiSelection( false );
  lvCategory->addColumn( tr("Category") );
  lvCategory->setTreeStepSize( 20 );
  lvCategory->setSorting( -1 );
  lvCategory->header()->setClickEnabled( false );
  lvCategory->header()->setMovingEnabled( false );
  lvCategory->header()->setResizeEnabled( false );
  lvCategory->setHScrollBarMode( QScrollView::AlwaysOff );
  lvCategory->setVScrollBarMode( QScrollView::AlwaysOn );
  lvCategory->setGeometry( 10, 10, 160, height()-40 );

  // Fill list view:
  RListViewItem *rootItem, *subItem1;

  rootItem = new RListViewItem( lvCategory, tr("Options"), true, WDG_OPTIONS );
  if(!drawingOnly) new RListViewItem( rootItem, tr("Paths"), false, WDG_PATHS );

#ifdef DEF_LINUX
  if(!drawingOnly) new RListViewItem( rootItem, tr("Printing"), false, WDG_PRINTING );
#endif

  new RListViewItem( rootItem,   tr("Page"), false,   WDG_PAPER );

#ifdef DEF_CAM_EXPERT
  if(!drawingOnly) new RListViewItem( rootItem, tr("CAM"), false, WDG_CAM );
#endif

  subItem1 = new RListViewItem( rootItem, tr("CAD"), true, WDG_CAD );
  new RListViewItem( subItem1, tr("Dimensions"), false, WDG_DIMENSIONS );
  new RListViewItem( subItem1, tr("Grid"), false, WDG_GRID );
  if(!drawingOnly) {
    new RListViewItem( rootItem, tr("Appearance"), false, WDG_FONTS );
    new RListViewItem( rootItem, tr("Language"), false, WDG_LANGUAGE );
  }

  // Connect:
  connect( lvCategory, SIGNAL(selectionChanged(QListViewItem*)),
           this, SLOT(slotChangePage(QListViewItem*)) );

  // Create widgets and put them into a stack:
  widgetStack = new QWidgetStack( this );
  widgetStack->setGeometry( 180, 10, width()-190, height()-40 );

  // Options widget:
  wdg = new QWidget( widgetStack );
  widgetStack->addWidget( wdg, WDG_OPTIONS );
  bl = new QBoxLayout( wdg, QBoxLayout::TopToBottom, border );

    // Measuring system:
    gb = new QGroupBox( 4, Horizontal, tr("Measuring System"), wdg );
    bl->addWidget( gb );

    // Type:
    new QLabel( tr("Type:"), gb );
    cb = new RComboBox( false, gb );
    cb->fillInScaleTypes( lMetric );
    if(!drawingOnly) cb->setEditText( mtScaleTypeToString( Absolute ) );
    else             cb->setEditText( mtScaleTypeToString( Absolute ) );
    cbScaleType = cb;

    connect( cbScaleType, SIGNAL(activated(const QString &)),
             this, SLOT(slotScaleTypeChanged(const QString &)) );

    // Dummy widgets
    new QWidget( gb );
    new QWidget( gb );

    // Drawing scale:
    new QLabel( tr("Drawing scale:"), gb );
    cb = new RComboBox( true, gb );
    cb->fillInScale( lMetric, mtStringToScaleType( cbScaleType->currentText() ) );
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:PaperSpace") );
    else             cb->setValue( gr->getPaperSpace() );
    cbPaperSpace = cb;

    connect( cbPaperSpace, SIGNAL(activated(const QString &)),
             this, SLOT(slotScaleChanged(const QString &)) );

    // Global unit:
    labelScale = new QLabel( "", gb );
    labelScale->setAlignment( AlignCenter );
    cb = new RComboBox( false, gb );
    cb->fillInUnits();
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:Unit") );
    else             cb->setEditText( mtUnitToString(gr->getUnit()) );
    cbUnit = cb;

    labelScale->setText( mtGetSignForUnit( mtStringToUnit(cb->currentText()) ) + "    =  1" );

    connect( cbUnit, SIGNAL(activated(const QString &)),
             this, SLOT(slotSystemChanged(const QString &)) );

    bl->addStretch( 100 );


  if(!drawingOnly) {
    // Language widget:
    wdg = new QWidget( widgetStack );
    widgetStack->addWidget( wdg, WDG_LANGUAGE );
    bl = new QBoxLayout( wdg, QBoxLayout::TopToBottom, border );

      // Language box:
      gb = new QGroupBox( 2, Horizontal, tr("Language"), wdg );
      bl->addWidget( gb );

      // Language:
      new QLabel( tr("Program language:"), gb );
      cbLanguage = new RComboBox( gb );
      QStringList languageList;
      for(int i=0; i<RCONFIG->getLanguageNum(); ++i) {
        languageList.append( RCONFIG->getLanguage(i) );
      }
      languageList.sort();
      for(QStringList::Iterator it = languageList.begin(); it != languageList.end(); ++it ) {
        cbLanguage->insertItem( *it );
      }
      cbLanguage->setEditText( RCONFIG->getLanguageName( RCONFIG->getSetting("Language:Language") ) );

      connect( cbLanguage, SIGNAL(activated(const QString &)),
             this, SLOT(slotLanguageChanged(const QString &)) );

      bl->addStretch( 100 );
  }


  if(!drawingOnly) {
    // Font widget:
    wdg = new QWidget( widgetStack );
    widgetStack->addWidget( wdg, WDG_FONTS );
    bl = new QBoxLayout( wdg, QBoxLayout::TopToBottom, border );

      // Font box:
      gb = new QGroupBox( 2, Horizontal, tr("Font"), wdg );
      bl->addWidget( gb );

      // Font name:
      new QLabel( tr("Font name"), gb );
      cb = new RComboBox( true, gb );
      cb->insertItem( "Helvetica" );
      cb->insertItem( "Arial" );
      cb->setEditText( RCONFIG->getSetting("Application:FontName") );
      cbFontName = cb;

      // Charset:
      new QLabel( tr("Charset"), gb );
      cb = new RComboBox( true, gb );
      cb->fillInCharsets();
      cb->setEditText( RCONFIG->getSetting("Application:Charset") );
      cbCharset = cb;

      // Font sizes box:
      gb = new QGroupBox( 2, Horizontal, tr("Font sizes"), wdg );
      bl->addWidget( gb );

      // Font size for GUI elements:
      new QLabel( tr("GUI font size:"), gb );
      cb = new RComboBox( true, gb );
      cb->insertItem( "8" );
      cb->insertItem( "10" );
      cb->insertItem( "12" );
      cb->setEditText( RCONFIG->getSetting("Application:FontSize0") );
      cbFontSize0 = cb;

      // Font size for Grid witdth:
      new QLabel( tr("Grid info font size:"), gb );
      cb = new RComboBox( true, gb );
      cb->insertItem( "8" );
      cb->insertItem( "10" );
      cb->insertItem( "12" );
      cb->setEditText( RCONFIG->getSetting("Application:FontSize1") );
      cbFontSize1 = cb;

      // Font size for Status panel:
      new QLabel( tr("Status panel font size:"), gb );
      cb = new RComboBox( true, gb );
      cb->insertItem( "8" );
      cb->insertItem( "10" );
      cb->insertItem( "12" );
      cb->setEditText( RCONFIG->getSetting("Application:FontSize2") );
      cbFontSize2 = cb;

      // Font size for Library Browser:
      new QLabel( tr("Library browser font size:"), gb );
      cb = new RComboBox( true, gb );
      cb->insertItem( "6" );
      cb->insertItem( "8" );
      cb->insertItem( "10" );
      cb->insertItem( "12" );
      cb->setEditText( RCONFIG->getSetting("Application:FontSize3") );
      cbFontSize3 = cb;

      bl->addStretch( 100 );
  }


  if(!drawingOnly) {
    // CAD widget:
    wdg = new QWidget( widgetStack );
    widgetStack->addWidget( wdg, WDG_CAD );
    bl = new QBoxLayout( wdg, QBoxLayout::TopToBottom, border );

      // Preview box:
      gb = new QGroupBox( 2, Horizontal, tr("Preview"), wdg );
      bl->addWidget( gb );

      // Preview elements:
      new QLabel( tr("Preview elements:"), gb );
      cb = new RComboBox( true, gb );
      cb->insertItem( "0" );
      cb->insertItem( "10" );
      cb->insertItem( "50" );
      cb->insertItem( "100" );
      cb->insertItem( "500" );
      cb->insertItem( "1000" );
      cb->setEditText( RCONFIG->getSetting("CAD:PreviewElements") );
      cbPreviewElements = cb;

      // Snap box:
      gb = new QGroupBox( 2, Horizontal, tr("Snap options"), wdg );
      bl->addWidget( gb );

      // Snap range:
      new QLabel( tr("Snap range (px):"), gb );
      cb = new RComboBox( true, gb );
      cb->insertItem( "10" );
      cb->insertItem( "20" );
      cb->insertItem( "50" );
      cb->insertItem( "100" );
      cb->insertItem( "200" );
      cb->insertItem( "1000" );
      cb->setEditText( RCONFIG->getSetting("CAD:SnapRange") );
      cbSnapRange = cb;

      // Mark size:
      new QLabel( tr("Mark size (px):"), gb );
      cb = new RComboBox( true, gb );
      cb->insertItem( "2" );
      cb->insertItem( "4" );
      cb->insertItem( "8" );
      cb->insertItem( "16" );
      cb->setEditText( RCONFIG->getSetting("CAD:MarkSize") );
      cbMarkSize = cb;

      // Edit box:
      gb = new QGroupBox( 2, Horizontal, tr("Edit"), wdg );
      bl->addWidget( gb );

      ckbAutoRedraw = new QCheckBox( tr("Auto redraw"), gb );
      ckbAutoRedraw->setFixedSize( 140, 20 );
      ckbAutoRedraw->setChecked( (RCONFIG->getSettingInt("CAD:AutoRedraw")==0 ? false : true) );

      bl->addStretch( 100 );
  }


  // Dimensions widget:
  wdg = new QWidget( widgetStack );
  widgetStack->addWidget( wdg, WDG_DIMENSIONS );
  bl = new QBoxLayout( wdg, QBoxLayout::TopToBottom, border );

    // Dimension label:
    gb = new QGroupBox( 2, Horizontal, tr("Dimension labels"), wdg );
    bl->addWidget( gb );

    // Dim unit:
    new QLabel( tr("Unit:"), gb );
    cb = new RComboBox( false, gb );
    cb->fillInUnits();
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:DimensionUnit") );
    else             cb->setEditText( mtUnitToString( gr->getDimensionUnit() ) );
    cbDimUnit = cb;

    // Dim format:
    new QLabel( tr("Format:"), gb );
    cb = new RComboBox( false, gb );
    cb->insertItem( "Fractional" );
    cb->insertItem( "Decimal" );
    cb->insertItem( "Scientific" );
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:DimensionFormat") );
    else             cb->setEditText( mtValueFormatToString( gr->getDimensionFormat() ) );
    cbDimFormat = cb;

    // Dim exactness:
    new QLabel( tr("Exactness:"), gb );
    cb = new RComboBox( true, gb );
    cb->fillInExactness( lMetric );
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:DimensionExactness") );
    else {
      if(!lMetric && gr->getDimensionExactness()<1.0) {
        cb->setEditText( mtGetFractionString(gr->getDimensionExactness()) );
      }
      else {
        cb->setValue( gr->getDimensionExactness() );
      }
    }
    cbDimExactness = cb;

    if(!drawingOnly) {
      // Show units:
      ckbDimShowUnits = new QCheckBox( tr("Show all units"), gb );
      ckbDimShowUnits->setFixedSize( 140, 20 );
      ckbDimShowUnits->setChecked( (RCONFIG->getSettingInt("CAD:DimensionShowUnits")==0 ? false : true) );
    }

    // Angle dimension box:
    gb = new QGroupBox( 2, Horizontal, tr("Angle dimensions"), wdg );
    bl->addWidget( gb );

    // Angle dim unit:
    new QLabel( tr("Unit:"), gb );
    cb = new RComboBox( false, gb );
    cb->insertItem( "Deg" );
    cb->insertItem( "Rad" );
    cb->insertItem( "Gra" );
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:AngleDimensionUnit") );
    else             cb->setEditText( mtAngleUnitToString( gr->getAngleDimensionUnit() ) );
    cbAngDimUnit = cb;

    // Angle dim format:
    new QLabel( tr("Format:"), gb );
    cb = new RComboBox( false, gb );
    cb->insertItem( "Decimal_Degrees" );
    cb->insertItem( "/'/\"" );
    cb->insertItem( "Radians" );
    cb->insertItem( "Gradians" );
    //cb->insertItem( "surveyor" );
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:AngleDimensionFormat") );
    else             cb->setEditText( mtAngleFormatToString( gr->getAngleDimensionFormat() ) );
    cbAngDimFormat = cb;

    // Angle dim exactness:
    new QLabel( tr("Exactness:"), gb );
    cb = new RComboBox( true, gb );
    cb->insertItem( "0.001" );
    cb->insertItem( "0.01" );
    cb->insertItem( "0.1" );
    cb->insertItem( "1" );
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:AngleDimensionExactness") );
    else             cb->setValue( gr->getAngleDimensionExactness() );
    cbAngDimExactness = cb;

    // Dimension measures:
    gb = new QGroupBox( 3, Horizontal, tr("Dimension measures"), wdg );
    bl->addWidget( gb );

    // Dim text height:
    new QLabel( tr("Text height:"), gb );
    cb = new RComboBox( true, gb );
    cb->insertItem( "1.0" );
    cb->insertItem( "2.0" );
    cb->insertItem( "2.5" );
    cb->insertItem( "3.0" );
    cb->insertItem( "4.0" );
    cb->insertItem( "5.0" );
    cb->insertItem( "2/8" );
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:DimensionTextHeight") );
    else             cb->setValue( gr->getDimensionTextHeight() );
    cbDimTextHeight = cb;
    labelUnit1 = new QLabel( (lMetric ? "mm" : "inch"), gb );

    // Dimension arrow length:
    new QLabel( tr("Arrow size:"), gb );
    cb = new RComboBox( true, gb );
    cb->insertItem( "0.5" );
    cb->insertItem( "1.0" );
    cb->insertItem( "2.0" );
    cb->insertItem( "3.0" );
    cb->insertItem( "4.0" );
    cb->insertItem( "5.0" );
    cb->insertItem( "3/8" );
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:DimensionArrowLength") );
    else             cb->setValue( gr->getDimensionArrowSize() );
    cbDimArrowSize=cb;
    labelUnit2 = new QLabel( (lMetric ? "mm" : "inch"), gb );

    // Dimension over length:
    new QLabel( tr("Over length:"), gb );
    cb = new RComboBox( true, gb );
    cb->insertItem( "0.0" );
    cb->insertItem( "0.5" );
    cb->insertItem( "1.0" );
    cb->insertItem( "1.5" );
    cb->insertItem( "1/8" );
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:DimensionOverLength") );
    else             cb->setValue( gr->getDimensionOverLength() );
    cbDimOverLength=cb;
    labelUnit3 = new QLabel( (lMetric ? "mm" : "inch"), gb );

    // Dimension under length:
    new QLabel( tr("Under length:"), gb );
    cb = new RComboBox( true, gb );
    cb->insertItem( "0.0" );
    cb->insertItem( "0.5" );
    cb->insertItem( "1.0" );
    cb->insertItem( "1.5" );
    cb->insertItem( "1/8" );
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:DimensionUnderLength") );
    else             cb->setValue( gr->getDimensionUnderLength() );
    cbDimUnderLength=cb;
    labelUnit4 = new QLabel( (lMetric ? "mm" : "inch"), gb );

    bl->addStretch( 100 );


  // Page:
  wdg = new QWidget( widgetStack );
  widgetStack->addWidget( wdg, WDG_PAPER );
  bl = new QBoxLayout( wdg, QBoxLayout::TopToBottom, border );

    // Settings box:
    gb = new QGroupBox( 2, Horizontal, tr("Page settings:"), wdg );
    bl->addWidget( gb );

    // Page format:
    new QLabel( tr("Page format:"), gb );
    cb = new RComboBox( false, gb );
    cb->fillInPageFormats();
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:PageFormat") );
    else             cb->setEditText( mtPageFormatToString(gr->getPageFormat()) );
    cbPageFormat = cb;

    connect( cbPageFormat, SIGNAL(activated(const QString &)),
             this, SLOT(slotPageFormatChanged(const QString &)) );

    // Page Orientation:
    rbPortrait = new QRadioButton( tr("Portrait"), gb );
    rbLandscape = new QRadioButton( tr("Landscape"), gb );

    if(!drawingOnly) {
      if( RCONFIG->getSetting("CAD:PageOrientation") == "Portrait" )
        rbPortrait->setChecked( true );
      else
        rbLandscape->setChecked( true );
    }
    else {
      if( gr->getPageOrientation() == QPrinter::Portrait )
        rbPortrait->setChecked( true );
      else
        rbLandscape->setChecked( true );
    }

    connect( rbPortrait, SIGNAL(clicked()),
             this, SLOT(slotPortrait()) );
    connect( rbLandscape, SIGNAL(clicked()),
             this, SLOT(slotLandscape()) );

    /*new QLabel( tr("Page orientation:"), gb );
    cb = new RComboBox( false, gb );
    cb->fillInPageOrientations();
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:PageOrientation") );
    else             cb->setEditText( mtOrientationToString(gr->getPageOrientation()) );
    cbPageOrientation = cb;*/

    // Page size x:
    new QLabel( tr("Page width:"), gb );
    le = new RLineEdit( gb );
    if(!drawingOnly) le->setText( RCONFIG->getSetting("CAD:PageSizeX") );
    else             le->setFloat( gr->getPageSizeX() );
    lePageSizeX = le;

    // Page size y:
    new QLabel( tr("Page height:"), gb );
    le = new RLineEdit( gb );
    if(!drawingOnly) le->setText( RCONFIG->getSetting("CAD:PageSizeY") );
    else             le->setFloat( gr->getPageSizeY() );
    lePageSizeY = le;

    // Page origin x:
    new QLabel( tr("Page origin X:"), gb );
    le = new RLineEdit( gb );
    if(!drawingOnly) le->setText( RCONFIG->getSetting("CAD:PageOriginX") );
    else             le->setFloat( gr->getPageOriginX() );
    lePageOriginX = le;

    // Page origin y:
    new QLabel( tr("Page origin Y:"), gb );
    le = new RLineEdit( gb );
    if(!drawingOnly) le->setText( RCONFIG->getSetting("CAD:PageOriginY") );
    else             le->setFloat( gr->getPageOriginY() );
    lePageOriginY = le;

    if(!drawingOnly) {
      // Settings box:
      gb = new QGroupBox( 2, Horizontal, tr("Page border"), wdg );
      bl->addWidget( gb );

      // Show page border:
      ckbShowPageBorder = new QCheckBox( tr("Show page border"), gb );
      ckbShowPageBorder->setFixedSize( 200, 20 );
      ckbShowPageBorder->setChecked( (RCONFIG->getSettingInt("CAD:ShowPageBorder")==0 ? false : true) );
    }

    bl->addStretch( 100 );

  // Grid widget:
  wdg = new QWidget( widgetStack );
  widgetStack->addWidget( wdg, WDG_GRID );
  bl = new QBoxLayout( wdg, QBoxLayout::TopToBottom, border );

    // Grid box:
    gb = new QGroupBox( 2, Horizontal, tr("Grid options"), wdg );
    bl->addWidget( gb );

    // Grid format:
    new QLabel( tr("Grid format"), gb );
    cb = new RComboBox( false, gb );
    cb->insertItem( "Fractional" );
    cb->insertItem( "Decimal" );
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:GridFormat") );
    else             cb->setEditText( mtValueFormatToString( gr->getGridFormat() ) );
    cbGridFormat = cb;

    connect( cbGridFormat, SIGNAL(activated(const QString &)),
             this, SLOT(slotGridFormatChanged(const QString &)) );

    // Grid unit:
    /*
    new QLabel( tr("Grid unit:"), gb );
    cb = new RComboBox( false, gb );
    cb->fillInUnits();
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:GridUnit") );
    else             cb->setEditText( mtUnitToString( gr->getGridUnit() ) );
    cbGridUnit = cb;
    */

    // Grid width X:
    new QLabel( tr("Minimum grid width X:"), gb );
    cb = new RComboBox( true, gb );
    cb->fillInExactness( lMetric );
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:MinGridX") );
    else {
      if( gr->getGridFormat()==Fractional && gr->getMinGridX()<1.0 ) {
        cb->setEditText( mtGetFractionString(gr->getMinGridX()) );
      }
      else {
        cb->setValue( gr->getMinGridX() );
      }
    }
    cbMinGridX = cb;

    // Grid width Y:
    new QLabel( tr("Minimum grid width Y:"), gb );
    cb = new RComboBox( true, gb );
    cb->fillInExactness( lMetric );
    if(!drawingOnly) cb->setEditText( RCONFIG->getSetting("CAD:MinGridY") );
    else {
      if(!lMetric && gr->getMinGridY()<1.0) {
        cb->setEditText( mtGetFractionString(gr->getMinGridY()) );
      }
      else {
        cb->setValue( gr->getMinGridY() );
      }
    }
    cbMinGridY = cb;

    if(!drawingOnly) {
      // Meta grid:
      ckbMetaGrid = new QCheckBox( tr("Show meta grid"), gb );
      ckbMetaGrid->setChecked( (RCONFIG->getSettingInt("CAD:MetaGrid")==0 ? false : true) );

      // Grid autoscale box:
      gb = new QGroupBox( 2, Horizontal, tr("Autoscaling options"), wdg );
      bl->addWidget( gb );

      // Auto scale grid:
      ckbAutoScaleGrid = new QCheckBox( tr("Autoscale grid"), gb );
      ckbAutoScaleGrid->setChecked( (RCONFIG->getSettingInt("CAD:AutoScaleGrid")==0 ? false : true) );

      // Dummy widget
      new QWidget( gb );

      // Grid space limit:
      new QLabel( tr("Minimum grid space:"), gb );
      cb = new RComboBox( true, gb );
      cb->insertItem( "4" );
      cb->insertItem( "6" );
      cb->insertItem( "8" );
      cb->insertItem( "10" );
      cb->insertItem( "12" );
      cb->insertItem( "16" );
      cb->insertItem( "20" );
      cb->insertItem( "26" );
      cb->insertItem( "32" );
      cb->setEditText( RCONFIG->getSetting("CAD:MinGridSpace") );
      cbMinGridSpace = cb;
    }

    bl->addStretch( 100 );

#ifdef DEF_CAM_EXPERT
  if(!drawingOnly) {
    // CAM widget:
    wdg = new QWidget( widgetStack );
    widgetStack->addWidget( wdg, WDG_CAM );
    bl = new QBoxLayout( wdg, QBoxLayout::TopToBottom, border );

      // Machine generator box:
      gb = new QGroupBox( 2, Horizontal, tr("CAM"), wdg );
      bl->addWidget( gb );

      // Machine generator:
      new QLabel( tr("Machine-Generator:"), gb );
      cb = new RComboBox( true, gb );
      cb->fillInMachineGenerators();
      cb->setEditText( RCONFIG->getSetting("CAM:MachineGenerator") );
      cbMachineGenerator = cb;

      bl->addStretch( 100 );
  }
#endif

  if(!drawingOnly) {
    // Paths widget:
    wdg = new QWidget( widgetStack );
    widgetStack->addWidget( wdg, WDG_PATHS );
    bl = new QBoxLayout( wdg, QBoxLayout::TopToBottom, border );

      // Path box:
      gb = new QGroupBox( 2, Horizontal, tr("Paths"), wdg );
      bl->addWidget( gb );

      // Browser path:
      //new QLabel( tr("Path to browser:"), gb );
      //leBrowserPath = new RLineEdit( gb );
      //leBrowserPath->setText( RCONFIG->getSetting("Paths:BrowserPath") );

      // Library path 1:
      new QLabel( tr("Additional library path"), gb );
      leLibraryPath1 = new RLineEdit( gb );
      leLibraryPath1->setText( RCONFIG->getSetting("Paths:LibraryPath1") );

      bl->addStretch( 100 );
  }

#ifdef DEF_LINUX
  if(!drawingOnly) {
    // Printing widget:
    wdg = new QWidget( widgetStack );
    widgetStack->addWidget( wdg, WDG_PRINTING );
    bl = new QBoxLayout( wdg, QBoxLayout::TopToBottom, border );

      // Printer box:
      gb = new QGroupBox( 2, Horizontal, tr("Printer"), wdg );
      bl->addWidget( gb );

      // Printer name:
      new QLabel( tr("Printer name:"), gb );
      cbPrinterName = new RComboBox( true, gb );
      cbPrinterName->insertItem( "lp" );
      cbPrinterName->insertItem( "raw" );
      cbPrinterName->setEditText( RCONFIG->getSetting("Print:PrinterName") );

      // Print program:
      new QLabel( tr("Print program:"), gb );
      cbPrintProgram = new RComboBox( true, gb );
      cbPrintProgram->insertItem( "lpr" );
      cbPrintProgram->insertItem( "gv" );
      cbPrintProgram->setEditText( RCONFIG->getSetting("Print:PrintProgram") );

      bl->addStretch( 100 );
  }
#endif


  // Buttons:
  QPushButton *ok, *cancel;

  ok = new QPushButton( tr("Ok"), this );
  ok->setGeometry( width()-210,height()-30, 100,22 );
  connect( ok, SIGNAL(clicked()), SLOT(accept()) );

  cancel = new QPushButton( tr("Cancel"), this );
  cancel->setGeometry( width()-110,height()-30, 100,22 );
  connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

  lastUnit = cbUnit->currentText();

  slotChangePage( rootItem );
  slotPageFormatChanged( "" );
  slotSystemChanged( "" );
  //slotGridFormatChanged( "" );
}


/*! Destructor
*/
RPreferencesDialog::~RPreferencesDialog()
{
}



/*! Slot accept.
*/
void
RPreferencesDialog::accept()
{
  slotScaleChanged("");

  // Global options:
  //
  if(!drawingOnly) {

    // Language:
    //
    RCONFIG->addSetting( "Language:Language",
                         RCONFIG->getLanguageSign(cbLanguage->currentText()) );

    // Fonts:
    //
    RCONFIG->addSetting( "Application:FontName",
                         cbFontName->currentText() );
    RCONFIG->addSetting( "Application:Charset",
                         cbCharset->currentText() );
    RCONFIG->addSetting( "Application:FontSize0",
                         cbFontSize0->currentText() );
    RCONFIG->addSetting( "Application:FontSize1",
                         cbFontSize1->currentText() );
    RCONFIG->addSetting( "Application:FontSize2",
                         cbFontSize2->currentText() );
    RCONFIG->addSetting( "Application:FontSize3",
                         cbFontSize3->currentText() );


    // CAD:
    //
    RCONFIG->addSetting( "CAD:Unit",
                         cbUnit->currentText() );
    RCONFIG->addSetting( "CAD:PreviewElements",
                         cbPreviewElements->currentText() );
    RCONFIG->addSetting( "CAD:SnapRange",
                         cbSnapRange->currentText() );
    RCONFIG->addSetting( "CAD:MarkSize",
                         cbMarkSize->currentText() );
    RCONFIG->addSetting( "CAD:AutoRedraw",
                         (ckbAutoRedraw->isChecked() ? "1" : "0") );
    RCONFIG->addSetting( "CAD:DimensionUnit",
                         cbDimUnit->currentText() );
    RCONFIG->addSetting( "CAD:DimensionFormat",
                         cbDimFormat->currentText() );
    RCONFIG->addSetting( "CAD:DimensionExactness",
                         cbDimExactness->currentText() );
    RCONFIG->addSetting( "CAD:DimensionShowUnits",
                         (ckbDimShowUnits->isChecked() ? "1" : "0") );
    RCONFIG->addSetting( "CAD:AngleDimensionUnit",
                         cbAngDimUnit->currentText() );
    RCONFIG->addSetting( "CAD:AngleDimensionFormat",
                         cbAngDimFormat->currentText() );
    RCONFIG->addSetting( "CAD:AngleDimensionExactness",
                         cbAngDimExactness->currentText() );
    RCONFIG->addSetting( "CAD:DimensionTextHeight",
                         cbDimTextHeight->currentText() );
    RCONFIG->addSetting( "CAD:DimensionArrowLength",
                         cbDimArrowSize->currentText() );
    RCONFIG->addSetting( "CAD:DimensionOverLength",
                         cbDimOverLength->currentText() );
    RCONFIG->addSetting( "CAD:DimensionUnderLength",
                         cbDimUnderLength->currentText() );
    //RCONFIG->addSetting( "CAD:DimensionScale",
    //                     cbPaperSpace->currentText() );

    //RCONFIG->addSetting( "CAD:GridUnit",
    //                     cbGridUnit->currentText() );
    RCONFIG->addSetting( "CAD:GridFormat",
                         cbGridFormat->currentText() );
    RCONFIG->addSetting( "CAD:MinGridX",
                         cbMinGridX->currentText() );
    RCONFIG->addSetting( "CAD:MinGridY",
                         cbMinGridY->currentText() );
    RCONFIG->addSetting( "CAD:MetaGrid",
                         (ckbMetaGrid->isChecked() ? "1" : "0") );
    RCONFIG->addSetting( "CAD:AutoScaleGrid",
                         (ckbAutoScaleGrid->isChecked() ? "1" : "0") );
    RCONFIG->addSetting( "CAD:MinGridSpace",
                         cbMinGridSpace->currentText() );

    RCONFIG->addSetting( "CAD:PaperSpace",
                         cbPaperSpace->currentText() );
    RCONFIG->addSetting( "CAD:PageFormat",
                         cbPageFormat->currentText() );
    RCONFIG->addSetting( "CAD:PageOrientation",
                         (rbPortrait->isChecked() ? "Portrait" : "Landscape") );
    RCONFIG->addSetting( "CAD:PageSizeX",
                         lePageSizeX->text() );
    RCONFIG->addSetting( "CAD:PageSizeY",
                         lePageSizeY->text() );
    RCONFIG->addSetting( "CAD:PageOriginX",
                         lePageOriginX->text() );
    RCONFIG->addSetting( "CAD:PageOriginY",
                         lePageOriginY->text() );
    RCONFIG->addSetting( "CAD:ShowPageBorder",
                         (ckbShowPageBorder->isChecked() ? "1" : "0") );

#ifdef DEF_CAM_EXPERT
    // CAM:
    //
    RCONFIG->addSetting( "CAM:MachineGenerator",
                         cbMachineGenerator->currentText() );
#endif

#ifdef DEF_LINUX
    // Printer name
    RCONFIG->addSetting( "Print:PrinterName",
                         cbPrinterName->currentText() );
    // Print program
    RCONFIG->addSetting( "Print:PrintProgram",
                         cbPrintProgram->currentText() );
#endif

    // Paths:
    //  		
    //RCONFIG->addSetting( "Paths:BrowserPath",
    //                     leBrowserPath->text() );
    RCONFIG->addSetting( "Paths:LibraryPath1",
                         leLibraryPath1->text() );

    // Please restart...:
    //
    QMessageBox::information( this, DEF_APPNAME, tr("Please restart the program to ensure\nthat all changes take effect!"), tr("Ok") );

    RAppWin::getRAppWin()->getStatusPanel()->reinit();
    //RAppWin::getRAppWin()->reinit();
  }


  // Drawing options:
  //
  else if( RAppWin::getRAppWin()->currentDoc() ) {
    RGraphic *gr = RAppWin::getRAppWin()->currentDoc();

    gr->setUnit( mtStringToUnit(cbUnit->currentText()) );

    gr->setDimensionUnit( cbDimUnit->currentText() );
    gr->setDimensionFormat( cbDimFormat->currentText() );
    gr->setDimensionExactness( mtCalculate( cbDimExactness->currentText(), 0.001 ) );
    gr->setAngleDimensionUnit( cbAngDimUnit->currentText() );
    gr->setAngleDimensionFormat( cbAngDimFormat->currentText() );
    gr->setAngleDimensionExactness( mtCalculate( cbAngDimExactness->currentText(), 0.01 ) );
    gr->setDimensionTextHeight( mtCalculate( cbDimTextHeight->currentText(), 5.0 ) );
    gr->setDimensionOverLength( mtCalculate( cbDimOverLength->currentText(), 2.0 ) );
    gr->setDimensionUnderLength( mtCalculate( cbDimUnderLength->currentText(), 2.0 ) );
    gr->setDimensionArrowSize( mtCalculate( cbDimArrowSize->currentText(), 1.0 ) );
    //gr->setDimensionScale( mtCalculate( cbPaperSpace->currentText(), 1.0 ) );

    gr->setPaperSpace( cbPaperSpace->currentText().toDouble() );
    gr->setPageFormat( mtStringToPageFormat(cbPageFormat->currentText()) );
    gr->setPageOrientation( (rbPortrait->isChecked() ? QPrinter::Portrait : QPrinter::Landscape) );
    gr->setPageSizeX( mtCalculate( lePageSizeX->text() ) );
    gr->setPageSizeY( mtCalculate( lePageSizeY->text() ) );
    gr->setPageOriginX( mtCalculate( lePageOriginX->text() ) );
    gr->setPageOriginY( mtCalculate( lePageOriginY->text() ) );

    //gr->setGridUnit( cbGridUnit->currentText() );
    gr->setGridFormat( cbGridFormat->currentText() );
    gr->setMinGridX( mtCalculate( cbMinGridX->currentText(), 0.001 ) );
    gr->setMinGridY( mtCalculate( cbMinGridY->currentText(), 0.001 ) );
  }

  QDialog::accept();

}


/** Slot change page (selected list view item changed):
*/
void
RPreferencesDialog::slotChangePage( QListViewItem* _item )
{
  if(_item) {
    widgetStack->raiseWidget( ((RListViewItem*)_item)->getId() );
  }
}


/** Slot unit system changed (global option)
*/
void
RPreferencesDialog::slotSystemChanged(const QString &)
{
  bool lMetric = mtIsUnitMetric( mtStringToUnit(cbUnit->currentText()) );
  QString unit = cbUnit->currentText();

  double factor = mtGetMM( 1.0, mtStringToUnit(lastUnit) ) /
                  mtGetMM( 1.0, mtStringToUnit(unit) );
  QString pSpace = cbPaperSpace->currentText();

  if( unit!=lastUnit ) {
    cbDimExactness->fillInExactness( lMetric );
    cbMinGridX->fillInExactness( lMetric );
    cbMinGridY->fillInExactness( lMetric );
    cbScaleType->fillInScaleTypes( lMetric );

    cbDimUnit->setEditText( unit );
    //cbGridUnit->setEditText( unit );

    labelScale->setText( mtGetSignForUnit( mtStringToUnit(unit) ) + "    =  1" );

    // Adjust page size for new unit
    QString v;
    v.setNum( lePageSizeX->text().toDouble()*factor );
    lePageSizeX->setText( v );
    v.setNum( lePageSizeY->text().toDouble()*factor );
    lePageSizeY->setText( v );

    // Adjust page origin for new unit
    v.setNum( lePageOriginX->text().toDouble()*factor );
    lePageOriginX->setText( v );
    v.setNum( lePageOriginY->text().toDouble()*factor );
    lePageOriginY->setText( v );

    if( lMetric ) {
      cbDimFormat->setEditText( "Decimal" );
      cbDimExactness->setEditText( "0.01" );
      cbGridFormat->setEditText( "Decimal" );
      cbMinGridX->setEditText( "0.001" );
      cbMinGridY->setEditText( "0.001" );

      cbDimTextHeight->setEditText( "5" );
      cbDimArrowSize->setEditText( "3" );
      cbDimOverLength->setEditText( "2" );
      cbDimUnderLength->setEditText( "2" );
      labelUnit1->setText( "mm" );
      labelUnit2->setText( "mm" );
      labelUnit3->setText( "mm" );
      labelUnit4->setText( "mm" );
    }
    else {
      cbDimFormat->setEditText( "Fractional" );
      cbDimExactness->setEditText( "1/128" );
      cbGridFormat->setEditText( "Fractional" );
      cbMinGridX->setEditText( "1/1024" );
      cbMinGridY->setEditText( "1/1024" );

      cbDimTextHeight->setEditText( "1/5" );
      cbDimArrowSize->setEditText( "1/8" );
      cbDimOverLength->setEditText( "1/12" );
      cbDimUnderLength->setEditText( "1/12" );
      labelUnit1->setText( "inch" );
      labelUnit2->setText( "inch" );
      labelUnit3->setText( "inch" );
      labelUnit4->setText( "inch" );
    }

  }

  slotScaleTypeChanged("");

  cbPaperSpace->setEditText(pSpace);

  lastUnit = unit;
}


/** Slot grid format changed (grid setting) -> change combobox contents for grid spaces
*/
void
RPreferencesDialog::slotGridFormatChanged(const QString &_str)
{
  if( cbGridFormat->currentText()=="Decimal" ) {
    cbMinGridX->fillInExactness( true );
    cbMinGridY->fillInExactness( true );
    if(!_str.isEmpty()) {
      cbMinGridX->setEditText( "0.001" );
      cbMinGridY->setEditText( "0.001" );
    }
  }
  else {
    cbMinGridX->fillInExactness( false );
    cbMinGridY->fillInExactness( false );
    if(!_str.isEmpty()) {
      cbMinGridX->setEditText( "1/1024" );
      cbMinGridY->setEditText( "1/1024" );
    }
  }
}



/** Slot scale type changed
*/
void
RPreferencesDialog::slotScaleTypeChanged(const QString &)
{
  bool lMetric = mtIsUnitMetric( mtStringToUnit(cbUnit->currentText()) );
  cbPaperSpace->fillInScale( lMetric, mtStringToScaleType( cbScaleType->currentText() ) );
  cbPaperSpace->setEditText( "1.0" );
}



/** Slot scale changed -> calculate scale factor
*/
void
RPreferencesDialog::slotScaleChanged(const QString &)
{
  QString text = cbPaperSpace->currentText();
  QString newText=text;

  // given in format 1:x
  if( text.contains( ':' ) ) {
    int p = text.find( ':' );
    QString left = text.left( p );
    QString right = text.right( text.length()-p-1 );

    double factor = left.toDouble() / right.toDouble();
    newText = strFloatToString( factor, 12 );
  }
  else if( text.contains( '=' ) ) {

    // format x" = 1'-0"
    if( text.contains( "1'-0\"" ) ) {
      int p = text.find( '"' );
      QString left = text.left( p );

      double factor = mtCalculate( left ) / 12.0;
      //newText.setNum( factor, 'f', 12 );
      newText = strFloatToString( factor, 12 );
    }

    else {
      int p = text.find( '=' );
      QString right = text.right( text.length()-p-1 );

      double rightFactor = (right.at( right.length()-1 ) == '"' ? 1.0 : 12.0);
      double factor = 1.0 / (right.left( right.length()-1 ).toDouble() * rightFactor);
      //newText.setNum( factor, 'f', 12 );
      newText = strFloatToString( factor, 12 );
    }
  }

  cbPaperSpace->setEditText( newText );
}



/** Slot page format changed -> insert sizes / (de)activate size boxes
*/
void
RPreferencesDialog::slotPageFormatChanged(const QString & _str)
{
  if(cbPageFormat->currentText()=="User") {
    lePageSizeX->setEnabled( true );
    lePageSizeY->setEnabled( true );
	rbPortrait->setEnabled( false );
	rbLandscape->setEnabled( false );
  }
  else {
    lePageSizeX->setEnabled( false );
    lePageSizeY->setEnabled( false );
	rbPortrait->setEnabled( true );
	rbLandscape->setEnabled( true );

    if(!_str.isEmpty()) {
      Unit unit = mtStringToUnit( cbUnit->currentText() );
      double sizeX = mtGetPageWidthMM(  mtStringToPageFormat( cbPageFormat->currentText() ) ) / mtGetMM( 1.0, unit );
      double sizeY = mtGetPageHeightMM( mtStringToPageFormat( cbPageFormat->currentText() ) ) / mtGetMM( 1.0, unit );

      if(rbLandscape->isChecked()) {
        lePageSizeX->setFloat( sizeY );
        lePageSizeY->setFloat( sizeX );
      }
      else {
        lePageSizeX->setFloat( sizeX );
        lePageSizeY->setFloat( sizeY );
      }
    }
  }
}



/** Slot language changed -> adjust charset
*/
void
RPreferencesDialog::slotLanguageChanged(const QString &)
{
  cbCharset->setEditText( mtCharsetToString( RCONFIG->getLanguageCharset( cbLanguage->currentText() ) ) );
}



/** Slot page orientation changed -> swap width/height
*/
void
RPreferencesDialog::slotPortrait()
{
  rbLandscape->setChecked( false );
  rbPortrait->setChecked( true );
  slotPageFormatChanged("_");
}


/** Slot page orientation changed -> swap width/height
*/
void
RPreferencesDialog::slotLandscape()
{
  rbLandscape->setChecked( true );
  rbPortrait->setChecked( false );
  slotPageFormatChanged("_");
}



// EOF
