#ifndef INC__GIOVANNI_DICANIO__EXPRPARSER_HPP_
#define INC__GIOVANNI_DICANIO__EXPRPARSER_HPP_


//===============================================================
//
//		ExprParser		-- Written by Giovanni Dicanio
//
//		A C++ class to parse mathematical expressions.
//
//		(v. 0.3)
//
//
//		Copyright (c) by Giovanni Dicanio
//
//===============================================================



//---------------------------------------------------------------
// File:			exprparser.hpp
// Description:		Class header file
// Author:			Giovanni Dicanio
// First date:		20th January 2000
// Last update:		1st February 2000
//---------------------------------------------------------------



//---------------------------------------------------------------
//				CONSTANT DEFINITION SECTION
//---------------------------------------------------------------

// Maximum number of variables an ExprParser object can store.
#define EXPRPARSER_MAX_VARIABLES		10

// Maximum length of a variable identifier.
#define EXPRPARSER_MAX_VARIABLE_IDENTIFIER_CHARS	32

// Default angle unit.
#define EXPRPARSER_DEFAULT_ANGLE_UNIT	ExprParser::DEGREES


//---------------------------------------------------------------
// Class name:		ExprParser
// Purpose:			Parse a mathematical expression.
//---------------------------------------------------------------

class ExprParser
{
public:

//---------------------------------------------------------------
//					PUBLIC PART OF CLASS
//---------------------------------------------------------------
	
	// Real data types = double floating points
	typedef double	real;	

	// Errors that can occurr:
	enum errorCode {		
		OK = 0,						// all right (no error)
		
		DIVIDE_BY_ZERO,				// division by 0

		PARENTHESES_NOT_BALANCED,	// parentheses non balanced
		OPEN_PARENTHESIS_EXPECTED,	// in that position of expression string,
									// a '(' parenthesis was expected
		CLOSE_PARENTHESIS_EXPECTED, // '(' expected
		
		COMMA_EXPECTED,				// a ',' was expected
		
		UNKNOWN_VARIABLE,			// unknown variable
		VARIABLE_MEMORY_LEAK,		// the variable array is full!
		
		BAD_IDENTIFIER,				// a bad identifier encountered

		DOMAIN_ERROR,				// value not in function domain
									// (e.g.: log(x) is defined for x > 0;
									// so log(-3) causes a DOMAIN_ERROR.)

		LAST_ERROR					// dummy
	};


	// Units for angles:
	enum angleUnit {
		RADIANS,					// radians
		DEGREES						// degrees

		// ... add more in the future ...
	};


	// Useful constants:

	// PI constant
	static const real pi; // = 3.1415926535897932384626433832795

	// 'e' (Neperian number) constant
	static const real e; // = 2.71828182845904523536028747135266


//
// CONSTRUCTION/DESTRUCTION
//
	// Default constructor.
	ExprParser();
	
	// Copy constructor.
	ExprParser(const ExprParser & ep);

	// Construct using an expression.
	ExprParser(const char * expr);
	
	// Destructor (free dynamically allocated resources).
	~ExprParser();


//
// ATTRIBUTES/STATUS
//

	// Set a new expression to be parsed.
	ExprParser & setExpression(const char * expr);

	// Return current expression.
	const char * getExpression() const;

	// Return the result of the expression.
	real	getValue();

	// Has any error occurred during parsing?
	bool	errorOccurred() const;

	// Get error code.
	errorCode getErrorCode() const;

	// Which is current angle unit?
	angleUnit getAngleMode() const;

	// Change current angle unit.
	ExprParser & setAngleMode(angleUnit unit);


//
// OVERLOADED OPERATORS
//
	// Copy from a parser object.
	ExprParser & operator=(const ExprParser & ep);



//---------------------------------------------------------------
// Variables managment
//---------------------------------------------------------------

	// This is the maximum number of variables storable in a parser object.
	static int maxNumberOfVariables;

	// This is the maximum length (number of characters) of a variable identifier.
	static int maxVarIdentifierLength;


	// Set the value of an existing variable or add new variable.
	bool setVariable(const char * name, real value);

	// Get the value of a varibale.
	bool getVariable(const char * name, real & value);

	// Get current number of variables.
	int  getNumberOfVariables() const;

	// Clear all variable memory.
	ExprParser & deleteAllVariables();


//---------------------------------------------------------------
//			PROTECTED DATA
//---------------------------------------------------------------

protected:

	char *		m_expr;		// expression to be parsed
	real		m_value;	// the result of parsing
	errorCode	m_error;	// error occurred
	bool		m_needParsing; // if true, we must reparse the expression
	angleUnit	m_angleMode;// angle unit (radians, degrees, etc.)


	//
	// Parsing process specific data members:
	//
	// The following data members are helper for the parsing process.
	// So, they are not copied when two parser objects are copied.
	char *		m_currChar;	// current character in the expression string
	int			m_errorPos; // position in expression string of character which caused error


	// This is used to stop the parsing process (see setError() for more details...)
	static char stopParsingChar;


	//
	// Variable managment
	//
	// In this version of the class, variables are stored in a fixed
	// dimension array. Better implementations could define a stand-alone
	// VariableManager class to manage variables. This class may support
	// variable numbers of variables (it could be implemented by a linked-list
	// or an hash table...).
	
	typedef char variableNameType[EXPRPARSER_MAX_VARIABLE_IDENTIFIER_CHARS+1];
	
	struct Variable {
		variableNameType identifier;	// variable identifier
		real			 value;			// variable value
	};									// a variable object
		
	// The place in which variables are stored.
	Variable m_variables[EXPRPARSER_MAX_VARIABLES];
	
	// 0-based index for current available variable slot in
	// the previous array.
	int			 m_freeVariableSlot;

	// Copy all variables from a source parser object to this one.
	bool	copyAllVariablesFromParser(const ExprParser & ep);

	// Free variables.
	void	freeVariables();


//---------------------------------------------------------------
// Helper for construction/copy/destruction
//---------------------------------------------------------------

	// Copy from another parser object to this.
	bool copyFromParser(const ExprParser & ep);

	// Init current parser object. (If expr == NULL, set m_expr to null string.)
	bool initParser(const char * expr);

	// Cleanup parser object resources.
	bool destroyParser();


//---------------------------------------------------------------
// Helper for parsing process
//---------------------------------------------------------------

	// Do the parsing.
	bool doParsing();

	// Set up variables and status for starting the parsing process.
	bool startParsing();

	// Parse an expression.
	real doExpression();

	// Parse a MulDiv expression.
	real doMulDiv();

	// Parse numbers and functions.
	real doNumberFunction();

	// Parse numbers (specific).
	real doNumber();

	// Parse functions and variables (specific).
	real doFunctionVariable();

	// Get an identifier string.
	char * getIdentifier();

	// Go ahead, jumping spaces and tabs.
	void skipBlanks();

	// Go ahead in parsing the string.
	void goAhead();

	// Set an error status (and stop parsing).
	bool setError(errorCode err);

	// Check for parentheses balancing.
	bool parenthesesBalanced();


	// Convert angles in radians 
	// (useful for input parameters of trig functions!).
	real convertAngleInRadians(real angle) const;

	// Convert angles to current angle mode
	// (useful for return value of inverse trig functions).
	real convertAngleInCurrMode(real angleRad) const;


//
// Specific functions parsing helpers:
//

	// sin(x)
	real doFunctionSin();

	// cos(x)
	real doFunctionCos();

	// tan(x)
	real doFunctionTan();

	// cot(x)
	real doFunctionCot();


	// asin(x)
	real doFunctionASin();

	// acos(x)
	real doFunctionACos();

	// atan(x)
	real doFunctionATan();


	// sqrt(x)
	real doFunctionSqrt();

	// sqr(x)
	real doFunctionSqr();

	// pow(x,y)
	real doFunctionPow();

	// exp(x)
	real doFunctionExp();

	// ln(x)
	real doFunctionLn();


//---------------------------------------------------------------
// Helper for string managment
//---------------------------------------------------------------
	
	// Create a string, copying from an exsisting one.
	static char * stringCreateFrom(const char * source);

	// Delete a string.
	static void	  stringDelete(char ** ptrString);
};

//---------------------------------------------------------------
// END OF CLASS: ExprParser
//---------------------------------------------------------------


#include "exprparser.inl"	// inline methods implementations


#endif // INC__GIOVANNI_DICANIO__EXPRPARSER_HPP_
