/***************************************************************************
  oldstore.cpp
  -------------------
  general purpose file storage class (deprecated)
  -------------------
  begin         June 3rd 1999
  author        David Johnson <david@usermode.org>
  -------------------

  Copyright 1999, 2001 David Johnson
  Please see header file for copyright and license information
 ***************************************************************************/

#include <qobject.h>
#include <qtextstream.h>
#include "oldstore.h"

// This is version 0.5 of the Store class

// NOTE: This class will soon be deprecated. I will be switching over to XML using
// QDom. Store will still be around for a while, but will only be used to convert
// old files to the new format

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous definitions

const unsigned MAX_LENGTH   = 512;        // if this isn't big enough, tough!
const char STORE_DELIM      = '=';
const char STORE_COMMENT    = '#';

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// OldStore()
// -------
// Constructor

OldStore::OldStore(int mode, const QString &filename, const QString &format, QString version)
{
    mode_ = mode;
    version_ = version;
    format_ = format;
    filename_ = filename;
    good_ = false;
    group_ = "";
    name_ = "";
    value_ = "";

    if (!((mode_ == IO_ReadOnly) || (mode_ == IO_WriteOnly))) {
        // invalid mode, don't proceed
        return;
    }
    // open file according to mode
    file_.setName(filename);
    if (!file_.open(mode_)) {
        // error opening file
        good_ = false;
        file_.close();
    } else {
        // file opened successfully
        good_ = true;
        if (!format_.isEmpty()) { // need to check for proper header
            if (!((mode_ == IO_ReadOnly) ? getHeader() : putHeader())) {
                // error in header
                good_ = false;
                file_.close();
            }
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// ~OldStore()
// --------
// Destructor

OldStore::~OldStore()
{
    // close any open files
    if (file_.isOpen())
        file_.close();
}

//////////////////////////////////////////////////////////////////////////////
// Utility Methods                                                          //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// good()
// ------
// Return true if no file error exists

bool OldStore::good()
{
    if (!good_) {
        // badness was set somewhere else
        return false;
    }
    // check for file errors
    return (file_.status() ==  IO_Ok);
}

//////////////////////////////////////////////////////////////////////////////
// getHeader()
// ------------
// Check for a valid header

bool OldStore::getHeader()
{
    if (mode_ != IO_ReadOnly) {
        // wrong mode
        return false;
    }
    QString line, token;
    // get first line of text
    file_.readLine(line, MAX_LENGTH);
    // set up text stream
    QTextStream istr(&line, IO_ReadOnly);
    // Check for comment character
    istr >> token;
    if (token[0] != STORE_COMMENT) {
        qWarning(QT_TR_NOOP("Store: Missing or bad header in file ") + filename_ + "\n");
        return false;
    }
    // check for format
    istr >> token;
    if (QString::compare(token.lower(), format_.lower())) { // reminder: with compare, zero equals true
        qWarning(QT_TR_NOOP("Store: Bad header in file ") + filename_ + "\n");
        return false;
    }
    // check for version keyword
    // may be "v" or "version"
    istr >> token;
    token.lower(); // take it to lower case
    if ((token.compare("v")) && (token.compare("version"))) {
        // no optional version
        version_ = "NONE";
    } else {
        // get version (accepting next token as valid)
        istr >> token;
        version_ = token;
    }
    return true;
}

//////////////////////////////////////////////////////////////////////////////
// putHeader()
// -----------
// Write a header to file

bool OldStore::putHeader()
{
    // check mode
    if (mode_ != IO_WriteOnly) {
        // wrong mode
        return false;
    }
    // write out the header (no newline)
    QTextStream ostr(&file_);
    ostr << STORE_COMMENT << " " << format_;
    if (version_.length() > 0)
        ostr << " v " << version_;
    // is everything still ok?
    return (file_.status() ==  IO_Ok);
}

//////////////////////////////////////////////////////////////////////////////
// Storage Methods                                                          //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// putGroup()
// ----------
// Write a new element group

bool OldStore::putGroup(const QString &group)
{
    if (mode_ != IO_WriteOnly) {
        // wrong mode
        return false;
    }
    // put group in brackets all by itself
    QTextStream ostr(&file_);
    ostr << "\n[" << group << "]";
    // is everything still ok?
    return (file_.status() == IO_Ok);
}

//////////////////////////////////////////////////////////////////////////////
// putLine()
// ---------
// Write a new line with name and value pair

bool OldStore::putLine(const QString &name, const QString &value)
{
    if (mode_ != IO_WriteOnly) {
        // wrong mode
        return false;
    }
    // start a newline
    QTextStream ostr(&file_);
    ostr << "\n";
    // write name, delimiter and value
    ostr << name << " " << STORE_DELIM << " " << value;
    // is everything still ok?
    return (file_.status() == IO_Ok);
}

//////////////////////////////////////////////////////////////////////////////
// putComment()
// ------------
// Write a comment line

bool OldStore::putComment(const QString &comment)
{
    if (mode_ != IO_WriteOnly) {
        // wrong mode
        return false;
    }
    QTextStream ostr(&file_);
    ostr << "\n" << STORE_COMMENT << " " << comment;
    // is everything still ok?
    return (file_.status() == IO_Ok);
}

//////////////////////////////////////////////////////////////////////////////
// putElement()
// ------------
// Write an element out

bool OldStore::putElement(const QString &element)
{
    if (mode_ != IO_WriteOnly) {
        // wrong mode
        return false;
    }
    QTextStream ostr(&file_);
    ostr << ", " << element;
    // is everything still ok?
    return (file_.status() == IO_Ok);
}

//////////////////////////////////////////////////////////////////////////////
// Retrieval Methods                                                        //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// nextValidLine()
// ---------------
// Read in next valid line

QString OldStore::nextValidLine()
{
    QString line ="";
    QString buffer = "";
    QString token = "";
    bool valid = false;
    while (!valid) {
        // check for eof
        if (file_.atEnd()) {
            // no more lines
            return "";
        }
        file_.readLine(line, MAX_LENGTH);
        if (line.length() == 0) {
            // line with no content
            valid = false;
        } else {
            // do some checking on line
            buffer = line;
            QTextStream istr(&buffer, IO_ReadOnly);
            // get first element
            istr >> token;
            if (token.length() == 0) {
                // line must be whitespace only
                valid = false;
            } else if (token[0] == STORE_COMMENT) {
                // it's a comment, ignore line
                valid = false;
            } else {
                // a valid token, thus valid line
                valid = true;
            }
        }
    }
    return line;
}

//////////////////////////////////////////////////////////////////////////////
// getLine()
// ---------
// Read the next line

bool OldStore::getLine()
{
    if (mode_ != IO_ReadOnly) {
        // wrong mode
        return false;
    }

    // read in next valid line
    QString line = nextValidLine();
    if (line.length() == 0) {
        //didn't get a valid line, return
        return false;
    }
    // line is now valid
    QString token = "";
    QString element = "";
    QTextStream istr(&line, IO_ReadOnly);

    // process name element
    // get first token
    istr >> token;
    if ((token[0] != STORE_DELIM) && (token.length() > 0))
        element = token;
    // get rest of tokens
    istr >> token;
    while ((token[0] != STORE_DELIM) && (token.length() > 0)) {
        element += " ";
        element += token;
        istr >> token;
    }
    // is element a group?
    if ((element[0] == '[') && (element[element.length()-1] == ']')) {
        //get rid of brackets
        int len = element.length();
        element = element.mid(1, len-2);
        group_ = element;
    }
    // set name to element
    name_ = element;
    // everything else is now the value element
    element = "";
    // get next token
    istr >> token;
    if (token.length() > 0)
        element = token;
    // get rest of tokens
    istr >> token;
    while (token.length() > 0) {
        element += " ";
        element += token;
        istr >> token;
    }
    value_ = element;

    return true;
}

//////////////////////////////////////////////////////////////////////////////
// getGroup()
// ----------
// Return the current element group

const QString OldStore::getGroup()
{
    if (mode_ != IO_ReadOnly) {
        // wrong mode
        return "";
    }
    return group_;
}

//////////////////////////////////////////////////////////////////////////////
// getName()
// ---------
// Return the current name element

const QString OldStore::getName()
{
    if (mode_ != IO_ReadOnly) {
        // wrong mode
        return "";
    }
    return name_;
}

//////////////////////////////////////////////////////////////////////////////
// getValue()
// ----------
// Return the current value element

const QString OldStore::getValue()
{
    if (mode_ != IO_ReadOnly) {
        // wrong mode
        return "";
    }
    return value_;
}    
