// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "sphere.h"
#include "sphmodel.h"
#include GL_INCLUDE    // see platform.h

namespace visual {

vector
sphere::getScale()
{
	return vector( radius, radius, radius);
}

void
sphere::set_radius( double r)
{
	write_lock L(mtx);
	radius = r;
}

double
sphere::rayIntersect( const vector &camera, const vector &ray)
{
	if (degenerate)
		return 0.0;

	vector delta(camera.x - mwt[0][3], camera.y - mwt[1][3], camera.z - mwt[2][3]);

	// quadratic formula
	//  A = |ray| = 1, since ray is normalized
	double B = 2 * ray.dot(delta);
	double C = delta.mag2() - scale.x*scale.x;

	double disc = B*B - 4*C;  // quadratic discriminant
	if (disc < 0.0)
		return 0.0;

	disc = std::sqrt(disc);

	// find the smaller positive root

	// disc is positive, so if this root is positive it
	//   is smallest
	double t = (-B - disc) * 0.5;

	if (t<0)  // otherwise try this one
		t = (-B + disc) * 0.5;

	return t;
}

void
sphere::glRender(rView &view)
{
	if (degenerate)
		return;

	view.ext_sphere( mwt * vector(0,0,0), scale.x );
	lighting lt(view.lights, wlt);
	tmatrix mct(mwt, view.wct);

	// Level-of-detail heuristic.
	//   xxx Figure out how this should actually work!
	vector a = mct*vector(0,0,0) / mct.w(vector(0,0,0));
	vector b = mct*vector(0.5,0,0) / mct.w(vector(0.5,0,0));
	vector c = mct*vector(0,0.5,0) / mct.w(vector(0,0.5,0));
	vector d = mct*vector(0,0,0.5) / mct.w(vector(0,0,0.5));
	float size = std::sqrt((a-b).mag2() + (a-c).mag2() + (a-d).mag2());
	int n;
	if (size < 0.02)
		n = 0;
	else if (size < 0.125)
		n = 1;
	else
		n = 2;

	sph_model& model = sph_model::get(n);

	// Projection and lighting loop
	float* col = model.color;
	float* mv = model.verts;
	for(int v=0; v<model.nverts; v++, mv+=3, col += 4) {
		double illum = lt.illuminate(mv[0], mv[1], mv[2]);

		col[0] = color.r*illum;
		col[1] = color.g*illum;
		col[2] = color.b*illum;
		col[3] = 1.0;
	}
	mct.gl_load();

	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);
	glVertexPointer( 3, GL_FLOAT, 3*sizeof(float), model.verts);
	glColorPointer(4, GL_FLOAT, 4*sizeof(float), model.color);
	glShadeModel(GL_SMOOTH);

	glDrawElements(GL_TRIANGLES, model.ni, GL_UNSIGNED_INT, model.indices);
	glLoadIdentity();
}

} // !namespace visual
