// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "faceset.h"
#include "display.h"
#include <boost/python/class.hpp>
#include <boost/python/return_internal_reference.hpp>
#include <boost/python/overloads.hpp>
#include <map>
#include GL_INCLUDE

#include "slice.h"

using boost::python::numeric::array;

namespace visual {
using python::slice;

namespace {	
// returns a pointer to the ith vector in the array.
double* index( array a, size_t i)
{
	// This is technically an unsafe cast since the alignment requirement
	// goes up for the cast.  It is made safe by padding actions within the Numeric
	// library itself, but the compiler doesn't know that, so I am just using a
	// raw cast vice a static_cast<>.
	return ((double*)data(a)) + i * 3;
}
} // !namespace (unnamed)
	
faces::faces()
	: pos(0), color(0), normal(0), degenerate(true), preallocated_size(256),
	count(0)
{
	std::vector<int> dims(2);
	dims[0] = 256;
	dims[1] = 3;
	pos = makeNum(dims);
	color = makeNum(dims);
	normal = makeNum(dims);
	double* i = index( pos, 0);
	i[0] = i[1] = i[2] = 0.0;
	i = index( color,0);
	i[0] = i[1] = i[2] = 1.0;
	i = index( normal,0);
	i[0] = i[1] = i[2] = 0.0;
}

faces::faces( const faces& other)
	: DisplayObject( other), pos( other.pos), color( other.color), 
	normal( other.normal), degenerate( true), 
	preallocated_size( other.preallocated_size),
	count( other.count)
{
	// Force a refreshCache() call.
	write_lock L(mtx);
}

void
faces::set_length( int length)
{
	int npoints = count;
	if (npoints > length) // A shrink operation - never done by VPython.
		npoints = length;
	if (npoints == 0) // The first allocation.
		npoints = 1;
		
	if (length > preallocated_size) {
		std::vector<int> dims(2);
		dims[0] = 2 * length;
		dims[1] = 3;
		array n_pos = makeNum( dims);
		array n_color = makeNum( dims);
		array n_normal = makeNum( dims);
		std::memcpy( data( n_pos), data( pos), sizeof(double) * 3 * npoints);
		std::memcpy( data( n_color), data( color), sizeof(double) * 3 * npoints);
		std::memcpy( data( n_normal), data( normal), sizeof(double) * 3*npoints);
		pos = n_pos;
		color = n_color;
		normal = n_normal;
		preallocated_size = dims[0];
	}
	if (length > npoints) {
		// Copy the last good element to the new positions.
		const double* last_element = index( pos, npoints-1);
		double* element_i = index( pos, npoints);
		double* element_end = index( pos, length);
		while (element_i < element_end) {
			element_i[0] = last_element[0];
			element_i[1] = last_element[1];
			element_i[2] = last_element[2];
			element_i += 3;
		}
		
		last_element = index( color, npoints-1);
		element_i = index( color, npoints);
		element_end = index( color, length);
		while (element_i < element_end) {
			element_i[0] = last_element[0];
			element_i[1] = last_element[1];
			element_i[2] = last_element[2];
			element_i += 3;
		}
		
		last_element = index( normal, npoints-1);
		element_i = index( normal, npoints);
		element_end = index( normal, length);
		while (element_i < element_end) {
			element_i[0] = last_element[0];
			element_i[1] = last_element[1];
			element_i[2] = last_element[2];
			element_i += 3;
		}
	}
	count = length;
}

void
faces::append( vector nv_pos, vector nv_normal, vector nv_color)
{
	write_lock L(mtx);
	set_length( count+1);
	double* pos_data = index( pos, count-1);
	double* norm_data = index(normal, count-1);
	double* color_data = index(color, count-1);
	pos_data[0] = nv_pos.get_x();
	pos_data[1] = nv_pos.get_y();
	pos_data[2] = nv_pos.get_z();
	norm_data[0] = nv_normal.get_x();
	norm_data[1] = nv_normal.get_y();
	norm_data[2] = nv_normal.get_z();
	color_data[0] = nv_color.get_x();
	color_data[1] = nv_color.get_y();
	color_data[2] = nv_color.get_z();
}

void
faces::append( vector n_pos, vector n_normal)
{
	write_lock L(mtx);
	set_length( count+1);
	double* pos_data = index( pos, count-1);
	double* norm_data = index(normal, count-1);
	pos_data[0] = n_pos.get_x();
	pos_data[1] = n_pos.get_y();
	pos_data[2] = n_pos.get_z();
	norm_data[0] = n_normal.get_x();
	norm_data[1] = n_normal.get_y();
	norm_data[2] = n_normal.get_z();
}

void
faces::smooth_shade(bool doublesided)
{
	if (shape(pos) != shape(normal))
		throw std::invalid_argument( "Dimension mismatch between pos and normal.");
	
	write_lock L(mtx);
	
	// positions -> normals
	std::map< const vector, vector> verticies;
	std::map< const vector, vector> verticies_backface;
	
	const double* pos_i = index(pos, 0);
	double* norm_i = index(normal, 0);
	const double* pos_end = index( pos, count);
	for ( ; pos_i < pos_end; pos_i+=3, norm_i+=3) {
		// If there isn't a normal at the specified position, it will be default
		// initialized to zero.  If there already is one, it will be returned.
		if (doublesided) {
			if (verticies[vector(pos_i)].dot( vector(norm_i)) >= 0.0) {
				verticies[vector(pos_i)] += vector(norm_i);
			}
			else {
				verticies_backface[vector(pos_i)] += vector(norm_i);
			}
		}
		else {
			verticies[vector(pos_i)] += vector(norm_i);
		}
	}
	
	pos_i = index(pos, 0);
	norm_i = index(normal, 0);
	vector tmp;
	for ( ; pos_i < pos_end; pos_i+=3, norm_i+=3) {
		if (doublesided) {
			if (verticies[vector(pos_i)].dot( vector(norm_i)) >= 0.0) {
				tmp = verticies[vector(pos_i)].norm();
			}
			else {
				tmp = verticies_backface[vector(pos_i)].norm();
			}
		}
		else {
			tmp = verticies[vector(pos_i)].norm();
		}
		norm_i[0] = tmp.get_x();
		norm_i[1] = tmp.get_y();
		norm_i[2] = tmp.get_z();
	}
}
	
void  
faces::set_pos( const array& n_pos)
{
	using namespace boost::python;
	using visual::python::slice;
	std::vector<int> n_dims = shape(n_pos);
	std::vector<int> dims = shape(this->pos);

	if (n_dims.size() == 1 && !n_dims[0]) {
		write_lock L(mtx);
		set_length(0);
		return;	
	}
	if (n_dims.size() != 2)
		throw std::invalid_argument( "Numeric.array members must be Nx3 arrays.");

	if (n_dims[1] == 2) {
		write_lock L(mtx);
		set_length( n_dims[0]);
		pos[make_tuple( slice(0,count), slice(0,2))] = n_pos;
		pos[make_tuple( slice(0,count), 2)] = 0.0;
	}
	else if (n_dims[1] == 3) {
		write_lock L(mtx);
		set_length( n_dims[0]);
		pos[slice(0, count)] = n_pos;
	}
	else
		throw std::invalid_argument( "Numeric.array members must be Nx3 arrays.");
}

void
faces::set_pos_l( boost::python::list l)
{
	set_pos( array( l));
}

boost::python::object 
faces::get_pos() 
{ 
	return pos[slice(0, count)];
}

boost::python::object 
faces::get_color()
{ 
	return color[slice(0, count)]; 
}

boost::python::object 
faces::get_normal() 
{ 
	return normal[slice(0, count)]; 
}


void  
faces::set_color( const array& n_color)
{
	using namespace boost::python;
	using visual::python::slice;
	std::vector<int> n_dims = shape(n_color);

	if (n_dims.size() != 2 && n_dims[1] != 3)
		throw std::invalid_argument( "color must be an Nx3 array.");
	if (n_dims[0] != count)
		throw std::invalid_argument( "color must be the same size as pos.");

	write_lock L(mtx);
	color[slice(0, count)] = n_color;
}

void
faces::set_color_l( boost::python::list color)
{
	set_color( array(color));	
}

void
faces::set_color_t( boost::python::tuple t)
{
	using visual::python::slice;
	// Broadcast the new color across the array.
	int npoints = count ? count : 1;
	write_lock L(mtx);
	color[slice(0, npoints)] = t;
}

void  
faces::set_normal( const array& n_normal)
{
	using namespace boost::python;
	using visual::python::slice;

	write_lock L(mtx);
	normal[slice(0, count)] = n_normal;
}

void
faces::set_normal_l( boost::python::list normals)
{
	set_normal( array(normals));	
}

void
faces::set_normal_v( vector v)
{
	using visual::python::slice;
	// Broadcast the new normal across the array.
	int npoints = count ? count : 1;
	write_lock L(mtx);
	normal[slice(0, npoints)] = v.as_tuple();
}


void
faces::glRender( rView& view)
{
	if (degenerate)
		return;
	
	// Initialize iterators and endpoints.
	const double* pos_i = index(pos, 0);
	const double* const pos_end = index( pos, count);
	const double* color_i = index(color, 0);
	const double* norm_i = index( normal, 0);


	glDisableClientState( GL_VERTEX_ARRAY);
	glDisableClientState( GL_COLOR_ARRAY);
	glShadeModel( GL_SMOOTH);
	glEnable( GL_CULL_FACE);

	glBegin( GL_TRIANGLES);

	for ( ; pos_i < pos_end; pos_i+=3, norm_i+=3, color_i+=3) {
		vertex vx;
		view.ext_point( vector(pos_i));
		view.wct.project( vector(pos_i), vx);

		// Manipulate colors when we are in stereo mode.
		rgb color( static_cast<float>(color_i[0]), 
		           static_cast<float>(color_i[1]),
		           static_cast<float>(color_i[2]));
		if (view.anaglyph) {
			if (view.coloranaglyph) {
				color = color.unsaturate();
			}
			else {
				color.r = 
				color.g = 
				color.b = color.grayscale();
			}
		}

		double lt = view.lights.illuminate( vector(norm_i).norm());
		glColor3f( lt*color.r, lt*color.g, lt*color.b);
		glVertex4d( vx.x, vx.y, vx.z, vx.w);
	}

	glEnd();
	glDisable( GL_CULL_FACE);
}

void
faces::refreshCache()
{
	if (count < 3) {
		degenerate = true;
		return;
	}
	degenerate = false;
}

namespace {
BOOST_PYTHON_MEMBER_FUNCTION_OVERLOADS( faces_smooth_shade, 
	faces::smooth_shade, 0, 1);
}

void
faces_init_type()
{
	using namespace boost::python;
		
	void (faces::* append_all_vectors)(vector, vector, vector) = &faces::append;
	void (faces::* append_default_color)( vector, vector) = &faces::append;
	
	class_<faces, bases<DisplayObject>, boost::shared_ptr<faces> >("faces")
		.def( init<const faces&>())
		.def( "append", append_default_color, args( "pos", "normal"))
		.def( "append", append_all_vectors, args("pos", "normal", "color"))
		.def( "_get_pos", &faces::get_pos)
		.def( "_set_pos", &faces::set_pos)
		.def( "_set_pos", &faces::set_pos_l)
		.def( "_get_normal", &faces::get_normal)
		.def( "_set_normal", &faces::set_normal_l)
		.def( "_set_normal", &faces::set_normal)
		.def( "_set_normal", &faces::set_normal_v)
		.def( "_get_color", &faces::get_color)
		.def( "_set_color", &faces::set_color_l)
		.def( "_set_color", &faces::set_color)
		.def( "_set_color", &faces::set_color_t)
		.def( "smooth_shade", &faces::smooth_shade, 
			faces_smooth_shade( args("doublesided"),
			"Average normal vectors at coincident vertexes."))
		;	
}

	
} // !namespace visual
