#
#

"""Unittests for the posix1e module"""

#  Copyright (C) 2002-2008 Iustin Pop <iusty@k1024.org>
#
#  This library is free software; you can redistribute it and/or
#  modify it under the terms of the GNU Lesser General Public
#  License as published by the Free Software Foundation; either
#  version 2.1 of the License, or (at your option) any later version.
#
#  This library is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  Lesser General Public License for more details.
#
#  You should have received a copy of the GNU Lesser General Public
#  License along with this library; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
#  02110-1301  USA


import unittest
import os
import tempfile

import posix1e
from posix1e import *

TEST_DIR = os.environ.get("TESTDIR", ".")

BASIC_ACL_TEXT = "u::rw,g::r,o::-"

def _skip_test(fn):
    """Wrapper to skip a test"""
    new_fn = lambda x: None
    new_fn.__doc__ = "SKIPPED %s" % fn.__doc__
    return new_fn


def has_ext(extension):
    """Decorator to skip tests based on platform support"""
    if not extension:
        return _skip_test
    else:
        return lambda x: x


class aclTest:
    """Support functions ACLs"""

    def setUp(self):
        """set up function"""
        self.rmfiles = []
        self.rmdirs = []

    def tearDown(self):
        """tear down function"""
        for fname in self.rmfiles:
            os.unlink(fname)
        for dname in self.rmdirs:
            os.rmdir(dname)

    def _getfile(self):
        """create a temp file"""
        fh, fname = tempfile.mkstemp(".test", "xattr-", TEST_DIR)
        self.rmfiles.append(fname)
        return fh, fname

    def _getdir(self):
        """create a temp dir"""
        dname = tempfile.mkdtemp(".test", "xattr-", TEST_DIR)
        self.rmdirs.append(dname)
        return dname

    def _getsymlink(self):
        """create a symlink"""
        fh, fname = self._getfile()
        os.close(fh)
        os.unlink(fname)
        os.symlink(fname + ".non-existent", fname)
        return fname


class LoadTests(aclTest, unittest.TestCase):
    """Load/create tests"""
    def testFromFile(self):
        """Test loading ACLs from a file"""
        _, fname = self._getfile()
        acl1 = posix1e.ACL(file=fname)
        self.failUnless(acl1.valid(), "ACL read from file should be valid")

    def testFromDir(self):
        """Test loading ACLs from a directory"""
        dname = self._getdir()
        acl1 = posix1e.ACL(file=dname)
        acl2 = posix1e.ACL(filedef=dname)
        self.failUnless(acl1.valid(),
                        "ACL read from directory should be valid")
        # default ACLs might or might not be valid; missing ones are
        # not valid, so we don't test acl2 for validity

    def testFromFd(self):
        """Test loading ACLs from a file descriptor"""
        fd, _ = self._getfile()
        acl1 = posix1e.ACL(fd=fd)
        self.failUnless(acl1.valid(), "ACL read from fd should be valid")

    def testFromEmpty(self):
        """Test creating an empty ACL"""
        acl1 = posix1e.ACL()
        self.failIf(acl1.valid(), "Empty ACL should not be valid")

    def testFromText(self):
        """Test creating an ACL from text"""
        acl1 = posix1e.ACL(text=BASIC_ACL_TEXT)
        self.failUnless(acl1.valid(),
                        "ACL based on standard description should be valid")

class AclExtensions(aclTest, unittest.TestCase):
    """ACL extensions checks"""

    @has_ext(HAS_ACL_FROM_MODE)
    def testFromMode(self):
        """Test loading ACLs from an octal mode"""
        acl1 = posix1e.ACL(mode=0644)
        self.failUnless(acl1.valid(),
                        "ACL created via octal mode shoule be valid")

    @has_ext(HAS_ACL_CHECK)
    def testAclCheck(self):
        """Test the acl_check method"""
        acl1 = posix1e.ACL(text=BASIC_ACL_TEXT)
        self.failIf(acl1.check(), "ACL is not valid")
        acl2 = posix1e.ACL()
        self.failUnless(acl2.check(), "Empty ACL should not be valid")

    @has_ext(HAS_EXTENDED_CHECK)
    def testExtended(self):
        """Test the acl_extended function"""
        fd, fname = self._getfile()
        basic_acl = posix1e.ACL(text=BASIC_ACL_TEXT)
        basic_acl.applyto(fd)
        for item in fd, fname:
            self.failIf(has_extended(item),
                        "A simple ACL should not be reported as extended")
        enhanced_acl = posix1e.ACL(text="u::rw,g::-,o::-,u:root:rw,mask::r")
        self.failUnless(enhanced_acl.valid(),
                        "Failure to build an extended ACL")
        enhanced_acl.applyto(fd)
        for item in fd, fname:
            self.failUnless(has_extended(item),
                        "An extended ACL should be reported as such")

    @has_ext(HAS_EQUIV_MODE)
    def testEquivMode(self):
        """Test the equiv_mode function"""
        if HAS_ACL_FROM_MODE:
            for mode in 0644, 0755:
                acl = posix1e.ACL(mode=mode)
                self.failUnlessEqual(acl.equiv_mode(), mode)
        acl = posix1e.ACL(text="u::rw,g::r,o::r")
        self.failUnlessEqual(acl.equiv_mode(), 0644)
        acl = posix1e.ACL(text="u::rx,g::-,o::-")
        self.failUnlessEqual(acl.equiv_mode(), 0500)


class WriteTests(aclTest, unittest.TestCase):
    """Write tests"""

    def testDeleteDefault(self):
        """Test removing the default ACL"""
        dname = self._getdir()
        posix1e.delete_default(dname)

    def testReapply(self):
        """Test re-applying an ACL"""
        fd, fname = self._getfile()
        acl1 = posix1e.ACL(fd=fd)
        acl1.applyto(fd)
        acl1.applyto(fname)
        dname = self._getdir()
        acl2 = posix1e.ACL(file=fname)
        acl2.applyto(dname)


class ModificationTests(aclTest, unittest.TestCase):
    """ACL modification tests"""

    @has_ext(HAS_ACL_ENTRY)
    def testAppend(self):
        """Test append a new Entry to the ACL"""
        acl = posix1e.ACL()
        e = acl.append()
        e.tag_type = posix1e.ACL_OTHER
        acl.calc_mask()

    @has_ext(HAS_ACL_ENTRY)
    def testDelete(self):
        """Test delete Entry from the ACL"""
        acl = posix1e.ACL()
        e = acl.append()
        e.tag_type = posix1e.ACL_OTHER
        acl.calc_mask()
        acl.delete_entry(e)
        acl.calc_mask()

    @has_ext(HAS_ACL_ENTRY)
    def testDoubleEntries(self):
        """Test double entries"""
        acl = posix1e.ACL(text=BASIC_ACL_TEXT)
        self.failUnless(acl.valid(), "ACL is not valid")
        for tag_type in (posix1e.ACL_USER_OBJ, posix1e.ACL_GROUP_OBJ,
                         posix1e.ACL_OTHER):
            e = acl.append()
            e.tag_type = tag_type
            e.permset.clear()
            self.failIf(acl.valid(),
                        "ACL containing duplicate entries should not be valid")
            acl.delete_entry(e)

    @has_ext(HAS_ACL_ENTRY)
    def testMultipleGoodEntries(self):
        """Test multiple valid entries"""
        acl = posix1e.ACL(text=BASIC_ACL_TEXT)
        self.failUnless(acl.valid(), "ACL is not valid")
        for tag_type in (posix1e.ACL_USER,
                         posix1e.ACL_GROUP):
            for obj_id in range(5):
                e = acl.append()
                e.tag_type = tag_type
                e.qualifier = obj_id
                e.permset.clear()
                acl.calc_mask()
                self.failUnless(acl.valid(),
                               "ACL should be able to hold multiple"
                                " user/group entries")

    @has_ext(HAS_ACL_ENTRY)
    def testMultipleBadEntries(self):
        """Test multiple invalid entries"""
        acl = posix1e.ACL(text=BASIC_ACL_TEXT)
        self.failUnless(acl.valid(), "ACL built from standard description"
                        " should be valid")
        for tag_type in (posix1e.ACL_USER,
                         posix1e.ACL_GROUP):
            e1 = acl.append()
            e1.tag_type = tag_type
            e1.qualifier = 0
            e1.permset.clear()
            acl.calc_mask()
            self.failUnless(acl.valid(), "ACL should be able to add a"
                            " user/group entry")
            e2 = acl.append()
            e2.tag_type = tag_type
            e2.qualifier = 0
            e2.permset.clear()
            acl.calc_mask()
            self.failIf(acl.valid(), "ACL should not validate when"
                        " containing two duplicate entries")
            acl.delete_entry(e1)
            acl.delete_entry(e2)

    @has_ext(HAS_ACL_ENTRY)
    def testPermset(self):
        """Test permissions"""
        acl = posix1e.ACL()
        e = acl.append()
        ps = e.permset
        ps.clear()
        pmap = {
            posix1e.ACL_READ: "read",
            posix1e.ACL_WRITE: "write",
            posix1e.ACL_EXECUTE: "execute",
            }
        for perm in pmap:
            self.failIf(ps.test(perm), "Empty permission set should not"
                        " have permission '%s'" % pmap[perm])
            ps.add(perm)
            self.failUnless(ps.test(perm), "Permission '%s' should exist"
                        " after addition" % pmap[perm])
            ps.delete(perm)
            self.failIf(ps.test(perm), "Permission '%s' should not exist"
                        " after deletion" % pmap[perm])


if __name__ == "__main__":
    unittest.main()
