#!/usr/bin/env python
#
# COPYRIGHT
#
#   Permission to use, copy, modify, and distribute this software and
#   its documentation for any purpose and without fee is hereby
#   granted, provided that the above copyright notice appear in all
#   copies and that both that copyright notice and this permission
#   notice appear in supporting documentation, and that the name of Doug
#   Hellmann not be used in advertising or publicity pertaining to
#   distribution of the software without specific, written prior
#   permission.
# 
# DISCLAIMER
#
#   DOUG HELLMANN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
#   INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
#   NO EVENT SHALL DOUG HELLMANN BE LIABLE FOR ANY SPECIAL, INDIRECT OR
#   CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
#   OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
#   NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
#   CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
# 


"""A base class for file-based formatters for HappyDoc.

"""

__rcs_info__ = {
    #
    #  Creation Information
    #
    'module_name':'$RCSfile: fileformatterbase.py,v $',
    'creator':'Doug Hellmann <doug@hellfly.net>',
    'project':'HappyDoc',
    'created':'Sat, 03-Jun-2000 17:56:22 EDT',
    #
    #  Current Information
    #
    'author':'$Author: doughellmann $',
    'version':'$Revision: 1.4 $',
    'date':'$Date: 2001/11/11 18:53:44 $',
    }
try:
    __version__ = __rcs_info__['version'].split(' ')[1]
except:
    __version__ = '0.0'

#
# Import system modules
#
import os
import string
import types

#
# Import Local modules
#
import happydoclib

#
# Module
#

    
class FileBasedFormatter(happydoclib.happyformatter.HappyFormatterBase):
    """A base class for file-based formatters for HappyDoc.
    """

    def __init__(self, docSet, filenamePrefix='', **extraNamedParameters):
        """Initialize a FileBasedFormatter.

        Parameters

          docSet -- The documentation set controlling the formatter.
          
          filenamePrefix -- A prefix to append to the base names of
          files and directories being created.  This is useful for
          situations where the names which would be automatically
          generated might cause a name clash or conflict.

          extraNamedParameters -- Parameters specified by name which
          were not supported by a subclass initialization.
          
        """
        #
        # Store parameters
        #
        self._filename_prefix = filenamePrefix
        #
        # Initialize the base class
        #
        apply(happydoclib.happyformatter.HappyFormatterBase.__init__,
              (self, docSet,),
              extraNamedParameters)
        return

    def openOutput(self, name, title1, title2=None):
        "Open the named output destination and give the title."
        happydoclib.path.rmkdir(happydoclib.path.dirname(name))
        f = open(name, 'wt')
        if not hasattr(self, 'open_root_file'):
            self.open_root_file = f
        return f

    def closeOutput(self, output):
        "Close the output handle."
        output.close()
        return

    def fixUpOutputFilename(self, filename):
        """Tweak the filename to remove relative references and add the safe prefix."""
        #
        # Remove preceding slashes to make name relative
        #
        filename = happydoclib.path.removeRelativePrefix(filename)
        #
        # Apply the path prefix, if required
        #
        if self.getFilenamePrefix():
            filename = happydoclib.path.applyPrefixToPath( filename,
                                                           self.getFilenamePrefix())
        return filename
    
    def getOutputNameForFile(self, filename):
        """
        Return the base name of the thing to which output should be
        written for a file.  This is usually a file name, but could be
        anything understood by the formatter as a name.  If infoObject
        is None, return the name for a root node for this formatter.
        """
        happydoclib.TRACE.into('FileBasedFormatter', 'getOutputNameForFile',
                               filename=filename)
        filename = self.fixUpOutputFilename(filename)
        #
        # Set the correct extension for the output file
        #
        extension = self.getFilenameExtension()
        #
        # Build the name
        #
        name = '%s.%s' % (filename, extension)
        happydoclib.TRACE.outof(name)
        return name
    
    def getOutputNameForObject(self, infoObject):
        """
        Return the base name of the thing to which output should be written
        for an info source.  This is usually a file name, but could
        be anything understood by the formatter as a name.  If
        infoObject is None, return the name for a root node for this
        formatter.
        """
        happydoclib.TRACE.into('FileBasedFormatter', 'getOutputNameForObject',
                               infoObject=infoObject)
        if type(infoObject) == types.StringType:
            happydoclib.TRACE.write('string')
            name = infoObject
        elif type(infoObject) == types.FileType:
            happydoclib.TRACE.write('file')
            name = infoObject.name
        elif infoObject is not None:
            happydoclib.TRACE.write('infoObject')
            name = self.getOutputNameForFile(infoObject.getFullyQualifiedName())
            happydoclib.TRACE.write('file for %s' % infoObject.getName())
            happydoclib.TRACE.write('is %s' % name)
        else:
            name = self.getRootNodeName()
        happydoclib.TRACE.outof(name)
        return name


    def getLocalOutputNameForObject(self, infoObject):
        """
        Return a local reference to base name of the thing to which
        output should be written for an info source.  This is usually
        a file name, but could be anything understood by the formatter
        as a name.  If infoObject is None, return the name for a root
        node for this formatter.
        """
        extension = self.getFilenameExtension()
        if infoObject is not None:
            name = '%s.%s' % ( infoObject.getQualifiedName(self.getFilenamePrefix()),
                               extension )
        else:
            name = self.getRootNodeName()
        return name


    def getFullOutputNameForObject(self, infoObject):
        "Get the full name, including path, to the object being output."
        happydoclib.TRACE.into('FileBasedFormatter', 'getFullOutputNameForObject',
                               infoObject=infoObject)
        
        obj_output_name = self.getOutputNameForObject(infoObject)
        docset_base_directory = self._docset.getDocsetBaseDirectory()
        output_base = self._docset.getOutputBaseDirectory()
        
        happydoclib.TRACE.writeVar(obj_output_name=obj_output_name)
        happydoclib.TRACE.writeVar(docset_base_directory=docset_base_directory)
        happydoclib.TRACE.writeVar(output_base=output_base)
        
        if (not infoObject) and docset_base_directory:
            happydoclib.TRACE.write('root node for docset')
            #
            # Root node for docset
            #
            output_subdir = os.sep + happydoclib.path.removePrefix(docset_base_directory, output_base)
            happydoclib.TRACE.writeVar(output_subdir=output_subdir)
            if output_subdir != output_base:
                if output_subdir[0] == '/':
                    output_subdir=output_subdir[1:]
                    happydoclib.TRACE.writeVar(output_subdir=output_subdir)
                obj_output_sub_path = happydoclib.path.join(output_subdir, obj_output_name)
                happydoclib.TRACE.writeVar(obj_output_sub_path=obj_output_sub_path)
                obj_output_sub_path = self.fixUpOutputFilename(obj_output_sub_path)
                happydoclib.TRACE.writeVar(obj_output_sub_path=obj_output_sub_path)
            
                name = happydoclib.path.join(output_base, obj_output_sub_path)
            else:
                name = happydoclib.path.join(output_base, obj_output_name)

        elif docset_base_directory:
            happydoclib.TRACE.write('subnode of docset')
            #
            # Subnode of docset
            #
            file_name = self.getOutputNameForObject(infoObject)
            happydoclib.TRACE.writeVar(file_name=file_name)
            name = happydoclib.path.join(output_base, file_name)
            #name = happydoclib.path.joinWithCommonMiddle(
            #    output_base,
            #    docset_base_directory,
            #    file_name,
            #    )
            
        else:
            happydoclib.TRACE.write('other')
            #
            # How can we get here?
            #
            name = self.getOutputNameForObject(infoObject)

        happydoclib.TRACE.outof(name)
        return name

    def getFullOutputNameForFile(self, filename):
        "Get the full name, including path, to the filename to convert."
        if self._docset.getOutputBaseDirectory():
            name = happydoclib.path.join(
                self._docset.getOutputBaseDirectory(),
                self.getOutputNameForFile(filename),
                )
        else:
            name = self.getOutputNameForFile(filename)
        return name

    def getRootNodeName(self):
        "Return the name of the root node for the documentation tree."
        self._requiredOfSubclass('getRootNodeName')
        return

    def getFilenamePrefix(self):
        "Return the filename prefix value for this formatter instance."
        return self._filename_prefix
