"""
RDF model-to-TriX DOM serialization, TriX XML-to-RDF model deserialization,
using SAX, where the serialized XML conforms to TriX syntax.
"""

from Ft.Rdf import RDF_MS_BASE, RDF_SCHEMA_BASE
from Ft.Rdf import OBJECT_TYPE_RESOURCE, OBJECT_TYPE_LITERAL
from Ft.Rdf import OBJECT_TYPE_UNKNOWN
from Ft.Rdf import RdfException, ParseException
from Ft.Rdf.Resource import Resource, BNODE_BASE
from Ft.Rdf.Statement import Statement
from Ft.Xml import Domlette

from Ft.Xml.Sax import ContentHandler, CreateParser
#from xml import sax

TRIX_NS = 'http://www.w3.org/2004/03/trix/trix-1/'

class TriXHandler(ContentHandler):
  def __init__(self, model):
    self.model = model

    self.currentGraph = None

    self.triplePieces = []
    self.objectType = OBJECT_TYPE_UNKNOWN

    self.text = None

  def characters(self, content):
    if self.text is None:
      self.text = content
    else:
      self.text = self.text + content

  def startElementNS(self, (uri, localName), qualifiedName, atts):
    self.text = None

    if uri == TRIX_NS:
      if localName == 'triple':
        self.startTriple()

  def startTriple(self):
    if self.currentGraph is None:
      self.currentGraph = ''

  def endElementNS(self, (uri, localName), qualifiedName):
    if uri == TRIX_NS:
      if localName == 'uri':
        self.useUri()
      elif localName == 'triple':
        self.useTriple()
      elif localName == 'id':
        self.useId()
      elif localName == 'plainLiteral':
        self.usePlainLiteral()
      elif localName == 'typedLiteral':
        self.useTypedLiteral()
      elif localName == 'graph':
        self.useGraph()

    self.text = None

  def useUri(self):
    if self.currentGraph is None:
      self.currentGraph = self.text
    else:
      self.triplePieces.append(self.text)

    if len(self.triplePieces) == 3:
      self.objectType = OBJECT_TYPE_RESOURCE

  def useId(self):
    if self.currentGraph is None:
      self.currentGraph = BNODE_BASE + self.text
    else:
      self.triplePieces.append(BNODE_BASE + self.text)

    if len(self.triplePieces) == 3:
      self.objectType = OBJECT_TYPE_RESOURCE

  def usePlainLiteral(self):
    self.triplePieces.append(self.text)

    if len(self.triplePieces) == 3:
      self.objectType = OBJECT_TYPE_LITERAL

  def useTypedLiteral(self):
    # Should this be an unimplemented error?  Currently we just silently
    # ignore it, but we should at least spit out a message.
    pass

  def useTriple(self):
    if len(self.triplePieces) == 3:
      newStatement = Statement(self.triplePieces[0], self.triplePieces[1],
                               self.triplePieces[2],
                               scope = self.currentGraph,
                               objectType = self.objectType)
      self.model.add(newStatement)
      self.triplePieces = []
      self.objectType = OBJECT_TYPE_UNKNOWN
    else:
      # Should this be an error?  Currently we just silently ignore it, but
      # we should at least spit out a message.
      pass

  def useGraph(self):
    self.currentGraph = None

class Serializer:
  """Serialize or deserialize a model based on the TriX serialization."""

  def __init__(self, reify=1):
    #self.reify = 0
    return

  def serialize(self, model, nsMap=None, selectUri=None, localResources=[],
                implementation=None, stmts=None):
    """Construct a DOM representing statements in the model."""
    implementation = implementation or Domlette.implementation
    doc = implementation.createDocument(TRIX_NS, 'graphset', None)

    nsMap = nsMap or {}
    if RDF_MS_BASE not in nsMap:
      nsMap[RDF_MS_BASE] = 'rdf'

    if stmts is None:
      stmts = model.statements()
      stmts = filter(lambda x: x.uri != RDF_SCHEMA_BASE, stmts)

      if selectUri:
        stmts = filter(lambda x, sel=selectUri: x.uri == sel, stmts)
    else:
      stmts = filter(lambda x: x.uri != RDF_SCHEMA_BASE, stmts)

    graphs = {}
    blank_graph_index = 0
    for stmt in stmts:
      tripleElement = doc.createElementNS(TRIX_NS, 'triple')

      if Resource(stmt.subject).isBnode():
        tripleElement.appendChild(self.makeTriXComponent(
          doc, 'id', stmt.subject[BNODE_BASE_LEN:]))
      else:
        tripleElement.appendChild(self.makeTriXComponent(
          doc, 'uri', stmt.subject))

      tripleElement.appendChild(self.makeTriXComponent(
        doc, 'uri', stmt.predicate))

      if stmt.objectType == OBJECT_TYPE_RESOURCE:
        tripleElement.appendChild(self.makeTriXComponent(
          doc, 'uri', stmt.object))
      else:
        tripleElement.appendChild(self.makeTriXComponent(
          doc, 'plainLiteral', stmt.object))

      if stmt.scope:
        if stmt.scope in graphs:
          graphs[stmt.scope].appendChild(tripleElement)
        else:
          graphElement = doc.createElementNS(TRIX_NS, 'graph')
          graphElement.appendChild(self.makeTriXComponent(
            doc, 'uri', stmt.scope))
          graphElement.appendChild(tripleElement)
          doc.documentElement.appendChild(graphElement)
          graphs[stmt.scope] = graphElement
      else:
        graphElement = doc.createElementNS(TRIX_NS, 'graph')
        graphElement.appendChild(self.makeTriXComponent(
          doc, 'id', 'g' + str(blank_graph_index)))
        blank_graph_index = blank_graph_index + 1
        graphElement.appendChild(tripleElement)
        doc.documentElement.appendChild(graphElement)

    return doc

  def makeTriXComponent(self, doc, name, content):
    element = doc.createElementNS(TRIX_NS, name)
    text = doc.createTextNode(content)
    element.appendChild(text)
    return element

  def deserialize(self, model, source):
    """
    Generate RDF statements from a TriX serialization and insert these
    into a Model.
    """
    #parser = sax.make_parser(['Ft.Xml.Sax'])
    parser = CreateParser()
    handler = TriXHandler(model)
    parser.setContentHandler(handler)
    parser.parse(source)
