# make_UID_dict.py
"""Reformat a UID list csv file (Table A-1 PS3.6-2008) to Python syntax

   Write the dict elements as:
   UID: (name, type, name_info, isRetired)
   in python format
   
   name_info is extra information extracted from very long names, e.g. 
       which bit size a particular transfer syntax is default for
   isRetired is 'Retired' if true, else is ''
"""
#
# Copyright 2008, Darcy Mason
# This file is part of pydicom.
# See the license.txt file for license information.

csv_filename = "UID_dictionary.csv"
pydict_filename = "_UID_dict.py"
dict_name = "UID_dictionary"


def write_dict(f, dict_name, attributes):
    entry_format = """'%s': ('%s', '%s', '%s', '%s')"""
    f.write("\n%s = {\n" % dict_name)
    f.write(",\n".join(entry_format % attribute for attribute in attributes))
    f.write("}\n")

if __name__ == "__main__":
    import csv  # comma-separated value module, python >=2.3

    csv_reader = csv.reader(file(csv_filename, 'rb'))

    attributes = []
    for row in csv_reader:
        UID, name, UIDtype, reference  = row
        name_info = ""
        isRetired = ""
        name = name.replace("\x96", "-") # non-ascii character
        if name.endswith("(Retired)"):
            name = name[:-10]
            isRetired = "Retired"
        if ":" in name:
            name, name_info = name.split(":")
        if "&" in name:
            name = name.replace("&", "and")
        name_info = name_info.strip()  # clear leading (and trailing, if any) whitespace
        attributes.append((UID, name, UIDtype, name_info, isRetired)) # leave Part reference out
              
    py_file = file(pydict_filename, "wb")
    py_file.write("# %s\n" % pydict_filename)
    py_file.write('"""\n%s\n"""\n' % "Dictionary of UID: (name, type, name_info, isRetired)\n")
    py_file.write('# Auto-generated by %s"""\n' % __file__)

    write_dict(py_file, dict_name, attributes)

    py_file.close()

    print "Finished creating python file %s containing the UID dictionary" % pydict_filename
    print "Wrote %d elements" % len(attributes)
