#!/usr/bin/env python
# arch-tag: 24f9b70a-eb7c-42a0-b325-5a69514002ba
# Copyright (C) 2004 David Allouche <david@allouche.net>
#               2005 Canonical Limited
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test suite for import and commit.
"""

import os
import shutil

import pybaz as arch
from pybaz import errors
from pybaz.pathname import DirName, FileName

import framework
import fixtures

from test_arch import explicit_files_import_listing


def set_tagging_method_rules(tree, rules):
    tagging_file = open(tree/'{arch}'/'=tagging-method', 'rb+')
    tagging = tagging_file.readlines()
    tagging_file.seek(0)
    for line in tagging:
        words = line.split()
        if not words:
            tagging_file.write(line)
            continue
        key = words[0]
        if key in rules:
            tagging_file.write('%s %s\n' % (key, rules[key]))
        else:
            tagging_file.write(line)
    tagging_file.truncate()
    tagging_file.close()


class Import(framework.NewTestCase):
    tests = []
    fixture = fixtures.WorkingTreeFixture()

    def import_without_log(self):
        """WorkingTree.import w/o log message produces expected message."""
        tree = self.fixture.tree
        version = self.fixture.version
        tree.import_()
        logs = tuple(tree.iter_logs())
        self.failUnlessEqual(1, len(logs))
        log = logs[0]
        self.failUnlessEqual(log.revision, version['base-0'])
        self.failUnlessEqual(log.summary, 'initial import')
        self.failUnlessEqual(log.description,
                             '\n(automatically generated log message)\n')
    tests.append('import_without_log')

    def _import_with_log_helper(self, m, import_):
        tree = self.fixture.tree
        version = self.fixture.version
        s = "empty initial import"
        m['Summary'] = s
        d = "Empty initial import.\n"
        m.description = d
        import_()
        logs = tuple(tree.iter_logs())
        self.failUnlessEqual(1, len(logs))
        log = logs[0]
        self.failUnlessEqual(log.revision, version['base-0'])
        self.failUnlessEqual(log.summary, s)
        self.failUnlessEqual(log.description, d)

    def import_with_log(self):
        """WorkingTree.import with default log message file name."""
        tree = self.fixture.tree
        m = tree.log_message()
        def import_():
            m.save()
            tree.import_()
        self._import_with_log_helper(m, import_)
    tests.append('import_with_log')

    def import_with_custom_log(self):
        """WorkingTree.import with specified log message file name."""
        tree = self.fixture.tree
        m = arch.LogMessage(tree/'+custom_log_message')
        m.clear()
        def import_():
            tree.import_(m)
        self._import_with_log_helper(m, import_)
    tests.append('import_with_custom_log')

    def import_not_empty(self):
        """WorkingTree.import fails on unrecognized and adds source."""
        # FIXME: depends on log parsing, should depend on checkout and
        # changeset instead -- David Allouche 2005-03-31
        tree = self.fixture.tree
        rules = {'junk': '^$', 'precious': '^$', 'backup': '^$',
                 'unrecognized': '^$', 'source': '^[_=a-zA-Z0-9].*$'}
        set_tagging_method_rules(tree, rules)
        # unrecognized file
        print >> file(tree/'!foo', 'w'), 'bar'
        def import_default():
            tree.import_()
        self.failUnlessRaises(errors.ExecProblem, import_default)
        os.remove(tree/'!foo')
        # source file
        print >> file(tree/'source', 'w'), 'hello'
        tree.add_tag('source')
        os.mkdir(tree/'dir')
        tree.add_tag('dir')
        tree.import_()
        log = tree.iter_logs().next()
        expected = explicit_files_import_listing(
            [FileName('source'), DirName('dir')])
        self.failUnlessEqual(list(log.new_files), expected)
    tests.append('import_not_empty')


class Commit(framework.NewTestCase):
    tests = []
    fixture = fixtures.WorkingTreeFixture()

    # FIXME: test commit and iter_logs at once and depends on log parsing.
    # Should depend on checkout and changeset, test that commit affects
    # checkout, test that iter_logs works, then test that commit creates
    # expected revisions. -- David Allouche 2005-03-31

    def commit_seal_fix(self):
        """Commit, seal, fix, create expected patchlevels."""
        tree = self.fixture.tree
        version = self.fixture.version
        tree.import_()
        # commit w/o log message
        self.failUnlessRaises(errors.ExecProblem, tree.commit)
        # commit revision
        m = tree.log_message()
        m['Summary'] = m.description = 'first revision'
        tree.commit(m)
        self.failUnlessEqual(map(lambda(x): x.revision, tree.iter_logs()),
                             map(lambda(x): version[x],
                                 ('base-0', 'patch-1')))
        # fix non-sealed, then seal
        m['Summary'] = m.description = 'second revision'
        self.failUnlessRaises(errors.ExecProblem,
                              lambda: tree.commit(fix=True, log=m))
        tree.commit(seal=True, log=m)
        self.failUnlessEqual(map(lambda(x): x.revision, tree.iter_logs()),
                             map(lambda(x): version[x],
                                 ('base-0', 'patch-1', 'version-0')))
        # commit sealed, then fix
        m['Summary'] = m.description = 'third revision'
        self.failUnlessRaises(errors.ExecProblem, tree.commit, m)
        tree.commit(fix=True, log=m)
        self.failUnlessEqual(map(lambda(x): x.revision, tree.iter_logs()),
                             map(lambda(x): version[x],
                                 ('base-0', 'patch-1',
                                  'version-0', 'versionfix-1')))
        # sanity catches trying to fix and seal at the same time
        m['Summary'] = m.description = 'error'
        self.failUnlessRaises(AssertionError,
                              lambda: tree.commit(seal=True, fix=True, log=m))
    tests.append('commit_seal_fix')

    def commit_relative_log(self):
        """Commit with a log file given as a relative path."""
        tree = self.fixture.tree
        tree.import_()
        tree_dir, tree_name = map(DirName, os.path.split(tree))
        os.chdir(tree_dir)
        log_name = tree_name/'+log'
        f = open(log_name, 'w')
        summary = 'first rev'
        f.write('Summary: %s\n\n' % summary)
        f.close()
        tree.commit(log=log_name)
        def lastlog(): return tree.iter_logs(reverse=True).next()
        self.failUnlessEqual(lastlog().summary, summary)
    tests.append('commit_relative_log')

    def commit_files(self):
        """Commit (maybe with limit) and Patchlog are consistent."""
        tree = self.fixture.tree
        # import two files
        print >> file(tree/'f1', 'w'), 'hello'
        print >> file(tree/'f2', 'w'), 'world'
        map(tree.add_tag, ('f1', 'f2'))
        tree.import_()
        def lastlog(): return tree.iter_logs(reverse=True).next()
        expected = explicit_files_import_listing(map(FileName, ('f1','f2')))
        self.failUnlessEqual(lastlog().new_files, expected)
        # modify both files
        print >> file(tree/'f1', 'w'), 'Hello,'
        print >> file(tree/'f2', 'w'), 'World!'
        # sanity catches non-null but empty file-list
        m = tree.log_message()
        m['Summary'] = m.description = 'first revision'
        def commit_empty(): tree.commit(log=m, file_list=[].__iter__())
        self.failUnlessRaises(AssertionError, commit_empty)
        # commit one file, absolute path
        tree.commit(m, file_list=[tree/'f2'].__iter__())
        self.failUnlessEqual(lastlog().modified_files, [FileName('f2')])
        # commit the other file, relative path
        m['Summary'] = m.description = 'third revision'
        tree.commit(m, file_list=['f1'])
        self.failUnlessEqual(lastlog().modified_files, [FileName('f1')])
    tests.append('commit_files')

    def commit_out_of_date(self):
        """Committing out-of-date fails unless forced."""
        tree = self.fixture.tree
        version = self.fixture.version
        tree_path = str(tree)
        tree.import_()
        m = tree.log_message()
        m['Summary'] = m.description = 'first revision'
        tree.commit(m)
        shutil.rmtree(tree_path)
        tree = version['base-0'].get(tree_path)
        m['Summary'] = m.description = 'out of date revision'
        # trying to commit an out-of-date tree fails
        self.failUnlessRaises(errors.ExecProblem, tree.commit, m)
        # unless the right option is set
        tree.commit(m, out_of_date_ok=True)
        self.failUnlessEqual(map(lambda(x): x.revision, tree.iter_logs()),
                             map(lambda(x): version[x],
                                 ('base-0', 'patch-2')))
    tests.append('commit_out_of_date')

    def commit_version(self):
        """Commit on a specified version."""
        tree = self.fixture.tree
        version = self.fixture.version
        other_version = arch.Version(version.fullname + '.1')
        tree.import_()
        tree.tree_version = other_version
        m = tree.log_message()
        m['Summary'] = m.description = 'first revision'
        self.assertRaises(errors.ExecProblem, tree.commit, m)
        tree.commit(m, version=version)
        tree.tree_version = version
        self.failUnlessEqual(map(lambda(x): x.revision, tree.iter_logs()),
                             map(lambda(x): version[x],
                                 ('base-0', 'patch-1')))
    tests.append('commit_version')

    def iter_commit(self):
        """iter_commit works"""
        tree = self.fixture.tree
        tree.tagging_method = 'names'
        tree.import_()
        print >> file(tree/'f1', 'w'), 'hello'
        m = tree.log_message()
        m['Summary'] = m.description = 'first revision'
        lines = tree.iter_commit(m)
        changes = [L for L in lines if isinstance(L, arch.TreeChange)]
        self.assertEqual(1, len(changes))
        self.assertEqual(arch.FileAddition, type(changes[0]))
        self.assertEqual('f1', changes[0].name)
    tests.append('iter_commit')

    def iter_commit_files(self):
        """Commit with limit print the right path."""
        tree = self.fixture.tree
        print >> file(tree/'f1', 'w'), 'hello'
        print >> file(tree/'f2', 'w'), 'world'
        map(tree.add_tag, ('f1', 'f2'))
        tree.import_()
        print >> file(tree/'f1', 'w'), 'Hello,'
        print >> file(tree/'f2', 'w'), 'World!'
        m = tree.log_message()
        m['Summary'] = m.description = 'first revision'
        lines = tree.iter_commit(m, file_list=['f2'].__iter__())
        changes = [L for L in lines if isinstance(L, arch.TreeChange)]
        self.assertEqual(1, len(changes))
        self.assertEqual(arch.FileModification, type(changes[0]))
        self.assertEqual('f2', changes[0].name)
        # commit the other file, relative path
        m['Summary'] = m.description = 'first revision'
        lines = tree.iter_commit(m, file_list=['f1'])
        changes = [L for L in lines if isinstance(L, arch.TreeChange)]
        self.assertEqual(1, len(changes))
        self.assertEqual(arch.FileModification, type(changes[0]))
        self.assertEqual('f1', changes[0].name)
    tests.append('iter_commit_files')


framework.register(__name__)
